"use strict";
// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.TableServiceClient = void 0;
const tslib_1 = require("tslib");
const core_auth_1 = require("@azure/core-auth");
const constants_js_1 = require("./utils/constants.js");
const secondaryEndpointPolicy_js_1 = require("./secondaryEndpointPolicy.js");
const core_xml_1 = require("@azure/core-xml");
const generatedClient_js_1 = require("./generated/generatedClient.js");
const apiVersionPolicy_js_1 = require("./utils/apiVersionPolicy.js");
const connectionString_js_1 = require("./utils/connectionString.js");
const errorHelpers_js_1 = require("./utils/errorHelpers.js");
const isCredential_js_1 = require("./utils/isCredential.js");
const logger_js_1 = require("./logger.js");
const challengeAuthenticationUtils_js_1 = require("./utils/challengeAuthenticationUtils.js");
const tablesNamedCredentialPolicy_js_1 = require("./tablesNamedCredentialPolicy.js");
const tablesSASTokenPolicy_js_1 = require("./tablesSASTokenPolicy.js");
const tracing_js_1 = require("./utils/tracing.js");
const isCosmosEndpoint_js_1 = require("./utils/isCosmosEndpoint.js");
/**
 * A TableServiceClient represents a Client to the Azure Tables service allowing you
 * to perform operations on the tables and the entities.
 */
class TableServiceClient {
    constructor(url, credentialOrOptions, options) {
        this.url = url;
        const isCosmos = (0, isCosmosEndpoint_js_1.isCosmosEndpoint)(this.url);
        const credential = (0, isCredential_js_1.isCredential)(credentialOrOptions) ? credentialOrOptions : undefined;
        const clientOptions = (!(0, isCredential_js_1.isCredential)(credentialOrOptions) ? credentialOrOptions : options) || {};
        const internalPipelineOptions = Object.assign(Object.assign({}, clientOptions), { endpoint: clientOptions.endpoint || this.url, loggingOptions: {
                logger: logger_js_1.logger.info,
                additionalAllowedHeaderNames: [...constants_js_1.TablesLoggingAllowedHeaderNames],
            }, deserializationOptions: {
                parseXML: core_xml_1.parseXML,
            }, serializationOptions: {
                stringifyXML: core_xml_1.stringifyXML,
            } });
        const client = new generatedClient_js_1.GeneratedClient(this.url, internalPipelineOptions);
        client.pipeline.addPolicy(secondaryEndpointPolicy_js_1.tablesSecondaryEndpointPolicy);
        if ((0, core_auth_1.isNamedKeyCredential)(credential)) {
            client.pipeline.addPolicy((0, tablesNamedCredentialPolicy_js_1.tablesNamedKeyCredentialPolicy)(credential));
        }
        else if ((0, core_auth_1.isSASCredential)(credential)) {
            client.pipeline.addPolicy((0, tablesSASTokenPolicy_js_1.tablesSASTokenPolicy)(credential));
        }
        if ((0, core_auth_1.isTokenCredential)(credential)) {
            const scope = isCosmos ? constants_js_1.COSMOS_SCOPE : constants_js_1.STORAGE_SCOPE;
            (0, challengeAuthenticationUtils_js_1.setTokenChallengeAuthenticationPolicy)(client.pipeline, credential, scope);
        }
        if (options === null || options === void 0 ? void 0 : options.version) {
            client.pipeline.addPolicy((0, apiVersionPolicy_js_1.apiVersionPolicy)(options.version));
        }
        this.pipeline = client.pipeline;
        this.table = client.table;
        this.service = client.service;
    }
    /**
     * Retrieves statistics related to replication for the Table service. It is only available on the
     * secondary location endpoint when read-access geo-redundant replication is enabled for the account.
     * @param options - The options parameters.
     */
    async getStatistics(options = {}) {
        return tracing_js_1.tracingClient.withSpan("TableServiceClient.getStatistics", options, (updatedOptions) => this.service.getStatistics((0, secondaryEndpointPolicy_js_1.injectSecondaryEndpointHeader)(updatedOptions)));
    }
    /**
     * Gets the properties of an account's Table service, including properties for Analytics and CORS
     * (Cross-Origin Resource Sharing) rules.
     * @param options - The options parameters.
     */
    getProperties(options = {}) {
        return tracing_js_1.tracingClient.withSpan("TableServiceClient.getProperties", options, (updatedOptions) => this.service.getProperties(updatedOptions));
    }
    /**
     * Sets properties for an account's Table service endpoint, including properties for Analytics and CORS
     * (Cross-Origin Resource Sharing) rules.
     * @param properties - The Table Service properties.
     * @param options - The options parameters.
     */
    setProperties(properties, options = {}) {
        return tracing_js_1.tracingClient.withSpan("TableServiceClient.setProperties", options, (updatedOptions) => this.service.setProperties(properties, updatedOptions));
    }
    /**
     * Creates a new table under the given account.
     * @param name - The name of the table.
     * @param options - The options parameters.
     */
    createTable(name, options = {}) {
        return tracing_js_1.tracingClient.withSpan("TableServiceClient.createTable", options, async (updatedOptions) => {
            try {
                await this.table.create({ name }, updatedOptions);
            }
            catch (e) {
                (0, errorHelpers_js_1.handleTableAlreadyExists)(e, Object.assign(Object.assign({}, updatedOptions), { logger: logger_js_1.logger, tableName: name }));
            }
        });
    }
    /**
     * Operation permanently deletes the specified table.
     * @param name - The name of the table.
     * @param options - The options parameters.
     */
    deleteTable(name, options = {}) {
        return tracing_js_1.tracingClient.withSpan("TableServiceClient.deleteTable", options, async (updatedOptions) => {
            try {
                await this.table.delete(name, updatedOptions);
            }
            catch (e) {
                if (e.statusCode === 404) {
                    logger_js_1.logger.info("TableServiceClient.deleteTable: Table doesn't exist");
                }
                else {
                    throw e;
                }
            }
        });
    }
    /**
     * Queries tables under the given account.
     * @param options - The options parameters.
     */
    listTables(
    // eslint-disable-next-line @azure/azure-sdk/ts-naming-options
    options) {
        const iter = this.listTablesAll(options);
        return {
            next() {
                return iter.next();
            },
            [Symbol.asyncIterator]() {
                return this;
            },
            byPage: (settings) => {
                const pageOptions = Object.assign(Object.assign({}, options), { queryOptions: Object.assign(Object.assign({}, options === null || options === void 0 ? void 0 : options.queryOptions), { top: settings === null || settings === void 0 ? void 0 : settings.maxPageSize }) });
                if (settings === null || settings === void 0 ? void 0 : settings.continuationToken) {
                    pageOptions.continuationToken = settings.continuationToken;
                }
                return this.listTablesPage(pageOptions);
            },
        };
    }
    listTablesAll(options) {
        return tslib_1.__asyncGenerator(this, arguments, function* listTablesAll_1() {
            var _a, e_1, _b, _c;
            const firstPage = yield tslib_1.__await(this._listTables(options));
            const { continuationToken } = firstPage;
            yield tslib_1.__await(yield* tslib_1.__asyncDelegator(tslib_1.__asyncValues(firstPage)));
            if (continuationToken) {
                const optionsWithContinuation = Object.assign(Object.assign({}, options), { continuationToken });
                try {
                    for (var _d = true, _e = tslib_1.__asyncValues(this.listTablesPage(optionsWithContinuation)), _f; _f = yield tslib_1.__await(_e.next()), _a = _f.done, !_a; _d = true) {
                        _c = _f.value;
                        _d = false;
                        const page = _c;
                        yield tslib_1.__await(yield* tslib_1.__asyncDelegator(tslib_1.__asyncValues(page)));
                    }
                }
                catch (e_1_1) { e_1 = { error: e_1_1 }; }
                finally {
                    try {
                        if (!_d && !_a && (_b = _e.return)) yield tslib_1.__await(_b.call(_e));
                    }
                    finally { if (e_1) throw e_1.error; }
                }
            }
        });
    }
    listTablesPage() {
        return tslib_1.__asyncGenerator(this, arguments, function* listTablesPage_1(options = {}) {
            let result = yield tslib_1.__await(tracing_js_1.tracingClient.withSpan("TableServiceClient.listTablesPage", options, (updatedOptions) => this._listTables(updatedOptions)));
            yield yield tslib_1.__await(result);
            while (result.continuationToken) {
                const optionsWithContinuation = Object.assign(Object.assign({}, options), { continuationToken: result.continuationToken });
                result = yield tslib_1.__await(tracing_js_1.tracingClient.withSpan("TableServiceClient.listTablesPage", optionsWithContinuation, async (updatedOptions, span) => {
                    span.setAttribute("continuationToken", updatedOptions.continuationToken);
                    return this._listTables(updatedOptions);
                }));
                yield yield tslib_1.__await(result);
            }
        });
    }
    async _listTables(options = {}) {
        const { continuationToken: nextTableName } = options, listOptions = tslib_1.__rest(options, ["continuationToken"]);
        const { xMsContinuationNextTableName: continuationToken, value = [] } = await this.table.query(Object.assign(Object.assign({}, listOptions), { nextTableName }));
        return Object.assign([...value], { continuationToken });
    }
    /**
     *
     * Creates an instance of TableServiceClient from connection string.
     *
     * @param connectionString - Account connection string or a SAS connection string of an Azure storage account.
     *                           [ Note - Account connection string can only be used in NODE.JS runtime. ]
     *                           Account connection string example -
     *                           `DefaultEndpointsProtocol=https;AccountName=myaccount;AccountKey=accountKey;EndpointSuffix=core.windows.net`
     *                           SAS connection string example -
     *                           `BlobEndpoint=https://myaccount.table.core.windows.net/;QueueEndpoint=https://myaccount.queue.core.windows.net/;FileEndpoint=https://myaccount.file.core.windows.net/;TableEndpoint=https://myaccount.table.core.windows.net/;SharedAccessSignature=sasString`
     * @param options - Options to configure the HTTP pipeline.
     * @returns A new TableServiceClient from the given connection string.
     */
    static fromConnectionString(connectionString, 
    // eslint-disable-next-line @azure/azure-sdk/ts-naming-options
    options) {
        const { url, options: clientOptions, credential, } = (0, connectionString_js_1.getClientParamsFromConnectionString)(connectionString, options);
        if (credential) {
            return new TableServiceClient(url, credential, clientOptions);
        }
        else {
            return new TableServiceClient(url, clientOptions);
        }
    }
}
exports.TableServiceClient = TableServiceClient;
//# sourceMappingURL=TableServiceClient.js.map