"use strict";
// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateTableSasQueryParameters = generateTableSasQueryParameters;
const sasIPRange_js_1 = require("./sasIPRange.js");
const sasQueryParameters_js_1 = require("./sasQueryParameters.js");
const tableSasPermisions_js_1 = require("./tableSasPermisions.js");
const constants_js_1 = require("../utils/constants.js");
const computeHMACSHA256_js_1 = require("../utils/computeHMACSHA256.js");
const truncateISO8061Date_js_1 = require("../utils/truncateISO8061Date.js");
/**
 * ONLY AVAILABLE IN NODE.JS RUNTIME.
 *
 * Creates an instance of SASQueryParameters.
 *
 * **Note**: When identifier is not provided, permissions has a default value of "read" and expiresOn of one hour from the time the token is generated.
 */
function generateTableSasQueryParameters(tableName, credential, tableSasSignatureValues) {
    var _a, _b, _c, _d, _e, _f;
    const version = (_a = tableSasSignatureValues.version) !== null && _a !== void 0 ? _a : constants_js_1.SERVICE_VERSION;
    if (credential === undefined) {
        throw TypeError("Invalid NamedKeyCredential");
    }
    if (!tableName) {
        throw new Error("Must provide a 'tableName'");
    }
    const signedPermissions = (0, tableSasPermisions_js_1.tableSasPermissionsToString)(tableSasSignatureValues.permissions);
    const signedStart = tableSasSignatureValues.startsOn
        ? (0, truncateISO8061Date_js_1.truncatedISO8061Date)(tableSasSignatureValues.startsOn, false /** withMilliseconds */)
        : "";
    const signedExpiry = tableSasSignatureValues.expiresOn
        ? (0, truncateISO8061Date_js_1.truncatedISO8061Date)(tableSasSignatureValues.expiresOn, false /** withMilliseconds */)
        : "";
    const canonicalizedResource = getCanonicalName(credential.name, tableName);
    const signedIdentifier = (_b = tableSasSignatureValues.identifier) !== null && _b !== void 0 ? _b : "";
    const signedIP = (0, sasIPRange_js_1.ipRangeToString)(tableSasSignatureValues.ipRange);
    const signedProtocol = tableSasSignatureValues.protocol || "";
    const startingPartitionKey = (_c = tableSasSignatureValues.startPartitionKey) !== null && _c !== void 0 ? _c : "";
    const startingRowKey = (_d = tableSasSignatureValues.startRowKey) !== null && _d !== void 0 ? _d : "";
    const endingPartitionKey = (_e = tableSasSignatureValues.endPartitionKey) !== null && _e !== void 0 ? _e : "";
    const endingRowKey = (_f = tableSasSignatureValues.endRowKey) !== null && _f !== void 0 ? _f : "";
    const stringToSign = [
        signedPermissions,
        signedStart,
        signedExpiry,
        canonicalizedResource,
        signedIdentifier,
        signedIP,
        signedProtocol,
        version,
        startingPartitionKey,
        startingRowKey,
        endingPartitionKey,
        endingRowKey,
    ].join("\n");
    const signature = (0, computeHMACSHA256_js_1.computeHMACSHA256)(stringToSign, credential.key);
    return new sasQueryParameters_js_1.SasQueryParameters(version, signature, {
        permissions: signedPermissions,
        protocol: tableSasSignatureValues.protocol,
        startsOn: tableSasSignatureValues.startsOn,
        expiresOn: tableSasSignatureValues.expiresOn,
        ipRange: tableSasSignatureValues.ipRange,
        identifier: tableSasSignatureValues.identifier,
        tableName,
        startPartitionKey: tableSasSignatureValues.startPartitionKey,
        startRowKey: tableSasSignatureValues.startRowKey,
        endPartitionKey: tableSasSignatureValues.endPartitionKey,
        endRowKey: tableSasSignatureValues.endRowKey,
    });
}
function getCanonicalName(accountName, tableName) {
    // Sample CanonicalName for URL = https://myaccount.table.core.windows.net/Employees(PartitionKey='Jeff',RowKey='Price'):
    //   canonicalizedResource = "/table/myaccount/employees"
    return `/table/${accountName}/${tableName.toLowerCase()}`;
}
//# sourceMappingURL=tableSasSignatureValues.js.map