// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.
import { __rest } from "tslib";
import { accountSasPermissionsFromString } from "./accountSasPermissions.js";
import { accountSasServicesFromString, accountSasServicesToString } from "./accountSasServices.js";
import { isNamedKeyCredential } from "@azure/core-auth";
import { generateAccountSasQueryParameters } from "./accountSasSignatureValues.js";
/**
 * Generates a Table Account Shared Access Signature (SAS) URI based on the client properties
 * and parameters passed in. The SAS is signed by the shared key credential of the client.
 *
 * @see https://learn.microsoft.com/en-us/rest/api/storageservices/create-account-sas
 *
 * @param options - Optional parameters.
 * @returns An account SAS token
 */
export function generateAccountSas(credential, options = {}) {
    const { expiresOn, permissions = accountSasPermissionsFromString("rl"), resourceTypes = "sco", services = accountSasServicesFromString("t") } = options, rest = __rest(options, ["expiresOn", "permissions", "resourceTypes", "services"]);
    if (!isNamedKeyCredential(credential)) {
        throw RangeError("Can only generate the account SAS when the client is initialized with a shared key credential");
    }
    let expiry = expiresOn;
    if (expiry === undefined) {
        const now = new Date();
        expiry = new Date(now.getTime() + 3600 * 1000);
    }
    const sas = generateAccountSasQueryParameters(Object.assign({ permissions, expiresOn: expiry, resourceTypes, services: accountSasServicesToString(services) }, rest), credential).toString();
    return sas;
}
//# sourceMappingURL=generateAccountSas.js.map