import type { SasIPRange } from "./sasIPRange.js";
import type { UserDelegationKey } from "./models.js";
/**
 * Protocols for generated SAS.
 */
export type SasProtocol = "https" | "https,http";
/**
 * Represents the components that make up an Azure SAS' query parameters. This type is not constructed directly
 * by the user; it is only generated by the {@link AccountSasSignatureValues} and {@link TableSasSignatureValues}
 * types. Once generated, it can be encoded into a `string` and appended to a URL directly (though caution should
 * be taken here in case there are existing query parameters, which might affect the appropriate means of appending
 * these query parameters).
 *
 * NOTE: Instances of this class are immutable.
 */
export declare class SasQueryParameters {
    /**
     * The Tables API version.
     */
    readonly version: string;
    /**
     * Optional. Table name to generate the SAS for
     */
    readonly tableName?: string;
    /**
     * Optional. The allowed HTTP protocol(s).
     */
    readonly protocol?: SasProtocol;
    /**
     * Optional. The start time for this SAS token.
     */
    readonly startsOn?: Date;
    /**
     * Optional only when identifier is provided. The expiry time for this SAS token.
     */
    readonly expiresOn?: Date;
    /**
     * Optional only when identifier is provided.
     * Please refer to {@link AccountSasPermissions}, or {@link TableSasPermissions} for
     * more details.
     */
    readonly permissions?: string;
    /**
     * Optional. The table services being accessed (only for Account SAS). Please refer to {@link AccountSasServices}
     * for more details.
     */
    readonly services?: string;
    /**
     * Optional. The table resource types being accessed (only for Account SAS). Please refer to
     * {@link AccountSasResourceTypes} for more details.
     */
    readonly resourceTypes?: string;
    /**
     * Optional. The signed identifier (only for {@link TableSasSignatureValues}).
     *
     * @see https://learn.microsoft.com/en-us/rest/api/storageservices/establishing-a-stored-access-policy
     */
    readonly identifier?: string;
    /**
     * The signature for the SAS token.
     */
    readonly signature: string;
    /**
     * Inner value of getter ipRange.
     */
    private readonly ipRangeInner?;
    /**
     * The Azure Active Directory object ID in GUID format.
     * Property of user delegation key.
     */
    private readonly signedOid?;
    /**
     * The Azure Active Directory tenant ID in GUID format.
     * Property of user delegation key.
     */
    private readonly signedTenantId?;
    /**
     * The date-time the key is active.
     * Property of user delegation key.
     */
    private readonly signedStartsOn?;
    /**
     * The date-time the key expires.
     * Property of user delegation key.
     */
    private readonly signedExpiresOn?;
    /**
     * Abbreviation of the Azure Table service that accepts the user delegation key.
     * Property of user delegation key.
     */
    private readonly signedService?;
    /**
     * The service version that created the user delegation key.
     * Property of user delegation key.
     */
    private readonly signedVersion?;
    /**
     * Authorized AAD Object ID in GUID format. The AAD Object ID of a user authorized by the owner of the User Delegation Key
     * to perform the action granted by the SAS. The Azure Table service will ensure that the owner of the user delegation key
     * has the required permissions before granting access but no additional permission check for the user specified in
     * this value will be performed. This is only used for User Delegation SAS.
     */
    readonly preauthorizedAgentObjectId?: string;
    /**
     * A GUID value that will be logged in the table diagnostic logs and can be used to correlate SAS generation with table resource access.
     * This is only used for User Delegation SAS.
     */
    readonly correlationId?: string;
    /**
     * Optional, but startPartitionKey must accompany startRowKey. The minimum partition and row keys that are accessible with this shared access signature.
     * Key values are inclusive. If they're omitted, there's no lower bound on the table entities that can be accessed.
     */
    readonly startPartitionKey?: string;
    /**
     * Optional, but startPartitionKey must accompany startRowKey. The minimum partition and row keys that are accessible with this shared access signature.
     * Key values are inclusive. If they're omitted, there's no lower bound on the table entities that can be accessed.
     */
    readonly startRowKey?: string;
    /**
     * Optional, but endPartitionKey must accompany endRowKey. The maximum partition and row keys that are accessible with this shared access signature.
     * Key values are inclusive. If they're omitted, there's no upper bound on the table entities that can be accessed.
     */
    readonly endPartitionKey?: string;
    /**
     * Optional, but endPartitionKey must accompany endRowKey. The maximum partition and row keys that are accessible with this shared access signature.
     * Key values are inclusive. If they're omitted, there's no upper bound on the table entities that can be accessed.
     */
    readonly endRowKey?: string;
    /**
     * Optional. IP range allowed for this SAS.
     *
     * @readonly
     */
    get ipRange(): SasIPRange | undefined;
    /**
     * Creates an instance of SASQueryParameters.
     *
     * @param version - Representing the table service version
     * @param signature - Representing the signature for the SAS token
     * @param options - Optional. Options to construct the SASQueryParameters.
     */
    constructor(version: string, signature: string, options?: SasQueryParametersOptions);
    /**
     * Encodes all SAS query parameters into a string that can be appended to a URL.
     *
     */
    toString(): string;
    /**
     * A private helper method used to filter and append query key/value pairs into an array.
     *
     * @param queries -
     * @param key -
     * @param value -
     */
    private tryAppendQueryParameter;
}
/**
 * Options to construct {@link SasQueryParameters}.
 */
export interface SasQueryParametersOptions {
    /**
     * Optional only when identifier is provided.
     * Please refer to {@link AccountSasPermissions}, or {@link TableSasPermissions} for
     * more details.
     */
    permissions?: string;
    /**
     * Optional. Table name to generate the SAS for
     */
    tableName?: string;
    /**
     * Optional. The storage services being accessed (only for Account SAS). Please refer to {@link AccountSasServices}
     * for more details.
     */
    services?: string;
    /**
     * Optional. The storage resource types being accessed (only for Account SAS). Please refer to
     * {@link AccountSasResourceTypes} for more details.
     */
    resourceTypes?: string;
    /**
     * Optional. The allowed HTTP protocol(s).
     */
    protocol?: SasProtocol;
    /**
     * Optional. The start time for this SAS token.
     */
    startsOn?: Date;
    /**
     * Optional only when identifier is provided. The expiry time for this SAS token.
     */
    expiresOn?: Date;
    /**
     * Optional. IP ranges allowed in this SAS.
     */
    ipRange?: SasIPRange;
    /**
     * Optional. The signed identifier for access policy
     *
     * @see https://learn.microsoft.com/rest/api/storageservices/establishing-a-stored-access-policy
     */
    identifier?: string;
    /**
     * Optional. Specifies which resources are accessible via the SAS (only for {@link AccountSasSignatureValues}).
     * @see https://learn.microsoft.com/rest/api/storageservices/create-service-sas#specifying-the-signed-resource-blob-service-only
     */
    resource?: string;
    /**
     * User delegation key properties.
     */
    userDelegationKey?: UserDelegationKey;
    /**
     * Authorized AAD Object ID in GUID format. The AAD Object ID of a user authorized by the owner of the User Delegation Key
     * to perform the action granted by the SAS. The Azure Table service will ensure that the owner of the user delegation key
     * has the required permissions before granting access but no additional permission check for the user specified in
     * this value will be performed. This cannot be used in conjuction with {@link signedUnauthorizedUserObjectId}.
     * This is only used for User Delegation SAS.
     */
    preauthorizedAgentObjectId?: string;
    /**
     * A GUID value that will be logged in the storage diagnostic logs and can be used to correlate SAS generation with storage resource access.
     * This is only used for User Delegation SAS.
     */
    correlationId?: string;
    /**
     * Optional, but startPartitionKey must accompany startRowKey. The minimum partition and row keys that are accessible with this shared access signature.
     * Key values are inclusive. If they're omitted, there's no lower bound on the table entities that can be accessed.
     */
    startPartitionKey?: string;
    /**
     * Optional, but startPartitionKey must accompany startRowKey. The minimum partition and row keys that are accessible with this shared access signature.
     * Key values are inclusive. If they're omitted, there's no lower bound on the table entities that can be accessed.
     */
    startRowKey?: string;
    /**
     * Optional, but endPartitionKey must accompany endRowKey. The maximum partition and row keys that are accessible with this shared access signature.
     * Key values are inclusive. If they're omitted, there's no upper bound on the table entities that can be accessed.
     */
    endPartitionKey?: string;
    /**
     * Optional, but endPartitionKey must accompany endRowKey. The maximum partition and row keys that are accessible with this shared access signature.
     * Key values are inclusive. If they're omitted, there's no upper bound on the table entities that can be accessed.
     */
    endRowKey?: string;
}
//# sourceMappingURL=sasQueryParameters.d.ts.map