var expect = require('chai').expect;
var vmWindowsAntiMalwareExtension = require('./vmWindowsAntiMalwareExtension');

const virtualMachines = [
    {
        'name': 'test-vm',
        'id': '/subscriptions/123/resourceGroups/AQUA-RESOURCE_GROUP/providers/Microsoft.Compute/virtualMachines/test-vm',
        'type': 'Microsoft.Compute/virtualMachines'
    }
];

const virtualMachineExtension = [
    {
        'name': 'TestExtension',
        'id': '/subscriptions/123/resourceGroups/AQUA-RESOURCE_GROUP/providers/Microsoft.Compute/virtualMachines/test-vm/extensions/TestExtension',
        'type': 'Microsoft.Compute/virtualMachines/extensions'
    },
    {
        'name': 'IaaSAntimalware',
        'id': '/subscriptions/123/resourceGroups/AQUA-RESOURCE_GROUP/providers/Microsoft.Compute/virtualMachines/test-vm/extensions/TestExtension',
        'type': 'Microsoft.Compute/virtualMachines/extensions'
    }
];

const createCache = (virtualMachines, virtualMachineExtension) => {
    let machine = {};
    let extension = {};
    if (virtualMachines) {
        machine['data'] = virtualMachines;
        if (virtualMachines.length && virtualMachineExtension) {
            extension[virtualMachines[0].id] = {
                'data': virtualMachineExtension
            };
        }
    }
    return {
        virtualMachines: {
            listAll: {
                'eastus': machine
            }
        },
        virtualMachineExtensions: {
            list: {
                'eastus': extension
            }
        }
    };
};

describe('vmWindowsAntiMalwareExtension', function() {
    describe('run', function() {
        it('should give passing result if no virtual machines', function(done) {
            const cache = createCache([]);
            vmWindowsAntiMalwareExtension.run(cache, {}, (err, results) => {
                expect(results.length).to.equal(1);
                expect(results[0].status).to.equal(0);
                expect(results[0].message).to.include('No Virtual Machines found');
                expect(results[0].region).to.equal('eastus');
                done();
            });
        });

        it('should give unknown result if unable to query for virtual machines', function(done) {
            const cache = createCache();
            vmWindowsAntiMalwareExtension.run(cache, {}, (err, results) => {
                expect(results.length).to.equal(1);
                expect(results[0].status).to.equal(3);
                expect(results[0].message).to.include('Unable to query for Virtual Machines');
                expect(results[0].region).to.equal('eastus');
                done();
            });
        });

        it('should give failing result if no virtual machine extensions', function(done) {
            const cache = createCache([virtualMachines[0]], []);
            vmWindowsAntiMalwareExtension.run(cache, {}, (err, results) => {
                expect(results.length).to.equal(1);
                expect(results[0].status).to.equal(2);
                expect(results[0].message).to.include('No VM Extensions found');
                expect(results[0].region).to.equal('eastus');
                done();
            });
        });

        it('should give unknown result if unable to query for virtual machine extensions', function(done) {
            const cache = createCache([virtualMachines[0]]);
            vmWindowsAntiMalwareExtension.run(cache, {}, (err, results) => {
                expect(results.length).to.equal(1);
                expect(results[0].status).to.equal(3);
                expect(results[0].message).to.include('Unable to query for VM Extensions');
                expect(results[0].region).to.equal('eastus');
                done();
            });
        });

        it('should give passing result if windows vm has antimalware extension installed', function(done) {
            const cache = createCache([virtualMachines[0]], [virtualMachineExtension[1]]);
            vmWindowsAntiMalwareExtension.run(cache, {}, (err, results) => {
                expect(results.length).to.equal(1);
                expect(results[0].status).to.equal(0);
                expect(results[0].message).to.include('Windows Virtual Machine has IaaS Antimalware extension installed');
                expect(results[0].region).to.equal('eastus');
                done();
            });
        });

        it('should give failing result if windows vm not have antimalware extension installed', function(done) {
            const cache = createCache([virtualMachines[0]], [virtualMachineExtension[0]]);
            vmWindowsAntiMalwareExtension.run(cache, {}, (err, results) => {
                expect(results.length).to.equal(1);
                expect(results[0].status).to.equal(2);
                expect(results[0].message).to.include('Windows Virtual Machine does not have IaaS Antimalware extension installed');
                expect(results[0].region).to.equal('eastus');
                done();
            });
        });
    });
});