var assert = require('assert');
var expect = require('chai').expect;
var plugin = require('./sqlNoPublicIps');

const createCache = (err, data) => {
    return {
            sql: {
                list: {
                    'global': {
                        err: err,
                        data: data
                    }
                }
        },
        projects: {
            get: {
                'global': {
                    data: [{ name: 'test-project' }]
                }
            }
        }
    }
};

describe('sqlPublicIps', function () {
    describe('run', function () {
        it('should give unknown result if a sql instance error is passed or no data is present', function (done) {
            const callback = (err, results) => {
                expect(results.length).to.be.above(0);
                expect(results[0].status).to.equal(3);
                expect(results[0].message).to.include('Unable to query SQL instances');
                expect(results[0].region).to.equal('global');
                done()
            };

            const cache = createCache(
                ['error'],
                null,
            );

            plugin.run(cache, {}, callback);
        });

        it('should give passing result if no sql instances are found', function (done) {
            const callback = (err, results) => {
                expect(results.length).to.be.above(0);
                expect(results[0].status).to.equal(0);
                expect(results[0].message).to.include('No SQL instances found');
                expect(results[0].region).to.equal('global');
                done()
            };

            const cache = createCache(
                null,
                [],
            );

            plugin.run(cache, {}, callback);
        });

        it('should give passing result if sql instances does not have public IPs', function (done) {
            const callback = (err, results) => {
                expect(results.length).to.be.above(0);
                expect(results[0].status).to.equal(0);
                expect(results[0].message).to.include('SQL instance does not have public IPs');
                expect(results[0].region).to.equal('global');
                done()
            };

            const cache = createCache(
                null,
                [{
                    instanceType: "CLOUD_SQL_INSTANCE",
                    name: "testing-instance",
                    databaseVersion: "POSTGRES_13",
                    ipAddresses: [
                        {
                          type: "PRIVATE",
                          ipAddress: "10.148.80.3",
                        },
                    ],
                }],
            );

            plugin.run(cache, {}, callback);
        });

        it('should give failing result if sql instances have public IPs', function (done) {
            const callback = (err, results) => {
                expect(results.length).to.be.above(0);
                expect(results[0].status).to.equal(2);
                expect(results[0].message).to.include('SQL instance has public IPs');
                expect(results[0].region).to.equal('global');
                done()
            };

            const cache = createCache(
                null,
                [{
                    instanceType: "CLOUD_SQL_INSTANCE",
                    name: "testing-instance",
                    databaseVersion: "POSTGRES_13",
                    ipAddresses: [
                        {
                          type: "PRIMARY",
                          ipAddress: "35.194.54.65",
                        },
                        {
                          type: "PRIVATE",
                          ipAddress: "10.148.80.3",
                        },
                    ],
                }],
            );

            plugin.run(cache, {}, callback);
        });
        
    })
})