var async = require('async');
var helpers = require('../../../helpers/google');

module.exports = {
    title: 'Storage Bucket All Users Policy',
    category: 'Storage',
    domain: 'Storage',
    severity: 'High',
    description: 'Ensures Storage bucket policies do not allow global write, delete, or read permissions',
    more_info: 'Storage buckets can be configured to allow the global principal to access the bucket via the bucket policy. This policy should be restricted only to known users or accounts.',
    link: 'https://cloud.google.com/storage/docs/access-control/iam',
    recommended_action: 'Ensure that each storage bucket is configured so that no member is set to allUsers or allAuthenticatedUsers.',
    apis: ['buckets:list','buckets:getIamPolicy'],
    realtime_triggers: ['storage.buckets.create', 'storage.buckets.delete', 'storage.buckets.update'],

    run: function(cache, settings, callback) {
        var results = [];
        var source = {};
        var regions = helpers.regions();

        async.each(regions.buckets, function(region, rcb){
            let buckets = helpers.addSource(
                cache, source, ['buckets', 'list', region]);

            if (!buckets) return rcb();

            if (buckets.err || !buckets.data) {
                helpers.addResult(results, 3, 'Unable to query storage buckets: ' + helpers.addError(buckets), region, null, null, buckets.err);
                return rcb();
            }

            if (!helpers.hasBuckets(buckets.data)) {
                helpers.addResult(results, 0, 'No storage buckets found', region);
                return rcb();
            }

            let bucketPolicyPolicies = helpers.addSource(cache, source,
                ['buckets', 'getIamPolicy', region]);

            if (!bucketPolicyPolicies) return rcb();

            if (bucketPolicyPolicies.err || !bucketPolicyPolicies.data) {
                helpers.addResult(results, 3, 'Unable to query bucket policies: ' + helpers.addError(bucketPolicyPolicies), region, null, null, bucketPolicyPolicies.err);
                return rcb();
            }

            if (!bucketPolicyPolicies.data.length) {
                helpers.addResult(results, 0, 'No storage buckets found', region);
                return rcb();
            }
            var badBuckets = [];
            var goodBuckets = [];
            bucketPolicyPolicies.data.forEach(bucketPolicy => {
                var hasAllUsers = false;
                var resourceIdArr = bucketPolicy.resourceId.split('/');
                var bucketName = resourceIdArr[resourceIdArr.length - 1];
                if (bucketPolicy.bindings &&
                    bucketPolicy.bindings.length) {
                    bucketPolicy.bindings.forEach(binding => {
                        if (binding.members &&
                           binding.members.length) {
                            binding.members.forEach(member => {
                                if (member === 'allUsers' ||
                                   member === 'allAuthenticatedUsers') {
                                    if (badBuckets.indexOf(bucketName) === -1) {
                                        badBuckets.push(bucketName);
                                        hasAllUsers = true;
                                    }
                                }
                            });
                        }
                    });
                }
                if (!hasAllUsers && badBuckets.indexOf(bucketName) === -1) {
                    goodBuckets.push(bucketName);
                }
            });

            if (badBuckets.length) {
                badBuckets.forEach(bucket => {
                    let resource = helpers.createResourceName('b', bucket);
                    helpers.addResult(results, 2,
                        'Bucket has anonymous or public access', region, resource);
                });
            } 
            if (goodBuckets.length) {
                goodBuckets.forEach(bucket => {
                    let resource = helpers.createResourceName('b', bucket);
                    helpers.addResult(results, 0,
                        'Bucket does not have anonymous or public access', region, resource);
                });
            }
            if (!goodBuckets.length && !badBuckets.length) {
                helpers.addResult(results, 0, 'No buckets found.', region);
            }

            rcb();
        }, function(){
            // Global checking goes here
            callback(null, results, source);
        });
    }
};