"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.useR2 = useR2;
const r2_mjs_1 = __importDefault(require("./r2.mjs"));
const react_1 = require("react");
let state = "unloaded";
let r2Module = null;
const pendingCommands = [];
const cachedPages = new Map([[0n, null]]);
function useR2({ source } = {}) {
    const sourceRef = (0, react_1.useRef)();
    (0, react_1.useEffect)(() => {
        if (source === undefined) {
            return;
        }
        sourceRef.current = source;
        if (state === "unloaded") {
            state = "loading";
            loadR2(sourceRef);
        }
    });
    const executeR2Command = (0, react_1.useCallback)((command) => {
        return new Promise(resolve => {
            pendingCommands.push({
                command,
                onComplete: resolve,
            });
            maybeProcessPendingCommands();
        });
    }, []);
    return {
        executeR2Command,
    };
}
async function loadR2(sourceRef) {
    const r2 = await (0, r2_mjs_1.default)({
        offset: "0",
        async onRead(offset, size) {
            const address = BigInt(offset);
            const pageSize = BigInt(sourceRef.current.pageSize);
            const firstPage = pageStart(address);
            const lastPage = pageStart(address + BigInt(size) - 1n);
            const pageAfterLastPage = lastPage + pageSize;
            const numPages = (pageAfterLastPage - firstPage) / pageSize;
            let allInCache = true;
            for (let page = firstPage; page !== pageAfterLastPage; page += pageSize) {
                const entry = cachedPages.get(page);
                if (entry === null) {
                    throw new Error("read failed");
                }
                if (entry === undefined) {
                    allInCache = false;
                    break;
                }
            }
            if (!allInCache) {
                try {
                    const block = await read(firstPage, Number(numPages * pageSize));
                    for (let page = firstPage; page !== pageAfterLastPage; page += pageSize) {
                        const offset = page - firstPage;
                        cachedPages.set(page, block.slice(Number(offset), Number(offset + pageSize)));
                    }
                }
                catch (e) {
                    for (let page = firstPage; page !== pageAfterLastPage; page += pageSize) {
                        cachedPages.set(page, null);
                    }
                    throw e;
                }
            }
            const result = new Uint8Array(size);
            let resultOffset = 0;
            for (let page = firstPage; page !== pageAfterLastPage; page += pageSize) {
                const remaining = size - resultOffset;
                const chunkSize = (remaining > pageSize) ? Number(pageSize) : remaining;
                const fromOffset = Number((page === firstPage) ? address % pageSize : 0n);
                const toOffset = fromOffset + chunkSize;
                const pageData = cachedPages.get(page);
                result.set(pageData.slice(fromOffset, toOffset), resultOffset);
                resultOffset += chunkSize;
            }
            return result;
            function pageStart(address) {
                const pageOffset = address % pageSize;
                return address - pageOffset;
            }
        },
    });
    function read(address, size) {
        return sourceRef.current.onReadRequest(address, size);
    }
    const { platform, arch, pointerSize } = sourceRef.current;
    await r2.ccall("r2_open", "void", ["string", "string", "int"], [platform, archFromFrida(arch), pointerSize * 8], { async: true });
    state = "loaded";
    r2Module = r2;
    maybeProcessPendingCommands();
}
async function maybeProcessPendingCommands() {
    if (state !== "loaded") {
        return;
    }
    state = "executing-command";
    const r = r2Module;
    const evaluate = r.cwrap("r2_execute", "number", ["string"], { async: true });
    let req;
    while ((req = pendingCommands.shift()) !== undefined) {
        const rawResult = await evaluate(req.command);
        try {
            const result = r.UTF8ToString(rawResult);
            req.onComplete(result);
        }
        finally {
            r._free(rawResult);
        }
    }
    state = "loaded";
}
function archFromFrida(arch) {
    switch (arch) {
        case "ia32":
        case "x64":
            return "x86";
        case "arm64":
            return "arm";
        default:
            return arch;
    }
}
