/// <reference types="react" />
import { CSSProperties } from 'react';
import { DetailedHTMLFactory } from 'react';
import { Easing as Easing_2 } from '@popmotion/easing';
import { ForwardRefExoticComponent } from 'react';
import { FunctionComponent } from 'react';
import { HTMLAttributes } from 'react';
import { PropsWithoutRef } from 'react';
import * as React from 'react';
import { ReactElement } from 'react';
import { ReactHTML } from 'react';
import { ReactNode } from 'react';
import { Ref } from 'react';
import { RefAttributes } from 'react';
import { RefObject } from 'react';
import { SpringProps } from 'popmotion';
import { SVGAttributes } from 'react';

/**
 * `AnimatePresence` enables the animation of components that have been removed from the tree.
 *
 * When adding/removing more than a single child, every child **must** be given a unique `key` prop.
 *
 * @library
 *
 * Any `Frame` components that have an `exit` property defined will animate out when removed from
 * the tree.
 *
 * ```jsx
 * import { Frame, AnimatePresence } from 'framer'
 *
 * // As items are added and removed from `items`
 * export function Items({ items }) {
 *   return (
 *     <AnimatePresence>
 *       {items.map(item => (
 *         <Frame
 *           key={item.id}
 *           initial={{ opacity: 0 }}
 *           animate={{ opacity: 1 }}
 *           exit={{ opacity: 0 }}
 *         />
 *       ))}
 *     </AnimatePresence>
 *   )
 * }
 * ```
 *
 * You can sequence exit animations throughout a tree using variants.
 *
 * @motion
 *
 * Any `motion` components that have an `exit` property defined will animate out when removed from
 * the tree.
 *
 * ```jsx
 * import { motion, AnimatePresence } from 'framer-motion'
 *
 * export const Items = ({ items }) => (
 *   <AnimatePresence>
 *     {items.map(item => (
 *       <motion.div
 *         key={item.id}
 *         initial={{ opacity: 0 }}
 *         animate={{ opacity: 1 }}
 *         exit={{ opacity: 0 }}
 *       />
 *     ))}
 *   </AnimatePresence>
 * )
 * ```
 *
 * You can sequence exit animations throughout a tree using variants.
 *
 * If a child contains multiple `motion` components with `exit` props, it will only unmount the child
 * once all `motion` components have finished animating out. Likewise, any components using
 * `usePresence` all need to call `safeToRemove`.
 *
 * @public
 */
export declare const AnimatePresence: FunctionComponent<AnimatePresenceProps>;

/**
 * @public
 */
export declare interface AnimatePresenceProps {
    /**
     * By passing `initial={false}`, `AnimatePresence` will disable any initial animations on children
     * that are present when the component is first rendered.
     *
     * @library
     *
     * ```jsx
     * <AnimatePresence initial={false}>
     *   {isVisible && (
     *     <Frame
     *       key="modal"
     *       initial={{ opacity: 0 }}
     *       animate={{ opacity: 1 }}
     *       exit={{ opacity: 0 }}
     *     />
     *   )}
     * </AnimatePresence>
     * ```
     *
     * @motion
     *
     * ```jsx
     * <AnimatePresence initial={false}>
     *   {isVisible && (
     *     <motion.div
     *       key="modal"
     *       initial={{ opacity: 0 }}
     *       animate={{ opacity: 1 }}
     *       exit={{ opacity: 0 }}
     *     />
     *   )}
     * </AnimatePresence>
     * ```
     *
     * @public
     */
    initial?: boolean;
    /**
     * When a component is removed, there's no longer a chance to update its props. So if a component's `exit`
     * prop is defined as a dynamic variant and you want to pass a new `custom` prop, you can do so via `AnimatePresence`.
     * This will ensure all leaving components animate using the latest data.
     *
     * @public
     */
    custom?: any;
    /**
     * Fires when all exiting nodes have completed animating out.
     *
     * @public
     */
    onExitComplete?: () => void;
    /**
     * `AnimatePresence` locally re-renders its children once exit animations are
     * complete. This means that if surrounding or parent components are also set to `positionTransition`,
     * they aren't informed of updates to the layout when they happen asynchronous to a render.
     *
     * This prop allows `AnimatePresence` to trigger re-renders at a higher level, so more
     * components can be made aware of the layout change and animate accordingly.
     *
     * In this example, the both the parent and sibling will animate to their new layout
     * once the div within `AnimatePresence` has finished animating:
     *
     * ```jsx
     * const MyComponent = ({ isVisible }) => {
     *   const forceUpdate = useForceUpdate() // Forces a set state or something
     *
     *   return (
     *     <motion.div positionTransition>
     *       <AnimatePresence _syncLayout={forceUpdate}>
     *         <motion.div positionTransition exit={{ opacity: 0 }} />
     *       </AnimatePresence>
     *       <motion.div positionTransition />
     *     </motion.div>
     *   )
     * }
     * ```
     *
     * In the final implementation `syncLayout` might be better as a component
     * that provides this function to children via context, or some other method
     * that obfuscates
     *
     * This isn't generally a problem for most use-cases but this capability will be useful
     * for advanced uses but also more so for phase 2 of `sizeTransition`, as we'd gain the power
     * to declaratively relayout entire parts of the page using only performant transforms.
     *
     * @internal
     */
    _syncLayout?: () => void;
    /**
     * If set to `true`, `AnimatePresence` will only render one component at a time. The exiting component
     * will finished its exit animation before the entering component is rendered.
     *
     * @library
     *
     * ```jsx
     * function MyComponent({ currentItem }) {
     *   return (
     *     <AnimatePresence exitBeforeEnter>
     *       <Frame key={currentItem} exit={{ opacity: 0 }} />
     *     </AnimatePresence>
     *   )
     * }
     * ```
     *
     * @motion
     *
     * ```jsx
     * const MyComponent = ({ currentItem }) => (
     *   <AnimatePresence exitBeforeEnter>
     *     <motion.div key={currentItem} exit={{ opacity: 0 }} />
     *   </AnimatePresence>
     * )
     * ```
     *
     * @beta
     */
    exitBeforeEnter?: boolean;
}

/**
 * Control animations on one or more components.
 *
 * @public
 */
export declare class AnimationControls {
    /**
     * Track whether the host component has mounted.
     *
     * @internal
     */
    private hasMounted;
    /**
     * A default `Transition` to set on linked components.
     *
     * @internal
     */
    private defaultTransition;
    /**
     * Pending animations that are started before a component is mounted.
     *
     * @internal
     */
    private pendingAnimations;
    /**
     * A collection of linked component animation controls.
     *
     * @internal
     */
    private componentControls;
    /**
     * A map of variants that can be later referenced via `start(variantLabel)`
     *
     * @internal
     */
    private variants;
    /**
     * Set variants on this and all child components.
     *
     * @param variants - The variants to set
     *
     * @internal
     */
    setVariants(variants: Variants): void;
    /**
     * Set a default transition on this and all child components
     *
     * @param transition - The default transition to set
     *
     * @internal
     */
    setDefaultTransition(transition: Transition): void;
    /**
     * Subscribes a component's animation controls to this.
     *
     * @param controls - The controls to subscribe
     * @returns An unsubscribe function.
     *
     * @internal
     */
    subscribe(controls: ValueAnimationControls): () => boolean;
    /**
     * Starts an animation on all linked components.
     *
     * @remarks
     *
     * ```jsx
     * controls.start("variantLabel")
     * controls.start({
     *   x: 0,
     *   transition: { duration: 1 }
     * })
     * ```
     *
     * @param definition - Properties or variant label to animate to
     * @param transition - Optional `transtion` to apply to a variant
     * @returns - A `Promise` that resolves when all animations have completed.
     *
     * @public
     */
    start(definition: AnimationDefinition, transitionOverride?: Transition): Promise<any>;
    /**
     * Instantly set to a set of properties or a variant.
     *
     * ```jsx
     * // With properties
     * controls.set({ opacity: 0 })
     *
     * // With variants
     * controls.set("hidden")
     * ```
     *
     * @internalremarks
     * We could perform a similar trick to `.start` where this can be called before mount
     * and we maintain a list of of pending actions that get applied on mount. But the
     * expectation of `set` is that it happens synchronously and this would be difficult
     * to do before any children have even attached themselves. It's also poor practise
     * and we should discourage render-synchronous `.start` calls rather than lean into this.
     *
     * @public
     */
    set(definition: AnimationDefinition): void;
    /**
     * Stops animations on all linked components.
     *
     * ```jsx
     * controls.stop()
     * ```
     *
     * @public
     */
    stop(): void;
    /**
     * Initialises the animation controls.
     *
     * @internal
     */
    mount(): void;
    /**
     * Stops all child animations when the host component unmounts.
     *
     * @internal
     */
    unmount(): void;
}

/**
 * @internal
 */
export declare const animationControls: () => AnimationControls;

declare type AnimationDefinition = VariantLabels | TargetAndTransition | TargetResolver;

declare type AnimationOptions = {
    delay?: number;
    priority?: number;
    transitionOverride?: Transition;
};

/**
 * @public
 */
export declare interface AnimationProps {
    /**
     * Values to animate to, variant label(s), or `AnimationControls`.
     *
     * @library
     *
     * ```jsx
     * // As values
     * <Frame animate={{ opacity: 1 }} />
     *
     * // As variant
     * <Frame animate="visible" variants={variants} />
     *
     * // Multiple variants
     * <Frame animate={["visible", "active"]} variants={variants} />
     *
     * // AnimationControls
     * <Frame animate={animation} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * // As values
     * <motion.div animate={{ opacity: 1 }} />
     *
     * // As variant
     * <motion.div animate="visible" variants={variants} />
     *
     * // Multiple variants
     * <motion.div animate={["visible", "active"]} variants={variants} />
     *
     * // AnimationControls
     * <motion.div animate={animation} />
     * ```
     */
    animate?: AnimationControls | TargetAndTransition | VariantLabels;
    /**
     * A target to animate to when this component is removed from the tree.
     *
     * This component **must** be the first animatable child of an `AnimatePresence` to enable this exit animation.
     *
     * This limitation exists because React doesn't allow components to defer unmounting until after
     * an animation is complete. Once this limitation is fixed, the `AnimatePresence` component will be unnecessary.
     *
     * @library
     *
     * ```jsx
     * import { Frame, AnimatePresence } from 'framer'
     *
     * export function MyComponent(props) {
     *   return (
     *     <AnimatePresence>
     *        {props.isVisible && (
     *          <Frame
     *            initial={{ opacity: 0 }}
     *            animate={{ opacity: 1 }}
     *            exit={{ opacity: 0 }}
     *          />
     *        )}
     *     </AnimatePresence>
     *   )
     * }
     * ```
     *
     * @motion
     *
     * ```jsx
     * import { AnimatePresence, motion } from 'framer-motion'
     *
     * export const MyComponent = ({ isVisible }) => {
     *   return (
     *     <AnimatePresence>
     *        {isVisible && (
     *          <motion.div
     *            initial={{ opacity: 0 }}
     *            animate={{ opacity: 1 }}
     *            exit={{ opacity: 0 }}
     *          />
     *        )}
     *     </AnimatePresence>
     *   )
     * }
     * ```
     */
    exit?: TargetAndTransition | VariantLabels | TargetResolver;
    /**
     * Variants allow you to define animation states and organise them by name. They allow
     * you to control animations throughout a component tree by switching a single `animate` prop.
     *
     * Using `transition` options like `delayChildren` and `staggerChildren`, you can orchestrate
     * when children animations play relative to their parent.
     *
     * @library
     *
     * After passing variants to one or more `Frame`'s `variants` prop, these variants
     * can be used in place of values on the `animate`, `initial`, `whileTap` and `whileHover` props.
     *
     * ```jsx
     * const variants = {
     *   active: {
     *     backgroundColor: "#f00"
     *   },
     *   inactive: {
     *     backgroundColor: "#fff",
     *     transition: { duration: 2 }
     *   }
     * }
     *
     * <Frame variants={variants} animate="active" />
     * ```
     *
     * @motion
     *
     * After passing variants to one or more `motion` component's `variants` prop, these variants
     * can be used in place of values on the `animate`, `initial`, `whileTap` and `whileHover` props.
     *
     * ```jsx
     * const variants = {
     *   active: {
     *       backgroundColor: "#f00"
     *   },
     *   inactive: {
     *     backgroundColor: "#fff",
     *     transition: { duration: 2 }
     *   }
     * }
     *
     * <motion.div variants={variants} animate="active" />
     * ```
     */
    variants?: Variants;
    /**
     * Default transition. If no `transition` is defined in `animate`, it will use the transition defined here.
     *
     * @library
     *
     * ```jsx
     * const spring = {
     *   type: "spring",
     *   damping: 10,
     *   stiffness: 100
     * }
     *
     * <Frame transition={spring} animate={{ scale: 1.2 }} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * const spring = {
     *   type: "spring",
     *   damping: 10,
     *   stiffness: 100
     * }
     *
     * <motion.div transition={spring} animate={{ scale: 1.2 }} />
     * ```
     */
    transition?: Transition;
    /**
     * @library
     *
     * When a `Frame` is the child of a `Stack`, the `Stack` is responsible for its layout. This makes it harder
     * for us to know when a `Frame`'s position changes within the `Stack` and make it animate to its new position.
     *
     * By adding `positionTransition` to a `Frame`, it'll automatically animate to its new position in the `Stack`,
     * whether the `Stack` layout has changed or the `Frame` has changed its order within it.
     *
     * It can either be set as a `Transition`, or just `true` to use the default layout transition.
     *
     * ```jsx
     * function MyComponent({ distribution = "space-around" }) {
     *   const spring = {
     *     type: "spring",
     *     damping: 10,
     *     stiffness: 100
     *   }
     *
     *   return (
     *     <Stack distribution={distribution}>
     *       <Frame layoutTransition={spring} />
     *     </Stack>
     *   )
     * }
     * ```
     *
     * @motion
     *
     * If `positionTransition` is defined on a `motion` component, it will automatically animate any changes to its layout
     * using a performant `x`/`y` transform.
     *
     * `positionTransition` can either be set as a `Transition`, or just `true` to use the default position transition, which is a snappy spring.
     *
     * It can also be set as a function that will resolve when the component has changed layout. This function
     * should return either a `Transition`, or `true`. For advanced use-cases where you want the component
     * to visually stay in its previous position, this function can also return `false`. This function will receive
     * the `delta` of the changed layout.
     *
     * ```jsx
     * const spring = {
     *   type: "spring",
     *   damping: 10,
     *   stiffness: 100
     * }
     *
     * // This component will animate position when `isVisible` is toggled.
     * const MyComponent = ({ isVisible }) => {
     *   return (
     *     <motion.div positionTransition={spring} style={{ left: isVisible ? 0 : 100 }} />
     *   )
     * }
     *
     * // This component will animate items to their new position if its place in `items` changes order.
     * const MyComponent = ({ items }) => {
     *   return items.map((item) => (
     *     <motion.div key={item.id} positionTransition={spring} />
     *   ))
     * }
     * ```
     *
     * @public
     */
    positionTransition?: Transition | boolean | ResolveLayoutTransition;
    /**
     * @library
     *
     * When a `Frame` is the child of a `Stack`, the `Stack` is responsible for its layout. This makes it
     * difficult for to know when the layout changes and smoothly animate components to their new positions.
     *
     * By adding `layoutTransition` to a child `Frame`, it'll automatically animate to its new position
     * when it moves in the `Stack`, whether the `Stack` layout has changed, or the `Frame` has changed order within it.
     *
     * It can either be set as a `Transition`, or just `true` to use the default layout transition.
     *
     * Animating size with `scale` can introduce visual distortion to the component's children. If unwanted,
     * the `useInvertedScale` function can be used to undo this distortion.
     *
     * ```jsx
     * function MyComponent({ distribution = "space-around" }) {
     *   const spring = {
     *     type: "spring",
     *     damping: 10,
     *     stiffness: 100
     *   }
     *
     *   return (
     *     <Stack distribution={distribution}>
     *       <Frame layoutTransition={spring} />
     *     </Stack>
     *   )
     * }
     * ```
     *
     * @motion
     *
     * If `layoutTransition` is defined on a `motion` component, the component will automatically
     * animate any changes to its position **and** size.
     *
     * It will do so using performant transforms. So if a `motion` component changes position, it'll animate
     * to its new position using `x` and `y`. If it changes size, it'll animate using `scaleX` and `scaleY`.
     *
     * Animating size with `scale` can introduce visual distortion to the component's children. If unwanted,
     * the `useInvertedScale` function can be used to undo this distortion.
     *
     * `layoutTransition` can either be set as a `Transition`, or just `true` to use the default layout transition,
     * which is a smooth `0.8` second ease.
     *
     * It can also be set as a function that will resolve when the component has changed layout. This function
     * should return either a `Transition`, or `true`. For advanced use-cases where you want the component
     * to visually stay in its previous position, this function can also return `false`. This function will receive
     * the `delta` of the changed layout.
     *
     * ```jsx
     * const spring = {
     *   type: "spring",
     *   damping: 10,
     *   stiffness: 100
     * }
     *
     * // This component will animate between sizes when `isVisible` is toggled.
     * const MyComponent = ({ isVisible }) => {
     *   return (
     *     <motion.div layoutTransition={spring}>
     *       {isVisible && <Content />}
     *     </motion.div>
     *   )
     * }
     * ```
     *
     * @beta
     */
    layoutTransition?: Transition | boolean | ResolveLayoutTransition;
}

declare type AnyPointerEvent = MouseEvent | TouchEvent | PointerEvent;

declare class ComponentDragControls {
    /**
     * Track whether we're currently dragging.
     *
     * @internal
     */
    private isDragging;
    /**
     * The current direction of drag, or `null` if both.
     *
     * @internal
     */
    private currentDirection;
    /**
     * The permitted t/r/b/l boundaries of travel, in pixels.
     *
     * @internal
     */
    private constraints;
    /**
     * If `true`, our constraints need to be resolved from a Element ref
     * passed to props.dragConstraints
     *
     * @internal
     */
    private constraintsNeedResolution;
    /**
     * A reference to the host component's latest props.
     *
     * @internal
     */
    private props;
    /**
     * @internal
     */
    private ref;
    /**
     * A reference to the host component's animation controls.
     *
     * @internal
     */
    private controls;
    /**
     * A reference to the host component's motion values.
     *
     * @internal
     */
    private values;
    /**
     * References to the MotionValues used for tracking the current dragged point.
     *
     * @internal
     */
    private point;
    /**
     * The origin point for the current drag gesture.
     *
     * @internal
     */
    private origin;
    private openGlobalLock;
    /**
     * @internal
     */
    private panSession;
    /**
     * A reference to the previous constraints bounding box
     *
     * @internal
     */
    private prevConstraintsBox;
    constructor({ ref, values, controls }: DragControlConfig);
    /**
     * Start dragging the host component.
     *
     * @param event - The originating pointer event.
     * @param options -
     *
     * @public
     */
    start(originEvent: AnyPointerEvent, { snapToCursor }?: DragControlOptions): void;
    cancelDrag(): void;
    stop(event: AnyPointerEvent, info: PanInfo): void;
    recordBoxInfo(constraints?: Constraints | false): void;
    snapToCursor(event: AnyPointerEvent): void;
    setPoint(axis: DragDirection, value: MotionValue<number>): void;
    updatePoint(axis: DragDirection, offset: {
        x: number;
        y: number;
    }): void;
    updateProps({ drag, dragDirectionLock, dragPropagation, dragConstraints, dragElastic, dragMomentum, ...remainingProps }: DragControlsProps): void;
    private applyConstraintsToPoint;
    private animateDragEnd;
    scalePoint(): void;
    mount(element: Element): () => void;
}

declare type Config<T> = {
    transformer?: Transformer<T>;
    parent?: MotionValue<T>;
};

declare type Constraints = {
    left?: number;
    right?: number;
    top?: number;
    bottom?: number;
};

declare interface ControlsProp {
    controls?: ValueAnimationControls;
}

/**
 * @internal
 */
export declare const createMotionComponent: <P extends {}>({ getValueControlsConfig, loadFunctionalityComponents, renderComponent, }: MotionComponentConfig) => React.ForwardRefExoticComponent<React.PropsWithoutRef<P & MotionProps> & React.RefAttributes<Element>>;

declare type CSSPropertiesWithoutTransitionOrSingleTransforms = Omit<CSSProperties, "transition" | "rotate" | "scale" | "perspective">;

declare interface CustomStyles {
    /**
     * Framer Library custom prop types. These are not actually supported in Motion - preferably
     * we'd have a way of external consumers injecting supported styles into this library.
     */
    size?: string | number;
    radius?: string | number;
    shadow?: string;
    image?: string;
}

/**
 * @public
 */
export declare interface CustomValueType {
    mix: (from: any, to: any) => (p: number) => number | string;
    toValue: () => number | string;
}

declare type Cycle = (i?: number) => void;

declare type CycleState<T> = [T, Cycle];

declare interface DragControlConfig {
    ref: RefObject<Element>;
    values: MotionValuesMap;
    controls: ValueAnimationControls;
}

declare interface DragControlOptions {
    snapToCursor?: boolean;
}

/**
 * Can manually trigger a drag gesture on one or more `drag`-enabled `motion` components.
 *
 * @library
 *
 * ```jsx
 * const dragControls = useDragControls()
 *
 * function startDrag(event) {
 *   dragControls.start(event, { snapToCursor: true })
 * }
 *
 * return (
 *   <>
 *     <Frame onTapStart={startDrag} />
 *     <Frame drag="x" dragControls={dragControls} />
 *   </>
 * )
 * ```
 *
 * @motion
 *
 * ```jsx
 * const dragControls = useDragControls()
 *
 * function startDrag(event) {
 *   dragControls.start(event, { snapToCursor: true })
 * }
 *
 * return (
 *   <>
 *     <div onMouseDown={startDrag} />
 *     <motion.div drag="x" dragControls={dragControls} />
 *   </>
 * )
 * ```
 *
 * @public
 */
export declare class DragControls {
    private componentControls;
    /**
     * Subscribe a component's internal `ComponentDragControls` to the user-facing API.
     *
     * @internal
     */
    subscribe(controls: ComponentDragControls): () => void;
    /**
     * Start a drag gesture on every `motion` component that has this set of drag controls
     * passed into it via the `dragControls` prop.
     *
     * ```jsx
     * dragControls.start(e, {
     *   snapToCursor: true
     * })
     * ```
     *
     * @param event - A mouse/touch/pointer event.
     * @param options - Options
     *
     * @public
     */
    start(event: React.MouseEvent | React.TouchEvent | React.PointerEvent | MouseEvent | TouchEvent | PointerEvent, options?: DragControlOptions): void;
}

declare interface DragControlsProps extends DraggableProps {
    transformPagePoint: (point: Point) => Point;
}

declare type DragDirection = "x" | "y";

/**
 * @public
 */
export declare interface DraggableProps extends DragHandlers {
    /**
     * Enable dragging for this element. Set to `false` by default.
     * Set `true` to drag in both directions.
     * Set `"x"` or `"y"` to only drag in a specific direction.
     *
     * @library
     *
     * ```jsx
     * <Frame drag="x" />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div drag="x" />
     * ```
     */
    drag?: boolean | "x" | "y";
    /**
     * If `true`, this will lock dragging to the initially-detected direction. Defaults to `false`.
     *
     * @library
     *
     * ```jsx
     * <Frame drag={true} dragDirectionLock={true} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div drag dragDirectionLock />
     * ```
     */
    dragDirectionLock?: boolean;
    /**
     * Allows drag gesture propagation to child components. Set to `false` by
     * default.
     *
     * @library
     *
     * ```jsx
     * <Frame drag="x" dragPropagation={true} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div drag="x" dragPropagation />
     * ```
     */
    dragPropagation?: boolean;
    /**
     * An object of optional `top`, `left`, `right`, `bottom` pixel values,
     * beyond which dragging is constrained.
     *
     * Another component can be used as drag constraints by creating a `ref` with React's `useRef`.hook.
     * This `ref` should be passed to that component's `ref` prop and to this component's `dragConstraints` prop.
     *
     * @library
     *
     * ```jsx
     * // In pixels
     * <Frame
     *   drag="x"
     *   dragConstraints={{ left: 0, right: 300 }}
     * />
     *
     * // As a ref to another component
     * function MyComponent() {
     *   const constraintsRef = useRef(null)
     *
     *   return (
     *      <Frame ref={constraintsRef} width={400} height={400}>
     *          <Frame drag dragConstraints={constraintsRef} />
     *      </Frame>
     *   )
     * }
     * ```
     *
     * @motion
     *
     * ```jsx
     * // In pixels
     * <motion.div
     *   drag="x"
     *   dragConstraints={{ left: 0, right: 300 }}
     * />
     *
     * // As a ref to another component
     * const MyComponent = () => {
     *   const constraintsRef = useRef(null)
     *
     *   return (
     *      <motion.div ref={constraintsRef}>
     *          <motion.div drag dragConstraints={constraintsRef} />
     *      </motion.div>
     *   )
     * }
     * ```
     */
    dragConstraints?: false | {
        top?: number;
        right?: number;
        bottom?: number;
        left?: number;
    } | RefObject<Element>;
    /**
     * The degree of movement allowed outside constraints. 0 = no movement, 1 =
     * full movement. Set to `0.5` by default.
     *
     * @library
     *
     * ```jsx
     * <Frame
     *   drag={true}
     *   dragConstraints={{ left: 0, right: 300 }}
     *   dragElastic={0.2}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   drag
     *   dragConstraints={{ left: 0, right: 300 }}
     *   dragElastic={0.2}
     * />
     * ```
     */
    dragElastic?: boolean | number;
    /**
     * Apply momentum from the pan gesture to the component when dragging
     * finishes. Set to `true` by default.
     *
     * @library
     *
     * ```jsx
     * <Frame
     *   drag={true}
     *   dragConstraints={{ left: 0, right: 300 }}
     *   dragMomentum={false}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   drag
     *   dragConstraints={{ left: 0, right: 300 }}
     *   dragMomentum={false}
     * />
     * ```
     */
    dragMomentum?: boolean;
    /**
     * Allows you to change dragging inertia parameters.
     * When releasing a draggable Frame, an animation with type `inertia` starts. The animation is based on your dragging velocity. This property allows you to customize it.
     * See {@link https://framer.com/api/animation/#inertia | Inertia} for all properties you can use.
     *
     * @library
     *
     * ```jsx
     * <Frame
     *   drag={true}
     *   dragTransition={{ bounceStiffness: 600, bounceDamping: 10 }}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   drag
     *   dragTransition={{ bounceStiffness: 600, bounceDamping: 10 }}
     * />
     * ```
     */
    dragTransition?: InertiaOptions;
    /**
     * @internalremarks
     *
     * _dragValueX, _dragValueY and _dragTransitionControls are a way of allowing this
     * component to be a drag target for another element.
     *
     * @internal
     */
    _dragValueX?: MotionValue<number>;
    /**
     * @internal
     */
    _dragValueY?: MotionValue<number>;
    /**
     * @internal
     */
    _dragTransitionControls?: AnimationControls;
    /**
     * Drag position is calculated by applying the pan offset to the x/y origin
     * measured when the drag gesture begins.
     *
     * By manually creating `dragOriginX` as a `MotionValue`, it can be updated
     * while the gesture is active, for instance to visually offset any movement should
     * the component change layout.
     *
     * @library
     *
     * ```jsx
     * const dragOriginX = useMotionValue(0)
     *
     * return <Frame dragOriginX={dragOriginX} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * const dragOriginX = useMotionValue(0)
     *
     * return <motion.div dragOriginX={dragOriginX} />
     * ```
     *
     * @public
     */
    dragOriginX?: MotionValue<number>;
    /**
     * Drag position is calculated by applying the pan offset to the x/y origin
     * measured when the drag gesture begins.
     *
     * By manually creating `dragOriginY` as a `MotionValue`, it can be updated
     * while the gesture is active, for instance to visually offset any movement should
     * the component change layout.
     *
     * @library
     *
     * ```jsx
     * const dragOriginY = useMotionValue(0)
     *
     * return <Frame dragOriginY={dragOriginY} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * const dragOriginY = useMotionValue(0)
     *
     * return <motion.div dragOriginY={dragOriginY} />
     * ```
     *
     * @public
     */
    dragOriginY?: MotionValue<number>;
    /**
     * Usually, dragging is initiated by pressing down on a component and moving it. For some
     * use-cases, for instance clicking at an arbitrary point on a video scrubber, we
     * might want to initiate dragging from a different component than the draggable one.
     *
     * By creating a `dragControls` using the `useDragControls` hook, we can pass this into
     * the draggable component's `dragControls` prop. It exposes a `start` method
     * that can start dragging from pointer events on other components.
     *
     * @library
     *
     * ```jsx
     * const dragControls = useDragControls()
     *
     * function startDrag(event) {
     *   dragControls.start(event, { snapToCursor: true })
     * }
     *
     * return (
     *   <>
     *     <Frame onTapStart={startDrag} />
     *     <Frame drag="x" dragControls={dragControls} />
     *   </>
     * )
     * ```
     *
     * @motion
     *
     * ```jsx
     * const dragControls = useDragControls()
     *
     * function startDrag(event) {
     *   dragControls.start(event, { snapToCursor: true })
     * }
     *
     * return (
     *   <>
     *     <div onMouseDown={startDrag} />
     *     <motion.div drag="x" dragControls={dragControls} />
     *   </>
     * )
     * ```
     */
    dragControls?: DragControls;
    /**
     * By default, if `drag` is defined on a component then an event listener will be attached
     * to automatically initiate dragging when a user presses down on it.
     *
     * By setting `dragListener` to `false`, this event listener will not be created.
     *
     * @library
     *
     * ```jsx
     * const dragControls = useDragControls()
     *
     * function startDrag(event) {
     *   dragControls.start(event, { snapToCursor: true })
     * }
     *
     * return (
     *   <>
     *     <Frame onTapStart={startDrag} />
     *     <Frame
     *       drag="x"
     *       dragControls={dragControls}
     *       dragListener={false}
     *     />
     *   </>
     * )
     * ```
     *
     * @motion
     *
     * ```jsx
     * const dragControls = useDragControls()
     *
     * function startDrag(event) {
     *   dragControls.start(event, { snapToCursor: true })
     * }
     *
     * return (
     *   <>
     *     <div onMouseDown={startDrag} />
     *     <motion.div
     *       drag="x"
     *       dragControls={dragControls}
     *       dragListener={false}
     *     />
     *   </>
     * )
     * ```
     */
    dragListener?: boolean;
}

/**
 * @public
 */
export declare interface DragHandlers {
    /**
     * Callback function that fires when dragging starts.
     *
     * @library
     *
     * ```jsx
     * function onDragStart(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <Frame drag onDragStart={onDragStart} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   drag
     *   onDragStart={
     *     (event, info) => console.log(info.point.x, info.point.y)
     *   }
     * />
     * ```
     *
     * @public
     */
    onDragStart?(event: MouseEvent | TouchEvent | PointerEvent, info: PanInfo): void;
    /**
     * Callback function that fires when dragging ends.
     *
     * @library
     *
     * ```jsx
     * function onDragEnd(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <Frame drag onDragEnd={onDragEnd} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   drag
     *   onDragEnd={
     *     (event, info) => console.log(info.point.x, info.point.y)
     *   }
     * />
     * ```
     *
     * @public
     */
    onDragEnd?(event: MouseEvent | TouchEvent | PointerEvent, info: PanInfo): void;
    /**
     * Callback function that fires when the component is dragged.
     *
     * @library
     *
     * ```jsx
     * function onDrag(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <Frame drag onDrag={onDrag} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   drag
     *   onDrag={
     *     (event, info) => console.log(info.point.x, info.point.y)
     *   }
     * />
     * ```
     *
     * @public
     */
    onDrag?(event: MouseEvent | TouchEvent | PointerEvent, info: PanInfo): void;
    /**
     * Callback function that fires a drag direction is determined.
     *
     * @library
     *
     * ```jsx
     * function onDirectionLock(axis) {
     *   console.log(axis)
     * }
     *
     * <Frame
     *   drag
     *   dragDirectionLock
     *   onDirectionLock={onDirectionLock}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   drag
     *   dragDirectionLock
     *   onDirectionLock={axis => console.log(axis)}
     * />
     * ```
     *
     * @public
     */
    onDirectionLock?(axis: "x" | "y"): void;
    /**
     * Callback function that fires when drag momentum/bounce transition finishes.
     *
     * @library
     *
     * ```jsx
     * function onDragTransitionEnd() {
     *   console.log('drag transition has ended')
     * }
     *
     * <Frame
     *   drag
     *   onDragTransitionEnd={onDragTransitionEnd}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   drag
     *   onDragTransitionEnd={() => console.log('Drag transition complete')}
     * />
     * ```
     *
     * @public
     */
    onDragTransitionEnd?(): void;
}

/**
 * The easing function to use. Set as one of:
 *
 * - The name of an in-built easing function.
 * - An array of four numbers to define a cubic bezier curve.
 * - An easing function, that accepts and returns a progress value between `0` and `1`.
 *
 * @public
 */
declare type Easing = [number, number, number, number] | "linear" | "easeIn" | "easeOut" | "easeInOut" | "circIn" | "circOut" | "circInOut" | "backIn" | "backOut" | "backInOut" | "anticipate" | EasingFunction;

/**
 * A function that accepts a progress value between `0` and `1` and returns a
 * new one.
 *
 * @library
 *
 * ```jsx
 * const transition = {
 *   ease: progress => progress * progress
 * }
 *
 * <Frame
 *   animate={{ opacity: 0 }}
 *   transition={transition}
 * />
 * ```
 *
 * @motion
 *
 * ```jsx
 * <motion.div
 *   animate={{ opacity: 0 }}
 *   transition={{
 *     duration: 1,
 *     ease: progress => progress * progress
 *   }}
 * />
 * ```
 *
 * @public
 */
export declare type EasingFunction = (v: number) => number;

/** @public */
export declare interface EventInfo {
    point: Point;
}

/**
 * @public
 */
export declare type ForwardRefComponent<T, P> = ForwardRefExoticComponent<PropsWithoutRef<P> & RefAttributes<T>>;

declare interface FunctionalProps extends MotionProps {
    controls: ValueAnimationControls;
    values: MotionValuesMap;
    innerRef: RefObject<Element>;
    parentContext: MotionContextProps;
}

/**
 * @public
 */
export declare type GestureHandlers = PanHandlers & TapHandlers & HoverHandlers;

/**
 * @public
 */
export declare interface HoverHandlers {
    /**
     * Properties or variant label to animate to while the hover gesture is recognised.
     *
     * @library
     *
     * ```jsx
     * <Frame whileHover={{ scale: 1.2 }} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div whileHover={{ scale: 1.2 }} />
     * ```
     */
    whileHover?: string | TargetAndTransition;
    /**
     * Callback function that fires when pointer starts hovering over the component.
     *
     * @library
     *
     * ```jsx
     * function onHoverStart(event) {
     *   console.log("Hover starts")
     * }
     *
     * <Frame onHoverStart={onHoverStart} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div onHoverStart={() => console.log('Hover starts')} />
     * ```
     */
    onHoverStart?(event: MouseEvent, info: EventInfo): void;
    /**
     * Callback function that fires when pointer stops hovering over the component.
     *
     * @library
     *
     * ```jsx
     * function onHoverEnd(event) {
     *   console.log("Hover ends")
     * }
     *
     * <Frame onHoverEnd={onHoverEnd} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div onHoverEnd={() => console.log("Hover ends")} />
     * ```
     */
    onHoverEnd?(event: MouseEvent, info: EventInfo): void;
}

declare type HTMLAttributesWithoutMotionProps<Attributes extends HTMLAttributes<Element>, Element extends HTMLElement> = {
    [K in Exclude<keyof Attributes, keyof MotionProps>]?: Attributes[K];
};

/**
 * @public
 */
export declare type HTMLMotionProps<TagName extends keyof ReactHTML> = HTMLAttributesWithoutMotionProps<UnwrapFactoryAttributes<ReactHTML[TagName]>, UnwrapFactoryElement<ReactHTML[TagName]>> & MotionProps;

/**
 * An animation that decelerates a value based on its initial velocity,
 * usually used to implement inertial scrolling.
 *
 * Optionally, `min` and `max` boundaries can be defined, and inertia
 * will snap to these with a spring animation.
 *
 * This animation will automatically precalculate a target value,
 * which can be modified with the `modifyTarget` property.
 *
 * This allows you to add snap-to-grid or similar functionality.
 *
 * Inertia is also the animation used for `dragTransition`, and can be configured via that prop.
 *
 * @public
 */
export declare interface Inertia {
    /**
     * Set `type` to animate using the inertia animation. Set to `"tween"` by
     * default. This can be used for natural deceleration, like momentum scrolling.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   type: "inertia",
     *   velocity: 50
     * }
     *
     * <Frame
     *   animate={{ rotate: 180 }}
     *   transition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   animate={{ rotate: 180 }}
     *   transition={{ type: "inertia", velocity: 50 }}
     * />
     * ```
     *
     * @public
     */
    type: "inertia";
    /**
     * A function that receives the automatically-calculated target and returns a new one. Useful for snapping the target to a grid.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   power: 0,
     *   // Snap calculated target to nearest 50 pixels
     *   modifyTarget: target => Math.round(target / 50) * 50
     * }
     *
     * <Frame
     *   drag
     *   dragTransition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   drag
     *   dragTransition={{
     *     power: 0,
     *     // Snap calculated target to nearest 50 pixels
     *     modifyTarget: target => Math.round(target / 50) * 50
     *   }}
     * />
     * ```
     *
     * @public
     */
    modifyTarget?(v: number): number;
    /**
     * If `min` or `max` is set, this affects the stiffness of the bounce
     * spring. Higher values will create more sudden movement. Set to `500` by
     * default.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   min: 0,
     *   max: 100,
     *   bounceStiffness: 100
     * }
     *
     * <Frame
     *   drag
     *   dragTransition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   drag
     *   dragTransition={{
     *     min: 0,
     *     max: 100,
     *     bounceStiffness: 100
     *   }}
     * />
     * ```
     *
     * @public
     */
    bounceStiffness?: number;
    /**
     * If `min` or `max` is set, this affects the damping of the bounce spring.
     * If set to `0`, spring will oscillate indefinitely. Set to `10` by
     * default.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   min: 0,
     *   max: 100,
     *   bounceDamping: 8
     * }
     *
     * <Frame
     *   drag
     *   dragTransition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   drag
     *   dragTransition={{
     *     min: 0,
     *     max: 100,
     *     bounceDamping: 8
     *   }}
     * />
     * ```
     *
     * @public
     */
    bounceDamping?: number;
    /**
     * A higher power value equals a further target. Set to `0.8` by default.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   min: 0,
     *   max: 100,
     *   power: 0.2
     * }
     *
     * <Frame
     *   drag
     *   dragTransition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   drag
     *   dragTransition={{ power: 0.2 }}
     * />
     * ```
     *
     * @public
     */
    power?: number;
    /**
     * Adjusting the time constant will change the duration of the
     * deceleration, thereby affecting its feel. Set to `700` by default.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   min: 0,
     *   max: 100,
     *   timeConstant: 200
     * }
     *
     * <Frame
     *   drag
     *   dragTransition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   drag
     *   dragTransition={{ timeConstant: 200 }}
     * />
     * ```
     *
     * @public
     */
    timeConstant?: number;
    /**
     * End the animation if the distance to the animation target is below this value, and the absolute speed is below `restSpeed`.
     * When the animation ends, the value gets snapped to the animation target. Set to `0.01` by default.
     * Generally the default values provide smooth animation endings, only in rare cases should you need to customize these.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   min: 0,
     *   max: 100,
     *   restDelta: 10
     * }
     *
     * <Frame
     *   drag
     *   dragTransition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   drag
     *   dragTransition={{ restDelta: 10 }}
     * />
     * ```
     *
     * @public
     */
    restDelta?: number;
    /**
     * Minimum constraint. If set, the value will "bump" against this value (or immediately spring to it if the animation starts as less than this value).
     *
     * @library
     *
     * ```jsx
     * <Frame
     *   drag
     *   dragTransition={{ min: 0, max: 100 }}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   drag
     *   dragTransition={{ min: 0, max: 100 }}
     * />
     * ```
     *
     * @public
     */
    min?: number;
    /**
     * Maximum constraint. If set, the value will "bump" against this value (or immediately snap to it, if the initial animation value exceeds this value).
     *
     * @library
     *
     * ```jsx
     * <Frame
     *   drag
     *   dragTransition={{ min: 0, max: 100 }}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   drag
     *   dragTransition={{ min: 0, max: 100 }}
     * />
     * ```
     *
     * @public
     */
    max?: number;
    /**
     * The value to animate from. By default, this is the current state of the animating value.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   min: 0,
     *   max: 100,
     *   from: 50
     * }
     *
     * <Frame
     *   drag
     *   dragTransition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <Frame
     *   drag
     *   dragTransition={{ from: 50 }}
     * />
     * ```
     *
     * @public
     */
    from?: number | string;
    /**
     * The initial velocity of the animation.
     * By default this is the current velocity of the component.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   type: "inertia",
     *   velocity: 200
     * }
     *
     * <Frame
     *   animate={{ rotate: 180 }}
     *   transition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   animate={{ rotate: 180 }}
     *   transition={{ type: 'inertia', velocity: 200 }}
     * />
     * ```
     *
     * @public
     */
    velocity?: number;
    /**
     * @internal
     */
    delay?: number;
}

/**
 * @public
 */
declare type InertiaOptions = Partial<Omit<Inertia, "velocity" | "type">>;

/**
 * Check whether a prop name is a valid `MotionProp` key.
 *
 * @param key - Name of the property to check
 * @returns `true` is key is a valid `MotionProp`.
 *
 * @public
 */
export declare function isValidMotionProp(key: string): boolean;

/**
 * @internal
 */
declare interface Just {
    type: "just";
    to?: number | string | ValueTarget;
    from?: number | string;
    delay?: number;
    velocity?: number;
}

/**
 * Keyframes tweens between multiple `values`.
 *
 * These tweens can be arranged using the `duration`, `easings`, and `times` properties.
 *
 * @internalremarks
 * We could possibly make the `type` property redundant, if not for all animations
 * then for this one quite easily.
 *
 * @internal
 */
export declare interface Keyframes {
    /**
     * Set `type` to `"keyframes"` to animate using the keyframes animation.
     * Set to `"tween"` by default. This can be used to animate between a series of values.
     *
     * @public
     */
    type: "keyframes";
    /**
     * An array of values to animate between.
     *
     * @internal
     */
    values: KeyframesTarget;
    /**
     * An array of numbers between 0 and 1, where `1` represents the `total` duration.
     *
     * Each value represents at which point during the animation each item in the animation target should be hit, so the array should be the same length as `values`.
     *
     * Defaults to an array of evenly-spread durations.
     *
     * @public
     */
    times?: number[];
    /**
     * An array of easing functions for each generated tween, or a single easing function applied to all tweens.
     *
     * This array should be one item less than `values`, as these easings apply to the transitions *between* the `values`.
     *
     * ```jsx
     * const transition = {
     *   backgroundColor: {
     *     type: 'keyframes',
     *     easings: ['circIn', 'circOut']
     *   }
     * }
     * ```
     *
     * @public
     */
    ease?: Easing | Easing[];
    /**
     * Popmotion's easing prop to define individual easings. `ease` will be mapped to this prop in keyframes animations.
     *
     * @internal
     */
    easings?: Easing | Easing[];
    /**
     * @internal
     */
    elapsed?: number;
    /**
     * The total duration of the animation. Set to `0.3` by default.
     *
     * ```jsx
     * const transition = {
     *   type: "keyframes",
     *   duration: 2
     * }
     *
     * <Frame
     *   animate={{ opacity: 0 }}
     *   transition={transition}
     * />
     * ```
     *
     * @public
     */
    duration?: number;
    /**
     * The number of times to loop the animation.
     *
     * Set to `Infinity` for perpetual looping.
     *
     * @public
     */
    loop?: number;
    /**
     * The number of times to flip the animation by swapping the `to` and `from` values.
     * Set to `Infinity` for perpetual flipping.
     *
     * ```jsx
     * const transition = {
     *   flip: Infinity,
     *   duration: 2
     * }
     *
     * <Frame
     *   animate={{ opacity: 0 }}
     *   transition={transition}
     * />
     * ```
     *
     * @public
     */
    flip?: number;
    /**
     * The number of times to reverse the animation.
     * Set to `Infinity` for perpetual reversing.
     *
     * ```jsx
     * const transition = {
     *   yoyo: Infinity,
     *   duration: 2
     * }
     *
     * <Frame
     *   animate={{ rotate: 180 }}
     *   transition={transition}
     * />
     *
     * ```
     * @public
     */
    yoyo?: number;
    /**
     * @public
     */
    repeatDelay?: number;
    /**
     * @internal
     */
    from?: number | string;
    /**
     * @internal
     */
    to?: number | string | ValueTarget;
    /**
     * @internal
     */
    velocity?: number;
    /**
     * @internal
     */
    delay?: number;
}

/**
 * @public
 */
export declare type KeyframesTarget = ResolvedKeyframesTarget | [null, ...CustomValueType[]] | CustomValueType[];

declare type LoadFunctionalityComponents<P = {}> = (ref: RefObject<Element>, values: MotionValuesMap, props: P & MotionProps, parentContext: MotionContextProps, controls: ValueAnimationControls<P>, inherit: boolean) => ReactElement<FunctionalProps>[];

declare type MakeCustomValueType<T> = {
    [K in keyof T]: T[K] | CustomValueType;
};

declare type MakeKeyframes<T> = {
    [K in keyof T]: T[K] | T[K][] | [null, ...T[K][]];
};

declare type MakeMotion<T> = MakeCustomValueType<{
    [K in keyof T]: T[K] | MotionValue<number> | MotionValue<string> | MotionValue<any>;
}>;

declare type MakeTargetAnimatable = (target: TargetWithKeyframes, transitionEnd?: Target | undefined) => {
    target: TargetWithKeyframes;
    transitionEnd?: Target | undefined;
};

/**
 * HTML & SVG components, optimised for use with gestures and animation. These can be used as
 * drop-in replacements for any HTML & SVG component, all CSS & SVG properties are supported.
 *
 * @internalremarks
 *
 * I'd like to make it possible for these to be loaded "on demand" - to reduce bundle size by only
 * including HTML/SVG stylers, animation and/or gesture support when necessary.
 *
 * ```jsx
 * <motion.div animate={{ x: 100 }} />
 *
 * <motion.p animate={{ height: 200 }} />
 *
 * <svg><motion.circle r={10} animate={{ r: 20 }} /></svg>
 * ```
 *
 * @public
 */
export declare const motion: {
    symbol: ForwardRefComponent<SVGSymbolElement, SVGMotionProps<SVGSymbolElement>>;
    clipPath: ForwardRefComponent<SVGClipPathElement, SVGMotionProps<SVGClipPathElement>>;
    filter: ForwardRefComponent<SVGFilterElement, SVGMotionProps<SVGFilterElement>>;
    mask: ForwardRefComponent<SVGMaskElement, SVGMotionProps<SVGMaskElement>>;
    marker: ForwardRefComponent<SVGMarkerElement, SVGMotionProps<SVGMarkerElement>>;
    image: ForwardRefComponent<SVGImageElement, SVGMotionProps<SVGImageElement>>;
    text: ForwardRefComponent<SVGTextElement, SVGMotionProps<SVGTextElement>>;
    circle: ForwardRefComponent<SVGCircleElement, SVGMotionProps<SVGCircleElement>>;
    svg: ForwardRefComponent<SVGSVGElement, SVGMotionProps<SVGSVGElement>>;
    animate: ForwardRefComponent<SVGElement, SVGMotionProps<SVGElement>>;
    defs: ForwardRefComponent<SVGDefsElement, SVGMotionProps<SVGDefsElement>>;
    desc: ForwardRefComponent<SVGDescElement, SVGMotionProps<SVGDescElement>>;
    ellipse: ForwardRefComponent<SVGEllipseElement, SVGMotionProps<SVGEllipseElement>>;
    feBlend: ForwardRefComponent<SVGFEBlendElement, SVGMotionProps<SVGFEBlendElement>>;
    feColorMatrix: ForwardRefComponent<SVGFEColorMatrixElement, SVGMotionProps<SVGFEColorMatrixElement>>;
    feComponentTransfer: ForwardRefComponent<SVGFEComponentTransferElement, SVGMotionProps<SVGFEComponentTransferElement>>;
    feComposite: ForwardRefComponent<SVGFECompositeElement, SVGMotionProps<SVGFECompositeElement>>;
    feConvolveMatrix: ForwardRefComponent<SVGFEConvolveMatrixElement, SVGMotionProps<SVGFEConvolveMatrixElement>>;
    feDiffuseLighting: ForwardRefComponent<SVGFEDiffuseLightingElement, SVGMotionProps<SVGFEDiffuseLightingElement>>;
    feDisplacementMap: ForwardRefComponent<SVGFEDisplacementMapElement, SVGMotionProps<SVGFEDisplacementMapElement>>;
    feDistantLight: ForwardRefComponent<SVGFEDistantLightElement, SVGMotionProps<SVGFEDistantLightElement>>;
    feDropShadow: ForwardRefComponent<SVGFEDropShadowElement, SVGMotionProps<SVGFEDropShadowElement>>;
    feFlood: ForwardRefComponent<SVGFEFloodElement, SVGMotionProps<SVGFEFloodElement>>;
    feFuncA: ForwardRefComponent<SVGFEFuncAElement, SVGMotionProps<SVGFEFuncAElement>>;
    feFuncB: ForwardRefComponent<SVGFEFuncBElement, SVGMotionProps<SVGFEFuncBElement>>;
    feFuncG: ForwardRefComponent<SVGFEFuncGElement, SVGMotionProps<SVGFEFuncGElement>>;
    feFuncR: ForwardRefComponent<SVGFEFuncRElement, SVGMotionProps<SVGFEFuncRElement>>;
    feGaussianBlur: ForwardRefComponent<SVGFEGaussianBlurElement, SVGMotionProps<SVGFEGaussianBlurElement>>;
    feImage: ForwardRefComponent<SVGFEImageElement, SVGMotionProps<SVGFEImageElement>>;
    feMerge: ForwardRefComponent<SVGFEMergeElement, SVGMotionProps<SVGFEMergeElement>>;
    feMergeNode: ForwardRefComponent<SVGFEMergeNodeElement, SVGMotionProps<SVGFEMergeNodeElement>>;
    feMorphology: ForwardRefComponent<SVGFEMorphologyElement, SVGMotionProps<SVGFEMorphologyElement>>;
    feOffset: ForwardRefComponent<SVGFEOffsetElement, SVGMotionProps<SVGFEOffsetElement>>;
    fePointLight: ForwardRefComponent<SVGFEPointLightElement, SVGMotionProps<SVGFEPointLightElement>>;
    feSpecularLighting: ForwardRefComponent<SVGFESpecularLightingElement, SVGMotionProps<SVGFESpecularLightingElement>>;
    feSpotLight: ForwardRefComponent<SVGFESpotLightElement, SVGMotionProps<SVGFESpotLightElement>>;
    feTile: ForwardRefComponent<SVGFETileElement, SVGMotionProps<SVGFETileElement>>;
    feTurbulence: ForwardRefComponent<SVGFETurbulenceElement, SVGMotionProps<SVGFETurbulenceElement>>;
    foreignObject: ForwardRefComponent<SVGForeignObjectElement, SVGMotionProps<SVGForeignObjectElement>>;
    g: ForwardRefComponent<SVGGElement, SVGMotionProps<SVGGElement>>;
    line: ForwardRefComponent<SVGLineElement, SVGMotionProps<SVGLineElement>>;
    linearGradient: ForwardRefComponent<SVGLinearGradientElement, SVGMotionProps<SVGLinearGradientElement>>;
    metadata: ForwardRefComponent<SVGMetadataElement, SVGMotionProps<SVGMetadataElement>>;
    path: ForwardRefComponent<SVGPathElement, SVGMotionProps<SVGPathElement>>;
    pattern: ForwardRefComponent<SVGPatternElement, SVGMotionProps<SVGPatternElement>>;
    polygon: ForwardRefComponent<SVGPolygonElement, SVGMotionProps<SVGPolygonElement>>;
    polyline: ForwardRefComponent<SVGPolylineElement, SVGMotionProps<SVGPolylineElement>>;
    radialGradient: ForwardRefComponent<SVGRadialGradientElement, SVGMotionProps<SVGRadialGradientElement>>;
    rect: ForwardRefComponent<SVGRectElement, SVGMotionProps<SVGRectElement>>;
    stop: ForwardRefComponent<SVGStopElement, SVGMotionProps<SVGStopElement>>;
    switch: ForwardRefComponent<SVGSwitchElement, SVGMotionProps<SVGSwitchElement>>;
    textPath: ForwardRefComponent<SVGTextPathElement, SVGMotionProps<SVGTextPathElement>>;
    tspan: ForwardRefComponent<SVGTSpanElement, SVGMotionProps<SVGTSpanElement>>;
    use: ForwardRefComponent<SVGUseElement, SVGMotionProps<SVGUseElement>>;
    view: ForwardRefComponent<SVGViewElement, SVGMotionProps<SVGViewElement>>;
    object: ForwardRefComponent<HTMLObjectElement, HTMLMotionProps<"object">>;
    style: ForwardRefComponent<HTMLStyleElement, HTMLMotionProps<"style">>;
    progress: ForwardRefComponent<HTMLProgressElement, HTMLMotionProps<"progress">>;
    ruby: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    table: ForwardRefComponent<HTMLTableElement, HTMLMotionProps<"table">>;
    small: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    sub: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    embed: ForwardRefComponent<HTMLEmbedElement, HTMLMotionProps<"embed">>;
    pre: ForwardRefComponent<HTMLPreElement, HTMLMotionProps<"pre">>;
    caption: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    menu: ForwardRefComponent<HTMLElement, HTMLMotionProps<"menu">>;
    button: ForwardRefComponent<HTMLButtonElement, HTMLMotionProps<"button">>;
    menuitem: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    meter: ForwardRefComponent<HTMLElement, HTMLMotionProps<"meter">>;
    textarea: ForwardRefComponent<HTMLTextAreaElement, HTMLMotionProps<"textarea">>;
    time: ForwardRefComponent<HTMLElement, HTMLMotionProps<"time">>;
    link: ForwardRefComponent<HTMLLinkElement, HTMLMotionProps<"link">>;
    dialog: ForwardRefComponent<HTMLDialogElement, HTMLMotionProps<"dialog">>;
    a: ForwardRefComponent<HTMLAnchorElement, HTMLMotionProps<"a">>;
    abbr: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    address: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    area: ForwardRefComponent<HTMLAreaElement, HTMLMotionProps<"area">>;
    article: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    aside: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    audio: ForwardRefComponent<HTMLAudioElement, HTMLMotionProps<"audio">>;
    b: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    base: ForwardRefComponent<HTMLBaseElement, HTMLMotionProps<"base">>;
    bdi: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    bdo: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    big: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    blockquote: ForwardRefComponent<HTMLElement, HTMLMotionProps<"blockquote">>;
    body: ForwardRefComponent<HTMLBodyElement, HTMLMotionProps<"body">>;
    br: ForwardRefComponent<HTMLBRElement, HTMLMotionProps<"br">>;
    canvas: ForwardRefComponent<HTMLCanvasElement, HTMLMotionProps<"canvas">>;
    cite: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    code: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    col: ForwardRefComponent<HTMLTableColElement, HTMLMotionProps<"col">>;
    colgroup: ForwardRefComponent<HTMLTableColElement, HTMLMotionProps<"colgroup">>;
    data: ForwardRefComponent<HTMLDataElement, HTMLMotionProps<"data">>;
    datalist: ForwardRefComponent<HTMLDataListElement, HTMLMotionProps<"datalist">>;
    dd: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    del: ForwardRefComponent<HTMLElement, HTMLMotionProps<"del">>;
    details: ForwardRefComponent<HTMLElement, HTMLMotionProps<"details">>;
    dfn: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    div: ForwardRefComponent<HTMLDivElement, HTMLMotionProps<"div">>;
    dl: ForwardRefComponent<HTMLDListElement, HTMLMotionProps<"dl">>;
    dt: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    em: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    fieldset: ForwardRefComponent<HTMLFieldSetElement, HTMLMotionProps<"fieldset">>;
    figcaption: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    figure: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    footer: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    form: ForwardRefComponent<HTMLFormElement, HTMLMotionProps<"form">>;
    h1: ForwardRefComponent<HTMLHeadingElement, HTMLMotionProps<"h1">>;
    h2: ForwardRefComponent<HTMLHeadingElement, HTMLMotionProps<"h1">>;
    h3: ForwardRefComponent<HTMLHeadingElement, HTMLMotionProps<"h1">>;
    h4: ForwardRefComponent<HTMLHeadingElement, HTMLMotionProps<"h1">>;
    h5: ForwardRefComponent<HTMLHeadingElement, HTMLMotionProps<"h1">>;
    h6: ForwardRefComponent<HTMLHeadingElement, HTMLMotionProps<"h1">>;
    head: ForwardRefComponent<HTMLHeadElement, HTMLMotionProps<"head">>;
    header: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    hgroup: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    hr: ForwardRefComponent<HTMLHRElement, HTMLMotionProps<"hr">>;
    html: ForwardRefComponent<HTMLHtmlElement, HTMLMotionProps<"html">>;
    i: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    iframe: ForwardRefComponent<HTMLIFrameElement, HTMLMotionProps<"iframe">>;
    img: ForwardRefComponent<HTMLImageElement, HTMLMotionProps<"img">>;
    input: ForwardRefComponent<HTMLInputElement, HTMLMotionProps<"input">>;
    ins: ForwardRefComponent<HTMLModElement, HTMLMotionProps<"ins">>;
    kbd: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    keygen: ForwardRefComponent<HTMLElement, HTMLMotionProps<"keygen">>;
    label: ForwardRefComponent<HTMLLabelElement, HTMLMotionProps<"label">>;
    legend: ForwardRefComponent<HTMLLegendElement, HTMLMotionProps<"legend">>;
    li: ForwardRefComponent<HTMLLIElement, HTMLMotionProps<"li">>;
    main: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    map: ForwardRefComponent<HTMLMapElement, HTMLMotionProps<"map">>;
    mark: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    meta: ForwardRefComponent<HTMLMetaElement, HTMLMotionProps<"meta">>;
    nav: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    noscript: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    ol: ForwardRefComponent<HTMLOListElement, HTMLMotionProps<"ol">>;
    optgroup: ForwardRefComponent<HTMLOptGroupElement, HTMLMotionProps<"optgroup">>;
    option: ForwardRefComponent<HTMLOptionElement, HTMLMotionProps<"option">>;
    output: ForwardRefComponent<HTMLElement, HTMLMotionProps<"output">>;
    p: ForwardRefComponent<HTMLParagraphElement, HTMLMotionProps<"p">>;
    param: ForwardRefComponent<HTMLParamElement, HTMLMotionProps<"param">>;
    picture: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    q: ForwardRefComponent<HTMLQuoteElement, HTMLMotionProps<"q">>;
    rp: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    rt: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    s: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    samp: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    script: ForwardRefComponent<HTMLScriptElement, HTMLMotionProps<"script">>;
    section: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    select: ForwardRefComponent<HTMLSelectElement, HTMLMotionProps<"select">>;
    source: ForwardRefComponent<HTMLSourceElement, HTMLMotionProps<"source">>;
    span: ForwardRefComponent<HTMLSpanElement, HTMLMotionProps<"span">>;
    strong: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    summary: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    sup: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    tbody: ForwardRefComponent<HTMLTableSectionElement, HTMLMotionProps<"tbody">>;
    td: ForwardRefComponent<HTMLTableDataCellElement, HTMLMotionProps<"td">>;
    tfoot: ForwardRefComponent<HTMLTableSectionElement, HTMLMotionProps<"tbody">>;
    th: ForwardRefComponent<HTMLTableHeaderCellElement, HTMLMotionProps<"th">>;
    thead: ForwardRefComponent<HTMLTableSectionElement, HTMLMotionProps<"tbody">>;
    title: ForwardRefComponent<HTMLTitleElement, HTMLMotionProps<"title">>;
    tr: ForwardRefComponent<HTMLTableRowElement, HTMLMotionProps<"tr">>;
    track: ForwardRefComponent<HTMLTrackElement, HTMLMotionProps<"track">>;
    u: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    ul: ForwardRefComponent<HTMLUListElement, HTMLMotionProps<"ul">>;
    var: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    video: ForwardRefComponent<HTMLVideoElement, HTMLMotionProps<"video">>;
    wbr: ForwardRefComponent<HTMLElement, HTMLMotionProps<"ruby">>;
    webview: ForwardRefComponent<HTMLWebViewElement, HTMLMotionProps<"webview">>;
    /**
     * Convert a custom React component into a `motion` component.
     *
     * It can also accept a string, to create [custom DOM elements](https://developer.mozilla.org/en-US/docs/Web/Web_Components/Using_custom_elements).
     *
     * ```jsx
     * const Component = React.forwardRef((props: Props, ref) => {
     *   return <div ref={ref} />
     * })
     *
     * const MotionComponent = motion.custom<Props>(Component)
     * ```
     *
     * @param Component -
     */
    custom: <Props>(Component: string | React.ComponentClass<Props, any> | React.FunctionComponent<Props>) => React.ForwardRefExoticComponent<React.PropsWithoutRef<Props & MotionProps> & React.RefAttributes<Element>>;
};

/**
 * @public
 */
export declare interface MotionAdvancedProps {
    /**
     * Custom data to use to resolve dynamic variants differently for each animating component.
     *
     * @library
     *
     * ```jsx
     * const variants = {
     *   visible: (custom) => ({
     *     opacity: 1,
     *     transition: { delay: custom * 0.2 }
     *   })
     * }
     *
     * <Frame custom={0} animate="visible" variants={variants} />
     * <Frame custom={1} animate="visible" variants={variants} />
     * <Frame custom={2} animate="visible" variants={variants} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * const variants = {
     *   visible: (custom) => ({
     *     opacity: 1,
     *     transition: { delay: custom * 0.2 }
     *   })
     * }
     *
     * <motion.div custom={0} animate="visible" variants={variants} />
     * <motion.div custom={1} animate="visible" variants={variants} />
     * <motion.div custom={2} animate="visible" variants={variants} />
     * ```
     *
     * @public
     */
    custom?: any;
    /**
     * @public
     * Set to `false` to prevent inheriting variant changes from its parent.
     */
    inherit?: boolean;
    /**
     * @internal
     * Set to `true` to block rendering motion values (`animate`, gestures, etcetera)
     * on the component. This can be used to temporarily disable animations for performance reasons.
     */
    static?: boolean;
}

/**
 * @public
 */
export declare interface MotionCallbacks {
    /**
     * Callback with latest motion values, fired max once per frame.
     *
     * @library
     *
     * ```jsx
     * function onUpdate(latest) {
     *   console.log(latest.x, latest.opacity)
     * }
     *
     * <Frame animate={{ x: 100, opacity: 0 }} onUpdate={onUpdate} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * function onUpdate(latest) {
     *   console.log(latest.x, latest.opacity)
     * }
     *
     * <motion.div animate={{ x: 100, opacity: 0 }} onUpdate={onUpdate} />
     * ```
     */
    onUpdate?(latest: {
        [key: string]: string | number;
    }): void;
    /**
     * Callback when animation defined in `animate` begins.
     *
     * @library
     *
     * ```jsx
     * function onStart() {
     *   console.log("Animation completed")
     * }
     *
     * <Frame animate={{ x: 100 }} onAnimationStart={onStart} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * function onStart() {
     *   console.log("Animation completed")
     * }
     *
     * <motion.div animate={{ x: 100 }} onAnimationStart={onStart} />
     * ```
     */
    onAnimationStart?(): void;
    /**
     * Callback when animation defined in `animate` is complete.
     *
     * @library
     *
     * ```jsx
     * function onComplete() {
     *   console.log("Animation completed")
     * }
     *
     * <Frame animate={{ x: 100 }} onAnimationComplete={onComplete} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * function onComplete() {
     *   console.log("Animation completed")
     * }
     *
     * <motion.div animate={{ x: 100 }} onAnimationComplete={onComplete} />
     * ```
     */
    onAnimationComplete?(): void;
}

declare interface MotionComponentConfig {
    loadFunctionalityComponents: LoadFunctionalityComponents;
    renderComponent: RenderComponent;
    getValueControlsConfig: (ref: RefObject<any>, values: MotionValuesMap) => ValueAnimationConfig;
}

/**
 * @internal
 */
export declare const MotionContext: React.Context<MotionContextProps>;

declare interface MotionContextProps {
    controls?: ValueAnimationControls;
    values?: MotionValuesMap;
    initial?: false | VariantLabels;
    animate?: VariantLabels;
    static?: boolean;
    hasMounted?: RefObject<boolean>;
    isReducedMotion?: boolean | undefined;
}

declare type MotionCSS = MakeMotion<Omit<CSSProperties, "rotate" | "scale" | "perspective">>;

/**
 * @internal
 */
export declare const MotionPluginContext: React.Context<MotionPluginsContext>;

declare interface MotionPluginProps extends MotionPluginsContext {
    children?: ReactNode;
}

/**
 * @remarks For now I think this should remain a private API for our own use
 * until we can figure out a nicer way of allowing people to add these
 *
 * @internal
 */
export declare function MotionPlugins({ children, ...props }: MotionPluginProps): JSX.Element;

declare interface MotionPluginsContext {
    transformPagePoint: (point: Point) => Point;
}

/**
 * Props for `motion` components.
 *
 * @public
 */
export declare interface MotionProps extends AnimationProps, MotionCallbacks, GestureHandlers, DraggableProps, MotionAdvancedProps {
    /**
     * Properties, variant label or array of variant labels to start in.
     *
     * Set to `false` to initialise with the values in `animate` (disabling the mount animation)
     *
     * @library
     *
     * ```jsx
     * // As values
     * <Frame initial={{ opacity: 1 }} />
     *
     * // As variant
     * <Frame initial="visible" variants={variants} />
     *
     * // Multiple variants
     * <Frame initial={["visible", "active"]} variants={variants} />
     *
     * // As false (disable mount animation)
     * <Frame initial={false} animate={{ opacity: 0 }} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * // As values
     * <motion.div initial={{ opacity: 1 }} />
     *
     * // As variant
     * <motion.div initial="visible" variants={variants} />
     *
     * // Multiple variants
     * <motion.div initial={["visible", "active"]} variants={variants} />
     *
     * // As false (disable mount animation)
     * <motion.div initial={false} animate={{ opacity: 0 }} />
     * ```
     */
    initial?: boolean | Target | VariantLabels;
    /**
     * @library
     *
     * The React DOM `style` prop, useful for setting CSS properties that aren't explicitly exposed by `Frame` props.
     *
     * ```jsx
     * <Frame style={{ mixBlendMode: "difference" }}  />
     * ```
     *
     * @motion
     *
     * The React DOM `style` prop, enhanced with support for `MotionValue`s and separate `transform` values.
     *
     * ```jsx
     * export const MyComponent = () => {
     *   const x = useMotionValue(0)
     *
     *   return <motion.div style={{ x, opacity: 1, scale: 0.5 }} />
     * }
     * ```
     */
    style?: MotionStyle;
    /**
     * By default, Framer Motion generates a `transform` property with a sensible transform order. `transformTemplate`
     * can be used to create a different order, or to append/preprend the automatically generated `transform` property.
     *
     * @library
     *
     * ```jsx
     * function transformTemplate({ x, rotate }) {
     *   return `rotate(${rotate}deg) translateX(${x}px)`
     * }
     *
     * <Frame x={0} rotate={180} transformTemplate={transformTemplate} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   style={{ x: 0, rotate: 180 }}
     *   transformTemplate={
     *     ({ x, rotate }) => `rotate(${rotate}deg) translateX(${x}px)`
     *   }
     * />
     * ```
     *
     * @param transform - The latest animated transform props.
     * @param generatedTransform - The transform string as automatically generated by Framer Motion
     *
     * @public
     */
    transformTemplate?(transform: TransformProperties, generatedTransform: string): string;
    /**
     * This allows values to be transformed before being animated or set as styles.
     *
     * For instance, this allows custom values in Framer Library like `size` to be converted into `width` and `height`.
     * It also allows us a chance to take a value like `Color` and convert it to an animatable color string.
     *
     * A few structural typing changes need making before this can be a public property:
     * - Allow `Target` values to be appended by user-defined types (delete `CustomStyles` - does `size` throw a type error?)
     * - Extract `CustomValueType` as a separate user-defined type (delete `CustomValueType` and animate a `Color` - does this throw a type error?).
     *
     * @param values -
     *
     * @internal
     */
    transformValues?<V extends any>(values: V): V;
}

/**
 * @public
 */
export declare type MotionStyle = MotionCSS & MotionTransform & MakeMotion<SVGPathProperties> & MakeCustomValueType<CustomStyles>;

/**
 * @public
 */
export declare type MotionTransform = MakeMotion<TransformProperties>;

/**
 * `MotionValue` is used to track the state and velocity of motion values.
 *
 * @public
 */
export declare class MotionValue<V = any> {
    /**
     * The current state of the `MotionValue`.
     *
     * @internal
     */
    private current;
    /**
     * The previous state of the `MotionValue`.
     *
     * @internal
     */
    private prev;
    /**
     * Duration, in milliseconds, since last updating frame.
     *
     * @internal
     */
    private timeDelta;
    /**
     * Timestamp of the last time this `MotionValue` was updated.
     *
     * @internal
     */
    private lastUpdated;
    /**
     * Collection of children `MotionValue`s to notify of updates.
     *
     * @internal
     */
    private children?;
    /**
     * A reference to this `MotionValue`'s parent.
     *
     * @internal
     */
    private parent?;
    /**
     * Functions to notify when the `MotionValue` updates.
     *
     * @internal
     */
    updateSubscribers?: Set<Subscriber<V>>;
    /**
     * Functions to notify when the `MotionValue` updates and `render` is set to `true`.
     *
     * @internal
     */
    private renderSubscribers?;
    /**
     * Add a passive effect to this `MotionValue`.
     *
     * A passive effect intercepts calls to `set`. For instance, `useSpring` adds
     * a passive effect that attaches a `spring` to the latest
     * set value. Hypothetically there could be a `useSmooth` that attaches an input smoothing effect.
     *
     * @internal
     */
    private passiveEffect?;
    /**
     * If defined, new values passed into `set` will be transformed through this function before being set.
     *
     * @internal
     */
    private transformer?;
    /**
     * A reference to the currently-controlling Popmotion animation
     *
     * @internal
     */
    private stopAnimation?;
    /**
     * Tracks whether this value can output a velocity. Currently this is only true
     * if the value is numerical, but we might be able to widen the scope here and support
     * other value types.
     *
     * @internal
     */
    private canTrackVelocity;
    /**
     * @param init - The initiating value
     * @param config - Optional configuration options
     *
     * -  `transformer`: A function to transform incoming values with.
     *
     * @internal
     */
    constructor(init: V, { transformer, parent }?: Config<V>);
    /**
     * Creates a new `MotionValue` that's subscribed to the output of this one.
     *
     * @param config - Optional configuration options
     *
     * -  `transformer`: A function to transform incoming values with.
     *
     * @internal
     */
    addChild(config?: Config<V>): MotionValue<V>;
    /**
     * Stops a `MotionValue` from being subscribed to this one.
     *
     * @param child - The subscribed `MotionValue`
     *
     * @internal
     */
    removeChild(child: MotionValue): void;
    /**
     * Subscribes a subscriber function to a subscription list.
     *
     * @param subscriptions - A `Set` of subscribers.
     * @param subscription - A subscriber function.
     */
    private subscribeTo;
    /**
     * Adds a function that will be notified when the `MotionValue` is updated.
     *
     * It returns a function that, when called, will cancel the subscription.
     *
     * When calling `onChange` inside a React component, it should be wrapped with the
     * `useEffect` hook. As it returns an unsubscribe function, this should be returned
     * from the `useEffect` function to ensure you don't add duplicate subscribers..
     *
     * @library
     *
     * ```jsx
     * function MyComponent() {
     *   const x = useMotionValue(0)
     *   const y = useMotionValue(0)
     *   const opacity = useMotionValue(1)
     *
     *   useEffect(() => {
     *     function updateOpacity() {
     *       const maxXY = Math.max(x.get(), y.get())
     *       const newOpacity = transform(maxXY, [0, 100], [1, 0])
     *       opacity.set(newOpacity)
     *     }
     *
     *     const unsubscribeX = x.onChange(updateOpacity)
     *     const unsubscribeY = y.onChange(updateOpacity)
     *
     *     return () => {
     *       unsubscribeX()
     *       unsubscribeY()
     *     }
     *   }, [])
     *
     *   return <Frame x={x} />
     * }
     * ```
     *
     * @motion
     *
     * ```jsx
     * export const MyComponent = () => {
     *   const x = useMotionValue(0)
     *   const y = useMotionValue(0)
     *   const opacity = useMotionValue(1)
     *
     *   useEffect(() => {
     *     function updateOpacity() {
     *       const maxXY = Math.max(x.get(), y.get())
     *       const newOpacity = transform(maxXY, [0, 100], [1, 0])
     *       opacity.set(newOpacity)
     *     }
     *
     *     const unsubscribeX = x.onChange(updateOpacity)
     *     const unsubscribeY = y.onChange(updateOpacity)
     *
     *     return () => {
     *       unsubscribeX()
     *       unsubscribeY()
     *     }
     *   }, [])
     *
     *   return <motion.div style={{ x }} />
     * }
     * ```
     *
     * @internalremarks
     *
     * We could look into a `useOnChange` hook if the above lifecycle management proves confusing.
     *
     * ```jsx
     * useOnChange(x, () => {})
     * ```
     *
     * @param subscriber - A function that receives the latest value.
     * @returns A function that, when called, will cancel this subscription.
     *
     * @public
     */
    onChange(subscription: Subscriber<V>): () => void;
    /**
     * Adds a function that will be notified when the `MotionValue` requests a render.
     *
     * @param subscriber - A function that's provided the latest value.
     * @returns A function that, when called, will cancel this subscription.
     *
     * @internal
     */
    onRenderRequest(subscription: Subscriber<V>): () => boolean;
    /**
     * Attaches a passive effect to the `MotionValue`.
     *
     * @internal
     */
    attach(passiveEffect: PassiveEffect<V>): void;
    /**
     * Sets the state of the `MotionValue`.
     *
     * @remarks
     *
     * ```jsx
     * const x = useMotionValue(0)
     * x.set(10)
     * ```
     *
     * @param latest - Latest value to set.
     * @param render - Whether to notify render subscribers. Defaults to `true`
     *
     * @public
     */
    set(v: V, render?: boolean): void;
    updateAndNotify: (v: V, render?: boolean) => void;
    /**
     * Returns the latest state of `MotionValue`
     *
     * @returns - The latest state of `MotionValue`
     *
     * @public
     */
    get(): V;
    /**
     * Returns the latest velocity of `MotionValue`
     *
     * @returns - The latest velocity of `MotionValue`. Returns `0` if the state is non-numerical.
     *
     * @public
     */
    getVelocity(): number;
    /**
     * Notify a subscriber with the latest value.
     *
     * This is an instanced and bound function to prevent generating a new
     * function once per frame.
     *
     * @param subscriber - The subscriber to notify.
     *
     * @internal
     */
    private notifySubscriber;
    /**
     * Schedule a velocity check for the next frame.
     *
     * This is an instanced and bound function to prevent generating a new
     * function once per frame.
     *
     * @internal
     */
    private scheduleVelocityCheck;
    /**
     * Updates `prev` with `current` if the value hasn't been updated this frame.
     * This ensures velocity calculations return `0`.
     *
     * This is an instanced and bound function to prevent generating a new
     * function once per frame.
     *
     * @internal
     */
    private velocityCheck;
    /**
     * Updates child `MotionValue`.
     *
     * @param child - Child `MotionValue`.
     *
     * @internal
     */
    private setChild;
    /**
     * Registers a new animation to control this `MotionValue`. Only one
     * animation can drive a `MotionValue` at one time.
     *
     * ```jsx
     * value.start()
     * ```
     *
     * @param animation - A function that starts the provided animation
     *
     * @internal
     */
    start(animation: StartAnimation): Promise<void>;
    /**
     * Stop the currently active animation.
     *
     * @public
     */
    stop(): void;
    /**
     * Returns `true` if this value is currently animating.
     *
     * @public
     */
    isAnimating(): boolean;
    private clearAnimation;
    /**
     * Destroy and clean up subscribers to this `MotionValue`.
     *
     * The `MotionValue` hooks like `useMotionValue` and `useTransform` automatically
     * handle the lifecycle of the returned `MotionValue`, so this method is only necessary if you've manually
     * created a `MotionValue` via the `motionValue` function.
     *
     * @public
     */
    destroy(): void;
}

/**
 * @internal
 */
export declare function motionValue<V>(init: V, opts?: Config<V>): MotionValue<V>;

declare class MotionValuesMap {
    private hasMounted;
    private transformTemplate;
    private onUpdate?;
    private values;
    private unsubscribers;
    private output;
    has(key: string): boolean;
    set(key: string, value: MotionValue): void;
    get<Value>(key: string): MotionValue<Value> | undefined;
    get<Value>(key: string, defaultValue: Value): MotionValue<Value>;
    forEach(callback: (value: MotionValue, key: string) => void): void;
    private bindValueToOutput;
    setOnUpdate(onUpdate?: OnUpdate): void;
    setTransformTemplate(transformTemplate?: TransformTemplate | undefined): void;
    getTransformTemplate(): TransformTemplate | undefined;
    updateTransformTemplate(): void;
    mount(output?: Output): void;
    unmount(): void;
}

/**
 * @public
 */
export declare interface None {
    /**
     * Set `type` to `false` for an instant transition.
     *
     * @public
     */
    type: false;
    /**
     * @internal
     */
    from?: number | string;
    /**
     * @internal
     */
    delay?: number;
    /**
     * @internal
     */
    velocity?: number;
}

declare type NotPresent = [false, () => void];

declare type Omit<T, K extends keyof T> = Pick<T, Exclude<keyof T, K>>;

declare type OnUpdate = (v: Target) => void;

/**
 * Options for orchestrating the timing of animations.
 *
 * @public
 */
export declare interface Orchestration {
    /**
     * Delay the animation by this duration (in seconds). Defaults to `0`.
     *
     * @remarks
     * ```javascript
     * const transition = {
     *   delay: 0.2
     * }
     * ```
     *
     * @public
     */
    delay?: number;
    /**
     * Describes the relationship between the transition and its children. Set
     * to `false` by default.
     *
     * @remarks
     * When using variants, the transition can be scheduled in relation to its
     * children with either `"beforeChildren"` to finish this transition before
     * starting children transitions, `"afterChildren"` to finish children
     * transitions before starting this transition.
     *
     * @library
     *
     * ```jsx
     * const container = {
     *   hidden: {
     *     opacity: 0,
     *     transition: { when: "afterChildren" }
     *   }
     * }
     *
     * const item = {
     *   hidden: {
     *     opacity: 0,
     *     transition: { duration: 2 }
     *   }
     * }
     *
     * return (
     *   <Frame variants={container} animate="hidden">
     *     <Frame variants={item} size={50} />
     *     <Frame variants={item} size={50} />
     *   </Frame>
     * )
     * ```
     *
     * @motion
     *
     * ```jsx
     * const list = {
     *   hidden: {
     *     opacity: 0,
     *     transition: { when: "afterChildren" }
     *   }
     * }
     *
     * const item = {
     *   hidden: {
     *     opacity: 0,
     *     transition: { duration: 2 }
     *   }
     * }
     *
     * return (
     *   <motion.ul variants={list} animate="hidden">
     *     <motion.li variants={item} />
     *     <motion.li variants={item} />
     *   </motion.ul>
     * )
     * ```
     *
     * @public
     */
    when?: false | "beforeChildren" | "afterChildren" | string;
    /**
     * When using variants, children animations will start after this duration
     * (in seconds). You can add the `transition` property to both the `Frame` and the `variant` directly. Adding it to the `variant` generally offers more flexibility, as it allows you to customize the delay per visual state.
     *
     * @library
     *
     * ```jsx
     * const container = {
     *   hidden: { opacity: 0 },
     *   show: {
     *     opacity: 1,
     *     transition: {
     *       delayChildren: 0.5
     *     }
     *   }
     * }
     *
     * const item = {
     *   hidden: { opacity: 0 },
     *   show: { opacity: 1 }
     * }
     *
     * return (
     *   <Frame
     *     variants={container}
     *     initial="hidden"
     *     animate="show"
     *   >
     *     <Frame variants={item} size={50} />
     *     <Frame variants={item} size={50} />
     *   </Frame>
     * )
     * ```
     *
     * @motion
     *
     * ```jsx
     * const container = {
     *   hidden: { opacity: 0 },
     *   show: {
     *     opacity: 1,
     *     transition: {
     *       delayChildren: 0.5
     *     }
     *   }
     * }
     *
     * const item = {
     *   hidden: { opacity: 0 },
     *   show: { opacity: 1 }
     * }
     *
     * return (
     *   <motion.ul
     *     variants={container}
     *     initial="hidden"
     *     animate="show"
     *   >
     *     <motion.li variants={item} />
     *     <motion.li variants={item} />
     *   </motion.ul>
     * )
     * ```
     *
     * @public
     */
    delayChildren?: number;
    /**
     * When using variants, animations of child components can be staggered by this
     * duration (in seconds).
     *
     * For instance, if `staggerChildren` is `0.01`, the first child will be
     * delayed by `0` seconds, the second by `0.01`, the third by `0.02` and so
     * on.
     *
     * The calculated stagger delay will be added to `delayChildren`.
     *
     * @library
     *
     * ```jsx
     * const container = {
     *   hidden: { opacity: 0 },
     *   show: {
     *     opacity: 1,
     *     transition: {
     *       staggerChildren: 0.5
     *     }
     *   }
     * }
     *
     * const item = {
     *   hidden: { opacity: 0 },
     *   show: { opacity: 1 }
     * }
     *
     * return (
     *   <Frame
     *     variants={container}
     *     initial="hidden"
     *     animate="show"
     *   >
     *     <Frame variants={item} size={50} />
     *     <Frame variants={item} size={50} />
     *   </Frame>
     * )
     * ```
     *
     * @motion
     *
     * ```jsx
     * const container = {
     *   hidden: { opacity: 0 },
     *   show: {
     *     opacity: 1,
     *     transition: {
     *       staggerChildren: 0.5
     *     }
     *   }
     * }
     *
     * const item = {
     *   hidden: { opacity: 0 },
     *   show: { opacity: 1 }
     * }
     *
     * return (
     *   <motion.ol
     *     variants={container}
     *     initial="hidden"
     *     animate="show"
     *   >
     *     <motion.li variants={item} />
     *     <motion.li variants={item} />
     *   </motion.ol>
     * )
     * ```
     *
     * @public
     */
    staggerChildren?: number;
    /**
     * The direction in which to stagger children.
     *
     * A value of `1` staggers from the first to the last while `-1`
     * staggers from the last to the first.
     *
     * @library
     *
     * ```jsx
     * const container = {
     *   hidden: { opacity: 0 },
     *   show: {
     *     opacity: 1,
     *     transition: {
     *       delayChildren: 0.5,
     *       staggerDirection: -1
     *     }
     *   }
     * }
     *
     * const item = {
     *   hidden: { opacity: 0 },
     *   show: { opacity: 1 }
     * }
     *
     * return (
     *   <Frame
     *     variants={container}
     *     initial="hidden"
     *     animate="show"
     *   >
     *     <Frame variants={item} size={50} />
     *     <Frame variants={item} size={50} />
     *   </Frame>
     * )
     * ```
     *
     * @motion
     *
     * ```jsx
     * const container = {
     *   hidden: { opacity: 0 },
     *   show: {
     *     opacity: 1,
     *     transition: {
     *       delayChildren: 0.5,
     *       staggerDirection: -1
     *     }
     *   }
     * }
     *
     * const item = {
     *   hidden: { opacity: 0 },
     *   show: { opacity: 1 }
     * }
     *
     * return (
     *   <motion.ul
     *     variants={container}
     *     initial="hidden"
     *     animate="show"
     *   >
     *     <motion.li variants={item} size={50} />
     *     <motion.li variants={item} size={50} />
     *   </motion.ul>
     * )
     * ```
     *
     * @public
     */
    staggerDirection?: number;
}

declare type Output = (key: string, value: string | number | TransformTemplate | undefined) => void;

/**
 * @public
 */
export declare interface PanHandlers {
    /**
     * Callback function that fires when the pan gesture is recognised on this element.
     *
     * @library
     *
     * ```jsx
     * function onPan(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <Frame onPan={onPan} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * function onPan(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <motion.div onPan={onPan} />
     * ```
     *
     * @param event - The originating pointer event.
     * @param info - A {@link PanInfo} object containing `x` and `y` values for:
     *
     *   - `point`: Relative to the device or page.
     *   - `delta`: Distance moved since the last event.
     *   - `offset`: Offset from the original pan event.
     *   - `velocity`: Current velocity of the pointer.
     */
    onPan?(event: MouseEvent | TouchEvent | PointerEvent, info: PanInfo): void;
    /**
     * Callback function that fires when the pan gesture begins on this element.
     *
     * @library
     *
     * ```jsx
     * function onPanStart(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <Frame onPanStart={onPanStart} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * function onPanStart(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <motion.div onPanStart={onPanStart} />
     * ```
     *
     * @param event - The originating pointer event.
     * @param info - A {@link PanInfo} object containing `x`/`y` values for:
     *
     *   - `point`: Relative to the device or page.
     *   - `delta`: Distance moved since the last event.
     *   - `offset`: Offset from the original pan event.
     *   - `velocity`: Current velocity of the pointer.
     */
    onPanStart?(event: MouseEvent | TouchEvent | PointerEvent, info: PanInfo): void;
    /**
     * Callback function that fires when we begin detecting a pan gesture. This
     * is analogous to `onMouseStart` or `onTouchStart`.
     *
     * @library
     *
     * ```jsx
     * function onPanSessionStart(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <Frame onPanSessionStart={onPanSessionStart} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * function onPanSessionStart(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <motion.div onPanSessionStart={onPanSessionStart} />
     * ```
     *
     * @param event - The originating pointer event.
     * @param info - An {@link EventInfo} object containing `x`/`y` values for:
     *
     *   - `point`: Relative to the device or page.
     */
    onPanSessionStart?(event: MouseEvent | TouchEvent | PointerEvent, info: EventInfo): void;
    /**
     * Callback function that fires when the pan gesture ends on this element.
     *
     * @library
     *
     * ```jsx
     * function onPanEnd(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <Frame onPanEnd={onPanEnd} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * function onPanEnd(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <motion.div onPanEnd={onPanEnd} />
     * ```
     *
     * @param event - The originating pointer event.
     * @param info - A {@link PanInfo} object containing `x`/`y` values for:
     *
     *   - `point`: Relative to the device or page.
     *   - `delta`: Distance moved since the last event.
     *   - `offset`: Offset from the original pan event.
     *   - `velocity`: Current velocity of the pointer.
     */
    onPanEnd?(event: MouseEvent | TouchEvent | PointerEvent, info: PanInfo): void;
}

/**
 * Passed in to pan event handlers like `onPan` the `PanInfo` object contains
 * information about the current state of the tap gesture such as its
 * `point`, `delta`, `offset` and `velocity`.
 *
 * @library
 *
 * ```jsx
 * function onPan(event, info) {
 *   console.log(info.point.x, info.point.y)
 * }
 *
 * <Frame onPan={onPan} />
 * ```
 *
 * @motion
 *
 * ```jsx
 * <motion.div onPan={(event, info) => {
 *   console.log(info.point.x, info.point.y)
 * }} />
 * ```
 *
 * @public
 */
export declare interface PanInfo {
    /**
     * Contains `x` and `y` values for the current pan position relative
     * to the device or page.
     *
     * @library
     *
     * ```jsx
     * function onPan(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <Frame onPan={onPan} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * function onPan(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <motion.div onPan={onPan} />
     * ```
     *
     * @public
     */
    point: Point;
    /**
     * Contains `x` and `y` values for the distance moved since
     * the last event.
     *
     * @library
     *
     * ```jsx
     * function onPan(event, info) {
     *   console.log(info.delta.x, info.delta.y)
     * }
     *
     * <Frame onPan={onPan} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * function onPan(event, info) {
     *   console.log(info.delta.x, info.delta.y)
     * }
     *
     * <motion.div onPan={onPan} />
     * ```
     *
     * @public
     */
    delta: Point;
    /**
     * Contains `x` and `y` values for the distance moved from
     * the first pan event.
     *
     * @library
     *
     * ```jsx
     * function onPan(event, info) {
     *   console.log(info.offset.x, info.offset.y)
     * }
     *
     * <Frame onPan={onPan} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * function onPan(event, info) {
     *   console.log(info.offset.x, info.offset.y)
     * }
     *
     * <motion.div onPan={onPan} />
     * ```
     *
     * @public
     */
    offset: Point;
    /**
     * Contains `x` and `y` values for the current velocity of the pointer.
     *
     * @library
     *
     * ```jsx
     * function onPan(event, info) {
     *   console.log(info.velocity.x, info.velocity.y)
     * }
     *
     * <Frame onPan={onPan} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * function onPan(event, info) {
     *   console.log(info.velocity.x, info.velocity.y)
     * }
     *
     * <motion.div onPan={onPan} />
     * ```
     *
     * @public
     */
    velocity: Point;
}

/**
 * @public
 */
export declare type PassiveEffect<T> = (v: T, safeSetter: (v: T) => void) => void;

declare type PermissiveTransitionDefinition = {
    [key: string]: any;
};

/** @public */
export declare interface Point {
    x: number;
    y: number;
}

/** @public */
export declare namespace Point {
    /** @beta */
    const subtract: (a: Point, b: Point) => Point;
    /** @beta */
    const relativeTo: (idOrElem: string | HTMLElement) => ({ x, y }: Point) => Point | undefined;
}

declare type Present = [true];

declare interface Props {
    children?: any;
    /**
     * Can be used to explicitly set whether we're in reduced motion mode. Set
     * as undefined to resume device detection.
     */
    enabled?: boolean | undefined;
}

declare type ReadValueFromSource = (key: string) => number | string;

/**
 * Define accessibility options for a tree. Can be used to force the tree into Reduced Motion mode,
 * or disable device detection.
 *
 * @internal
 */
export declare function ReducedMotion({ children, enabled }: Props): JSX.Element;

/**
 * @public
 */
export declare interface RelayoutInfo {
    delta: {
        x: number;
        y: number;
        width: number;
        height: number;
    };
}

declare type RenderComponent<P = {}> = (ref: RefObject<Element>, style: CSSProperties, values: MotionValuesMap, props: P, isStatic?: boolean) => ReactElement;

/**
 * @public
 */
export declare type ResolvedKeyframesTarget = [null, ...number[]] | number[] | [null, ...string[]] | string[];

/**
 * @public
 */
export declare type ResolvedSingleTarget = string | number;

/**
 * @public
 */
export declare type ResolvedValueTarget = ResolvedSingleTarget | ResolvedKeyframesTarget;

/**
 * @public
 */
export declare type ResolveLayoutTransition = (info: RelayoutInfo) => Transition | boolean;

declare interface ScaleMotionValues {
    scaleX: MotionValue<number>;
    scaleY: MotionValue<number>;
}

export declare interface ScrollMotionValues {
    scrollX: MotionValue<number>;
    scrollY: MotionValue<number>;
    scrollXProgress: MotionValue<number>;
    scrollYProgress: MotionValue<number>;
}

/**
 * @public
 */
export declare type SingleTarget = ResolvedSingleTarget | CustomValueType;

/**
 * An animation that simulates spring physics for realistic motion.
 * This is the default animation for physical values like `x`, `y`, `scale` and `rotate`.
 *
 * @public
 */
export declare interface Spring {
    /**
     * Set `type` to `"spring"` to animate using spring physics for natural
     * movement. Type is set to `"spring"` by default.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   type: "spring"
     * }
     *
     * <Frame
     *   animate={{ rotate: 180 }}
     *   transition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   animate={{ rotate: 180 }}
     *   transition={{ type: 'spring' }}
     * />
     * ```
     *
     * @public
     */
    type: "spring";
    /**
     * Stiffness of the spring. Higher values will create more sudden movement.
     * Set to `100` by default.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   type: "spring",
     *   stiffness: 50
     * }
     *
     * <Frame
     *   animate={{ rotate: 180 }}
     *   transition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.section
     *   animate={{ rotate: 180 }}
     *   transition={{ type: 'spring', stiffness: 50 }}
     * />
     * ```
     *
     * @public
     */
    stiffness?: number;
    /**
     * Strength of opposing force. If set to 0, spring will oscillate
     * indefinitely. Set to `10` by default.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   type: "spring",
     *   damping: 300
     * }
     *
     * <Frame
     *   animate={{ rotate: 180 }}
     *   transition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.a
     *   animate={{ rotate: 180 }}
     *   transition={{ type: 'spring', damping: 300 }}
     * />
     * ```
     *
     * @public
     */
    damping?: number;
    /**
     * Mass of the moving object. Higher values will result in more lethargic
     * movement. Set to `1` by default.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   type: "spring",
     *   mass: 0.5
     * }
     *
     * <Frame
     *   animate={{ rotate: 180 }}
     *   transition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.feTurbulence
     *   animate={{ baseFrequency: 0.5 } as any}
     *   transition={{ type: "spring", mass: 0.5 }}
     * />
     * ```
     *
     * @public
     */
    mass?: number;
    /**
     * End animation if absolute speed (in units per second) drops below this
     * value and delta is smaller than `restDelta`. Set to `0.01` by default.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   type: "spring",
     *   restSpeed: 0.5
     * }
     *
     * <Frame
     *   animate={{ rotate: 180 }}
     *   transition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   animate={{ rotate: 180 }}
     *   transition={{ type: 'spring', restSpeed: 0.5 }}
     * />
     * ```
     *
     * @public
     */
    restSpeed?: number;
    /**
     * End animation if distance is below this value and speed is below
     * `restSpeed`. When animation ends, spring gets “snapped” to. Set to
     * `0.01` by default.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   type: "spring",
     *   restDelta: 0.5
     * }
     *
     * <Frame
     *   animate={{ rotate: 180 }}
     *   transition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   animate={{ rotate: 180 }}
     *   transition={{ type: 'spring', restDelta: 0.5 }}
     * />
     * ```
     *
     * @public
     */
    restDelta?: number;
    /**
     * The value to animate from.
     * By default, this is the initial state of the animating value.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   type: "spring",
     *   from: 90
     * }
     *
     * <Frame
     *   animate={{ rotate: 180 }}
     *   transition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   animate={{ rotate: 180 }}
     *   transition={{ type: 'spring', from: 90 }}
     * />
     * ```
     *
     * @public
     */
    from?: number | string;
    /**
     * @internal
     */
    to?: number | string | ValueTarget;
    /**
     * The initial velocity of the spring. By default this is the current velocity of the component.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   type: "spring",
     *   velocity: 2
     * }
     *
     * <Frame
     *   animate={{ rotate: 180 }}
     *   transition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   animate={{ rotate: 180 }}
     *   transition={{ type: 'spring', velocity: 2 }}
     * />
     * ```
     *
     * @public
     */
    velocity?: number;
    /**
     * @internal
     */
    delay?: number;
}

declare type StartAnimation = (complete: () => void) => () => void;

/**
 * @public
 */
export declare type Subscriber<T> = (v: T) => void;

/**
 * Blanket-accept any SVG attribute as a `MotionValue`
 * @public
 */
export declare type SVGAttributesAsMotionValues<T> = MakeMotion<SVGAttributesWithoutMotionProps<T>>;

declare interface SVGAttributesWithoutMotionProps<T> extends Pick<SVGAttributes<T>, Exclude<keyof SVGAttributes<T>, keyof MotionProps>> {
}

/**
 * @public
 */
export declare interface SVGMotionProps<T> extends SVGAttributesAsMotionValues<T>, Omit<MotionProps, "positionTransition"> {
}

/**
 * @public
 */
declare interface SVGPathProperties {
    pathLength?: number;
    pathOffset?: number;
    pathSpacing?: number;
}

declare interface SyncLayoutProps {
    children: React.ReactNode;
}

/**
 * @public
 */
export declare interface TapHandlers {
    /**
     * Callback when the tap gesture successfully ends on this element.
     *
     * @library
     *
     * ```jsx
     * function onTap(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <Frame onTap={onTap} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * function onTap(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <motion.div onTap={onTap} />
     * ```
     *
     * @param event - The originating pointer event.
     * @param info - An {@link TapInfo} object containing `x` and `y` values for the `point` relative to the device or page.
     */
    onTap?(event: MouseEvent | TouchEvent | PointerEvent, info: TapInfo): void;
    /**
     * Callback when the tap gesture starts on this element.
     *
     * @library
     *
     * ```jsx
     * function onTapStart(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <Frame onTapStart={onTapStart} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * function onTapStart(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <motion.div onTapStart={onTapStart} />
     * ```
     *
     * @param event - The originating pointer event.
     * @param info - An {@link TapInfo} object containing `x` and `y` values for the `point` relative to the device or page.
     */
    onTapStart?(event: MouseEvent | TouchEvent | PointerEvent, info: TapInfo): void;
    /**
     * Callback when the tap gesture ends outside this element.
     *
     * @library
     *
     * ```jsx
     * function onTapCancel(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <Frame onTapCancel={onTapCancel} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * function onTapCancel(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <motion.div onTapCancel={onTapCancel} />
     * ```
     *
     * @param event - The originating pointer event.
     * @param info - An {@link TapInfo} object containing `x` and `y` values for the `point` relative to the device or page.
     */
    onTapCancel?(event: MouseEvent | TouchEvent | PointerEvent, info: TapInfo): void;
    /**
     * Properties or variant label to animate to while the component is pressed.
     *
     * @library
     *
     * ```jsx
     * <Frame whileTap={{ scale: 0.8 }} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div whileTap={{ scale: 0.8 }} />
     * ```
     */
    whileTap?: string | TargetAndTransition;
}

/**
 * Passed in to tap event handlers like `onTap` the `TapInfo` object contains
 * information about the tap gesture such as it‘s location.
 *
 * @library
 *
 * ```jsx
 * function onTap(event, info) {
 *   console.log(info.point.x, info.point.y)
 * }
 *
 * <Frame onTap={onTap} />
 * ```
 *
 * @motion
 *
 * ```jsx
 * function onTap(event, info) {
 *   console.log(info.point.x, info.point.y)
 * }
 *
 * <motion.div onTap={onTap} />
 * ```
 *
 * @public
 */
export declare interface TapInfo {
    /**
     * Contains `x` and `y` values for the tap gesture relative to the
     * device or page.
     *
     * @library
     *
     * ```jsx
     * function onTapStart(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <Frame onTapStart={onTapStart} />
     * ```
     *
     * @motion
     *
     * ```jsx
     * function onTapStart(event, info) {
     *   console.log(info.point.x, info.point.y)
     * }
     *
     * <motion.div onTapStart={onTapStart} />
     * ```
     *
     * @public
     */
    point: Point;
}

declare type Target = MakeCustomValueType<TargetProperties>;

/**
 * An object that specifies values to animate to. Each value may be set either as
 * a single value, or an array of values.
 *
 * It may also option contain these properties:
 *
 * - `transition`: Specifies transitions for all or individual values.
 * - `transitionEnd`: Specifies values to set when the animation finishes.
 *
 * ```jsx
 * const target = {
 *   x: "0%",
 *   opacity: 0,
 *   transition: { duration: 1 },
 *   transitionEnd: { display: "none" }
 * }
 * ```
 *
 * @public
 */
export declare type TargetAndTransition = TargetWithKeyframes & {
    transition?: Transition;
    transitionEnd?: Target;
};

declare type TargetProperties = CSSPropertiesWithoutTransitionOrSingleTransforms & SVGAttributes<SVGElement> & TransformProperties & CustomStyles;

declare type TargetResolver = (custom: any, current: Target, velocity: Target) => TargetAndTransition;

declare type TargetWithKeyframes = MakeKeyframes<Target>;

/**
 * Transforms numbers into other values by mapping them from an input range to an output range.
 * Returns the type of the input provided.
 *
 * @remarks
 *
 * Given an input range of `[0, 200]` and an output range of
 * `[0, 1]`, this function will return a value between `0` and `1`.
 * The input range must be a linear series of numbers. The output range
 * can be any supported value type, such as numbers, colors, shadows, arrays, objects and more.
 * Every value in the output range must be of the same type and in the same format.
 *
 * @library
 *
 * ```jsx
 * import * as React from "react"
 * import { Frame, transform } from "framer"
 *
 * export function MyComponent() {
 *    const inputRange = [0, 200]
 *    const outputRange = [0, 1]
 *    const output = transform(100, inputRange, outputRange)
 *
 *    // Returns 0.5
 *    return <Frame>{output}</Frame>
 * }
 * ```
 *
 * @motion
 *
 * ```jsx
 * import * as React from "react"
 * import { transform } from "framer-motion"
 *
 * export function MyComponent() {
 *    const inputRange = [0, 200]
 *    const outputRange = [0, 1]
 *    const output = transform(100, inputRange, outputRange)
 *
 *    // Returns 0.5
 *    return <div>{output}</div>
 * }
 * ```
 *
 * @param inputValue - A number to transform between the input and output ranges.
 * @param inputRange - A linear series of numbers (either all increasing or decreasing).
 * @param outputRange - A series of numbers, colors, strings, or arrays/objects of those. Must be the same length as `inputRange`.
 * @param options - Clamp: Clamp values to within the given range. Defaults to `true`.
 *
 * @public
 */
export declare function transform<T>(inputValue: number, inputRange: number[], outputRange: T[], options?: TransformOptions<T>): T;

/**
 * @library
 *
 * For improved performance, `transform` can pre-calculate the function that will transform a value between two ranges.
 * Returns a function.
 *
 * ```jsx
 * import * as React from "react"
 * import { Frame, transform } from "framer"
 *
 * export function MyComponent() {
 *     const inputRange = [-200, -100, 100, 200]
 *     const outputRange = [0, 1, 1, 0]
 *     const convertRange = transform(inputRange, outputRange)
 *     const output = convertRange(-150)
 *
 *     // Returns 0.5
 *     return <Frame>{output}</Frame>
 * }
 *
 * ```
 *
 * @motion
 *
 * Transforms numbers into other values by mapping them from an input range to an output range.
 *
 * Given an input range of `[0, 200]` and an output range of
 * `[0, 1]`, this function will return a value between `0` and `1`.
 * The input range must be a linear series of numbers. The output range
 * can be any supported value type, such as numbers, colors, shadows, arrays, objects and more.
 * Every value in the output range must be of the same type and in the same format.
 *
 * ```jsx
 * import * as React from "react"
 * import { Frame, transform } from "framer"
 *
 * export function MyComponent() {
 *     const inputRange = [-200, -100, 100, 200]
 *     const outputRange = [0, 1, 1, 0]
 *     const convertRange = transform(inputRange, outputRange)
 *     const output = convertRange(-150)
 *
 *     // Returns 0.5
 *     return <div>{output}</div>
 * }
 *
 * ```
 *
 * @param inputRange - A linear series of numbers (either all increasing or decreasing).
 * @param outputRange - A series of numbers, colors or strings. Must be the same length as `inputRange`.
 * @param options - Clamp: clamp values to within the given range. Defaults to `true`.
 *
 * @public
 */
export declare function transform<T>(inputRange: number[], outputRange: T[], options?: TransformOptions<T>): (inputValue: number) => T;

declare type Transformer<T> = (v: T) => T;

declare type Transformer_2<T> = (v: any) => T;

/**
 * @public
 */
declare interface TransformOptions<T> {
    /**
     * Clamp values to within the given range. Defaults to `true`
     *
     * @public
     */
    clamp?: boolean;
    /**
     * Easing functions to use on the interpolations between each value in the input and output ranges.
     *
     * If provided as an array, the array must be one item shorter than the input and output ranges, as the easings apply to the transition **between** each.
     *
     * @public
     */
    ease?: Easing_2 | Easing_2[];
    /**
     * @internal
     */
    mixer?: (from: T, to: T) => (v: number) => any;
}

declare interface TransformProperties {
    x?: string | number;
    y?: string | number;
    z?: string | number;
    translateX?: string | number;
    translateY?: string | number;
    translateZ?: string | number;
    rotate?: string | number;
    rotateX?: string | number;
    rotateY?: string | number;
    rotateZ?: string | number;
    scale?: string | number;
    scaleX?: string | number;
    scaleY?: string | number;
    scaleZ?: string | number;
    skew?: string | number;
    skewX?: string | number;
    skewY?: string | number;
    originX?: string | number;
    originY?: string | number;
    originZ?: string | number;
    perspective?: string | number;
}

declare type TransformTemplate = (transform: TransformProperties, generatedTransform: string) => string;

/**
 * Transition props
 *
 * @public
 */
export declare type Transition = (Orchestration & TransitionDefinition) | (Orchestration & TransitionMap);

/**
 * @public
 */
declare type TransitionDefinition = Tween | Spring | Keyframes | Inertia | Just | None | PermissiveTransitionDefinition;

declare type TransitionMap = Orchestration & {
    [key: string]: TransitionDefinition;
};

/**
 * An animation that animates between two or more values over a specific duration of time.
 * This is the default animation for non-physical values like `color` and `opacity`.
 *
 * @public
 */
export declare interface Tween {
    /**
     * Set `type` to `"tween"` to use a duration-based tween animation.
     * If any non-orchestration `transition` values are set without a `type` property,
     * this is used as the default animation.
     *
     * @library
     *
     * ```jsx
     * <Frame
     *   animate={{ opacity: 0 }}
     *   transition={{ duration: 2, type: "tween" }}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.path
     *   animate={{ pathLength: 1 }}
     *   transition={{ duration: 2, type: "tween" }}
     * />
     * ```
     *
     * @public
     */
    type?: "tween";
    /**
     * The duration of the tween animation. Set to `0.3` by default, 0r `0.8` if animating a series of keyframes.
     *
     * @library
     *
     * ```jsx
     * <Frame
     *   animate={{ opacity: 0 }}
     *   transition={{ duration: 2 }}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * const variants = {
     *   visible: {
     *     opacity: 1,
     *     transition: { duration: 2 }
     *   }
     * }
     * ```
     *
     * @public
     */
    duration?: number;
    /**
     * The easing function to use. Set as one of the below.
     *
     * - The name of an existing easing function.
     * - An array of four numbers to define a cubic bezier curve.
     * - An easing function, that accepts and returns a value `0-1`.
     *
     * If the animating value is set as an array of multiple values for a keyframes
     * animation, `ease` can be set as an array of easing functions to set different easings between
     * each of those values.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   ease: [0.17, 0.67, 0.83, 0.67]
     * }
     *
     * <Frame
     *   animate={{ opacity: 0 }}
     *   transition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   animate={{ opacity: 0 }}
     *   transition={{ ease: [0.17, 0.67, 0.83, 0.67] }}
     * />
     * ```
     *
     * @public
     */
    ease?: Easing | Easing[];
    /**
     * The duration of time already elapsed in the animation. Set to `0` by
     * default.
     *
     * @internal
     */
    elapsed?: number;
    /**
     * When animating keyframes, `times` can be used to determine where in the animation each keyframe is reached.
     * Each value in `times` is a value between `0` and `1`, representing `duration`.
     *
     * There must be the same number of `times` as there are keyframes.
     * Defaults to an array of evenly-spread durations.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   times: [0, 0.1, 0.9, 1]
     * }
     *
     * <Frame
     *   animate={{ scale: [0, 1, 0.5, 1] }}
     *   transition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   animate={{ scale: [0, 1, 0.5, 1] }}
     *   transition={{ times: [0, 0.1, 0.9, 1] }}
     * />
     * ```
     *
     * @public
     */
    times?: number[];
    /**
     * When animating keyframes, `easings` can be used to define easing functions between each keyframe. This array should be one item fewer than the number of keyframes, as these easings apply to the transitions between the keyframes.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   easings: ["easeIn", "easeOut"]
     * }
     *
     * <Frame
     *   animate={{ backgroundColor: ["#0f0", "#00f", "#f00"] }}
     *   transition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   animate={{ backgroundColor: ["#0f0", "#00f", "#f00"] }}
     *   transition={{ easings: ["easeIn", "easeOut"] }}
     * />
     * ```
     *
     * @public
     */
    easings?: Easing[];
    /**
     * The number of times to loop the animation.
     * Set to `Infinity` for perpetual looping.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   loop: Infinity,
     *   ease: "linear",
     *   duration: 2
     * }
     *
     * <Frame
     *   animate={{ rotate: 360 }}
     *   transition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   animate={{ rotate: 360 }}
     *   transition={{
     *     loop: Infinity,
     *     ease: "linear",
     *     duration: 2
     *   }}
     * />
     * ```
     *
     * @public
     */
    loop?: number;
    /**
     * The number of times to flip the animation by swapping the `to` and `from` values.
     * Set to `Infinity` for perpetual flipping.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   flip: Infinity,
     *   duration: 2
     * }
     *
     * <Frame
     *   animate={{ opacity: 0 }}
     *   transition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   animate={{ opacity: 0 }}
     *   transition={{ flip: Infinity, duration: 2 }}
     * />
     * ```
     *
     * @public
     */
    flip?: number;
    /**
     * The number of times to reverse the animation.
     * Set to `Infinity` for perpetual reversing.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   yoyo: Infinity,
     *   duration: 2
     * }
     *
     * <Frame
     *   animate={{ rotate: 180 }}
     *   transition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   animate={{ rotate: 180 }}
     *   transition={{ yoyo: Infinity, duration: 2 }}
     * />
     * ```
     *
     * @public
     */
    yoyo?: number;
    /**
     * When repeating an animation using `loop`, `flip`, or `yoyo`, `repeatDelay` can set the
     * duration of the time to wait, in seconds, between each repetition.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   yoyo: Infinity,
     *   repeatDelay: 1
     * }
     *
     * <Frame
     *   animate={{ rotate: 180 }}
     *   transition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   animate={{ rotate: 180 }}
     *   transition={{ yoyo: Infinity, repeatDelay: 1 }}
     * />
     * ```
     *
     *
     * @public
     */
    repeatDelay?: number;
    /**
     * The value to animate from.
     * By default, this is the current state of the animating value.
     *
     * @library
     *
     * ```jsx
     * const transition = {
     *   from: 90,
     *   duration: 2
     * }
     *
     * <Frame
     *   animate={{ rotate: 180 }}
     *   transition={transition}
     * />
     * ```
     *
     * @motion
     *
     * ```jsx
     * <motion.div
     *   animate={{ rotate: 180 }}
     *   transition={{ from: 90, duration: 2 }}
     * />
     * ```
     *
     * @public
     */
    from?: number | string;
    /**
     * @internal
     */
    to?: number | string | ValueTarget;
    /**
     * @internal
     */
    velocity?: number;
    /**
     * @internal
     */
    delay?: number;
}

/**
 * When layout changes happen asynchronously to their instigating render (ie when exiting
 * children of `AnimatePresence` are removed), `SyncLayout` can wrap parent and sibling
 * components that need to animate as a result of this layout change.
 *
 * @motion
 *
 * ```jsx
 * const MyComponent = ({ isVisible }) => {
 *   return (
 *     <SyncLayout>
 *       <AnimatePresence>
 *         {isVisible && (
 *           <motion.div exit={{ opacity: 0 }} />
 *         )}
 *       </AnimatePresence>
 *       <motion.div positionTransition />
 *     </SyncLayout>
 *   )
 * }
 * ```
 *
 * @internalremarks
 *
 * The way this component works is by memoising a function and passing it down via context.
 * The function, when called, updates the local state, which is used to invalidate the
 * memoisation cache. A new function is called, performing a synced re-render of components
 * that are using the SyncLayoutContext.
 *
 * @internal
 */
export declare const UnstableSyncLayout: ({ children }: SyncLayoutProps) => JSX.Element;

declare type UnwrapFactoryAttributes<F> = F extends DetailedHTMLFactory<infer P, any> ? P : never;

declare type UnwrapFactoryElement<F> = F extends DetailedHTMLFactory<any, infer P> ? P : never;

/**
 * If the provided value is a MotionValue, this returns the actual value, otherwise just the value itself
 *
 * TODO: Remove and move to library
 *
 * @internal
 */
export declare function unwrapMotionValue(value?: string | number | CustomValueType | MotionValue): string | number;

/**
 * Experimental API.
 *
 * Makes an animated version of `useState`.
 *
 * @remarks
 *
 * When the returned state setter is called, values will be animated to their new target.
 *
 * This allows the animation of arbitrary React components.
 *
 * **Note:** When animating DOM components, it's always preferable to use the `animate` prop, as Framer
 * will bypass React's rendering cycle with one optimised for 60fps motion. This Hook is specifically
 * for animating props on arbitrary React components, or for animating text content.
 *
 * ```jsx
 * const [state, setState] = useAnimatedState({ percentage: 0 })
 *
 * return (
 *   <Graph
 *     percentage={state.percentage}
 *     onTap={() => setState({ percentage: 50 })}
 *   />
 * )
 * ```
 *
 * @internalremarks
 *
 * TODO:
 * - Make hook accept a typed version of Target that accepts any value (not just DOM values)
 * - Allow hook to accept single values. ie useAnimatedState(0)
 * - Allow providing MotionValues via initialState.
 *
 * @beta
 */
export declare function useAnimatedState(initialState: any): any[];

/**
 * Creates `AnimationControls`, which can be used to manually start, stop
 * and sequence animations on one or more components.
 *
 * The returned `AnimationControls` should be passed to the `animate` property
 * of the components you want to animate.
 *
 * These components can then be animated with the `start` method.
 *
 * @library
 *
 * ```jsx
 * import * as React from 'react'
 * import { Frame, useAnimation } from 'framer'
 *
 * export function MyComponent(props) {
 *    const controls = useAnimation()
 *
 *    controls.start({
 *        x: 100,
 *        transition: { duration: 0.5 },
 *    })
 *
 *    return <Frame animate={controls} />
 * }
 * ```
 *
 * @motion
 *
 * ```jsx
 * import * as React from 'react'
 * import { motion, useAnimation } from 'framer-motion'
 *
 * export function MyComponent(props) {
 *    const controls = useAnimation()
 *
 *    controls.start({
 *        x: 100,
 *        transition: { duration: 0.5 },
 *    })
 *
 *    return <motion.div animate={controls} />
 * }
 * ```
 *
 * @returns Animation controller with `start` and `stop` methods
 *
 * @public
 */
export declare function useAnimation(): AnimationControls;

/**
 * Cycles through a series of visual properties. Can be used to toggle between or cycle through animations. It works similar to `useState` in React. It is provided an initial array of possible states, and returns an array of two arguments.
 *
 * @library
 *
 * ```jsx
 * import * as React from "react"
 * import { Frame, useCycle } from "framer"
 *
 * export function MyComponent() {
 *   const [x, cycleX] = useCycle(0, 50, 100)
 *
 *   return (
 *     <Frame
 *       animate={{ x: x }}
 *       onTap={() => cycleX()}
 *      />
 *    )
 * }
 * ```
 *
 * @motion
 *
 * An index value can be passed to the returned `cycle` function to cycle to a specific index.
 *
 * ```jsx
 * import * as React from "react"
 * import { motion, useCycle } from "framer-motion"
 *
 * export const MyComponent = () => {
 *   const [x, cycleX] = useCycle(0, 50, 100)
 *
 *   return (
 *     <motion.div
 *       animate={{ x: x }}
 *       onTap={() => cycleX()}
 *      />
 *    )
 * }
 * ```
 *
 * @param items - items to cycle through
 * @returns [currentState, cycleState]
 *
 * @public
 */
export declare function useCycle<T>(...items: T[]): CycleState<T>;

/**
 * Attaches an event listener directly to the provided DOM element.
 *
 * Bypassing React's event system can be desirable, for instance when attaching non-passive
 * event handlers.
 *
 * ```jsx
 * const ref = useRef(null)
 *
 * useDomEvent(ref, 'wheel', onWheel, { passive: false })
 *
 * return <div ref={ref} />
 * ```
 *
 * @param ref - React.RefObject that's been provided to the element you want to bind the listener to.
 * @param eventName - Name of the event you want listen for.
 * @param handler - Function to fire when receiving the event.
 * @param options - Options to pass to `Event.addEventListener`.
 *
 * @public
 */
export declare function useDomEvent(ref: RefObject<Element>, eventName: string, handler?: EventListener | undefined, options?: AddEventListenerOptions): void;

/**
 * Usually, dragging is initiated by pressing down on a `motion` component with a `drag` prop
 * and moving it. For some use-cases, for instance clicking at an arbitrary point on a video scrubber, we
 * might want to initiate that dragging from a different component than the draggable one.
 *
 * By creating a `dragControls` using the `useDragControls` hook, we can pass this into
 * the draggable component's `dragControls` prop. It exposes a `start` method
 * that can start dragging from pointer events on other components.
 *
 * @library
 *
 * ```jsx
 * const dragControls = useDragControls()
 *
 * function startDrag(event) {
 *   dragControls.start(event, { snapToCursor: true })
 * }
 *
 * return (
 *   <>
 *     <Frame onTapStart={startDrag} />
 *     <Frame drag="x" dragControls={dragControls} />
 *   </>
 * )
 * ```
 *
 * @motion
 *
 * ```jsx
 * const dragControls = useDragControls()
 *
 * function startDrag(event) {
 *   dragControls.start(event, { snapToCursor: true })
 * }
 *
 * return (
 *   <>
 *     <div onMouseDown={startDrag} />
 *     <motion.div drag="x" dragControls={dragControls} />
 *   </>
 * )
 * ```
 *
 * @public
 */
export declare function useDragControls(): DragControls;

/**
 * Returns MotionValues that update when the provided element scrolls:
 *
 * - `scrollX` — Horizontal scroll distance in pixels.
 * - `scrollY` — Vertical scroll distance in pixels.
 * - `scrollXProgress` — Horizontal scroll progress between `0` and `1`.
 * - `scrollYProgress` — Vertical scroll progress between `0` and `1`.
 *
 * @library
 *
 * ```jsx
 * import * as React from "react"
 * import {
 *   Frame,
 *   useElementScroll,
 *   useTransform
 * } from "framer"
 *
 * export function MyComponent() {
 *   const ref = React.useRef()
 *   const { scrollYProgress } = useElementScroll(ref)
 *
 *   return (
 *     <Frame ref={ref}>
 *       <Frame scaleX={scrollYProgress} />
 *     </Frame>
 *   )
 * }
 * ```
 *
 * @motion
 *
 * ```jsx
 * export const MyComponent = () => {
 *   const ref = useRef()
 *   const { scrollYProgress } = useElementScroll(ref)
 *
 *   return (
 *     <div ref={ref}>
 *       <motion.div style={{ scaleX: scrollYProgress }} />
 *     </div>
 *   )
 * }
 * ```
 *
 * @public
 */
export declare function useElementScroll(ref: RefObject<HTMLElement>): ScrollMotionValues;

/**
 * Uses the ref that is passed in, or creates a new one
 * @param external - External ref
 * @internal
 */
export declare function useExternalRef<E = Element>(externalRef?: Ref<E>): RefObject<E>;

/**
 * Add pan and tap gesture recognition to an element.
 *
 * @param props - Gesture event handlers
 * @param ref - React `ref` containing a DOM `Element`
 * @public
 */
export declare function useGestures<GestureHandlers>(props: GestureHandlers, ref: RefObject<Element>): void;

/**
 * Returns a `MotionValue` each for `scaleX` and `scaleY` that update with the inverse
 * of their respective parent scales.
 *
 * This is useful for undoing the distortion of content when scaling a parent component.
 *
 * By default, `useInvertedScale` will automatically fetch `scaleX` and `scaleY` from the nearest parent.
 * By passing other `MotionValue`s in as `useInvertedScale({ scaleX, scaleY })`, it will invert the output
 * of those instead.
 *
 * @motion
 *
 * ```jsx
 * const MyComponent = () => {
 *   const { scaleX, scaleY } = useInvertedScale()
 *   return <motion.div style={{ scaleX, scaleY }} />
 * }
 * ```
 *
 * @library
 *
 * ```jsx
 * function MyComponent() {
 *   const { scaleX, scaleY } = useInvertedScale()
 *   return <Frame scaleX={scaleX} scaleY={scaleY} />
 * }
 * ```
 *
 * @public
 */
export declare function useInvertedScale(scale?: Partial<ScaleMotionValues>): ScaleMotionValues;

/**
 * Creates a `MotionValue` to track the state and velocity of a value.
 *
 * Usually, these are created automatically. For advanced use-cases, like use with `useTransform`, you can create `MotionValue`s externally and pass them into the animated component via the `style` prop.
 *
 * @library
 *
 * ```jsx
 * export function MyComponent() {
 *   const scale = useMotionValue(1)
 *
 *   return <Frame scale={scale} />
 * }
 * ```
 *
 * @motion
 *
 * ```jsx
 * export const MyComponent = () => {
 *   const scale = useMotionValue(1)
 *
 *   return <motion.div style={{ scale }} />
 * }
 * ```
 *
 * @param initial - The initial state.
 *
 * @public
 */
export declare function useMotionValue<T>(initial: T): MotionValue<T>;

/**
 *
 * @param handlers -
 * @param ref -
 *
 * @internalremarks
 * Currently this sets new pan gesture functions every render. The memo route has been explored
 * in the past but ultimately we're still creating new functions every render. An optimisation
 * to explore is creating the pan gestures and loading them into a `ref`.
 *
 * @internal
 */
export declare function usePanGesture({ onPan, onPanStart, onPanEnd, onPanSessionStart }: PanHandlers, ref: RefObject<Element>): void;

/**
 * When a component is the child of `AnimatePresence`, it can use `usePresence`
 * to access information about whether it's still present in the React tree.
 *
 * ```jsx
 * import { usePresence } from "framer-motion"
 *
 * export const Component = () => {
 *   const [isPresent, safeToRemove] = usePresence()
 *
 *   useEffect(() => {
 *     !isPresent setTimeout(safeToRemove, 1000)
 *   }, [isPresent])
 *
 *   return <div />
 * }
 * ```
 *
 * If `isPresent` is `false`, it means that a component has been removed the tree, but
 * `AnimatePresence` won't really remove it until `safeToRemove` has been called.
 *
 * @public
 */
export declare function usePresence(): Present | NotPresent;

/**
 * A hook that returns `true` if we should be using reduced motion based on the current device's Reduced Motion setting.
 *
 * This can be used to implement changes to your UI based on Reduced Motion. For instance, replacing motion-sickness inducing
 * `x`/`y` animations with `opacity`, disabling the autoplay of background videos, or turning off parallax motion.
 *
 * It will actively respond to changes and re-render your components with the latest setting.
 *
 * ```jsx
 * export function Sidebar({ isOpen }) {
 *   const shouldReduceMotion = useReducedMotion()
 *   const closedX = shouldReduceMotion ? 0 : "-100%"
 *
 *   return (
 *     <motion.div animate={{
 *       opacity: isOpen ? 1 : 0,
 *       x: isOpen ? 0 : closedX
 *     }} />
 *   )
 * }
 * ```
 *
 * @return boolean
 *
 * @public
 */
export declare function useReducedMotion(): boolean;

/**
 * Creates a `MotionValue` that, when `set`, will use a spring animation to animate to its new state.
 *
 * It can either work as a stand-alone `MotionValue` by initialising it with a value, or as a subscriber
 * to another `MotionValue`.
 *
 * @remarks
 *
 * ```jsx
 * const x = useSpring(0, { stiffness: 300 })
 * const y = useSpring(x, { damping: 10 })
 * ```
 *
 * @param inputValue - `MotionValue` or number. If provided a `MotionValue`, when the input `MotionValue` changes, the created `MotionValue` will spring towards that value.
 * @param springConfig - Configuration options for the spring.
 * @returns `MotionValue`
 *
 * @public
 */
export declare function useSpring(source: MotionValue | number, config?: SpringProps): MotionValue<any>;

/**
 * @param handlers -
 * @internal
 */
export declare function useTapGesture({ onTap, onTapStart, onTapCancel, whileTap, controls, }: TapHandlers & ControlsProp, ref: RefObject<Element>): void;

/**
 * Create a `MotionValue` that transforms the output of another `MotionValue` through a function.
 * In this example, `y` will always be double `x`.
 *
 * @library
 *
 * ```jsx
 * import * as React from "react"
 * import { Frame, useMotionValue, useTransform } from "framer"
 *
 * export function MyComponent() {
 *   const x = useMotionValue(10)
 *   const y = useTransform(x, value => value * 2)
 *
 *   return <Frame x={x} y={y} />
 * }
 * ```
 *
 * @motion
 *
 * ```jsx
 * export const MyComponent = () => {
 *   const x = useMotionValue(10)
 *   const y = useTransform(x, value => value * 2)
 *
 *   return <motion.div style={{ x, y }} />
 * }
 * ```
 *
 * @param value - The `MotionValue` to transform the output of.
 * @param transform - Function that accepts the output of `value` and returns a new value.
 * @returns `MotionValue`
 *
 * @public
 */
export declare function useTransform<T>(parent: MotionValue, transform: Transformer_2<T>): MotionValue;

/**
 * Create a `MotionValue` that transforms the output of another `MotionValue` by mapping it from one range of values into another.
 *
 * @remarks
 *
 * Given an input range of `[-200, -100, 100, 200]` and an output range of
 * `[0, 1, 1, 0]`, the returned `MotionValue` will:
 *
 * - When provided a value between `-200` and `-100`, will return a value between `0` and  `1`.
 * - When provided a value between `-100` and `100`, will return `1`.
 * - When provided a value between `100` and `200`, will return a value between `1` and  `0`
 *
 *
 * The input range must be a linear series of numbers. The output range
 * can be any value type supported by Framer Motion: numbers, colors, shadows, etc.
 *
 * Every value in the output range must be of the same type and in the same format.
 *
 * @library
 *
 * ```jsx
 * export function MyComponent() {
 *   const x = useMotionValue(0)
 *   const xRange = [-200, -100, 100, 200]
 *   const opacityRange = [0, 1, 1, 0]
 *   const opacity = useTransform(x, xRange, opacityRange)
 *
 *   return <Frame drag="x" x={x} opacity={opacity} />
 * }
 * ```
 *
 * @motion
 *
 * ```jsx
 * export const MyComponent = () => {
 *   const x = useMotionValue(0)
 *   const xRange = [-200, -100, 100, 200]
 *   const opacityRange = [0, 1, 1, 0]
 *   const opacity = useTransform(x, xRange, opacityRange)
 *
 *   return <motion.div drag="x" style={{ opacity, x }} />
 * }
 * ```
 *
 * @param inputValue - `MotionValue`
 * @param inputRange - A linear series of numbers (either all increasing or decreasing)
 * @param outputRange - A series of numbers, colors or strings. Must be the same length as `inputRange`.
 * @param options -
 *
 *  - clamp: boolean - Clamp values to within the given range. Defaults to `true`
 *  - ease: EasingFunction[] - Easing functions to use on the interpolations between each value in the input and output ranges. If provided as an array, the array must be one item shorter than the input and output ranges, as the easings apply to the transition **between** each.
 *
 * @returns `MotionValue`
 *
 * @public
 */
export declare function useTransform<T>(parent: MotionValue<number>, from: number[], to: T[], options?: TransformOptions<T>): MotionValue<T>;

/**
 * Returns MotionValues that update when the viewport scrolls:
 *
 * - `scrollX` — Horizontal scroll distance in pixels.
 * - `scrollY` — Vertical scroll distance in pixels.
 * - `scrollXProgress` — Horizontal scroll progress between `0` and `1`.
 * - `scrollYProgress` — Vertical scroll progress between `0` and `1`.
 *
 * @library
 *
 * ```jsx
 * import * as React from "react"
 * import {
 *   Frame,
 *   useViewportScroll,
 *   useTransform
 * } from "framer"
 *
 * export function MyComponent() {
 *   const { scrollYProgress } = useViewportScroll()
 *   return <Frame scaleX={scrollYProgress} />
 * }
 * ```
 *
 * @motion
 *
 * ```jsx
 * export const MyComponent = () => {
 *   const { scrollYProgress } = useViewportScroll()
 *   return <motion.div style={{ scaleX: scrollYProgress }} />
 * }
 * ```
 *
 * @public
 */
export declare function useViewportScroll(): ScrollMotionValues;

declare interface ValueAnimationConfig {
    values: MotionValuesMap;
    readValueFromSource: ReadValueFromSource;
    makeTargetAnimatable?: MakeTargetAnimatable;
}

/**
 * Control animations for a single component
 *
 * @internal
 */
declare class ValueAnimationControls<P extends {} = {}, V extends {} = {}> {
    /**
     * A reference to the component's latest props. We could probably ditch this in
     * favour to a reference to the `custom` prop now we don't send all props through
     * to target resolvers.
     */
    private props;
    /**
     * A reference to the component's motion values
     */
    private values;
    /**
     * The default transition to use for `Target`s without any `transition` prop.
     */
    private defaultTransition?;
    /**
     * The component's variants, as provided by `variants`
     */
    private variants;
    /**
     * A set of values that we animate back to when a value is cleared of all overrides.
     */
    private baseTarget;
    /**
     * A series of target overrides that we can animate to/from when overrides are set/cleared.
     */
    private overrides;
    /**
     * A series of target overrides as they were originally resolved.
     */
    private resolvedOverrides;
    /**
     * A Set of currently active override indexes
     */
    private activeOverrides;
    /**
     * A Set of children component controls for variant propagation.
     */
    private children?;
    /**
     * A Set of value keys that are currently animating.
     */
    private isAnimating;
    /**
     * In the event we attempt to animate a value that doesn't exist yet, we use this
     * function to attempt to read it from the source (ie the DOM, or React state etc)
     */
    private readValueFromSource;
    /**
     * A chance
     */
    private makeTargetAnimatable;
    constructor({ values, readValueFromSource, makeTargetAnimatable, }: ValueAnimationConfig);
    /**
     * Set the reference to the component's props.
     * @param props -
     */
    setProps(props: P & MotionProps): void;
    /**
     * Set the reference to the component's variants
     * @param variants -
     */
    setVariants(variants?: Variants): void;
    /**
     * Set the component's default transition
     * @param transition -
     */
    setDefaultTransition(transition?: Transition): void;
    /**
     * Set motion values without animation.
     *
     * @param definition -
     * @param isActive -
     */
    private setValues;
    /**
     * Allows `transformValues` to be set by a component that allows us to
     * transform the values in a given `Target`. This allows Framer Library
     * to extend Framer Motion to animate `Color` variables etc. Currently we have
     * to manually support these extended types here in Framer Motion.
     *
     * @param values -
     */
    private transformValues;
    /**
     * Check a `Target` for new values we haven't animated yet, and add them
     * to the `MotionValueMap`.
     *
     * Currently there's functionality here that is DOM-specific, we should allow
     * this functionality to be injected by the factory that creates DOM-specific
     * components.
     *
     * @param target -
     */
    private checkForNewValues;
    /**
     * Check if the associated `MotionValueMap` has a key with the provided string.
     * Pre-bound to the class so we can provide directly to the `filter` in `checkForNewValues`.
     */
    private hasValue;
    /**
     * Resolve a variant from its label or resolver into an actual `Target` we can animate to.
     * @param variant -
     */
    private resolveVariant;
    /**
     * Get the highest active override priority index
     */
    private getHighestPriority;
    /**
     * Set an override. We add this layer of indirection so if, for instance, a tap gesture
     * starts and overrides a hover gesture, when we clear the tap gesture and fallback to the
     * hover gesture, if that hover gesture has changed in the meantime we can go to that rather
     * than the one that was resolved when the hover gesture animation started.
     *
     * @param definition -
     * @param overrideIndex -
     */
    setOverride(definition: AnimationDefinition, overrideIndex: number): void;
    /**
     * Start an override animation.
     * @param overrideIndex -
     */
    startOverride(overrideIndex: number): Promise<void> | undefined;
    /**
     * Clear an override. We check every value we animated to in this override to see if
     * its present on any lower-priority overrides. If not, we animate it back to its base target.
     * @param overrideIndex -
     */
    clearOverride(overrideIndex: number): void;
    /**
     * Apply a target/variant without any animation
     */
    apply(definition: AnimationDefinition): void;
    /**
     * Apply variant labels without animation
     */
    private applyVariantLabels;
    start(definition: AnimationDefinition, opts?: AnimationOptions): Promise<void>;
    private animate;
    private animateVariantLabels;
    private animateVariant;
    private animateChildren;
    private onStart;
    private onComplete;
    private checkOverrideIsAnimating;
    private resetIsAnimating;
    stop(): void;
    /**
     * Add the controls of a child component.
     * @param controls -
     */
    addChild(controls: ValueAnimationControls): void;
    removeChild(controls: ValueAnimationControls): void;
    resetChildren(): void;
}

/**
 * @public
 */
export declare type ValueTarget = SingleTarget | KeyframesTarget;

/**
 * @public
 */
export declare type Variant = TargetAndTransition | TargetResolver;

/**
 * Either a string, or array of strings, that reference variants defined via the `variants` prop.
 * @public
 */
export declare type VariantLabels = string | string[];

/**
 * @public
 */
export declare type Variants = {
    [key: string]: Variant;
};

export { }
