import type { DeleteTableEntityOptions, TableEntity, TableTransactionResponse, TransactionAction, UpdateMode, UpdateTableEntityOptions } from "./models.js";
import type { NamedKeyCredential, SASCredential, TokenCredential } from "@azure/core-auth";
import type { OperationOptions, ServiceClient } from "@azure/core-client";
import type { Pipeline, PipelineResponse } from "@azure/core-rest-pipeline";
import type { TableClientLike } from "./utils/internalModels.js";
/**
 * Helper to build a list of transaction actions
 */
export declare class TableTransaction {
    /**
     * List of actions to perform in a transaction
     */
    actions: TransactionAction[];
    constructor(actions?: TransactionAction[]);
    /**
     * Adds a create action to the transaction
     * @param entity - entity to create
     */
    createEntity<T extends object = Record<string, unknown>>(entity: TableEntity<T>): void;
    /**
     * Adds a delete action to the transaction
     * @param partitionKey - partition key of the entity to delete
     * @param rowKey - rowKey of the entity to delete
     */
    deleteEntity(partitionKey: string, rowKey: string): void;
    /**
     * Adds an update action to the transaction
     * @param entity - entity to update
     * @param updateOptions - options for the update operation
     */
    updateEntity<T extends object = Record<string, unknown>>(entity: TableEntity<T>, updateOptions?: UpdateTableEntityOptions): void;
    /**
     * Adds an update action to the transaction
     * @param entity - entity to update
     * @param updateMode - update mode
     * @param updateOptions - options for the update operation
     */
    updateEntity<T extends object = Record<string, unknown>>(entity: TableEntity<T>, updateMode: UpdateMode, updateOptions?: UpdateTableEntityOptions): void;
    /**
     * Adds an upsert action to the transaction, which inserts if the entity doesn't exist or updates the existing one
     * @param entity - entity to upsert
     * @param updateMode - update mode
     */
    upsertEntity<T extends object = Record<string, unknown>>(entity: TableEntity<T>, updateMode?: UpdateMode): void;
}
/**
 * TableTransaction collects sub-operations that can be submitted together via submitTransaction
 */
export declare class InternalTableTransaction {
    /**
     * Table Account URL
     */
    url: string;
    /**
     * State that holds the information about a particular transation
     */
    private state;
    private interceptClient;
    private allowInsecureConnection;
    private client;
    /**
     * @param url - Tables account url
     * @param partitionKey - partition key
     * @param credential - credential to authenticate the transaction request
     */
    constructor(url: string, partitionKey: string, transactionId: string, changesetId: string, client: ServiceClient, interceptClient: TableClientLike, credential?: NamedKeyCredential | SASCredential | TokenCredential, allowInsecureConnection?: boolean);
    private initializeState;
    /**
     * Adds a createEntity operation to the transaction
     * @param entity - Entity to create
     */
    createEntity<T extends object>(entity: TableEntity<T>): void;
    /**
     * Adds a createEntity operation to the transaction per each entity in the entities array
     * @param entities - Array of entities to create
     */
    createEntities<T extends object>(entities: TableEntity<T>[]): void;
    /**
     * Adds a deleteEntity operation to the transaction
     * @param partitionKey - Partition key of the entity to delete
     * @param rowKey - Row key of the entity to delete
     * @param options - Options for the delete operation
     */
    deleteEntity(partitionKey: string, rowKey: string, options?: DeleteTableEntityOptions): void;
    /**
     * Adds an updateEntity operation to the transaction
     * @param entity - Entity to update
     * @param mode - Update mode (Merge or Replace)
     * @param options - Options for the update operation
     */
    updateEntity<T extends object>(entity: TableEntity<T>, mode: UpdateMode, options?: UpdateTableEntityOptions): void;
    /**
     * Adds an upsertEntity operation to the transaction
     * @param entity - The properties for the table entity.
     * @param mode   - The different modes for updating the entity:
     *               - Merge: Updates an entity by updating the entity's properties without replacing the existing entity.
     *               - Replace: Updates an existing entity by replacing the entire entity.
     * @param options - The options parameters.
     */
    upsertEntity<T extends object>(entity: TableEntity<T>, mode: UpdateMode, options?: OperationOptions): void;
    /**
     * Submits the operations in the transaction
     * @param options - Options for the request.
     */
    submitTransaction(options?: OperationOptions): Promise<TableTransactionResponse>;
    private checkPartitionKey;
    private getUrlWithSlash;
}
export declare function parseTransactionResponse(transactionResponse: PipelineResponse): TableTransactionResponse;
/**
 * Prepares the transaction pipeline to intercept operations
 * @param pipeline - Client pipeline
 */
export declare function prepateTransactionPipeline(pipeline: Pipeline, bodyParts: string[], changesetId: string, isCosmos: boolean): void;
//# sourceMappingURL=TableTransaction.d.ts.map