import type { AbortSignalLike } from "@azure/abort-controller";
import type { HttpHeaders } from "@azure/core-rest-pipeline";
import type { ListFilesAndDirectoriesSegmentResponse as ListFilesAndDirectoriesSegmentResponseInternal, ListHandlesResponse as ListHandlesResponseInternal, SharePermission, StringEncoded } from "../generated/src/models/index.js";
import type { ListFilesAndDirectoriesSegmentResponse, ListHandlesResponse } from "../generatedModels.js";
import type { HttpAuthorization, NfsFileMode, PosixRolePermissions } from "../models.js";
import type { HttpHeadersLike, WebResourceLike } from "@azure/core-http-compat";
/**
 * Reserved URL characters must be properly escaped for Storage services like Blob or File.
 *
 * ## URL encode and escape strategy for JS SDKs
 *
 * When customers pass a URL string into XXXClient classes constructor, the URL string may already be URL encoded or not.
 * But before sending to Azure Storage server, the URL must be encoded. However, it's hard for a SDK to guess whether the URL
 * string has been encoded or not. We have 2 potential strategies, and chose strategy two for the XXXClient constructors.
 *
 * ### Strategy One: Assume the customer URL string is not encoded, and always encode URL string in SDK.
 *
 * This is what legacy V2 SDK does, simple and works for most of the cases.
 * - When customer URL string is "http://account.blob.core.windows.net/con/b:",
 *   SDK will encode it to "http://account.blob.core.windows.net/con/b%3A" and send to server. A blob named "b:" will be created.
 * - When customer URL string is "http://account.blob.core.windows.net/con/b%3A",
 *   SDK will encode it to "http://account.blob.core.windows.net/con/b%253A" and send to server. A blob named "b%3A" will be created.
 *
 * But this strategy will make it not possible to create a blob with "?" in it's name. Because when customer URL string is
 * "http://account.blob.core.windows.net/con/blob?name", the "?name" will be treated as URL paramter instead of blob name.
 * If customer URL string is "http://account.blob.core.windows.net/con/blob%3Fname", a blob named "blob%3Fname" will be created.
 * V2 SDK doesn't have this issue because it doesn't allow customer pass in a full URL, it accepts a separate blob name and encodeURIComponent for it.
 * We cannot accept a SDK cannot create a blob name with "?". So we implement strategy two:
 *
 * ### Strategy Two: SDK doesn't assume the URL has been encoded or not. It will just escape the special characters.
 *
 * This is what V10 Blob Go SDK does. It accepts a URL type in Go, and call url.EscapedPath() to escape the special chars unescaped.
 * - When customer URL string is "http://account.blob.core.windows.net/con/b:",
 *   SDK will escape ":" like "http://account.blob.core.windows.net/con/b%3A" and send to server. A blob named "b:" will be created.
 * - When customer URL string is "http://account.blob.core.windows.net/con/b%3A",
 *   There is no special characters, so send "http://account.blob.core.windows.net/con/b%3A" to server. A blob named "b:" will be created.
 * - When customer URL string is "http://account.blob.core.windows.net/con/b%253A",
 *   There is no special characters, so send "http://account.blob.core.windows.net/con/b%253A" to server. A blob named "b%3A" will be created.
 *
 * This strategy gives us flexibility to create with any special characters. But "%" will be treated as a special characters, if the URL string
 * is not encoded, there shouldn't a "%" in the URL string, otherwise the URL is not a valid URL.
 * If customer needs to create a blob with "%" in it's blob name, use "%25" insead of "%". Just like above 3rd sample.
 * And following URL strings are invalid:
 * - "http://account.blob.core.windows.net/con/b%"
 * - "http://account.blob.core.windows.net/con/b%2"
 * - "http://account.blob.core.windows.net/con/b%G"
 *
 * Another special character is "?", use "%2F" to represent a blob name with "?" in a URL string.
 *
 * ### Strategy for containerName, blobName or other specific XXXName parameters in methods such as `ContainerClient.getBlobClient(blobName)`
 *
 * We will apply strategy one, and call encodeURIComponent for these parameters like blobName. Because what customers passes in is a plain name instead of a URL.
 *
 * @see https://learn.microsoft.com/rest/api/storageservices/naming-and-referencing-containers--blobs--and-metadata
 * @see https://learn.microsoft.com/rest/api/storageservices/naming-and-referencing-shares--directories--files--and-metadata
 *
 * @param url -
 */
export declare function escapeURLPath(url: string): string;
export interface ConnectionString {
    kind: "AccountConnString" | "SASConnString";
    url: string;
    accountName: string;
    accountKey?: any;
    accountSas?: string;
}
/**
 * Extracts the parts of an Azure Storage account connection string.
 *
 * @param connectionString - Connection string.
 * @returns String key value pairs of the storage account's url and credentials.
 */
export declare function extractConnectionStringParts(connectionString: string): ConnectionString;
/**
 * Append a string to URL path. Will remove duplicated "/" in front of the string
 * when URL path ends with a "/".
 *
 * @param url - Source URL string
 * @param name - String to be appended to URL
 * @returns An updated URL string
 */
export declare function appendToURLPath(url: string, name: string): string;
/**
 * Append a string to URL query.
 *
 * @param url - Source URL string.
 * @param queryParts - String to be appended to the URL query.
 * @returns An updated URL string.
 */
export declare function appendToURLQuery(url: string, queryParts: string): string;
/**
 * Set URL parameter name and value. If name exists in URL parameters, old value
 * will be replaced by name key. If not provide value, the parameter will be deleted.
 *
 * @param url - Source URL string
 * @param name - Parameter name
 * @param value - Parameter value
 * @returns An updated URL string
 */
export declare function setURLParameter(url: string, name: string, value?: string): string;
/**
 * Get URL parameter by name.
 *
 * @param url -
 * @param name -
 */
export declare function getURLParameter(url: string, name: string): string | string[] | undefined;
/**
 * Set URL host.
 *
 * @param url - Source URL string
 * @param host - New host string
 * @returns An updated URL string
 */
export declare function setURLHost(url: string, host: string): string;
/**
 * Get URL path from an URL string.
 *
 * @param url - Source URL string
 */
export declare function getURLPath(url: string): string | undefined;
/**
 * Get URL query key value pairs from an URL string.
 *
 * @param url -
 */
export declare function getURLQueries(url: string): {
    [key: string]: string;
};
/**
 * Rounds a date off to seconds.
 *
 * @param date -
 * @param withMilliseconds - If true, YYYY-MM-DDThh:mm:ss.fffffffZ will be returned;
 *                                          If false, YYYY-MM-DDThh:mm:ssZ will be returned.
 * @returns Date string in ISO8061 format, with or without 7 milliseconds component
 */
export declare function truncatedISO8061Date(date: Date, withMilliseconds?: boolean): string;
/**
 * Base64 encode.
 *
 * @param content -
 */
export declare function base64encode(content: string): string;
/**
 * Base64 decode.
 *
 * @param encodedString -
 */
export declare function base64decode(encodedString: string): string;
/**
 * Delay specified time interval.
 *
 * @param timeInMs -
 * @param aborter -
 * @param abortError -
 */
export declare function delay(timeInMs: number, aborter?: AbortSignalLike, abortError?: Error): Promise<void>;
export declare function sanitizeURL(url: string): string;
export declare function sanitizeHeaders(originalHeader: HttpHeaders): HttpHeaders;
/**
 * Extracts account name from the url
 * @param url - url to extract the account name from
 * @returns with the account name
 */
export declare function getAccountNameFromUrl(url: string): string;
export declare function isIpEndpointStyle(parsedUrl: URL): boolean;
export declare function getShareNameAndPathFromUrl(url: string): {
    baseName: string;
    shareName: string;
    path: string;
};
export declare function httpAuthorizationToString(httpAuthorization?: HttpAuthorization): string | undefined;
/**
 * Set URL path.
 *
 * @param url - URL to change path to.
 * @param path - Path to set into the URL.
 */
export declare function setURLPath(url: string, path: string): string;
/**
 * Set URL query string.
 *
 * @param url - URL to set query string to.
 * @param queryString - Query string to set to the URL.
 */
export declare function setURLQueries(url: string, queryString: string): string;
/**
 * Escape the file or directory name but keep path separator ('/').
 */
export declare function EscapePath(pathName: string): string;
/**
 * A representation of an HTTP response that
 * includes a reference to the request that
 * originated it.
 */
export interface HttpResponse {
    /**
     * The headers from the response.
     */
    headers: HttpHeadersLike;
    /**
     * The original request that resulted in this response.
     */
    request: WebResourceLike;
    /**
     * The HTTP status code returned from the service.
     */
    status: number;
}
/**
 * An object with a _response property that has
 * headers already parsed into a typed object.
 */
export interface ResponseWithHeaders<Headers> {
    /**
     * The underlying HTTP response.
     */
    _response: HttpResponse & {
        /**
         * The parsed HTTP response headers.
         */
        parsedHeaders: Headers;
    };
}
/**
 * An object with a _response property that has body
 * and headers already parsed into known types.
 */
export interface ResponseWithBody<Headers, Body> {
    /**
     * The underlying HTTP response.
     */
    _response: HttpResponse & {
        /**
         * The parsed HTTP response headers.
         */
        parsedHeaders: Headers;
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: Body;
    };
}
/**
 * An object with a simple _response property.
 */
export interface ResponseLike {
    /**
     * The underlying HTTP response.
     */
    _response: HttpResponse;
}
/**
 * A type that represents an operation result with a known _response property.
 */
export type WithResponse<T, Headers = undefined, Body = undefined> = T & (Body extends object ? ResponseWithBody<Headers, Body> : Headers extends object ? ResponseWithHeaders<Headers> : ResponseLike);
/**
 * A typesafe helper for ensuring that a given response object has
 * the original _response attached.
 * @param response - A response object from calling a client operation
 * @returns The same object, but with known _response property
 */
export declare function assertResponse<T extends object, Headers = undefined, Body = undefined>(response: T): WithResponse<T, Headers, Body>;
export declare function StringEncodedToString(name: StringEncoded): string;
export declare function ConvertInternalResponseOfListFiles(internalResponse: ListFilesAndDirectoriesSegmentResponseInternal): ListFilesAndDirectoriesSegmentResponse;
export declare function ConvertInternalResponseOfListHandles(internalResponse: ListHandlesResponseInternal): ListHandlesResponse;
/**
 * A small helper to handle converting an empty string "" into undefined
 * This is used in the case of query parameters (like continuation token) where
 * we don't want to send an empty query parameter to the service since the signing
 * policy for shared key will fail.
 * @internal
 */
export declare function removeEmptyString(value: string | undefined): string | undefined;
export declare function asSharePermission(value: string | SharePermission): SharePermission;
/**
 * Parse 4-digit octal string representation of a File Mode to a {@link NfsFileMode} structure.
 */
export declare function parseOctalFileMode(input?: string): NfsFileMode | undefined;
/**
 * Convert {@link NfsFileMode} structure to a 4-digit octal string represenation.
 */
export declare function toOctalFileMode(input?: NfsFileMode): string | undefined;
/**
 * Convert a {@link NfsFileMode} to a string in symbolic notation.
 */
export declare function toSymbolicFileMode(input?: NfsFileMode): string | undefined;
/**
 * Parse a 9-character symbolic string representation of a File Mode to a {@link NfsFileMode} structure.
 */
export declare function parseSymbolicFileMode(input?: string): NfsFileMode | undefined;
export declare function parseOctalRolePermissions(c: string): PosixRolePermissions;
export declare function toOctalRolePermissions(rolePermissions: PosixRolePermissions): string;
export declare function toSymbolicRolePermissions(rolePermissions: PosixRolePermissions): string;
export declare function parseSymbolicRolePermissions(input: string): {
    rolePermissions: PosixRolePermissions;
    setSticky: boolean;
};
//# sourceMappingURL=utils.common.d.ts.map