var expect = require('chai').expect;
var systemAssignedIdentityEnabled = require('./systemAssignedIdentityEnabled');

const virtualMachines = [
    {
        'name': 'test-vm',
        'id': '/subscriptions/123/resourceGroups/AQUA-RESOURCE_GROUP/providers/Microsoft.Compute/virtualMachines/test-vm',
        'type': 'Microsoft.Compute/virtualMachines',
        'osProfile': {
            'adminUsername': 'aquauser',
            'linuxConfiguration': {
                'disablePasswordAuthentication': true
            }
        },
        "identity": {
            "type": "SystemAssigned",
            "principalId": "824f877b-e90f-43ab-b135-36e7cfff154",
            "tenantId": "d207c7bd-fcb1-4dd3-855a-cfd2ffff123"
        }
    },
    {
        'name': 'test-vm',
        'id': '/subscriptions/123/resourceGroups/AQUA-RESOURCE_GROUP/providers/Microsoft.Compute/virtualMachines/test-vm',
        'type': 'Microsoft.Compute/virtualMachines',
        'osProfile': {
            'computerName': 'test-vm',
            'linuxConfiguration': {
                'disablePasswordAuthentication': false
            }
        }
    }
];

const createCache = (virtualMachines) => {
    let machine = {};
    if (virtualMachines) {
        machine['data'] = virtualMachines;
    }
    return {
        virtualMachines: {
            listAll: {
                'eastus': machine
            }
        }
    };
};

describe('systemAssignedIdentityEnabled', function() {
    describe('run', function() {
        it('should give passing result if no virtual machines', function(done) {
            const cache = createCache([]);
            systemAssignedIdentityEnabled.run(cache, {}, (err, results) => {
                expect(results.length).to.equal(1);
                expect(results[0].status).to.equal(0);
                expect(results[0].message).to.include('No existing Virtual Machines found');
                expect(results[0].region).to.equal('eastus');
                done();
            });
        });

        it('should give unknown result if unable to query for virtual machines', function(done) {
            const cache = createCache(null);
            systemAssignedIdentityEnabled.run(cache, {}, (err, results) => {
                expect(results.length).to.equal(1);
                expect(results[0].status).to.equal(3);
                expect(results[0].message).to.include('Unable to query for virtualMachines');
                expect(results[0].region).to.equal('eastus');
                done();
            });
        });

        it('should give passing result if system assigned managed identity enabled', function(done) {
            const cache = createCache([virtualMachines[0]]);
            systemAssignedIdentityEnabled.run(cache, {}, (err, results) => {
                expect(results.length).to.equal(1);
                expect(results[0].status).to.equal(0);
                expect(results[0].message).to.include('Virtual Machine has system assigned managed identity enabled');
                expect(results[0].region).to.equal('eastus');
                done();
            });
        });

        it('should give failing result if system assigned managed identity not enabled', function(done) {
            const cache = createCache([virtualMachines[1]]);
            systemAssignedIdentityEnabled.run(cache, {}, (err, results) => {
                expect(results.length).to.equal(1);
                expect(results[0].status).to.equal(2);
                expect(results[0].message).to.include('Virtual Machine does not have system assigned managed identity enabled');
                expect(results[0].region).to.equal('eastus');
                done();
            });
        });
    });
});