/*
 * SpotBugs - Find bugs in Java programs
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package edu.umd.cs.findbugs.detect;

import java.util.HashSet;
import java.util.Set;

/**
 * This class contains all the public identifiers that are part of the
 * Java Standard Library. It is used to detect any shadowing or obscuring
 * of these identifiers.
 * Since there are so many identifiers, they are loaded from separate
 * functions to avoid compiler limits. The functions load roughly the same number
 * of identifiers.
 */
public class PublicIdentifiers {
    private static final Set<String> STANDARD_LIBRARY_PACKAGES = new HashSet<>();
    public static final Set<String> PUBLIC_IDENTIFIERS = new HashSet<>();

    static {
        addStandardPackages();
        addPublicIdentifiersAB();
        addPublicIdentifiersCC();
        addPublicIdentifiersDE();
        addPublicIdentifiersFH();
        addPublicIdentifiersIJ();
        addPublicIdentifiersKM();
        addPublicIdentifiersNR();
        addPublicIdentifiersSS();
        addPublicIdentifiersTU();
        addPublicIdentifiersVZ();
    }

    public static boolean isPartOfStandardLibrary(String containingPackage) {
        return STANDARD_LIBRARY_PACKAGES.stream().anyMatch(containingPackage::startsWith);
    }

    private static void addStandardPackages() {
        STANDARD_LIBRARY_PACKAGES.add("com.sun");

        STANDARD_LIBRARY_PACKAGES.add("java.applet");
        STANDARD_LIBRARY_PACKAGES.add("java.awt");
        STANDARD_LIBRARY_PACKAGES.add("java.beans");
        STANDARD_LIBRARY_PACKAGES.add("java.io");
        STANDARD_LIBRARY_PACKAGES.add("java.lang");
        STANDARD_LIBRARY_PACKAGES.add("java.math");
        STANDARD_LIBRARY_PACKAGES.add("java.net");
        STANDARD_LIBRARY_PACKAGES.add("java.nio");
        STANDARD_LIBRARY_PACKAGES.add("java.rmi");
        STANDARD_LIBRARY_PACKAGES.add("java.security");
        STANDARD_LIBRARY_PACKAGES.add("java.sql");
        STANDARD_LIBRARY_PACKAGES.add("java.text");
        STANDARD_LIBRARY_PACKAGES.add("java.time");
        STANDARD_LIBRARY_PACKAGES.add("java.util");

        STANDARD_LIBRARY_PACKAGES.add("javax.accessibility");
        STANDARD_LIBRARY_PACKAGES.add("javax.annotation");
        STANDARD_LIBRARY_PACKAGES.add("javax.crypto");
        STANDARD_LIBRARY_PACKAGES.add("javax.imageio");
        STANDARD_LIBRARY_PACKAGES.add("javax.lang");
        STANDARD_LIBRARY_PACKAGES.add("javax.management");
        STANDARD_LIBRARY_PACKAGES.add("javax.naming");
        STANDARD_LIBRARY_PACKAGES.add("javax.net");
        STANDARD_LIBRARY_PACKAGES.add("javax.print");
        STANDARD_LIBRARY_PACKAGES.add("javax.rmi");
        STANDARD_LIBRARY_PACKAGES.add("javax.script");
        STANDARD_LIBRARY_PACKAGES.add("javax.security");
        STANDARD_LIBRARY_PACKAGES.add("javax.smartcardio");
        STANDARD_LIBRARY_PACKAGES.add("javax.sound");
        STANDARD_LIBRARY_PACKAGES.add("javax.sql");
        STANDARD_LIBRARY_PACKAGES.add("javax.swing");
        STANDARD_LIBRARY_PACKAGES.add("javax.tools");
        STANDARD_LIBRARY_PACKAGES.add("javax.transaction");
        STANDARD_LIBRARY_PACKAGES.add("javax.xml");

        STANDARD_LIBRARY_PACKAGES.add("jdk.dynalink");
        STANDARD_LIBRARY_PACKAGES.add("jdk.editpad");
        STANDARD_LIBRARY_PACKAGES.add("jdk.incubator");
        STANDARD_LIBRARY_PACKAGES.add("jdk.internal");
        STANDARD_LIBRARY_PACKAGES.add("jdk.javadoc");
        STANDARD_LIBRARY_PACKAGES.add("jdk.jfr");
        STANDARD_LIBRARY_PACKAGES.add("jdk.jpackage");
        STANDARD_LIBRARY_PACKAGES.add("jdk.jshell");
        STANDARD_LIBRARY_PACKAGES.add("jdk.management");
        STANDARD_LIBRARY_PACKAGES.add("jdk.net");
        STANDARD_LIBRARY_PACKAGES.add("jdk.nio");
        STANDARD_LIBRARY_PACKAGES.add("jdk.random");
        STANDARD_LIBRARY_PACKAGES.add("jdk.security");
        STANDARD_LIBRARY_PACKAGES.add("jdk.swing");
        STANDARD_LIBRARY_PACKAGES.add("jdk.tools");
        STANDARD_LIBRARY_PACKAGES.add("jdk.vm");
        STANDARD_LIBRARY_PACKAGES.add("jdk.xml");

        STANDARD_LIBRARY_PACKAGES.add("netscape.javascript");

        STANDARD_LIBRARY_PACKAGES.add("org.ietf");
        STANDARD_LIBRARY_PACKAGES.add("org.jcp");
        STANDARD_LIBRARY_PACKAGES.add("org.w3c");
        STANDARD_LIBRARY_PACKAGES.add("org.xml");

        STANDARD_LIBRARY_PACKAGES.add("sun.awt");
        STANDARD_LIBRARY_PACKAGES.add("sun.datatransfer");
        STANDARD_LIBRARY_PACKAGES.add("sun.font");
        STANDARD_LIBRARY_PACKAGES.add("sun.instrument");
        STANDARD_LIBRARY_PACKAGES.add("sun.invoke");
        STANDARD_LIBRARY_PACKAGES.add("sun.java2d");
        STANDARD_LIBRARY_PACKAGES.add("sun.jvm");
        STANDARD_LIBRARY_PACKAGES.add("sun.jvmstat");
        STANDARD_LIBRARY_PACKAGES.add("sun.launcher");
        STANDARD_LIBRARY_PACKAGES.add("sun.management");
        STANDARD_LIBRARY_PACKAGES.add("sun.misc");
        STANDARD_LIBRARY_PACKAGES.add("sun.net");
        STANDARD_LIBRARY_PACKAGES.add("sun.nio");
        STANDARD_LIBRARY_PACKAGES.add("sun.print");
        STANDARD_LIBRARY_PACKAGES.add("sun.reflect");
        STANDARD_LIBRARY_PACKAGES.add("sun.rmi");
        STANDARD_LIBRARY_PACKAGES.add("sun.security");
        STANDARD_LIBRARY_PACKAGES.add("sun.swing");
        STANDARD_LIBRARY_PACKAGES.add("sun.text");
        STANDARD_LIBRARY_PACKAGES.add("sun.tools");
        STANDARD_LIBRARY_PACKAGES.add("sun.usagetracker");
        STANDARD_LIBRARY_PACKAGES.add("sun.util");

        STANDARD_LIBRARY_PACKAGES.add("toolbarButtonGraphics.development");
        STANDARD_LIBRARY_PACKAGES.add("toolbarButtonGraphics.general");
        STANDARD_LIBRARY_PACKAGES.add("toolbarButtonGraphics.navigation");
        STANDARD_LIBRARY_PACKAGES.add("toolbarButtonGraphics.text");
    }

    private static void addPublicIdentifiersAB() {
        PUBLIC_IDENTIFIERS.add("AALOAD");
        PUBLIC_IDENTIFIERS.add("AArch64");
        PUBLIC_IDENTIFIERS.add("AArch64Architecture");
        PUBLIC_IDENTIFIERS.add("AARCH64CurrentFrameGuess");
        PUBLIC_IDENTIFIERS.add("AARCH64Frame");
        PUBLIC_IDENTIFIERS.add("AArch64HotSpotJVMCIBackendFactory");
        PUBLIC_IDENTIFIERS.add("AArch64HotSpotRegisterConfig");
        PUBLIC_IDENTIFIERS.add("AARCH64JavaCallWrapper");
        PUBLIC_IDENTIFIERS.add("AARCH64RegisterMap");
        PUBLIC_IDENTIFIERS.add("AARCH64ThreadContext");
        PUBLIC_IDENTIFIERS.add("AASTORE");
        PUBLIC_IDENTIFIERS.add("AATextRenderer");
        PUBLIC_IDENTIFIERS.add("AATileGenerator");
        PUBLIC_IDENTIFIERS.add("AbbreviationMap");
        PUBLIC_IDENTIFIERS.add("ABIDescriptor");
        PUBLIC_IDENTIFIERS.add("Abort");
        PUBLIC_IDENTIFIERS.add("AbortException");
        PUBLIC_IDENTIFIERS.add("AbortPolicy");
        PUBLIC_IDENTIFIERS.add("AboutAction");
        PUBLIC_IDENTIFIERS.add("AboutDialog");
        PUBLIC_IDENTIFIERS.add("AboutHandler");
        PUBLIC_IDENTIFIERS.add("AbsentInformationException");
        PUBLIC_IDENTIFIERS.add("AbstractAction");
        PUBLIC_IDENTIFIERS.add("AbstractAlgorithmConstraints");
        PUBLIC_IDENTIFIERS.add("AbstractAnnotationValueVisitor14");
        PUBLIC_IDENTIFIERS.add("AbstractAnnotationValueVisitor6");
        PUBLIC_IDENTIFIERS.add("AbstractAnnotationValueVisitor7");
        PUBLIC_IDENTIFIERS.add("AbstractAnnotationValueVisitor8");
        PUBLIC_IDENTIFIERS.add("AbstractAnnotationValueVisitor9");
        PUBLIC_IDENTIFIERS.add("AbstractAppImageBuilder");
        PUBLIC_IDENTIFIERS.add("AbstractArbitrarilyJumpableGenerator");
        PUBLIC_IDENTIFIERS.add("AbstractBorder");
        PUBLIC_IDENTIFIERS.add("AbstractBufferStatisticsEvent");
        PUBLIC_IDENTIFIERS.add("AbstractBuilder");
        PUBLIC_IDENTIFIERS.add("AbstractButton");
        PUBLIC_IDENTIFIERS.add("AbstractCalendar");
        PUBLIC_IDENTIFIERS.add("AbstractCellEditor");
        PUBLIC_IDENTIFIERS.add("AbstractCharsetProvider");
        PUBLIC_IDENTIFIERS.add("AbstractChronology");
        PUBLIC_IDENTIFIERS.add("AbstractClassLoaderValue");
        PUBLIC_IDENTIFIERS.add("AbstractCollection");
        PUBLIC_IDENTIFIERS.add("AbstractColorChooserPanel");
        PUBLIC_IDENTIFIERS.add("AbstractCounter");
        PUBLIC_IDENTIFIERS.add("AbstractDateTimeDV");
        PUBLIC_IDENTIFIERS.add("AbstractDelegateHttpsURLConnection");
        PUBLIC_IDENTIFIERS.add("AbstractDiagnosticFormatter");
        PUBLIC_IDENTIFIERS.add("AbstractDoclet");
        PUBLIC_IDENTIFIERS.add("AbstractDocument");
        PUBLIC_IDENTIFIERS.add("AbstractDOMParser");
        PUBLIC_IDENTIFIERS.add("AbstractDoubleSpliterator");
        PUBLIC_IDENTIFIERS.add("AbstractDrbg");
        PUBLIC_IDENTIFIERS.add("AbstractElement");
        PUBLIC_IDENTIFIERS.add("AbstractElementVisitor14");
        PUBLIC_IDENTIFIERS.add("AbstractElementVisitor6");
        PUBLIC_IDENTIFIERS.add("AbstractElementVisitor7");
        PUBLIC_IDENTIFIERS.add("AbstractElementVisitor8");
        PUBLIC_IDENTIFIERS.add("AbstractElementVisitor9");
        PUBLIC_IDENTIFIERS.add("AbstractEndPosTable");
        PUBLIC_IDENTIFIERS.add("AbstractEventStream");
        PUBLIC_IDENTIFIERS.add("AbstractExecutableMemberWriter");
        PUBLIC_IDENTIFIERS.add("AbstractExecutorService");
        PUBLIC_IDENTIFIERS.add("AbstractFileSystemProvider");
        PUBLIC_IDENTIFIERS.add("AbstractFileTypeDetector");
        PUBLIC_IDENTIFIERS.add("AbstractFormatter");
        PUBLIC_IDENTIFIERS.add("AbstractFormatterFactory");
        PUBLIC_IDENTIFIERS.add("AbstractGroupLayout");
        PUBLIC_IDENTIFIERS.add("AbstractHashDrbg");
        PUBLIC_IDENTIFIERS.add("AbstractHeapGraphWriter");
        PUBLIC_IDENTIFIERS.add("AbstractInsnNode");
        PUBLIC_IDENTIFIERS.add("AbstractInterruptibleChannel");
        PUBLIC_IDENTIFIERS.add("AbstractIntSpliterator");
        PUBLIC_IDENTIFIERS.add("AbstractJavaProfile");
        PUBLIC_IDENTIFIERS.add("AbstractLayout");
        PUBLIC_IDENTIFIERS.add("AbstractLayoutCache");
        PUBLIC_IDENTIFIERS.add("AbstractLinker");
        PUBLIC_IDENTIFIERS.add("AbstractList");
        PUBLIC_IDENTIFIERS.add("AbstractListModel");
        PUBLIC_IDENTIFIERS.add("AbstractLog");
        PUBLIC_IDENTIFIERS.add("AbstractLongSpliterator");
        PUBLIC_IDENTIFIERS.add("AbstractMap");
        PUBLIC_IDENTIFIERS.add("AbstractMemberBuilder");
        PUBLIC_IDENTIFIERS.add("AbstractMemberWriter");
        PUBLIC_IDENTIFIERS.add("AbstractMemorySegmentImpl");
        PUBLIC_IDENTIFIERS.add("AbstractMethodError");
        PUBLIC_IDENTIFIERS.add("AbstractMidiDeviceProvider");
        PUBLIC_IDENTIFIERS.add("AbstractMonitor");
        PUBLIC_IDENTIFIERS.add("AbstractMonitoredVm");
        PUBLIC_IDENTIFIERS.add("AbstractMultiResolutionImage");
        PUBLIC_IDENTIFIERS.add("AbstractNode");
        PUBLIC_IDENTIFIERS.add("AbstractNotificationHandler");
        PUBLIC_IDENTIFIERS.add("AbstractOptionSpec");
        PUBLIC_IDENTIFIERS.add("AbstractOverviewIndexWriter");
        PUBLIC_IDENTIFIERS.add("AbstractOwnableSynchronizer");
        PUBLIC_IDENTIFIERS.add("AbstractPerfDataBuffer");
        PUBLIC_IDENTIFIERS.add("AbstractPerfDataBufferPrologue");
        PUBLIC_IDENTIFIERS.add("AbstractPlugin");
        PUBLIC_IDENTIFIERS.add("AbstractPosixTerminal");
        PUBLIC_IDENTIFIERS.add("AbstractPreferences");
        PUBLIC_IDENTIFIERS.add("AbstractProcessor");
        PUBLIC_IDENTIFIERS.add("AbstractProfiledItem");
        PUBLIC_IDENTIFIERS.add("AbstractPty");
        PUBLIC_IDENTIFIERS.add("AbstractQueue");
        PUBLIC_IDENTIFIERS.add("AbstractQueuedLongSynchronizer");
        PUBLIC_IDENTIFIERS.add("AbstractQueuedSynchronizer");
        PUBLIC_IDENTIFIERS.add("AbstractRegionPainter");
        PUBLIC_IDENTIFIERS.add("AbstractRelinkableCallSite");
        PUBLIC_IDENTIFIERS.add("AbstractRepository");
        PUBLIC_IDENTIFIERS.add("AbstractResourceBundleProvider");
        PUBLIC_IDENTIFIERS.add("AbstractSaslImpl");
        PUBLIC_IDENTIFIERS.add("AbstractSAXParser");
        PUBLIC_IDENTIFIERS.add("AbstractScope");
        PUBLIC_IDENTIFIERS.add("AbstractScriptEngine");
        PUBLIC_IDENTIFIERS.add("AbstractSelectableChannel");
        PUBLIC_IDENTIFIERS.add("AbstractSelectionKey");
        PUBLIC_IDENTIFIERS.add("AbstractSelector");
        PUBLIC_IDENTIFIERS.add("AbstractSequentialList");
        PUBLIC_IDENTIFIERS.add("AbstractSet");
        PUBLIC_IDENTIFIERS.add("AbstractSpinnerModel");
        PUBLIC_IDENTIFIERS.add("AbstractSpliterator");
        PUBLIC_IDENTIFIERS.add("AbstractSpliteratorGenerator");
        PUBLIC_IDENTIFIERS.add("AbstractSplittableGenerator");
        PUBLIC_IDENTIFIERS.add("AbstractSplittableWithBrineGenerator");
        PUBLIC_IDENTIFIERS.add("AbstractTableModel");
        PUBLIC_IDENTIFIERS.add("AbstractTerminal");
        PUBLIC_IDENTIFIERS.add("AbstractTranslet");
        PUBLIC_IDENTIFIERS.add("AbstractTreeTableModel");
        PUBLIC_IDENTIFIERS.add("AbstractTreeWriter");
        PUBLIC_IDENTIFIERS.add("AbstractTubePublisher");
        PUBLIC_IDENTIFIERS.add("AbstractTubeSubscriber");
        PUBLIC_IDENTIFIERS.add("AbstractTypeVisitor14");
        PUBLIC_IDENTIFIERS.add("AbstractTypeVisitor6");
        PUBLIC_IDENTIFIERS.add("AbstractTypeVisitor7");
        PUBLIC_IDENTIFIERS.add("AbstractTypeVisitor8");
        PUBLIC_IDENTIFIERS.add("AbstractTypeVisitor9");
        PUBLIC_IDENTIFIERS.add("AbstractUndoableEdit");
        PUBLIC_IDENTIFIERS.add("AbstractView");
        PUBLIC_IDENTIFIERS.add("AbstractWindowsConsoleWriter");
        PUBLIC_IDENTIFIERS.add("AbstractWindowsTerminal");
        PUBLIC_IDENTIFIERS.add("AbstractWriter");
        PUBLIC_IDENTIFIERS.add("AbstractXMLDocumentParser");
        PUBLIC_IDENTIFIERS.add("AccelGraphicsConfig");
        PUBLIC_IDENTIFIERS.add("AccelSurface");
        PUBLIC_IDENTIFIERS.add("AccelTypedVolatileImage");
        PUBLIC_IDENTIFIERS.add("AcceptAllFileFilter");
        PUBLIC_IDENTIFIERS.add("AcceptPendingException");
        PUBLIC_IDENTIFIERS.add("AccessControl");
        PUBLIC_IDENTIFIERS.add("AccessControlException");
        PUBLIC_IDENTIFIERS.add("AccessDeniedException");
        PUBLIC_IDENTIFIERS.add("AccessException");
        PUBLIC_IDENTIFIERS.add("AccessFlags");
        PUBLIC_IDENTIFIERS.add("AccessibilityEventMonitor");
        PUBLIC_IDENTIFIERS.add("AccessibilityListenerList");
        PUBLIC_IDENTIFIERS.add("AccessibilityProvider");
        PUBLIC_IDENTIFIERS.add("Accessible");
        PUBLIC_IDENTIFIERS.add("AccessibleAbstractButton");
        PUBLIC_IDENTIFIERS.add("AccessibleAction");
        PUBLIC_IDENTIFIERS.add("AccessibleApplet");
        PUBLIC_IDENTIFIERS.add("AccessibleAttributeSequence");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTButton");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTCanvas");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTCheckbox");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTCheckboxMenuItem");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTChoice");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTComponent");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTComponentHandler");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTContainer");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTDialog");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTFocusHandler");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTFrame");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTLabel");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTList");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTListChild");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTMenu");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTMenuBar");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTMenuComponent");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTMenuItem");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTPanel");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTPopupMenu");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTScrollBar");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTScrollPane");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTTextArea");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTTextComponent");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTTextField");
        PUBLIC_IDENTIFIERS.add("AccessibleAWTWindow");
        PUBLIC_IDENTIFIERS.add("AccessibleBox");
        PUBLIC_IDENTIFIERS.add("AccessibleBoxFiller");
        PUBLIC_IDENTIFIERS.add("AccessibleBundle");
        PUBLIC_IDENTIFIERS.add("AccessibleBundleAccessor");
        PUBLIC_IDENTIFIERS.add("(AccessibleContainerHandler");
        PUBLIC_IDENTIFIERS.add("AccessibleCellRendererPane");
        PUBLIC_IDENTIFIERS.add("AccessibleComponent");
        PUBLIC_IDENTIFIERS.add("AccessibleContainerHandler");
        PUBLIC_IDENTIFIERS.add("AccessibleContext");
        PUBLIC_IDENTIFIERS.add("AccessibleContextAccessor");
        PUBLIC_IDENTIFIERS.add("AccessibleEditableText");
        PUBLIC_IDENTIFIERS.add("AccessibleExtendedComponent");
        PUBLIC_IDENTIFIERS.add("AccessibleExtendedTable");
        PUBLIC_IDENTIFIERS.add("AccessibleExtendedText");
        PUBLIC_IDENTIFIERS.add("AccessibleFocusHandler");
        PUBLIC_IDENTIFIERS.add("AccessibleHeadersTable");
        PUBLIC_IDENTIFIERS.add("AccessibleHyperlink");
        PUBLIC_IDENTIFIERS.add("AccessibleHypertext");
        PUBLIC_IDENTIFIERS.add("AccessibleIcon");
        PUBLIC_IDENTIFIERS.add("AccessibleImageIcon");
        PUBLIC_IDENTIFIERS.add("AccessibleJApplet");
        PUBLIC_IDENTIFIERS.add("AccessibleJButton");
        PUBLIC_IDENTIFIERS.add("AccessibleJCheckBox");
        PUBLIC_IDENTIFIERS.add("AccessibleJCheckBoxMenuItem");
        PUBLIC_IDENTIFIERS.add("AccessibleJColorChooser");
        PUBLIC_IDENTIFIERS.add("AccessibleJComboBox");
        PUBLIC_IDENTIFIERS.add("AccessibleJComponent");
        PUBLIC_IDENTIFIERS.add("AccessibleJDesktopIcon");
        PUBLIC_IDENTIFIERS.add("AccessibleJDesktopPane");
        PUBLIC_IDENTIFIERS.add("AccessibleJDialog");
        PUBLIC_IDENTIFIERS.add("AccessibleJEditorPane");
        PUBLIC_IDENTIFIERS.add("AccessibleJEditorPaneHTML");
        PUBLIC_IDENTIFIERS.add("AccessibleJFileChooser");
        PUBLIC_IDENTIFIERS.add("AccessibleJFrame");
        PUBLIC_IDENTIFIERS.add("AccessibleJInternalFrame");
        PUBLIC_IDENTIFIERS.add("AccessibleJLabel");
        PUBLIC_IDENTIFIERS.add("AccessibleJLayeredPane");
        PUBLIC_IDENTIFIERS.add("AccessibleJList");
        PUBLIC_IDENTIFIERS.add("AccessibleJListChild");
        PUBLIC_IDENTIFIERS.add("AccessibleJMenu");
        PUBLIC_IDENTIFIERS.add("AccessibleJMenuBar");
        PUBLIC_IDENTIFIERS.add("AccessibleJMenuItem");
        PUBLIC_IDENTIFIERS.add("AccessibleJOptionPane");
        PUBLIC_IDENTIFIERS.add("AccessibleJPanel");
        PUBLIC_IDENTIFIERS.add("AccessibleJPasswordField");
        PUBLIC_IDENTIFIERS.add("AccessibleJPopupMenu");
        PUBLIC_IDENTIFIERS.add("AccessibleJProgressBar");
        PUBLIC_IDENTIFIERS.add("AccessibleJRadioButton");
        PUBLIC_IDENTIFIERS.add("AccessibleJRadioButtonMenuItem");
        PUBLIC_IDENTIFIERS.add("AccessibleJRootPane");
        PUBLIC_IDENTIFIERS.add("AccessibleJScrollBar");
        PUBLIC_IDENTIFIERS.add("AccessibleJScrollPane");
        PUBLIC_IDENTIFIERS.add("AccessibleJSeparator");
        PUBLIC_IDENTIFIERS.add("AccessibleJSlider");
        PUBLIC_IDENTIFIERS.add("AccessibleJSpinner");
        PUBLIC_IDENTIFIERS.add("AccessibleJSplitPane");
        PUBLIC_IDENTIFIERS.add("AccessibleJTabbedPane");
        PUBLIC_IDENTIFIERS.add("AccessibleJTable");
        PUBLIC_IDENTIFIERS.add("AccessibleJTableCell");
        PUBLIC_IDENTIFIERS.add("AccessibleJTableHeader");
        PUBLIC_IDENTIFIERS.add("AccessibleJTableHeaderEntry");
        PUBLIC_IDENTIFIERS.add("AccessibleJTableModelChange");
        PUBLIC_IDENTIFIERS.add("AccessibleJTextArea");
        PUBLIC_IDENTIFIERS.add("AccessibleJTextComponent");
        PUBLIC_IDENTIFIERS.add("AccessibleJTextField");
        PUBLIC_IDENTIFIERS.add("AccessibleJToggleButton");
        PUBLIC_IDENTIFIERS.add("AccessibleJToolBar");
        PUBLIC_IDENTIFIERS.add("AccessibleJToolTip");
        PUBLIC_IDENTIFIERS.add("AccessibleJTree");
        PUBLIC_IDENTIFIERS.add("AccessibleJTreeNode");
        PUBLIC_IDENTIFIERS.add("AccessibleJViewport");
        PUBLIC_IDENTIFIERS.add("AccessibleJWindow");
        PUBLIC_IDENTIFIERS.add("AccessibleKeyBinding");
        PUBLIC_IDENTIFIERS.add("AccessibleObject");
        PUBLIC_IDENTIFIERS.add("AccessiblePlotter");
        PUBLIC_IDENTIFIERS.add("AccessiblePlotterPanel");
        PUBLIC_IDENTIFIERS.add("AccessiblePoolChart");
        PUBLIC_IDENTIFIERS.add("AccessibleProgressMonitor");
        PUBLIC_IDENTIFIERS.add("AccessibleRelation");
        PUBLIC_IDENTIFIERS.add("AccessibleRelationSet");
        PUBLIC_IDENTIFIERS.add("AccessibleResourceBundle");
        PUBLIC_IDENTIFIERS.add("AccessibleRole");
        PUBLIC_IDENTIFIERS.add("AccessibleSelection");
        PUBLIC_IDENTIFIERS.add("AccessibleState");
        PUBLIC_IDENTIFIERS.add("AccessibleStateSet");
        PUBLIC_IDENTIFIERS.add("AccessibleStreamable");
        PUBLIC_IDENTIFIERS.add("AccessibleTable");
        PUBLIC_IDENTIFIERS.add("AccessibleTableModelChange");
        PUBLIC_IDENTIFIERS.add("AccessibleText");
        PUBLIC_IDENTIFIERS.add("AccessibleTextSequence");
        PUBLIC_IDENTIFIERS.add("AccessibleValue");
        PUBLIC_IDENTIFIERS.add("AccessorUtils");
        PUBLIC_IDENTIFIERS.add("AccessWatchpointEvent");
        PUBLIC_IDENTIFIERS.add("AccessWatchpointRequest");
        PUBLIC_IDENTIFIERS.add("AccountException");
        PUBLIC_IDENTIFIERS.add("AccountExpiredException");
        PUBLIC_IDENTIFIERS.add("AccountLockedException");
        PUBLIC_IDENTIFIERS.add("AccountNotFoundException");
        PUBLIC_IDENTIFIERS.add("AccumulativeRunnable");
        PUBLIC_IDENTIFIERS.add("AclFileAttributeView");
        PUBLIC_IDENTIFIERS.add("ACONST_NULL");
        PUBLIC_IDENTIFIERS.add("Action");
        PUBLIC_IDENTIFIERS.add("ActionEvent");
        PUBLIC_IDENTIFIERS.add("ActionListener");
        PUBLIC_IDENTIFIERS.add("ActionManager");
        PUBLIC_IDENTIFIERS.add("ActionMap");
        PUBLIC_IDENTIFIERS.add("ActionMapUIResource");
        PUBLIC_IDENTIFIERS.add("ActionScroller");
        PUBLIC_IDENTIFIERS.add("ActiveEvent");
        PUBLIC_IDENTIFIERS.add("ActiveValue");
        PUBLIC_IDENTIFIERS.add("AdapterBlob");
        PUBLIC_IDENTIFIERS.add("AdaptFailure");
        PUBLIC_IDENTIFIERS.add("AdaptiveResultTreeImpl");
        PUBLIC_IDENTIFIERS.add("AddOp");
        PUBLIC_IDENTIFIERS.add("Address");
        PUBLIC_IDENTIFIERS.add("AddressDataSource");
        PUBLIC_IDENTIFIERS.add("AddressException");
        PUBLIC_IDENTIFIERS.add("AddressField");
        PUBLIC_IDENTIFIERS.add("AddressOps");
        PUBLIC_IDENTIFIERS.add("AddressVisitor");
        PUBLIC_IDENTIFIERS.add("AdjacencyList");
        PUBLIC_IDENTIFIERS.add("Adjustable");
        PUBLIC_IDENTIFIERS.add("AdjustmentEvent");
        PUBLIC_IDENTIFIERS.add("AdjustmentListener");
        PUBLIC_IDENTIFIERS.add("Adler32");
        PUBLIC_IDENTIFIERS.add("AdviceAdapter");
        PUBLIC_IDENTIFIERS.add("AEADBadTagException");
        PUBLIC_IDENTIFIERS.add("Aes128");
        PUBLIC_IDENTIFIERS.add("Aes128Sha2");
        PUBLIC_IDENTIFIERS.add("Aes256");
        PUBLIC_IDENTIFIERS.add("Aes256Sha2");
        PUBLIC_IDENTIFIERS.add("AesDkCrypto");
        PUBLIC_IDENTIFIERS.add("AesSha2DkCrypto");
        PUBLIC_IDENTIFIERS.add("AffinePoint");
        PUBLIC_IDENTIFIERS.add("AffineTransform");
        PUBLIC_IDENTIFIERS.add("AffineTransformOp");
        PUBLIC_IDENTIFIERS.add("Agent");
        PUBLIC_IDENTIFIERS.add("AgentConfigurationError");
        PUBLIC_IDENTIFIERS.add("AgentInitializationException");
        PUBLIC_IDENTIFIERS.add("AgentLoadException");
        PUBLIC_IDENTIFIERS.add("AggregateCompleter");
        PUBLIC_IDENTIFIERS.add("aiff");
        PUBLIC_IDENTIFIERS.add("Algorithm");
        PUBLIC_IDENTIFIERS.add("AlgorithmAlreadyRegisteredException");
        PUBLIC_IDENTIFIERS.add("AlgorithmConstraints");
        PUBLIC_IDENTIFIERS.add("AlgorithmDecomposer");
        PUBLIC_IDENTIFIERS.add("AlgorithmId");
        PUBLIC_IDENTIFIERS.add("AlgorithmMethod");
        PUBLIC_IDENTIFIERS.add("AlgorithmParameterGenerator");
        PUBLIC_IDENTIFIERS.add("AlgorithmParameterGeneratorSpi");
        PUBLIC_IDENTIFIERS.add("AlgorithmParameters");
        PUBLIC_IDENTIFIERS.add("AlgorithmParameterSpec");
        PUBLIC_IDENTIFIERS.add("AlgorithmParametersSpi");
        PUBLIC_IDENTIFIERS.add("AliasFileParser");
        PUBLIC_IDENTIFIERS.add("AlignCenterAction");
        PUBLIC_IDENTIFIERS.add("AlignLeftAction");
        PUBLIC_IDENTIFIERS.add("Alignment");
        PUBLIC_IDENTIFIERS.add("AlignmentAction");
        PUBLIC_IDENTIFIERS.add("AlignRightAction");
        PUBLIC_IDENTIFIERS.add("AllClassesIndexWriter");
        PUBLIC_IDENTIFIERS.add("AllocatableValue");
        PUBLIC_IDENTIFIERS.add("AllocationInstruction");
        PUBLIC_IDENTIFIERS.add("AllPackagesIndexWriter");
        PUBLIC_IDENTIFIERS.add("ALOAD");
        PUBLIC_IDENTIFIERS.add("AlphaColorPipe");
        PUBLIC_IDENTIFIERS.add("AlphaPaintPipe");
        PUBLIC_IDENTIFIERS.add("AlreadyBoundException");
        PUBLIC_IDENTIFIERS.add("AlreadyConnectedException");
        PUBLIC_IDENTIFIERS.add("AltPlatformInfo");
        PUBLIC_IDENTIFIERS.add("AmbiguousMethodException");
        PUBLIC_IDENTIFIERS.add("AMD64");
        PUBLIC_IDENTIFIERS.add("AMD64CurrentFrameGuess");
        PUBLIC_IDENTIFIERS.add("AMD64HotSpotJVMCIBackendFactory");
        PUBLIC_IDENTIFIERS.add("AMD64HotSpotRegisterConfig");
        PUBLIC_IDENTIFIERS.add("AMD64JavaCallWrapper");
        PUBLIC_IDENTIFIERS.add("AMD64ThreadContext");
        PUBLIC_IDENTIFIERS.add("Analyzer");
        PUBLIC_IDENTIFIERS.add("AnalyzerAdapter");
        PUBLIC_IDENTIFIERS.add("AnalyzerException");
        PUBLIC_IDENTIFIERS.add("AncestorEvent");
        PUBLIC_IDENTIFIERS.add("AncestorIterator");
        PUBLIC_IDENTIFIERS.add("AncestorListener");
        PUBLIC_IDENTIFIERS.add("AnchorCertificates");
        PUBLIC_IDENTIFIERS.add("And");
        PUBLIC_IDENTIFIERS.add("ANEWARRAY");
        PUBLIC_IDENTIFIERS.add("AnnoConstruct");
        PUBLIC_IDENTIFIERS.add("Annotate");
        PUBLIC_IDENTIFIERS.add("AnnotatedArrayType");
        PUBLIC_IDENTIFIERS.add("AnnotatedConstruct");
        PUBLIC_IDENTIFIERS.add("AnnotatedElement");
        PUBLIC_IDENTIFIERS.add("AnnotatedMemoryPanel");
        PUBLIC_IDENTIFIERS.add("AnnotatedParameterizedType");
        PUBLIC_IDENTIFIERS.add("AnnotatedType");
        PUBLIC_IDENTIFIERS.add("AnnotatedTypeTree");
        PUBLIC_IDENTIFIERS.add("AnnotatedTypeVariable");
        PUBLIC_IDENTIFIERS.add("AnnotatedWildcardType");
        PUBLIC_IDENTIFIERS.add("Annotation");
        PUBLIC_IDENTIFIERS.add("Annotation_element_value");
        PUBLIC_IDENTIFIERS.add("AnnotationDefault");
        PUBLIC_IDENTIFIERS.add("AnnotationDefault_attribute");
        PUBLIC_IDENTIFIERS.add("AnnotationElementValue");
        PUBLIC_IDENTIFIERS.add("AnnotationElementValueGen");
        PUBLIC_IDENTIFIERS.add("AnnotationEntry");
        PUBLIC_IDENTIFIERS.add("AnnotationEntryGen");
        PUBLIC_IDENTIFIERS.add("AnnotationFormatError");
        PUBLIC_IDENTIFIERS.add("AnnotationMirror");
        PUBLIC_IDENTIFIERS.add("AnnotationNode");
        PUBLIC_IDENTIFIERS.add("AnnotationParser");
        PUBLIC_IDENTIFIERS.add("AnnotationProcessingError");
        PUBLIC_IDENTIFIERS.add("AnnotationProxyMaker");
        PUBLIC_IDENTIFIERS.add("AnnotationRemapper");
        PUBLIC_IDENTIFIERS.add("Annotations");
        PUBLIC_IDENTIFIERS.add("AnnotationTree");
        PUBLIC_IDENTIFIERS.add("AnnotationType");
        PUBLIC_IDENTIFIERS.add("AnnotationTypeCompleter");
        PUBLIC_IDENTIFIERS.add("AnnotationTypeMemberBuilder");
        PUBLIC_IDENTIFIERS.add("AnnotationTypeMemberWriter");
        PUBLIC_IDENTIFIERS.add("AnnotationTypeMemberWriterImpl");
        PUBLIC_IDENTIFIERS.add("AnnotationTypeMetadata");
        PUBLIC_IDENTIFIERS.add("AnnotationTypeMismatchException");
        PUBLIC_IDENTIFIERS.add("AnnotationTypeVisitor");
        PUBLIC_IDENTIFIERS.add("AnnotationValue");
        PUBLIC_IDENTIFIERS.add("AnnotationValueVisitor");
        PUBLIC_IDENTIFIERS.add("AnnotationVisitor");
        PUBLIC_IDENTIFIERS.add("AnnotationWriter");
        PUBLIC_IDENTIFIERS.add("AnsiWriter");
        PUBLIC_IDENTIFIERS.add("AnyNodeCounter");
        PUBLIC_IDENTIFIERS.add("AnySimpleDV");
        PUBLIC_IDENTIFIERS.add("AnyURIDV");
        PUBLIC_IDENTIFIERS.add("ApacheCanonicalizer");
        PUBLIC_IDENTIFIERS.add("ApacheData");
        PUBLIC_IDENTIFIERS.add("ApacheNodeSetData");
        PUBLIC_IDENTIFIERS.add("ApacheOctetStreamData");
        PUBLIC_IDENTIFIERS.add("ApacheTransform");
        PUBLIC_IDENTIFIERS.add("APOptions");
        PUBLIC_IDENTIFIERS.add("AppConfigurationEntry");
        PUBLIC_IDENTIFIERS.add("append_frame");
        PUBLIC_IDENTIFIERS.add("Appendable");
        PUBLIC_IDENTIFIERS.add("AppEvent");
        PUBLIC_IDENTIFIERS.add("AppForegroundListener");
        PUBLIC_IDENTIFIERS.add("AppHiddenListener");
        PUBLIC_IDENTIFIERS.add("Applet");
        PUBLIC_IDENTIFIERS.add("AppletContext");
        PUBLIC_IDENTIFIERS.add("AppletInitializer");
        PUBLIC_IDENTIFIERS.add("AppletStub");
        PUBLIC_IDENTIFIERS.add("ApplyAction");
        PUBLIC_IDENTIFIERS.add("AppReopenedListener");
        PUBLIC_IDENTIFIERS.add("ApproveSelectionAction");
        PUBLIC_IDENTIFIERS.add("APRep");
        PUBLIC_IDENTIFIERS.add("APReq");
        PUBLIC_IDENTIFIERS.add("ArbitrarilyJumpableGenerator");
        PUBLIC_IDENTIFIERS.add("Arc2D");
        PUBLIC_IDENTIFIERS.add("ARCFOURCipher");
        PUBLIC_IDENTIFIERS.add("ArcFourCrypto");
        PUBLIC_IDENTIFIERS.add("ArcFourHmac");
        PUBLIC_IDENTIFIERS.add("Architecture");
        PUBLIC_IDENTIFIERS.add("Archive");
        PUBLIC_IDENTIFIERS.add("Area");
        PUBLIC_IDENTIFIERS.add("AreaAveragingScaleFilter");
        PUBLIC_IDENTIFIERS.add("AreaOp");
        PUBLIC_IDENTIFIERS.add("Arena");
        PUBLIC_IDENTIFIERS.add("ARETURN");
        PUBLIC_IDENTIFIERS.add("Arg");
        PUBLIC_IDENTIFIERS.add("Argb");
        PUBLIC_IDENTIFIERS.add("ArgbBm");
        PUBLIC_IDENTIFIERS.add("ArgbPre");
        PUBLIC_IDENTIFIERS.add("ArgInfoData");
        PUBLIC_IDENTIFIERS.add("Argument");
        PUBLIC_IDENTIFIERS.add("ArgumentAcceptingOptionSpec");
        PUBLIC_IDENTIFIERS.add("ArgumentAttr");
        PUBLIC_IDENTIFIERS.add("ArgumentCompleter");
        PUBLIC_IDENTIFIERS.add("ArgumentLine");
        PUBLIC_IDENTIFIERS.add("ArgumentList");
        PUBLIC_IDENTIFIERS.add("Arguments");
        PUBLIC_IDENTIFIERS.add("ArgumentSizeComputer");
        PUBLIC_IDENTIFIERS.add("ArithmeticException");
        PUBLIC_IDENTIFIERS.add("ArithmeticInstruction");
        PUBLIC_IDENTIFIERS.add("Array_element_value");
        PUBLIC_IDENTIFIERS.add("ArrayAccessTree");
        PUBLIC_IDENTIFIERS.add("ArrayBlockingQueue");
        PUBLIC_IDENTIFIERS.add("ArrayCache");
        PUBLIC_IDENTIFIERS.add("ArrayDecoder");
        PUBLIC_IDENTIFIERS.add("ArrayDeque");
        PUBLIC_IDENTIFIERS.add("ArrayElementValue");
        PUBLIC_IDENTIFIERS.add("ArrayElementValueGen");
        PUBLIC_IDENTIFIERS.add("ArrayEncoder");
        PUBLIC_IDENTIFIERS.add("ArrayIndexOutOfBoundsException");
        PUBLIC_IDENTIFIERS.add("ArrayInfo");
        PUBLIC_IDENTIFIERS.add("ArrayInstruction");
        PUBLIC_IDENTIFIERS.add("ArrayKlass");
        PUBLIC_IDENTIFIERS.add("ARRAYLENGTH");
        PUBLIC_IDENTIFIERS.add("ArrayList");
        PUBLIC_IDENTIFIERS.add("ArrayNodeListIterator");
        PUBLIC_IDENTIFIERS.add("ArrayNotificationBuffer");
        PUBLIC_IDENTIFIERS.add("ArrayQueue");
        PUBLIC_IDENTIFIERS.add("ArrayReference");
        PUBLIC_IDENTIFIERS.add("ArrayReferenceImpl");
        PUBLIC_IDENTIFIERS.add("Arrays");
        PUBLIC_IDENTIFIERS.add("ArraysSupport");
        PUBLIC_IDENTIFIERS.add("ArrayStoreException");
        PUBLIC_IDENTIFIERS.add("ArrayType");
        PUBLIC_IDENTIFIERS.add("ArrayTypeImpl");
        PUBLIC_IDENTIFIERS.add("ArrayTypeSignature");
        PUBLIC_IDENTIFIERS.add("ArrayTypeTree");
        PUBLIC_IDENTIFIERS.add("ArrayUtils");
        PUBLIC_IDENTIFIERS.add("ArrowButtonListener");
        PUBLIC_IDENTIFIERS.add("ASCIIReader");
        PUBLIC_IDENTIFIERS.add("ASMifier");
        PUBLIC_IDENTIFIERS.add("ASMifierSupport");
        PUBLIC_IDENTIFIERS.add("Asn1Exception");
        PUBLIC_IDENTIFIERS.add("ASRep");
        PUBLIC_IDENTIFIERS.add("ASReq");
        PUBLIC_IDENTIFIERS.add("Assert");
        PUBLIC_IDENTIFIERS.add("AssertionError");
        PUBLIC_IDENTIFIERS.add("AssertionFailure");
        PUBLIC_IDENTIFIERS.add("AssertTree");
        PUBLIC_IDENTIFIERS.add("AssignAnalyzer");
        PUBLIC_IDENTIFIERS.add("AssignmentTree");
        PUBLIC_IDENTIFIERS.add("AssignPendingExit");
        PUBLIC_IDENTIFIERS.add("Association");
        PUBLIC_IDENTIFIERS.add("AssociationChange");
        PUBLIC_IDENTIFIERS.add("AssociationChangeNotification");
        PUBLIC_IDENTIFIERS.add("AssociationImpl");
        PUBLIC_IDENTIFIERS.add("Associative");
        PUBLIC_IDENTIFIERS.add("Assumption");
        PUBLIC_IDENTIFIERS.add("AssumptionResult");
        PUBLIC_IDENTIFIERS.add("Assumptions");
        PUBLIC_IDENTIFIERS.add("ASTORE");
        PUBLIC_IDENTIFIERS.add("AsyncBoxView");
        PUBLIC_IDENTIFIERS.add("AsynchronousByteChannel");
        PUBLIC_IDENTIFIERS.add("AsynchronousChannel");
        PUBLIC_IDENTIFIERS.add("AsynchronousChannelGroup");
        PUBLIC_IDENTIFIERS.add("AsynchronousChannelProvider");
        PUBLIC_IDENTIFIERS.add("AsynchronousCloseException");
        PUBLIC_IDENTIFIERS.add("AsynchronousCompletionTask");
        PUBLIC_IDENTIFIERS.add("AsynchronousFileChannel");
        PUBLIC_IDENTIFIERS.add("AsynchronousServerSocketChannel");
        PUBLIC_IDENTIFIERS.add("AsynchronousSocketChannel");
        PUBLIC_IDENTIFIERS.add("ATHROW");
        PUBLIC_IDENTIFIERS.add("AtomicBoolean");
        PUBLIC_IDENTIFIERS.add("AtomicContext");
        PUBLIC_IDENTIFIERS.add("AtomicDirContext");
        PUBLIC_IDENTIFIERS.add("AtomicInteger");
        PUBLIC_IDENTIFIERS.add("AtomicIntegerArray");
        PUBLIC_IDENTIFIERS.add("AtomicIntegerFieldUpdater");
        PUBLIC_IDENTIFIERS.add("AtomicLong");
        PUBLIC_IDENTIFIERS.add("AtomicLongArray");
        PUBLIC_IDENTIFIERS.add("AtomicLongFieldUpdater");
        PUBLIC_IDENTIFIERS.add("AtomicMarkableReference");
        PUBLIC_IDENTIFIERS.add("AtomicMoveNotSupportedException");
        PUBLIC_IDENTIFIERS.add("AtomicReference");
        PUBLIC_IDENTIFIERS.add("AtomicReferenceArray");
        PUBLIC_IDENTIFIERS.add("AtomicReferenceFieldUpdater");
        PUBLIC_IDENTIFIERS.add("AtomicStampedReference");
        PUBLIC_IDENTIFIERS.add("AttachingConnector");
        PUBLIC_IDENTIFIERS.add("AttachNotSupportedException");
        PUBLIC_IDENTIFIERS.add("AttachOperationFailedException");
        PUBLIC_IDENTIFIERS.add("AttachProvider");
        PUBLIC_IDENTIFIERS.add("AttachProviderImpl");
        PUBLIC_IDENTIFIERS.add("AttList");
        PUBLIC_IDENTIFIERS.add("Attr");
        PUBLIC_IDENTIFIERS.add("AttrCompare");
        PUBLIC_IDENTIFIERS.add("AttrContext");
        PUBLIC_IDENTIFIERS.add("AttrContextEnv");
        PUBLIC_IDENTIFIERS.add("AttrFilter");
        PUBLIC_IDENTIFIERS.add("Attribute");
        PUBLIC_IDENTIFIERS.add("AttributeChangeNotification");
        PUBLIC_IDENTIFIERS.add("AttributeChangeNotificationFilter");
        PUBLIC_IDENTIFIERS.add("AttributeClass");
        PUBLIC_IDENTIFIERS.add("AttributeContext");
        PUBLIC_IDENTIFIERS.add("AttributedCharacterIterator");
        PUBLIC_IDENTIFIERS.add("AttributedCharSequence");
        PUBLIC_IDENTIFIERS.add("AttributedString");
        PUBLIC_IDENTIFIERS.add("AttributedStringBuilder");
        PUBLIC_IDENTIFIERS.add("AttributedStyle");
        PUBLIC_IDENTIFIERS.add("AttributeException");
        PUBLIC_IDENTIFIERS.add("AttributeImpl");
        PUBLIC_IDENTIFIERS.add("AttributeInUseException");
        PUBLIC_IDENTIFIERS.add("AttributeIterator");
        PUBLIC_IDENTIFIERS.add("AttributeList");
        PUBLIC_IDENTIFIERS.add("AttributeListImpl");
        PUBLIC_IDENTIFIERS.add("AttributeMap");
        PUBLIC_IDENTIFIERS.add("AttributeModificationException");
        PUBLIC_IDENTIFIERS.add("AttributeNotFoundException");
        PUBLIC_IDENTIFIERS.add("AttributePSVI");
        PUBLIC_IDENTIFIERS.add("AttributePSVImpl");
        PUBLIC_IDENTIFIERS.add("AttributeReader");
        PUBLIC_IDENTIFIERS.add("Attributes");
        PUBLIC_IDENTIFIERS.add("Attributes2");
        PUBLIC_IDENTIFIERS.add("Attributes2Impl");
        PUBLIC_IDENTIFIERS.add("AttributeSet");
        PUBLIC_IDENTIFIERS.add("AttributesImpl");
        PUBLIC_IDENTIFIERS.add("AttributeTree");
        PUBLIC_IDENTIFIERS.add("AttributeUndoableEdit");
        PUBLIC_IDENTIFIERS.add("AttributeValueExp");
        PUBLIC_IDENTIFIERS.add("AttributeView");
        PUBLIC_IDENTIFIERS.add("AttributeWriter");
        PUBLIC_IDENTIFIERS.add("AttrImpl");
        PUBLIC_IDENTIFIERS.add("AttrNSImpl");
        PUBLIC_IDENTIFIERS.add("AttrRecover");
        PUBLIC_IDENTIFIERS.add("Attrs");
        PUBLIC_IDENTIFIERS.add("AudioClip");
        PUBLIC_IDENTIFIERS.add("AudioFileFormat");
        PUBLIC_IDENTIFIERS.add("AudioFileReader");
        PUBLIC_IDENTIFIERS.add("AudioFileWriter");
        PUBLIC_IDENTIFIERS.add("AudioFloatConverter");
        PUBLIC_IDENTIFIERS.add("AudioFloatInputStream");
        PUBLIC_IDENTIFIERS.add("AudioFormat");
        PUBLIC_IDENTIFIERS.add("AudioInputStream");
        PUBLIC_IDENTIFIERS.add("AudioPermission");
        PUBLIC_IDENTIFIERS.add("AudioSynthesizer");
        PUBLIC_IDENTIFIERS.add("AudioSystem");
        PUBLIC_IDENTIFIERS.add("Augmentations");
        PUBLIC_IDENTIFIERS.add("AugmentationsImpl");
        PUBLIC_IDENTIFIERS.add("AugmentVisitor");
        PUBLIC_IDENTIFIERS.add("AuthCache");
        PUBLIC_IDENTIFIERS.add("AuthCacheImpl");
        PUBLIC_IDENTIFIERS.add("AuthCacheValue");
        PUBLIC_IDENTIFIERS.add("Authenticate");
        PUBLIC_IDENTIFIERS.add("AuthenticationException");
        PUBLIC_IDENTIFIERS.add("AuthenticationHeader");
        PUBLIC_IDENTIFIERS.add("AuthenticationInfo");
        PUBLIC_IDENTIFIERS.add("AuthenticationNotSupportedException");
        PUBLIC_IDENTIFIERS.add("Authenticator");
        PUBLIC_IDENTIFIERS.add("AuthenticatorKeyAccess");
        PUBLIC_IDENTIFIERS.add("AuthFilter");
        PUBLIC_IDENTIFIERS.add("AuthList");
        PUBLIC_IDENTIFIERS.add("AuthorityInfoAccessExtension");
        PUBLIC_IDENTIFIERS.add("AuthorityKeyIdentifierExtension");
        PUBLIC_IDENTIFIERS.add("AuthorizationData");
        PUBLIC_IDENTIFIERS.add("AuthorizationDataEntry");
        PUBLIC_IDENTIFIERS.add("AuthorizeCallback");
        PUBLIC_IDENTIFIERS.add("AuthorTree");
        PUBLIC_IDENTIFIERS.add("AuthProvider");
        PUBLIC_IDENTIFIERS.add("AuthResources");
        PUBLIC_IDENTIFIERS.add("AuthResources_de");
        PUBLIC_IDENTIFIERS.add("AuthResources_ja");
        PUBLIC_IDENTIFIERS.add("AuthResources_zh_CN");
        PUBLIC_IDENTIFIERS.add("AuthTime");
        PUBLIC_IDENTIFIERS.add("AuthTimeWithHash");
        PUBLIC_IDENTIFIERS.add("AutoCloseable");
        PUBLIC_IDENTIFIERS.add("AutoConnectSequencer");
        PUBLIC_IDENTIFIERS.add("Autoscroll");
        PUBLIC_IDENTIFIERS.add("AuxBfEfRecord");
        PUBLIC_IDENTIFIERS.add("AuxFileRecord");
        PUBLIC_IDENTIFIERS.add("AuxFunctionDefinitionRecord");
        PUBLIC_IDENTIFIERS.add("AuxLocaleProviderAdapter");
        PUBLIC_IDENTIFIERS.add("AuxSectionDefinitionsRecord");
        PUBLIC_IDENTIFIERS.add("AuxSymbolRecord");
        PUBLIC_IDENTIFIERS.add("AuxWeakExternalRecord");
        PUBLIC_IDENTIFIERS.add("AVA");
        PUBLIC_IDENTIFIERS.add("AvailableLanguageTags");
        PUBLIC_IDENTIFIERS.add("AWTError");
        PUBLIC_IDENTIFIERS.add("AWTEvent");
        PUBLIC_IDENTIFIERS.add("AWTEventAccessor");
        PUBLIC_IDENTIFIERS.add("AWTEventListener");
        PUBLIC_IDENTIFIERS.add("AWTEventListenerProxy");
        PUBLIC_IDENTIFIERS.add("AWTEventMonitor");
        PUBLIC_IDENTIFIERS.add("AWTEventMulticaster");
        PUBLIC_IDENTIFIERS.add("AWTException");
        PUBLIC_IDENTIFIERS.add("AwtGraphicsConfigData");
        PUBLIC_IDENTIFIERS.add("AWTIcon32_java_icon16_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon32_java_icon24_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon32_java_icon32_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon32_java_icon48_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon32_security_icon_bw16_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon32_security_icon_bw24_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon32_security_icon_bw32_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon32_security_icon_bw48_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon32_security_icon_interim16_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon32_security_icon_interim24_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon32_security_icon_interim32_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon32_security_icon_interim48_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon32_security_icon_yellow16_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon32_security_icon_yellow24_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon32_security_icon_yellow32_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon32_security_icon_yellow48_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon64_java_icon16_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon64_java_icon24_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon64_java_icon32_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon64_java_icon48_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon64_security_icon_bw16_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon64_security_icon_bw24_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon64_security_icon_bw32_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon64_security_icon_bw48_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon64_security_icon_interim16_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon64_security_icon_interim24_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon64_security_icon_interim32_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon64_security_icon_interim48_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon64_security_icon_yellow16_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon64_security_icon_yellow24_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon64_security_icon_yellow32_png");
        PUBLIC_IDENTIFIERS.add("AWTIcon64_security_icon_yellow48_png");
        PUBLIC_IDENTIFIERS.add("awtImageData");
        PUBLIC_IDENTIFIERS.add("AWTKeyStroke");
        PUBLIC_IDENTIFIERS.add("AwtScreenData");
        PUBLIC_IDENTIFIERS.add("AxesWalker");
        PUBLIC_IDENTIFIERS.add("Axis");
        PUBLIC_IDENTIFIERS.add("BackAction");
        PUBLIC_IDENTIFIERS.add("BackBufferCapsProvider");
        PUBLIC_IDENTIFIERS.add("BackgroundLookupListener");
        PUBLIC_IDENTIFIERS.add("BackgroundServiceLookup");
        PUBLIC_IDENTIFIERS.add("BackingStoreException");
        PUBLIC_IDENTIFIERS.add("BadAddressTreeNodeAdapter");
        PUBLIC_IDENTIFIERS.add("BadArgs");
        PUBLIC_IDENTIFIERS.add("BadAttributeValueExpException");
        PUBLIC_IDENTIFIERS.add("BadBinaryOpValueExpException");
        PUBLIC_IDENTIFIERS.add("BadClassFile");
        PUBLIC_IDENTIFIERS.add("BadEnclosingMethodAttr");
        PUBLIC_IDENTIFIERS.add("BadLocationException");
        PUBLIC_IDENTIFIERS.add("BadPaddingException");
        PUBLIC_IDENTIFIERS.add("BadStringOperationException");
        PUBLIC_IDENTIFIERS.add("BailoutException");
        PUBLIC_IDENTIFIERS.add("BakedArrayList");
        PUBLIC_IDENTIFIERS.add("Balloon");
        PUBLIC_IDENTIFIERS.add("BALOAD");
        PUBLIC_IDENTIFIERS.add("BandCombineOp");
        PUBLIC_IDENTIFIERS.add("Base64");
        PUBLIC_IDENTIFIERS.add("Base64BinaryDV");
        PUBLIC_IDENTIFIERS.add("Base64DecodingException");
        PUBLIC_IDENTIFIERS.add("BaseCalendar");
        PUBLIC_IDENTIFIERS.add("BaseClass");
        PUBLIC_IDENTIFIERS.add("BaseConfiguration");
        PUBLIC_IDENTIFIERS.add("BaseDVFactory");
        PUBLIC_IDENTIFIERS.add("BaseFileManager");
        PUBLIC_IDENTIFIERS.add("BaseLocaleDataMetaInfo");
        PUBLIC_IDENTIFIERS.add("BaseMarkupSerializer");
        PUBLIC_IDENTIFIERS.add("BaseMultiResolutionImage");
        PUBLIC_IDENTIFIERS.add("BaseOperatingSystemImpl");
        PUBLIC_IDENTIFIERS.add("BaseOptions");
        PUBLIC_IDENTIFIERS.add("BaseRow");
        PUBLIC_IDENTIFIERS.add("BaseRowSet");
        PUBLIC_IDENTIFIERS.add("BaseSchemaDVFactory");
        PUBLIC_IDENTIFIERS.add("BaseStream");
        PUBLIC_IDENTIFIERS.add("BaseTaglet");
        PUBLIC_IDENTIFIERS.add("BaseType");
        PUBLIC_IDENTIFIERS.add("basic");
        PUBLIC_IDENTIFIERS.add("BasicAddressFieldWrapper");
        PUBLIC_IDENTIFIERS.add("BasicArrayType");
        PUBLIC_IDENTIFIERS.add("BasicArrowButton");
        PUBLIC_IDENTIFIERS.add("BasicAttribute");
        PUBLIC_IDENTIFIERS.add("BasicAttributes");
        PUBLIC_IDENTIFIERS.add("BasicAuthenticator");
        PUBLIC_IDENTIFIERS.add("BasicBaseClass");
        PUBLIC_IDENTIFIERS.add("BasicBitType");
        PUBLIC_IDENTIFIERS.add("BasicBlockSym");
        PUBLIC_IDENTIFIERS.add("BasicBorders");
        PUBLIC_IDENTIFIERS.add("BasicBundlers");
        PUBLIC_IDENTIFIERS.add("BasicButtonListener");
        PUBLIC_IDENTIFIERS.add("BasicButtonUI");
        PUBLIC_IDENTIFIERS.add("BasicCaret");
        PUBLIC_IDENTIFIERS.add("BasicCDebugInfoDataBase");
        PUBLIC_IDENTIFIERS.add("BasicCFrame");
        PUBLIC_IDENTIFIERS.add("BasicCheckBoxMenuItemUI");
        PUBLIC_IDENTIFIERS.add("BasicCheckBoxUI");
        PUBLIC_IDENTIFIERS.add("BasicCIntegerField");
        PUBLIC_IDENTIFIERS.add("BasicCIntegerType");
        PUBLIC_IDENTIFIERS.add("BasicColorChooserUI");
        PUBLIC_IDENTIFIERS.add("BasicComboBoxEditor");
        PUBLIC_IDENTIFIERS.add("BasicComboBoxRenderer");
        PUBLIC_IDENTIFIERS.add("BasicComboBoxUI");
        PUBLIC_IDENTIFIERS.add("BasicComboPopup");
        PUBLIC_IDENTIFIERS.add("BasicComment");
        PUBLIC_IDENTIFIERS.add("BasicCompoundType");
        PUBLIC_IDENTIFIERS.add("BasicConfiguration");
        PUBLIC_IDENTIFIERS.add("BasicConstant");
        PUBLIC_IDENTIFIERS.add("BasicConstraintsExtension");
        PUBLIC_IDENTIFIERS.add("BasicControl");
        PUBLIC_IDENTIFIERS.add("BasicDebugEvent");
        PUBLIC_IDENTIFIERS.add("BasicDesktopIconUI");
        PUBLIC_IDENTIFIERS.add("BasicDesktopPaneUI");
        PUBLIC_IDENTIFIERS.add("BasicDiagnosticFormatter");
        PUBLIC_IDENTIFIERS.add("BasicDirectoryModel");
        PUBLIC_IDENTIFIERS.add("BasicDoubleType");
        PUBLIC_IDENTIFIERS.add("BasicEditorPaneUI");
        PUBLIC_IDENTIFIERS.add("BasicEnumType");
        PUBLIC_IDENTIFIERS.add("BasicField");
        PUBLIC_IDENTIFIERS.add("BasicFieldWrapper");
        PUBLIC_IDENTIFIERS.add("BasicFileAttributes");
        PUBLIC_IDENTIFIERS.add("BasicFileAttributesHolder");
        PUBLIC_IDENTIFIERS.add("BasicFileAttributeView");
        PUBLIC_IDENTIFIERS.add("BasicFileChooserUI");
        PUBLIC_IDENTIFIERS.add("BasicFileView");
        PUBLIC_IDENTIFIERS.add("BasicFloatType");
        PUBLIC_IDENTIFIERS.add("BasicFormattedTextFieldUI");
        PUBLIC_IDENTIFIERS.add("BasicFunctionSym");
        PUBLIC_IDENTIFIERS.add("BasicFunctionType");
        PUBLIC_IDENTIFIERS.add("BasicGlobalSym");
        PUBLIC_IDENTIFIERS.add("BasicGraphicsUtils");
        PUBLIC_IDENTIFIERS.add("BasicHighlighter");
        PUBLIC_IDENTIFIERS.add("BasicHorizontalLayoutManager");
        PUBLIC_IDENTIFIERS.add("BasicHTML");
        PUBLIC_IDENTIFIERS.add("BasicIconFactory");
        PUBLIC_IDENTIFIERS.add("BasicImageReader");
        PUBLIC_IDENTIFIERS.add("BasicIndexableFieldIdentifier");
        PUBLIC_IDENTIFIERS.add("BasicInternalFrameListener");
        PUBLIC_IDENTIFIERS.add("BasicInternalFrameTitlePane");
        PUBLIC_IDENTIFIERS.add("BasicInternalFrameUI");
        PUBLIC_IDENTIFIERS.add("BasicInterpreter");
        PUBLIC_IDENTIFIERS.add("BasicIntType");
        PUBLIC_IDENTIFIERS.add("BasicJavacTask");
        PUBLIC_IDENTIFIERS.add("BasicJBooleanField");
        PUBLIC_IDENTIFIERS.add("BasicJByteField");
        PUBLIC_IDENTIFIERS.add("BasicJCharField");
        PUBLIC_IDENTIFIERS.add("BasicJDoubleField");
        PUBLIC_IDENTIFIERS.add("BasicJFloatField");
        PUBLIC_IDENTIFIERS.add("BasicJIntField");
        PUBLIC_IDENTIFIERS.add("BasicJLongField");
        PUBLIC_IDENTIFIERS.add("BasicJShortField");
        PUBLIC_IDENTIFIERS.add("BasicLabelUI");
        PUBLIC_IDENTIFIERS.add("BasicLineNumberInfo");
        PUBLIC_IDENTIFIERS.add("BasicLineNumberMapping");
        PUBLIC_IDENTIFIERS.add("BasicListUI");
        PUBLIC_IDENTIFIERS.add("BasicLocalSym");
        PUBLIC_IDENTIFIERS.add("BasicLock");
        PUBLIC_IDENTIFIERS.add("BasicLookAndFeel");
        PUBLIC_IDENTIFIERS.add("BasicMemberFunctionType");
        PUBLIC_IDENTIFIERS.add("BasicMenuBarUI");
        PUBLIC_IDENTIFIERS.add("BasicMenuItemUI");
        PUBLIC_IDENTIFIERS.add("BasicMenuUI");
        PUBLIC_IDENTIFIERS.add("BasicNamedFieldIdentifier");
        PUBLIC_IDENTIFIERS.add("BasicNarrowOopField");
        PUBLIC_IDENTIFIERS.add("BasicObjectLock");
        PUBLIC_IDENTIFIERS.add("BasicOopField");
        PUBLIC_IDENTIFIERS.add("BasicOptionPaneUI");
        PUBLIC_IDENTIFIERS.add("BasicPanelUI");
        PUBLIC_IDENTIFIERS.add("BasicParserConfiguration");
        PUBLIC_IDENTIFIERS.add("BasicPasswordFieldUI");
        PUBLIC_IDENTIFIERS.add("BasicPermission");
        PUBLIC_IDENTIFIERS.add("BasicPointerType");
        PUBLIC_IDENTIFIERS.add("BasicPopupMenuSeparatorUI");
        PUBLIC_IDENTIFIERS.add("BasicPopupMenuUI");
        PUBLIC_IDENTIFIERS.add("BasicProgressBarUI");
        PUBLIC_IDENTIFIERS.add("BasicRadioButtonMenuItemUI");
        PUBLIC_IDENTIFIERS.add("BasicRadioButtonUI");
        PUBLIC_IDENTIFIERS.add("BasicRefType");
        PUBLIC_IDENTIFIERS.add("BasicRootPaneUI");
        PUBLIC_IDENTIFIERS.add("BasicScrollBarUI");
        PUBLIC_IDENTIFIERS.add("BasicScrollPaneUI");
        PUBLIC_IDENTIFIERS.add("BasicSeparatorUI");
        PUBLIC_IDENTIFIERS.add("BasicSliderUI");
        PUBLIC_IDENTIFIERS.add("BasicSpinnerUI");
        PUBLIC_IDENTIFIERS.add("BasicSplitPaneDivider");
        PUBLIC_IDENTIFIERS.add("BasicSplitPaneUI");
        PUBLIC_IDENTIFIERS.add("BasicStroke");
        PUBLIC_IDENTIFIERS.add("BasicSym");
        PUBLIC_IDENTIFIERS.add("BasicTabbedPaneUI");
        PUBLIC_IDENTIFIERS.add("BasicTableHeaderUI");
        PUBLIC_IDENTIFIERS.add("BasicTableUI");
        PUBLIC_IDENTIFIERS.add("BasicTestIterator");
        PUBLIC_IDENTIFIERS.add("BasicTextAreaUI");
        PUBLIC_IDENTIFIERS.add("BasicTextFieldUI");
        PUBLIC_IDENTIFIERS.add("BasicTextPaneUI");
        PUBLIC_IDENTIFIERS.add("BasicTextUI");
        PUBLIC_IDENTIFIERS.add("BasicToggleButtonUI");
        PUBLIC_IDENTIFIERS.add("BasicToolBarSeparatorUI");
        PUBLIC_IDENTIFIERS.add("BasicToolBarUI");
        PUBLIC_IDENTIFIERS.add("BasicToolTipUI");
        PUBLIC_IDENTIFIERS.add("BasicTreeUI");
        PUBLIC_IDENTIFIERS.add("BasicType");
        PUBLIC_IDENTIFIERS.add("BasicTypeDataBase");
        PUBLIC_IDENTIFIERS.add("BasicTypeSize");
        PUBLIC_IDENTIFIERS.add("BasicValue");
        PUBLIC_IDENTIFIERS.add("BasicVerifier");
        PUBLIC_IDENTIFIERS.add("BasicVerticalLayoutManager");
        PUBLIC_IDENTIFIERS.add("BasicViewportUI");
        PUBLIC_IDENTIFIERS.add("BasicVoidType");
        PUBLIC_IDENTIFIERS.add("BasicVtblAccess");
        PUBLIC_IDENTIFIERS.add("BasicWriter");
        PUBLIC_IDENTIFIERS.add("BASTORE");
        PUBLIC_IDENTIFIERS.add("BatchUpdateException");
        PUBLIC_IDENTIFIERS.add("BCELComparator");
        PUBLIC_IDENTIFIERS.add("BCELifier");
        PUBLIC_IDENTIFIERS.add("BCSChild");
        PUBLIC_IDENTIFIERS.add("BCSSChild");
        PUBLIC_IDENTIFIERS.add("BCSSProxyServiceProvider");
        PUBLIC_IDENTIFIERS.add("BCSSServiceProvider");
        PUBLIC_IDENTIFIERS.add("BeanContext");
        PUBLIC_IDENTIFIERS.add("BeanContextChild");
        PUBLIC_IDENTIFIERS.add("BeanContextChildComponentProxy");
        PUBLIC_IDENTIFIERS.add("BeanContextChildSupport");
        PUBLIC_IDENTIFIERS.add("BeanContextContainerProxy");
        PUBLIC_IDENTIFIERS.add("BeanContextEvent");
        PUBLIC_IDENTIFIERS.add("BeanContextMembershipEvent");
        PUBLIC_IDENTIFIERS.add("BeanContextMembershipListener");
        PUBLIC_IDENTIFIERS.add("BeanContextProxy");
        PUBLIC_IDENTIFIERS.add("BeanContextServiceAvailableEvent");
        PUBLIC_IDENTIFIERS.add("BeanContextServiceProvider");
        PUBLIC_IDENTIFIERS.add("BeanContextServiceProviderBeanInfo");
        PUBLIC_IDENTIFIERS.add("BeanContextServiceRevokedEvent");
        PUBLIC_IDENTIFIERS.add("BeanContextServiceRevokedListener");
        PUBLIC_IDENTIFIERS.add("BeanContextServices");
        PUBLIC_IDENTIFIERS.add("BeanContextServicesListener");
        PUBLIC_IDENTIFIERS.add("BeanContextServicesSupport");
        PUBLIC_IDENTIFIERS.add("BeanContextSupport");
        PUBLIC_IDENTIFIERS.add("BeanDescriptor");
        PUBLIC_IDENTIFIERS.add("BeanInfo");
        PUBLIC_IDENTIFIERS.add("BeanProperty");
        PUBLIC_IDENTIFIERS.add("Beans");
        PUBLIC_IDENTIFIERS.add("BeansLinker");
        PUBLIC_IDENTIFIERS.add("BeepAction");
        PUBLIC_IDENTIFIERS.add("BeforeDrag");
        PUBLIC_IDENTIFIERS.add("Ber");
        PUBLIC_IDENTIFIERS.add("BevelBorder");
        PUBLIC_IDENTIFIERS.add("BevelBorderUIResource");
        PUBLIC_IDENTIFIERS.add("Bgrx");
        PUBLIC_IDENTIFIERS.add("BiConsumer");
        PUBLIC_IDENTIFIERS.add("BidiBase");
        PUBLIC_IDENTIFIERS.add("BidiPairedBracketType");
        PUBLIC_IDENTIFIERS.add("BiFunction");
        PUBLIC_IDENTIFIERS.add("Big5");
        PUBLIC_IDENTIFIERS.add("Big5_HKSCS");
        PUBLIC_IDENTIFIERS.add("Big5_HKSCS_2001");
        PUBLIC_IDENTIFIERS.add("Big5_Solaris");
        PUBLIC_IDENTIFIERS.add("BigDecimal");
        PUBLIC_IDENTIFIERS.add("BigInteger");
        PUBLIC_IDENTIFIERS.add("Binary");
        PUBLIC_IDENTIFIERS.add("BinaryOperation");
        PUBLIC_IDENTIFIERS.add("BinaryOperator");
        PUBLIC_IDENTIFIERS.add("BinaryRefAddr");
        PUBLIC_IDENTIFIERS.add("BinaryToASCIIConverter");
        PUBLIC_IDENTIFIERS.add("BinaryTree");
        PUBLIC_IDENTIFIERS.add("BindException");
        PUBLIC_IDENTIFIERS.add("Binding");
        PUBLIC_IDENTIFIERS.add("BindingInterpreter");
        PUBLIC_IDENTIFIERS.add("BindingPatternTree");
        PUBLIC_IDENTIFIERS.add("BindingReader");
        PUBLIC_IDENTIFIERS.add("Bindings");
        PUBLIC_IDENTIFIERS.add("BindingSpecializer");
        PUBLIC_IDENTIFIERS.add("BindingSymbol");
        PUBLIC_IDENTIFIERS.add("BiPredicate");
        PUBLIC_IDENTIFIERS.add("BIPUSH");
        PUBLIC_IDENTIFIERS.add("BitArray");
        PUBLIC_IDENTIFIERS.add("BitData");
        PUBLIC_IDENTIFIERS.add("BitFile");
        PUBLIC_IDENTIFIERS.add("BitMap");
        PUBLIC_IDENTIFIERS.add("BitMapClosure");
        PUBLIC_IDENTIFIERS.add("BitMapInterface");
        PUBLIC_IDENTIFIERS.add("BitMapSegmented");
        PUBLIC_IDENTIFIERS.add("Bitmask");
        PUBLIC_IDENTIFIERS.add("Bits");
        PUBLIC_IDENTIFIERS.add("BitSet");
        PUBLIC_IDENTIFIERS.add("BitType");
        PUBLIC_IDENTIFIERS.add("Blit");
        PUBLIC_IDENTIFIERS.add("BlitBg");
        PUBLIC_IDENTIFIERS.add("Blob");
        PUBLIC_IDENTIFIERS.add("Block");
        PUBLIC_IDENTIFIERS.add("Block_Array");
        PUBLIC_IDENTIFIERS.add("Block_List");
        PUBLIC_IDENTIFIERS.add("BlockAction");
        PUBLIC_IDENTIFIERS.add("BlockElement");
        PUBLIC_IDENTIFIERS.add("Blocker");
        PUBLIC_IDENTIFIERS.add("BlockingDeque");
        PUBLIC_IDENTIFIERS.add("BlockingQueue");
        PUBLIC_IDENTIFIERS.add("BlockSym");
        PUBLIC_IDENTIFIERS.add("BlockTagTree");
        PUBLIC_IDENTIFIERS.add("BlockTree");
        PUBLIC_IDENTIFIERS.add("BlockView");
        PUBLIC_IDENTIFIERS.add("BltBufferStrategy");
        PUBLIC_IDENTIFIERS.add("BMPattern");
        PUBLIC_IDENTIFIERS.add("BMPCompressionTypes");
        PUBLIC_IDENTIFIERS.add("BMPConstants");
        PUBLIC_IDENTIFIERS.add("BMPImageReader");
        PUBLIC_IDENTIFIERS.add("BMPImageReaderSpi");
        PUBLIC_IDENTIFIERS.add("BMPImageWriteParam");
        PUBLIC_IDENTIFIERS.add("BMPImageWriter");
        PUBLIC_IDENTIFIERS.add("BMPImageWriterSpi");
        PUBLIC_IDENTIFIERS.add("BMPMetadata");
        PUBLIC_IDENTIFIERS.add("BMPMetadataFormat");
        PUBLIC_IDENTIFIERS.add("BMPMetadataFormatResources");
        PUBLIC_IDENTIFIERS.add("BodyContents");
        PUBLIC_IDENTIFIERS.add("BodyHandler");
        PUBLIC_IDENTIFIERS.add("BodyHandlers");
        PUBLIC_IDENTIFIERS.add("BodyPublisher");
        PUBLIC_IDENTIFIERS.add("BodyPublishers");
        PUBLIC_IDENTIFIERS.add("BodySubscriber");
        PUBLIC_IDENTIFIERS.add("BodySubscribers");
        PUBLIC_IDENTIFIERS.add("BogusColorSpace");
        PUBLIC_IDENTIFIERS.add("BoldAction");
        PUBLIC_IDENTIFIERS.add("Book");
        PUBLIC_IDENTIFIERS.add("Bool");
        PUBLIC_IDENTIFIERS.add("Boolean");
        PUBLIC_IDENTIFIERS.add("BooleanArgument");
        PUBLIC_IDENTIFIERS.add("BooleanControl");
        PUBLIC_IDENTIFIERS.add("BooleanDV");
        PUBLIC_IDENTIFIERS.add("BooleanEditor");
        PUBLIC_IDENTIFIERS.add("BooleanField");
        PUBLIC_IDENTIFIERS.add("BooleanFlag");
        PUBLIC_IDENTIFIERS.add("BooleanSignature");
        PUBLIC_IDENTIFIERS.add("BooleanSupplier");
        PUBLIC_IDENTIFIERS.add("BooleanTreeNodeAdapter");
        PUBLIC_IDENTIFIERS.add("BooleanType");
        PUBLIC_IDENTIFIERS.add("BooleanTypeImpl");
        PUBLIC_IDENTIFIERS.add("BooleanValue");
        PUBLIC_IDENTIFIERS.add("BooleanValueImpl");
        PUBLIC_IDENTIFIERS.add("BootFilter");
        PUBLIC_IDENTIFIERS.add("BootLoader");
        PUBLIC_IDENTIFIERS.add("Bootstrap");
        PUBLIC_IDENTIFIERS.add("BootstrapMethod");
        PUBLIC_IDENTIFIERS.add("BootstrapMethodError");
        PUBLIC_IDENTIFIERS.add("BootstrapMethodInvocation");
        PUBLIC_IDENTIFIERS.add("BootstrapMethods");
        PUBLIC_IDENTIFIERS.add("BootstrapMethods_attribute");
        PUBLIC_IDENTIFIERS.add("BootstrapMethodSpecifier");
        PUBLIC_IDENTIFIERS.add("Border");
        PUBLIC_IDENTIFIERS.add("BorderedComponent");
        PUBLIC_IDENTIFIERS.add("BorderFactory");
        PUBLIC_IDENTIFIERS.add("BorderLayout");
        PUBLIC_IDENTIFIERS.add("BorderListener");
        PUBLIC_IDENTIFIERS.add("BorderUIResource");
        PUBLIC_IDENTIFIERS.add("BottomSignature");
        PUBLIC_IDENTIFIERS.add("BoundedRangeModel");
        PUBLIC_IDENTIFIERS.add("BoundFilter");
        PUBLIC_IDENTIFIERS.add("Box");
        PUBLIC_IDENTIFIERS.add("BoxLayout");
        PUBLIC_IDENTIFIERS.add("BoxView");
        PUBLIC_IDENTIFIERS.add("BranchData");
        PUBLIC_IDENTIFIERS.add("BranchElement");
        PUBLIC_IDENTIFIERS.add("BranchInstruction");
        PUBLIC_IDENTIFIERS.add("BreakIterator");
        PUBLIC_IDENTIFIERS.add("BreakIteratorInfo");
        PUBLIC_IDENTIFIERS.add("BreakIteratorInfo_th");
        PUBLIC_IDENTIFIERS.add("BreakIteratorInfoProvider");
        PUBLIC_IDENTIFIERS.add("BreakIteratorProvider");
        PUBLIC_IDENTIFIERS.add("BreakIteratorProviderImpl");
        PUBLIC_IDENTIFIERS.add("BreakIteratorResourceBundle");
        PUBLIC_IDENTIFIERS.add("BreakIteratorResources");
        PUBLIC_IDENTIFIERS.add("BreakIteratorResources_th");
        PUBLIC_IDENTIFIERS.add("BREAKPOINT");
        PUBLIC_IDENTIFIERS.add("BreakpointEvent");
        PUBLIC_IDENTIFIERS.add("BreakpointInfo");
        PUBLIC_IDENTIFIERS.add("BreakpointRequest");
        PUBLIC_IDENTIFIERS.add("BreakTree");
        PUBLIC_IDENTIFIERS.add("Bridge");
        PUBLIC_IDENTIFIERS.add("BrokenBarrierException");
        PUBLIC_IDENTIFIERS.add("BsdAARCH64JavaThreadPDAccess");
        PUBLIC_IDENTIFIERS.add("BsdAARCH64ThreadContext");
        PUBLIC_IDENTIFIERS.add("BsdAMD64JavaThreadPDAccess");
        PUBLIC_IDENTIFIERS.add("BsdAMD64ThreadContext");
        PUBLIC_IDENTIFIERS.add("BsdDebugger");
        PUBLIC_IDENTIFIERS.add("BsdDebuggerLocal");
        PUBLIC_IDENTIFIERS.add("BsdSignals");
        PUBLIC_IDENTIFIERS.add("BsdVtblAccess");
        PUBLIC_IDENTIFIERS.add("BsdX86JavaThreadPDAccess");
        PUBLIC_IDENTIFIERS.add("BsdX86ThreadContext");
        PUBLIC_IDENTIFIERS.add("BsmKey");
        PUBLIC_IDENTIFIERS.add("BuddhistCalendar");
        PUBLIC_IDENTIFIERS.add("Buffer");
        PUBLIC_IDENTIFIERS.add("BufferAllocator");
        PUBLIC_IDENTIFIERS.add("BufferBlob");
        PUBLIC_IDENTIFIERS.add("BufferCapabilities");
        PUBLIC_IDENTIFIERS.add("BufferedBufImgOps");
        PUBLIC_IDENTIFIERS.add("BufferedContext");
        PUBLIC_IDENTIFIERS.add("BufferedContextProvider");
        PUBLIC_IDENTIFIERS.add("BufferedImage");
        PUBLIC_IDENTIFIERS.add("BufferedImageFilter");
        PUBLIC_IDENTIFIERS.add("BufferedImageOp");
        PUBLIC_IDENTIFIERS.add("BufferedInputStream");
        PUBLIC_IDENTIFIERS.add("BufferedMaskBlit");
        PUBLIC_IDENTIFIERS.add("BufferedMaskFill");
        PUBLIC_IDENTIFIERS.add("BufferedMonitoredVm");
        PUBLIC_IDENTIFIERS.add("BufferedOutputStream");
        PUBLIC_IDENTIFIERS.add("BufferedPaints");
        PUBLIC_IDENTIFIERS.add("BufferedReader");
        PUBLIC_IDENTIFIERS.add("BufferedRenderPipe");
        PUBLIC_IDENTIFIERS.add("BufferedTextPipe");
        PUBLIC_IDENTIFIERS.add("BufferedWriter");
        PUBLIC_IDENTIFIERS.add("BufferImpl");
        PUBLIC_IDENTIFIERS.add("BufferingSubscriber");
        PUBLIC_IDENTIFIERS.add("BufferOverflowException");
        PUBLIC_IDENTIFIERS.add("BufferPool");
        PUBLIC_IDENTIFIERS.add("BufferPoolMXBean");
        PUBLIC_IDENTIFIERS.add("BufferSource");
        PUBLIC_IDENTIFIERS.add("BufferStrategy");
        PUBLIC_IDENTIFIERS.add("BufferSupplier");
        PUBLIC_IDENTIFIERS.add("BufferUnderflowException");
        PUBLIC_IDENTIFIERS.add("BufImgSurfaceData");
        PUBLIC_IDENTIFIERS.add("BufImgSurfaceManager");
        PUBLIC_IDENTIFIERS.add("BufImgVolatileSurfaceManager");
        PUBLIC_IDENTIFIERS.add("Builder");
        PUBLIC_IDENTIFIERS.add("BuilderFactory");
        PUBLIC_IDENTIFIERS.add("BuildInEditorProvider");
        PUBLIC_IDENTIFIERS.add("BuildStep");
        PUBLIC_IDENTIFIERS.add("BuiltinClassLoader");
        PUBLIC_IDENTIFIERS.add("BuiltinHelpFormatter");
        PUBLIC_IDENTIFIERS.add("BuiltinSchemaGrammar");
        PUBLIC_IDENTIFIERS.add("BundleParams");
        PUBLIC_IDENTIFIERS.add("Bundler");
        PUBLIC_IDENTIFIERS.add("Bundlers");
        PUBLIC_IDENTIFIERS.add("Bundles");
        PUBLIC_IDENTIFIERS.add("Button");
        PUBLIC_IDENTIFIERS.add("ButtonActionListener");
        PUBLIC_IDENTIFIERS.add("ButtonAreaLayout");
        PUBLIC_IDENTIFIERS.add("ButtonBorder");
        PUBLIC_IDENTIFIERS.add("ButtonChangeListener");
        PUBLIC_IDENTIFIERS.add("ButtonGroup");
        PUBLIC_IDENTIFIERS.add("ButtonModel");
        PUBLIC_IDENTIFIERS.add("ButtonPeer");
        PUBLIC_IDENTIFIERS.add("ButtonTranslator");
        PUBLIC_IDENTIFIERS.add("ButtonUI");
        PUBLIC_IDENTIFIERS.add("Byte");
        PUBLIC_IDENTIFIERS.add("BYTE_ARRAY");
        PUBLIC_IDENTIFIERS.add("ByteArrayChannel");
        PUBLIC_IDENTIFIERS.add("ByteArrayCounter");
        PUBLIC_IDENTIFIERS.add("ByteArrayImageSource");
        PUBLIC_IDENTIFIERS.add("ByteArrayInputStream");
        PUBLIC_IDENTIFIERS.add("ByteArrayLexOrder");
        PUBLIC_IDENTIFIERS.add("ByteArrayMonitor");
        PUBLIC_IDENTIFIERS.add("ByteArrayOutputStream");
        PUBLIC_IDENTIFIERS.add("ByteArrayPublisher");
        PUBLIC_IDENTIFIERS.add("ByteArrays");
        PUBLIC_IDENTIFIERS.add("ByteArraySubscriber");
        PUBLIC_IDENTIFIERS.add("ByteArrayTagOrder");
        PUBLIC_IDENTIFIERS.add("ByteBandedRaster");
        PUBLIC_IDENTIFIERS.add("ByteBuffer");
        PUBLIC_IDENTIFIERS.add("ByteBuffered");
        PUBLIC_IDENTIFIERS.add("ByteChannel");
        PUBLIC_IDENTIFIERS.add("Bytecode");
        PUBLIC_IDENTIFIERS.add("BytecodeANewArray");
        PUBLIC_IDENTIFIERS.add("BytecodeBipush");
        PUBLIC_IDENTIFIERS.add("BytecodeCheckCast");
        PUBLIC_IDENTIFIERS.add("BytecodeDescriptor");
        PUBLIC_IDENTIFIERS.add("BytecodeDisassembler");
        PUBLIC_IDENTIFIERS.add("BytecodeGetField");
        PUBLIC_IDENTIFIERS.add("BytecodeGetPut");
        PUBLIC_IDENTIFIERS.add("BytecodeGetStatic");
        PUBLIC_IDENTIFIERS.add("BytecodeGoto");
        PUBLIC_IDENTIFIERS.add("BytecodeGotoW");
        PUBLIC_IDENTIFIERS.add("BytecodeIf");
        PUBLIC_IDENTIFIERS.add("BytecodeIinc");
        PUBLIC_IDENTIFIERS.add("BytecodeInstanceOf");
        PUBLIC_IDENTIFIERS.add("BytecodeInvoke");
        PUBLIC_IDENTIFIERS.add("BytecodeJmp");
        PUBLIC_IDENTIFIERS.add("BytecodeJsr");
        PUBLIC_IDENTIFIERS.add("BytecodeJsrW");
        PUBLIC_IDENTIFIERS.add("BytecodeLoad");
        PUBLIC_IDENTIFIERS.add("BytecodeLoadConstant");
        PUBLIC_IDENTIFIERS.add("BytecodeLoadStore");
        PUBLIC_IDENTIFIERS.add("BytecodeLookupswitch");
        PUBLIC_IDENTIFIERS.add("BytecodeMultiANewArray");
        PUBLIC_IDENTIFIERS.add("BytecodeName");
        PUBLIC_IDENTIFIERS.add("BytecodeNew");
        PUBLIC_IDENTIFIERS.add("BytecodeNewArray");
        PUBLIC_IDENTIFIERS.add("BytecodePosition");
        PUBLIC_IDENTIFIERS.add("BytecodePutField");
        PUBLIC_IDENTIFIERS.add("BytecodePutStatic");
        PUBLIC_IDENTIFIERS.add("BytecodeRet");
        PUBLIC_IDENTIFIERS.add("ByteCodeRewriter");
        PUBLIC_IDENTIFIERS.add("ByteCodes");
        PUBLIC_IDENTIFIERS.add("Bytecodes");
        PUBLIC_IDENTIFIERS.add("BytecodeSipush");
        PUBLIC_IDENTIFIERS.add("BytecodeStore");
        PUBLIC_IDENTIFIERS.add("BytecodeStream");
        PUBLIC_IDENTIFIERS.add("BytecodeTableswitch");
        PUBLIC_IDENTIFIERS.add("BytecodeVisitor");
        PUBLIC_IDENTIFIERS.add("BytecodeWideable");
        PUBLIC_IDENTIFIERS.add("BytecodeWithCPIndex");
        PUBLIC_IDENTIFIERS.add("BytecodeWithKlass");
        PUBLIC_IDENTIFIERS.add("ByteComponentRaster");
        PUBLIC_IDENTIFIERS.add("ByteData");
        PUBLIC_IDENTIFIERS.add("ByteEditor");
        PUBLIC_IDENTIFIERS.add("ByteField");
        PUBLIC_IDENTIFIERS.add("ByteGray");
        PUBLIC_IDENTIFIERS.add("ByteInterleavedRaster");
        PUBLIC_IDENTIFIERS.add("ByteList");
        PUBLIC_IDENTIFIERS.add("ByteListImpl");
        PUBLIC_IDENTIFIERS.add("ByteLookupTable");
        PUBLIC_IDENTIFIERS.add("BytePackedRaster");
        PUBLIC_IDENTIFIERS.add("Bytes");
        PUBLIC_IDENTIFIERS.add("ByteSignature");
        PUBLIC_IDENTIFIERS.add("ByteType");
        PUBLIC_IDENTIFIERS.add("ByteTypeImpl");
        PUBLIC_IDENTIFIERS.add("ByteValue");
        PUBLIC_IDENTIFIERS.add("ByteValueImpl");
        PUBLIC_IDENTIFIERS.add("ByteVector");

    }

    private static void addPublicIdentifiersCC() {
        PUBLIC_IDENTIFIERS.add("C14NMethodParameterSpec");
        PUBLIC_IDENTIFIERS.add("Cache");
        PUBLIC_IDENTIFIERS.add("CachedPainter");
        PUBLIC_IDENTIFIERS.add("CachedRowSet");
        PUBLIC_IDENTIFIERS.add("CachedRowSetImpl");
        PUBLIC_IDENTIFIERS.add("CachedRowSetReader");
        PUBLIC_IDENTIFIERS.add("CachedRowSetWriter");
        PUBLIC_IDENTIFIERS.add("CachedXPathAPI");
        PUBLIC_IDENTIFIERS.add("CacheFSInfo");
        PUBLIC_IDENTIFIERS.add("CacheRequest");
        PUBLIC_IDENTIFIERS.add("CacheResponse");
        PUBLIC_IDENTIFIERS.add("CacheVisitor");
        PUBLIC_IDENTIFIERS.add("CachingParserPool");
        PUBLIC_IDENTIFIERS.add("CAGOp");
        PUBLIC_IDENTIFIERS.add("Calendar");
        PUBLIC_IDENTIFIERS.add("CalendarData");
        PUBLIC_IDENTIFIERS.add("CalendarDataProvider");
        PUBLIC_IDENTIFIERS.add("CalendarDataProviderImpl");
        PUBLIC_IDENTIFIERS.add("CalendarDataUtility");
        PUBLIC_IDENTIFIERS.add("CalendarDate");
        PUBLIC_IDENTIFIERS.add("CalendarNameProvider");
        PUBLIC_IDENTIFIERS.add("CalendarNameProviderImpl");
        PUBLIC_IDENTIFIERS.add("CalendarProvider");
        PUBLIC_IDENTIFIERS.add("CalendarProviderImpl");
        PUBLIC_IDENTIFIERS.add("CalendarSystem");
        PUBLIC_IDENTIFIERS.add("CalendarUtils");
        PUBLIC_IDENTIFIERS.add("Callable");
        PUBLIC_IDENTIFIERS.add("CallableStatement");
        PUBLIC_IDENTIFIERS.add("CallArranger");
        PUBLIC_IDENTIFIERS.add("Callback");
        PUBLIC_IDENTIFIERS.add("CallbackHandler");
        PUBLIC_IDENTIFIERS.add("CallbackHandlerProtection");
        PUBLIC_IDENTIFIERS.add("CallDynamicJavaNode");
        PUBLIC_IDENTIFIERS.add("CallerRunsPolicy");
        PUBLIC_IDENTIFIERS.add("CallerSensitive");
        PUBLIC_IDENTIFIERS.add("CallerSensitiveAdapter");
        PUBLIC_IDENTIFIERS.add("CallingConvention");
        PUBLIC_IDENTIFIERS.add("CallingSequence");
        PUBLIC_IDENTIFIERS.add("CallingSequenceBuilder");
        PUBLIC_IDENTIFIERS.add("CallJavaNode");
        PUBLIC_IDENTIFIERS.add("CallNode");
        PUBLIC_IDENTIFIERS.add("CallRuntimeNode");
        PUBLIC_IDENTIFIERS.add("CallSite");
        PUBLIC_IDENTIFIERS.add("CallSiteDescriptor");
        PUBLIC_IDENTIFIERS.add("CallStaticJavaNode");
        PUBLIC_IDENTIFIERS.add("CallTypeData");
        PUBLIC_IDENTIFIERS.add("CallTypeDataInterface");
        PUBLIC_IDENTIFIERS.add("CALOAD");
        PUBLIC_IDENTIFIERS.add("Cancelable");
        PUBLIC_IDENTIFIERS.add("CancelablePrintJob");
        PUBLIC_IDENTIFIERS.add("CancelAction");
        PUBLIC_IDENTIFIERS.add("CancellationException");
        PUBLIC_IDENTIFIERS.add("CancelledKeyException");
        PUBLIC_IDENTIFIERS.add("CancelSelectionAction");
        PUBLIC_IDENTIFIERS.add("Candidate");
        PUBLIC_IDENTIFIERS.add("CandidatesCache");
        PUBLIC_IDENTIFIERS.add("CannotCreateUriError");
        PUBLIC_IDENTIFIERS.add("CannotProceedException");
        PUBLIC_IDENTIFIERS.add("CannotRedoException");
        PUBLIC_IDENTIFIERS.add("CannotUndoException");
        PUBLIC_IDENTIFIERS.add("CanonicalizationException");
        PUBLIC_IDENTIFIERS.add("CanonicalizationMethod");
        PUBLIC_IDENTIFIERS.add("Canonicalizer11_OmitComments");
        PUBLIC_IDENTIFIERS.add("Canonicalizer11_WithComments");
        PUBLIC_IDENTIFIERS.add("Canonicalizer20010315");
        PUBLIC_IDENTIFIERS.add("Canonicalizer20010315Excl");
        PUBLIC_IDENTIFIERS.add("Canonicalizer20010315ExclOmitComments");
        PUBLIC_IDENTIFIERS.add("Canonicalizer20010315ExclWithComments");
        PUBLIC_IDENTIFIERS.add("Canonicalizer20010315OmitComments");
        PUBLIC_IDENTIFIERS.add("Canonicalizer20010315WithComments");
        PUBLIC_IDENTIFIERS.add("CanonicalizerBase");
        PUBLIC_IDENTIFIERS.add("CanonicalizerPhysical");
        PUBLIC_IDENTIFIERS.add("CanonicalizerSpi");
        PUBLIC_IDENTIFIERS.add("Canvas");
        PUBLIC_IDENTIFIERS.add("CanvasPeer");
        PUBLIC_IDENTIFIERS.add("Capabilities");
        PUBLIC_IDENTIFIERS.add("CaptureCallState");
        PUBLIC_IDENTIFIERS.add("CapturedType");
        PUBLIC_IDENTIFIERS.add("Card");
        PUBLIC_IDENTIFIERS.add("CardChannel");
        PUBLIC_IDENTIFIERS.add("CardException");
        PUBLIC_IDENTIFIERS.add("CardLayout");
        PUBLIC_IDENTIFIERS.add("CardNotPresentException");
        PUBLIC_IDENTIFIERS.add("CardPermission");
        PUBLIC_IDENTIFIERS.add("CardTerminal");
        PUBLIC_IDENTIFIERS.add("CardTerminals");
        PUBLIC_IDENTIFIERS.add("Caret");
        PUBLIC_IDENTIFIERS.add("CaretEvent");
        PUBLIC_IDENTIFIERS.add("CaretListener");
        PUBLIC_IDENTIFIERS.add("CaretPolicy");
        PUBLIC_IDENTIFIERS.add("CarrierThread");
        PUBLIC_IDENTIFIERS.add("CarrierThreadLocal");
        PUBLIC_IDENTIFIERS.add("CaseLabelTree");
        PUBLIC_IDENTIFIERS.add("CaseTree");
        PUBLIC_IDENTIFIERS.add("Cast");
        PUBLIC_IDENTIFIERS.add("CastExpr");
        PUBLIC_IDENTIFIERS.add("CASTORE");
        PUBLIC_IDENTIFIERS.add("Catalog");
        PUBLIC_IDENTIFIERS.add("CatalogException");
        PUBLIC_IDENTIFIERS.add("CatalogFeatures");
        PUBLIC_IDENTIFIERS.add("CatalogResolver");
        PUBLIC_IDENTIFIERS.add("CatchTree");
        PUBLIC_IDENTIFIERS.add("Category");
        PUBLIC_IDENTIFIERS.add("Cbrt");
        PUBLIC_IDENTIFIERS.add("CCacheInputStream");
        PUBLIC_IDENTIFIERS.add("CCacheOutputStream");
        PUBLIC_IDENTIFIERS.add("CDATASection");
        PUBLIC_IDENTIFIERS.add("CDATASectionImpl");
        PUBLIC_IDENTIFIERS.add("CDebugger");
        PUBLIC_IDENTIFIERS.add("CDebugInfoDataBase");
        PUBLIC_IDENTIFIERS.add("CDS");
        PUBLIC_IDENTIFIERS.add("CellEditor");
        PUBLIC_IDENTIFIERS.add("CellEditorHandler");
        PUBLIC_IDENTIFIERS.add("CellEditorListener");
        PUBLIC_IDENTIFIERS.add("CellRendererPane");
        PUBLIC_IDENTIFIERS.add("CellTypeState");
        PUBLIC_IDENTIFIERS.add("CellTypeStateList");
        PUBLIC_IDENTIFIERS.add("CertException");
        PUBLIC_IDENTIFIERS.add("CertId");
        PUBLIC_IDENTIFIERS.add("Certificate");
        PUBLIC_IDENTIFIERS.add("CertificateAlgorithmId");
        PUBLIC_IDENTIFIERS.add("CertificateEncodingException");
        PUBLIC_IDENTIFIERS.add("CertificateException");
        PUBLIC_IDENTIFIERS.add("CertificateExpiredException");
        PUBLIC_IDENTIFIERS.add("CertificateExtensions");
        PUBLIC_IDENTIFIERS.add("CertificateFactory");
        PUBLIC_IDENTIFIERS.add("CertificateFactorySpi");
        PUBLIC_IDENTIFIERS.add("CertificateId");
        PUBLIC_IDENTIFIERS.add("CertificateIssuerExtension");
        PUBLIC_IDENTIFIERS.add("CertificateNotYetValidException");
        PUBLIC_IDENTIFIERS.add("CertificateParsingException");
        PUBLIC_IDENTIFIERS.add("CertificatePoliciesExtension");
        PUBLIC_IDENTIFIERS.add("CertificatePolicyId");
        PUBLIC_IDENTIFIERS.add("CertificatePolicyMap");
        PUBLIC_IDENTIFIERS.add("CertificatePolicySet");
        PUBLIC_IDENTIFIERS.add("CertificateRep");
        PUBLIC_IDENTIFIERS.add("CertificateRevokedException");
        PUBLIC_IDENTIFIERS.add("CertificateSerialNumber");
        PUBLIC_IDENTIFIERS.add("CertificateSubjectName");
        PUBLIC_IDENTIFIERS.add("CertificateValidity");
        PUBLIC_IDENTIFIERS.add("CertificateVersion");
        PUBLIC_IDENTIFIERS.add("CertificateX509Key");
        PUBLIC_IDENTIFIERS.add("CertPath");
        PUBLIC_IDENTIFIERS.add("CertPathBuilder");
        PUBLIC_IDENTIFIERS.add("CertPathBuilderException");
        PUBLIC_IDENTIFIERS.add("CertPathBuilderResult");
        PUBLIC_IDENTIFIERS.add("CertPathBuilderSpi");
        PUBLIC_IDENTIFIERS.add("CertPathChecker");
        PUBLIC_IDENTIFIERS.add("CertPathConstraintsParameters");
        PUBLIC_IDENTIFIERS.add("CertPathHelper");
        PUBLIC_IDENTIFIERS.add("CertPathParameters");
        PUBLIC_IDENTIFIERS.add("CertPathRep");
        PUBLIC_IDENTIFIERS.add("CertPathTrustManagerParameters");
        PUBLIC_IDENTIFIERS.add("CertPathValidator");
        PUBLIC_IDENTIFIERS.add("CertPathValidatorException");
        PUBLIC_IDENTIFIERS.add("CertPathValidatorResult");
        PUBLIC_IDENTIFIERS.add("CertPathValidatorSpi");
        PUBLIC_IDENTIFIERS.add("CertSelector");
        PUBLIC_IDENTIFIERS.add("CertStore");
        PUBLIC_IDENTIFIERS.add("CertStoreException");
        PUBLIC_IDENTIFIERS.add("CertStoreParameters");
        PUBLIC_IDENTIFIERS.add("CertStoreSpi");
        PUBLIC_IDENTIFIERS.add("CFrame");
        PUBLIC_IDENTIFIERS.add("CgroupInfo");
        PUBLIC_IDENTIFIERS.add("CgroupMetrics");
        PUBLIC_IDENTIFIERS.add("CgroupSubsystem");
        PUBLIC_IDENTIFIERS.add("CgroupSubsystemController");
        PUBLIC_IDENTIFIERS.add("CgroupSubsystemFactory");
        PUBLIC_IDENTIFIERS.add("CgroupV1MemorySubSystemController");
        PUBLIC_IDENTIFIERS.add("CgroupV1Metrics");
        PUBLIC_IDENTIFIERS.add("CgroupV1MetricsImpl");
        PUBLIC_IDENTIFIERS.add("CgroupV1Subsystem");
        PUBLIC_IDENTIFIERS.add("CgroupV1SubsystemController");
        PUBLIC_IDENTIFIERS.add("CgroupV2Subsystem");
        PUBLIC_IDENTIFIERS.add("CgroupV2SubsystemController");
        PUBLIC_IDENTIFIERS.add("Chain");
        PUBLIC_IDENTIFIERS.add("ChainedCallSite");
        PUBLIC_IDENTIFIERS.add("ChainedDouble");
        PUBLIC_IDENTIFIERS.add("ChainedInt");
        PUBLIC_IDENTIFIERS.add("ChainedLong");
        PUBLIC_IDENTIFIERS.add("ChainedReference");
        PUBLIC_IDENTIFIERS.add("ChangedCharSetException");
        PUBLIC_IDENTIFIERS.add("ChangeEvent");
        PUBLIC_IDENTIFIERS.add("ChangeHandler");
        PUBLIC_IDENTIFIERS.add("ChangeListener");
        PUBLIC_IDENTIFIERS.add("ChangesCurrentThread");
        PUBLIC_IDENTIFIERS.add("ChangeToParentDirectoryAction");
        PUBLIC_IDENTIFIERS.add("Channel");
        PUBLIC_IDENTIFIERS.add("ChannelBinding");
        PUBLIC_IDENTIFIERS.add("ChannelBindingException");
        PUBLIC_IDENTIFIERS.add("CHAR_ARRAY");
        PUBLIC_IDENTIFIERS.add("Character");
        PUBLIC_IDENTIFIERS.add("Character.Subset");
        PUBLIC_IDENTIFIERS.add("Character.UnicodeBlock");
        PUBLIC_IDENTIFIERS.add("Character.UnicodeScript");
        PUBLIC_IDENTIFIERS.add("CharacterAction");
        PUBLIC_IDENTIFIERS.add("CharacterAttribute");
        PUBLIC_IDENTIFIERS.add("CharacterCodingException");
        PUBLIC_IDENTIFIERS.add("CharacterData");
        PUBLIC_IDENTIFIERS.add("CharacterDataImpl");
        PUBLIC_IDENTIFIERS.add("CharacterEvent");
        PUBLIC_IDENTIFIERS.add("Characteristics");
        PUBLIC_IDENTIFIERS.add("CharacterIterator");
        PUBLIC_IDENTIFIERS.add("CharacterIteratorWrapper");
        PUBLIC_IDENTIFIERS.add("CharacterNodeHandler");
        PUBLIC_IDENTIFIERS.add("CharacterRangeTable_attribute");
        PUBLIC_IDENTIFIERS.add("Characters");
        PUBLIC_IDENTIFIERS.add("CharArrayReader");
        PUBLIC_IDENTIFIERS.add("CharArrayWrapper");
        PUBLIC_IDENTIFIERS.add("CharArrayWriter");
        PUBLIC_IDENTIFIERS.add("CharBuffer");
        PUBLIC_IDENTIFIERS.add("CharConversionException");
        PUBLIC_IDENTIFIERS.add("CharField");
        PUBLIC_IDENTIFIERS.add("CharKey");
        PUBLIC_IDENTIFIERS.add("CharSequence");
        PUBLIC_IDENTIFIERS.add("Charset");
        PUBLIC_IDENTIFIERS.add("CharsetDecoder");
        PUBLIC_IDENTIFIERS.add("CharsetEncoder");
        PUBLIC_IDENTIFIERS.add("CharsetMapping");
        PUBLIC_IDENTIFIERS.add("CharsetProvider");
        PUBLIC_IDENTIFIERS.add("CharsetString");
        PUBLIC_IDENTIFIERS.add("CharSignature");
        PUBLIC_IDENTIFIERS.add("CharToGlyphMapper");
        PUBLIC_IDENTIFIERS.add("CharTreeNodeAdapter");
        PUBLIC_IDENTIFIERS.add("CharTrie");
        PUBLIC_IDENTIFIERS.add("CharType");
        PUBLIC_IDENTIFIERS.add("CharTypeImpl");
        PUBLIC_IDENTIFIERS.add("CharValue");
        PUBLIC_IDENTIFIERS.add("CharValueImpl");
        PUBLIC_IDENTIFIERS.add("Check");
        PUBLIC_IDENTIFIERS.add("CheckAnnotationAdapter");
        PUBLIC_IDENTIFIERS.add("Checkbox");
        PUBLIC_IDENTIFIERS.add("CheckboxGroup");
        PUBLIC_IDENTIFIERS.add("CheckboxMenuItem");
        PUBLIC_IDENTIFIERS.add("CheckboxMenuItemAccessor");
        PUBLIC_IDENTIFIERS.add("CheckboxMenuItemPeer");
        PUBLIC_IDENTIFIERS.add("CheckboxPeer");
        PUBLIC_IDENTIFIERS.add("CheckboxTranslator");
        PUBLIC_IDENTIFIERS.add("CHECKCAST");
        PUBLIC_IDENTIFIERS.add("CheckClassAdapter");
        PUBLIC_IDENTIFIERS.add("CheckContext");
        PUBLIC_IDENTIFIERS.add("CheckedExceptionElement");
        PUBLIC_IDENTIFIERS.add("CheckedInputStream");
        PUBLIC_IDENTIFIERS.add("CheckedOutputStream");
        PUBLIC_IDENTIFIERS.add("Checker");
        PUBLIC_IDENTIFIERS.add("CheckFieldAdapter");
        PUBLIC_IDENTIFIERS.add("CheckMethodAdapter");
        PUBLIC_IDENTIFIERS.add("CheckModuleAdapter");
        PUBLIC_IDENTIFIERS.add("CheckRecordComponentAdapter");
        PUBLIC_IDENTIFIERS.add("CheckSignatureAdapter");
        PUBLIC_IDENTIFIERS.add("Checksum");
        PUBLIC_IDENTIFIERS.add("ChildIterator");
        PUBLIC_IDENTIFIERS.add("ChildLocator");
        PUBLIC_IDENTIFIERS.add("ChildNode");
        PUBLIC_IDENTIFIERS.add("ChildState");
        PUBLIC_IDENTIFIERS.add("ChildTestIterator");
        PUBLIC_IDENTIFIERS.add("Choice");
        PUBLIC_IDENTIFIERS.add("ChoiceCallback");
        PUBLIC_IDENTIFIERS.add("ChoiceFormat");
        PUBLIC_IDENTIFIERS.add("ChoicePeer");
        PUBLIC_IDENTIFIERS.add("chop_frame");
        PUBLIC_IDENTIFIERS.add("Chromaticities");
        PUBLIC_IDENTIFIERS.add("ChronoLocalDate");
        PUBLIC_IDENTIFIERS.add("ChronoLocalDateTime");
        PUBLIC_IDENTIFIERS.add("Chronology");
        PUBLIC_IDENTIFIERS.add("ChronoPeriod");
        PUBLIC_IDENTIFIERS.add("ChronoZonedDateTime");
        PUBLIC_IDENTIFIERS.add("ChunkedInputStream");
        PUBLIC_IDENTIFIERS.add("ChunkedOutputStream");
        PUBLIC_IDENTIFIERS.add("ciArrayKlass");
        PUBLIC_IDENTIFIERS.add("ciBaseObject");
        PUBLIC_IDENTIFIERS.add("ciConstant");
        PUBLIC_IDENTIFIERS.add("ciEnv");
        PUBLIC_IDENTIFIERS.add("ciField");
        PUBLIC_IDENTIFIERS.add("ciInstance");
        PUBLIC_IDENTIFIERS.add("ciInstanceKlass");
        PUBLIC_IDENTIFIERS.add("ciKlass");
        PUBLIC_IDENTIFIERS.add("ciMetadata");
        PUBLIC_IDENTIFIERS.add("ciMethod");
        PUBLIC_IDENTIFIERS.add("ciMethodData");
        PUBLIC_IDENTIFIERS.add("CIntegerField");
        PUBLIC_IDENTIFIERS.add("CIntegerType");
        PUBLIC_IDENTIFIERS.add("CIntField");
        PUBLIC_IDENTIFIERS.add("ciObjArrayKlass");
        PUBLIC_IDENTIFIERS.add("ciObject");
        PUBLIC_IDENTIFIERS.add("ciObjectFactory");
        PUBLIC_IDENTIFIERS.add("Cipher");
        PUBLIC_IDENTIFIERS.add("CipherInputStream");
        PUBLIC_IDENTIFIERS.add("CipherOutputStream");
        PUBLIC_IDENTIFIERS.add("CipherSpi");
        PUBLIC_IDENTIFIERS.add("ciSymbol");
        PUBLIC_IDENTIFIERS.add("ciType");
        PUBLIC_IDENTIFIERS.add("ciTypeArrayKlass");
        PUBLIC_IDENTIFIERS.add("CK_AES_CTR_PARAMS");
        PUBLIC_IDENTIFIERS.add("CK_ATTRIBUTE");
        PUBLIC_IDENTIFIERS.add("CK_C_INITIALIZE_ARGS");
        PUBLIC_IDENTIFIERS.add("CK_CCM_PARAMS");
        PUBLIC_IDENTIFIERS.add("CK_CREATEMUTEX");
        PUBLIC_IDENTIFIERS.add("CK_DATE");
        PUBLIC_IDENTIFIERS.add("CK_DESTROYMUTEX");
        PUBLIC_IDENTIFIERS.add("CK_ECDH1_DERIVE_PARAMS");
        PUBLIC_IDENTIFIERS.add("CK_ECDH2_DERIVE_PARAMS");
        PUBLIC_IDENTIFIERS.add("CK_GCM_PARAMS");
        PUBLIC_IDENTIFIERS.add("CK_INFO");
        PUBLIC_IDENTIFIERS.add("CK_LOCKMUTEX");
        PUBLIC_IDENTIFIERS.add("CK_MECHANISM");
        PUBLIC_IDENTIFIERS.add("CK_MECHANISM_INFO");
        PUBLIC_IDENTIFIERS.add("CK_NOTIFY");
        PUBLIC_IDENTIFIERS.add("CK_PBE_PARAMS");
        PUBLIC_IDENTIFIERS.add("CK_PKCS5_PBKD2_PARAMS");
        PUBLIC_IDENTIFIERS.add("CK_RSA_PKCS_OAEP_PARAMS");
        PUBLIC_IDENTIFIERS.add("CK_RSA_PKCS_PSS_PARAMS");
        PUBLIC_IDENTIFIERS.add("CK_SALSA20_CHACHA20_POLY1305_PARAMS");
        PUBLIC_IDENTIFIERS.add("CK_SESSION_INFO");
        PUBLIC_IDENTIFIERS.add("CK_SLOT_INFO");
        PUBLIC_IDENTIFIERS.add("CK_SSL3_KEY_MAT_OUT");
        PUBLIC_IDENTIFIERS.add("CK_SSL3_KEY_MAT_PARAMS");
        PUBLIC_IDENTIFIERS.add("CK_SSL3_MASTER_KEY_DERIVE_PARAMS");
        PUBLIC_IDENTIFIERS.add("CK_SSL3_RANDOM_DATA");
        PUBLIC_IDENTIFIERS.add("CK_TLS12_KEY_MAT_PARAMS");
        PUBLIC_IDENTIFIERS.add("CK_TLS12_MASTER_KEY_DERIVE_PARAMS");
        PUBLIC_IDENTIFIERS.add("CK_TLS_MAC_PARAMS");
        PUBLIC_IDENTIFIERS.add("CK_TLS_PRF_PARAMS");
        PUBLIC_IDENTIFIERS.add("CK_TOKEN_INFO");
        PUBLIC_IDENTIFIERS.add("CK_UNLOCKMUTEX");
        PUBLIC_IDENTIFIERS.add("CK_VERSION");
        PUBLIC_IDENTIFIERS.add("CK_X9_42_DH1_DERIVE_PARAMS");
        PUBLIC_IDENTIFIERS.add("CK_X9_42_DH2_DERIVE_PARAMS");
        PUBLIC_IDENTIFIERS.add("CksumType");
        PUBLIC_IDENTIFIERS.add("Class");
        PUBLIC_IDENTIFIERS.add("Class2HTML");
        PUBLIC_IDENTIFIERS.add("Class_element_value");
        PUBLIC_IDENTIFIERS.add("ClassAndLoaderVisitor");
        PUBLIC_IDENTIFIERS.add("ClassBrowserPanel");
        PUBLIC_IDENTIFIERS.add("ClassBuilder");
        PUBLIC_IDENTIFIERS.add("ClassCastException");
        PUBLIC_IDENTIFIERS.add("ClassCircularityError");
        PUBLIC_IDENTIFIERS.add("ClassConstants");
        PUBLIC_IDENTIFIERS.add("ClassDesc");
        PUBLIC_IDENTIFIERS.add("ClassDump");
        PUBLIC_IDENTIFIERS.add("ClassElementValue");
        PUBLIC_IDENTIFIERS.add("ClassElementValueGen");
        PUBLIC_IDENTIFIERS.add("ClassFile");
        PUBLIC_IDENTIFIERS.add("ClassFileConstants");
        PUBLIC_IDENTIFIERS.add("ClassFileError");
        PUBLIC_IDENTIFIERS.add("ClassFileInfo");
        PUBLIC_IDENTIFIERS.add("ClassFileNotFoundException");
        PUBLIC_IDENTIFIERS.add("ClassFileReader");
        PUBLIC_IDENTIFIERS.add("ClassFileTransformer");
        PUBLIC_IDENTIFIERS.add("ClassFilter");
        PUBLIC_IDENTIFIERS.add("ClassFinder");
        PUBLIC_IDENTIFIERS.add("ClassFormatError");
        PUBLIC_IDENTIFIERS.add("ClassFormatException");
        PUBLIC_IDENTIFIERS.add("ClassGen");
        PUBLIC_IDENTIFIERS.add("ClassGenerator");
        PUBLIC_IDENTIFIERS.add("ClassGenException");
        PUBLIC_IDENTIFIERS.add("ClassInstallException");
        PUBLIC_IDENTIFIERS.add("ClassLoader");
        PUBLIC_IDENTIFIERS.add("ClassLoaderData");
        PUBLIC_IDENTIFIERS.add("ClassLoaderDataGraph");
        PUBLIC_IDENTIFIERS.add("ClassLoaderReference");
        PUBLIC_IDENTIFIERS.add("ClassLoaderReferenceImpl");
        PUBLIC_IDENTIFIERS.add("ClassLoaderRepository");
        PUBLIC_IDENTIFIERS.add("ClassLoaders");
        PUBLIC_IDENTIFIERS.add("ClassLoaderStats");
        PUBLIC_IDENTIFIERS.add("ClassLoaderWithRepository");
        PUBLIC_IDENTIFIERS.add("ClassLoadingMXBean");
        PUBLIC_IDENTIFIERS.add("ClassLogger");
        PUBLIC_IDENTIFIERS.add("ClassNameSimplifier");
        PUBLIC_IDENTIFIERS.add("ClassNode");
        PUBLIC_IDENTIFIERS.add("ClassNotFoundException");
        PUBLIC_IDENTIFIERS.add("ClassNotLoadedException");
        PUBLIC_IDENTIFIERS.add("ClassNotPreparedException");
        PUBLIC_IDENTIFIERS.add("ClassObjectReference");
        PUBLIC_IDENTIFIERS.add("ClassObjectReferenceImpl");
        PUBLIC_IDENTIFIERS.add("ClassObserver");
        PUBLIC_IDENTIFIERS.add("ClassOutputStream");
        PUBLIC_IDENTIFIERS.add("ClassPrepareEvent");
        PUBLIC_IDENTIFIERS.add("ClassPrepareRequest");
        PUBLIC_IDENTIFIERS.add("ClassQueue");
        PUBLIC_IDENTIFIERS.add("ClassReader");
        PUBLIC_IDENTIFIERS.add("ClassRemapper");
        PUBLIC_IDENTIFIERS.add("ClassRepository");
        PUBLIC_IDENTIFIERS.add("ClassScope");
        PUBLIC_IDENTIFIERS.add("ClassSet");
        PUBLIC_IDENTIFIERS.add("ClassSignature");
        PUBLIC_IDENTIFIERS.add("ClassSigType");
        PUBLIC_IDENTIFIERS.add("ClassStack");
        PUBLIC_IDENTIFIERS.add("ClassState");
        PUBLIC_IDENTIFIERS.add("ClassSymbol");
        PUBLIC_IDENTIFIERS.add("ClassTranslator");
        PUBLIC_IDENTIFIERS.add("ClassTree");
        PUBLIC_IDENTIFIERS.add("ClassType");
        PUBLIC_IDENTIFIERS.add("ClassTypeSignature");
        PUBLIC_IDENTIFIERS.add("ClassUnloadEvent");
        PUBLIC_IDENTIFIERS.add("ClassUnloadRequest");
        PUBLIC_IDENTIFIERS.add("ClassUseMapper");
        PUBLIC_IDENTIFIERS.add("ClassUseWriter");
        PUBLIC_IDENTIFIERS.add("ClassValue");
        PUBLIC_IDENTIFIERS.add("ClassVisitor");
        PUBLIC_IDENTIFIERS.add("ClassWriter");
        PUBLIC_IDENTIFIERS.add("ClassWriterImpl");
        PUBLIC_IDENTIFIERS.add("CLDRBaseLocaleDataMetaInfo");
        PUBLIC_IDENTIFIERS.add("CLDRCalendarDataProviderImpl");
        PUBLIC_IDENTIFIERS.add("CLDRCalendarNameProviderImpl");
        PUBLIC_IDENTIFIERS.add("CLDRLocaleDataMetaInfo");
        PUBLIC_IDENTIFIERS.add("CLDRLocaleProviderAdapter");
        PUBLIC_IDENTIFIERS.add("CLDRTimeZoneNameProviderImpl");
        PUBLIC_IDENTIFIERS.add("Cleanable");
        PUBLIC_IDENTIFIERS.add("Cleaner");
        PUBLIC_IDENTIFIERS.add("CLHSDB");
        PUBLIC_IDENTIFIERS.add("ClientCodeException");
        PUBLIC_IDENTIFIERS.add("ClientCodeWrapper");
        PUBLIC_IDENTIFIERS.add("ClientCommunicatorAdmin");
        PUBLIC_IDENTIFIERS.add("ClientListenerInfo");
        PUBLIC_IDENTIFIERS.add("ClientNotifForwarder");
        PUBLIC_IDENTIFIERS.add("ClientPropertyKeyAccessor");
        PUBLIC_IDENTIFIERS.add("ClientProvider");
        PUBLIC_IDENTIFIERS.add("CLIHelp");
        PUBLIC_IDENTIFIERS.add("Clip");
        PUBLIC_IDENTIFIERS.add("Clipboard");
        PUBLIC_IDENTIFIERS.add("ClipboardOwner");
        PUBLIC_IDENTIFIERS.add("ClipboardTransferable");
        PUBLIC_IDENTIFIERS.add("Clob");
        PUBLIC_IDENTIFIERS.add("Clock");
        PUBLIC_IDENTIFIERS.add("Cloneable");
        PUBLIC_IDENTIFIERS.add("CloneNotSupportedException");
        PUBLIC_IDENTIFIERS.add("Closeable");
        PUBLIC_IDENTIFIERS.add("CloseableDisposerRecord");
        PUBLIC_IDENTIFIERS.add("CloseAction");
        PUBLIC_IDENTIFIERS.add("ClosedByInterruptException");
        PUBLIC_IDENTIFIERS.add("ClosedChannelException");
        PUBLIC_IDENTIFIERS.add("ClosedConnectionException");
        PUBLIC_IDENTIFIERS.add("ClosedDirectoryStreamException");
        PUBLIC_IDENTIFIERS.add("ClosedException");
        PUBLIC_IDENTIFIERS.add("ClosedFileSystemException");
        PUBLIC_IDENTIFIERS.add("ClosedSelectorException");
        PUBLIC_IDENTIFIERS.add("ClosedWatchServiceException");
        PUBLIC_IDENTIFIERS.add("ClosestSymbol");
        PUBLIC_IDENTIFIERS.add("Closure");
        PUBLIC_IDENTIFIERS.add("CMAny");
        PUBLIC_IDENTIFIERS.add("CMBinOp");
        PUBLIC_IDENTIFIERS.add("CMBuilder");
        PUBLIC_IDENTIFIERS.add("CMLeaf");
        PUBLIC_IDENTIFIERS.add("CMMException");
        PUBLIC_IDENTIFIERS.add("CMMServiceProvider");
        PUBLIC_IDENTIFIERS.add("CMMTracer");
        PUBLIC_IDENTIFIERS.add("CMNode");
        PUBLIC_IDENTIFIERS.add("CMNodeFactory");
        PUBLIC_IDENTIFIERS.add("CMStateSet");
        PUBLIC_IDENTIFIERS.add("CMUniOp");
        PUBLIC_IDENTIFIERS.add("Code");
        PUBLIC_IDENTIFIERS.add("Code_attribute");
        PUBLIC_IDENTIFIERS.add("CodeBlob");
        PUBLIC_IDENTIFIERS.add("CodeCache");
        PUBLIC_IDENTIFIERS.add("CodeCacheProvider");
        PUBLIC_IDENTIFIERS.add("CodeCacheVisitor");
        PUBLIC_IDENTIFIERS.add("CodeConstraint");
        PUBLIC_IDENTIFIERS.add("CodeHeap");
        PUBLIC_IDENTIFIERS.add("CodePointIterator");
        PUBLIC_IDENTIFIERS.add("CodePointMap");
        PUBLIC_IDENTIFIERS.add("CodePointTrie");
        PUBLIC_IDENTIFIERS.add("CoderMalfunctionError");
        PUBLIC_IDENTIFIERS.add("CoderResult");
        PUBLIC_IDENTIFIERS.add("CodeSizeEvaluator");
        PUBLIC_IDENTIFIERS.add("CodeSizeOverflow");
        PUBLIC_IDENTIFIERS.add("CodeSource");
        PUBLIC_IDENTIFIERS.add("CodeTaglet");
        PUBLIC_IDENTIFIERS.add("CodeUtil");
        PUBLIC_IDENTIFIERS.add("CodeViewerPanel");
        PUBLIC_IDENTIFIERS.add("CodeWriter");
        PUBLIC_IDENTIFIERS.add("CodingErrorAction");
        PUBLIC_IDENTIFIERS.add("COFFException");
        PUBLIC_IDENTIFIERS.add("COFFFile");
        PUBLIC_IDENTIFIERS.add("COFFFileParser");
        PUBLIC_IDENTIFIERS.add("COFFHeader");
        PUBLIC_IDENTIFIERS.add("COFFLineNumber");
        PUBLIC_IDENTIFIERS.add("COFFRelocation");
        PUBLIC_IDENTIFIERS.add("COFFSymbol");
        PUBLIC_IDENTIFIERS.add("COFFSymbolConstants");
        PUBLIC_IDENTIFIERS.add("CollationData");
        PUBLIC_IDENTIFIERS.add("CollationData_ar");
        PUBLIC_IDENTIFIERS.add("CollationData_be");
        PUBLIC_IDENTIFIERS.add("CollationData_bg");
        PUBLIC_IDENTIFIERS.add("CollationData_ca");
        PUBLIC_IDENTIFIERS.add("CollationData_cs");
        PUBLIC_IDENTIFIERS.add("CollationData_da");
        PUBLIC_IDENTIFIERS.add("CollationData_el");
        PUBLIC_IDENTIFIERS.add("CollationData_es");
        PUBLIC_IDENTIFIERS.add("CollationData_et");
        PUBLIC_IDENTIFIERS.add("CollationData_fi");
        PUBLIC_IDENTIFIERS.add("CollationData_fr");
        PUBLIC_IDENTIFIERS.add("CollationData_he");
        PUBLIC_IDENTIFIERS.add("CollationData_hi");
        PUBLIC_IDENTIFIERS.add("CollationData_hr");
        PUBLIC_IDENTIFIERS.add("CollationData_hu");
        PUBLIC_IDENTIFIERS.add("CollationData_is");
        PUBLIC_IDENTIFIERS.add("CollationData_ja");
        PUBLIC_IDENTIFIERS.add("CollationData_ko");
        PUBLIC_IDENTIFIERS.add("CollationData_lt");
        PUBLIC_IDENTIFIERS.add("CollationData_lv");
        PUBLIC_IDENTIFIERS.add("CollationData_mk");
        PUBLIC_IDENTIFIERS.add("CollationData_no");
        PUBLIC_IDENTIFIERS.add("CollationData_pl");
        PUBLIC_IDENTIFIERS.add("CollationData_ro");
        PUBLIC_IDENTIFIERS.add("CollationData_ru");
        PUBLIC_IDENTIFIERS.add("CollationData_sk");
        PUBLIC_IDENTIFIERS.add("CollationData_sl");
        PUBLIC_IDENTIFIERS.add("CollationData_sq");
        PUBLIC_IDENTIFIERS.add("CollationData_sr");
        PUBLIC_IDENTIFIERS.add("CollationData_sr_Latn");
        PUBLIC_IDENTIFIERS.add("CollationData_sv");
        PUBLIC_IDENTIFIERS.add("CollationData_th");
        PUBLIC_IDENTIFIERS.add("CollationData_tr");
        PUBLIC_IDENTIFIERS.add("CollationData_uk");
        PUBLIC_IDENTIFIERS.add("CollationData_vi");
        PUBLIC_IDENTIFIERS.add("CollationData_zh");
        PUBLIC_IDENTIFIERS.add("CollationData_zh_HK");
        PUBLIC_IDENTIFIERS.add("CollationData_zh_TW");
        PUBLIC_IDENTIFIERS.add("CollationDataProvider");
        PUBLIC_IDENTIFIERS.add("CollationKey");
        PUBLIC_IDENTIFIERS.add("Collator");
        PUBLIC_IDENTIFIERS.add("CollatorFactory");
        PUBLIC_IDENTIFIERS.add("CollatorFactoryBase");
        PUBLIC_IDENTIFIERS.add("CollatorProvider");
        PUBLIC_IDENTIFIERS.add("CollatorProviderImpl");
        PUBLIC_IDENTIFIERS.add("CollatorUtilities");
        PUBLIC_IDENTIFIERS.add("CollectedHeap");
        PUBLIC_IDENTIFIERS.add("CollectedHeapName");
        PUBLIC_IDENTIFIERS.add("Collection");
        PUBLIC_IDENTIFIERS.add("CollectionCertStore");
        PUBLIC_IDENTIFIERS.add("CollectionCertStoreParameters");
        PUBLIC_IDENTIFIERS.add("Collections");
        PUBLIC_IDENTIFIERS.add("Collector");
        PUBLIC_IDENTIFIERS.add("Color");
        PUBLIC_IDENTIFIERS.add("ColorAttribute");
        PUBLIC_IDENTIFIERS.add("ColorChooserComponentFactory");
        PUBLIC_IDENTIFIERS.add("ColorChooserUI");
        PUBLIC_IDENTIFIERS.add("ColorConvertOp");
        PUBLIC_IDENTIFIERS.add("ColorData");
        PUBLIC_IDENTIFIERS.add("ColorEditor");
        PUBLIC_IDENTIFIERS.add("ColorEntry");
        PUBLIC_IDENTIFIERS.add("ColorModel");
        PUBLIC_IDENTIFIERS.add("ColorNode");
        PUBLIC_IDENTIFIERS.add("ColorPalette");
        PUBLIC_IDENTIFIERS.add("Colors");
        PUBLIC_IDENTIFIERS.add("ColorSelectionModel");
        PUBLIC_IDENTIFIERS.add("ColorSpace");
        PUBLIC_IDENTIFIERS.add("ColorTransform");
        PUBLIC_IDENTIFIERS.add("ColorType");
        PUBLIC_IDENTIFIERS.add("ColorUIResource");
        PUBLIC_IDENTIFIERS.add("ColumnAlignment");
        PUBLIC_IDENTIFIERS.add("ColumnFormat");
        PUBLIC_IDENTIFIERS.add("ComboBoxEditor");
        PUBLIC_IDENTIFIERS.add("ComboBoxLayoutManager");
        PUBLIC_IDENTIFIERS.add("ComboBoxModel");
        PUBLIC_IDENTIFIERS.add("ComboBoxUI");
        PUBLIC_IDENTIFIERS.add("ComboPopup");
        PUBLIC_IDENTIFIERS.add("COMDATSelectionTypes");
        PUBLIC_IDENTIFIERS.add("CommandLine");
        PUBLIC_IDENTIFIERS.add("CommandProcessor");
        PUBLIC_IDENTIFIERS.add("CommandProcessorPanel");
        PUBLIC_IDENTIFIERS.add("Comment");
        PUBLIC_IDENTIFIERS.add("CommentEvent");
        PUBLIC_IDENTIFIERS.add("CommentHelper");
        PUBLIC_IDENTIFIERS.add("CommentImpl");
        PUBLIC_IDENTIFIERS.add("CommentTree");
        PUBLIC_IDENTIFIERS.add("CommentUtils");
        PUBLIC_IDENTIFIERS.add("CommonDataSource");
        PUBLIC_IDENTIFIERS.add("CommonMenuBar");
        PUBLIC_IDENTIFIERS.add("CommonResourceBundleProvider");
        PUBLIC_IDENTIFIERS.add("CommonToolBar");
        PUBLIC_IDENTIFIERS.add("CommonUI");
        PUBLIC_IDENTIFIERS.add("CommunicationException");
        PUBLIC_IDENTIFIERS.add("CompactHashTable");
        PUBLIC_IDENTIFIERS.add("CompactibleSpace");
        PUBLIC_IDENTIFIERS.add("Comparable");
        PUBLIC_IDENTIFIERS.add("Comparator");
        PUBLIC_IDENTIFIERS.add("Comparators");
        PUBLIC_IDENTIFIERS.add("Comparison");
        PUBLIC_IDENTIFIERS.add("Compilable");
        PUBLIC_IDENTIFIERS.add("CompilationEvent");
        PUBLIC_IDENTIFIERS.add("CompilationID_attribute");
        PUBLIC_IDENTIFIERS.add("CompilationMXBean");
        PUBLIC_IDENTIFIERS.add("CompilationRequest");
        PUBLIC_IDENTIFIERS.add("CompilationRequestResult");
        PUBLIC_IDENTIFIERS.add("CompilationTask");
        PUBLIC_IDENTIFIERS.add("CompilationUnitTree");
        PUBLIC_IDENTIFIERS.add("Compile");
        PUBLIC_IDENTIFIERS.add("CompiledCode");
        PUBLIC_IDENTIFIERS.add("CompiledICHolder");
        PUBLIC_IDENTIFIERS.add("CompiledMethod");
        PUBLIC_IDENTIFIERS.add("CompiledScript");
        PUBLIC_IDENTIFIERS.add("CompiledVFrame");
        PUBLIC_IDENTIFIERS.add("Compiler");
        PUBLIC_IDENTIFIERS.add("CompilerFailureEvent");
        PUBLIC_IDENTIFIERS.add("CompilerProperties");
        PUBLIC_IDENTIFIERS.add("CompilerThread");
        PUBLIC_IDENTIFIERS.add("CompilerThreadStat");
        PUBLIC_IDENTIFIERS.add("CompileStates");
        PUBLIC_IDENTIFIERS.add("CompileTask");
        PUBLIC_IDENTIFIERS.add("CompletableFuture");
        PUBLIC_IDENTIFIERS.add("Completer");
        PUBLIC_IDENTIFIERS.add("CompleteRestartableTask");
        PUBLIC_IDENTIFIERS.add("CompletingParsedLine");
        PUBLIC_IDENTIFIERS.add("Completion");
        PUBLIC_IDENTIFIERS.add("CompletionException");
        PUBLIC_IDENTIFIERS.add("CompletionFailure");
        PUBLIC_IDENTIFIERS.add("CompletionHandler");
        PUBLIC_IDENTIFIERS.add("CompletionInfo");
        PUBLIC_IDENTIFIERS.add("CompletionMatcher");
        PUBLIC_IDENTIFIERS.add("CompletionMatcherImpl");
        PUBLIC_IDENTIFIERS.add("CompletionNode");
        PUBLIC_IDENTIFIERS.add("Completions");
        PUBLIC_IDENTIFIERS.add("CompletionService");
        PUBLIC_IDENTIFIERS.add("CompletionStage");
        PUBLIC_IDENTIFIERS.add("Component");
        PUBLIC_IDENTIFIERS.add("ComponentAccessor");
        PUBLIC_IDENTIFIERS.add("ComponentAdapter");
        PUBLIC_IDENTIFIERS.add("ComponentBeanInfo");
        PUBLIC_IDENTIFIERS.add("ComponentColorModel");
        PUBLIC_IDENTIFIERS.add("ComponentContext");
        PUBLIC_IDENTIFIERS.add("ComponentDirContext");
        PUBLIC_IDENTIFIERS.add("ComponentEvent");
        PUBLIC_IDENTIFIERS.add("ComponentFactory");
        PUBLIC_IDENTIFIERS.add("ComponentInfo");
        PUBLIC_IDENTIFIERS.add("ComponentInputMap");
        PUBLIC_IDENTIFIERS.add("ComponentInputMapUIResource");
        PUBLIC_IDENTIFIERS.add("ComponentListener");
        PUBLIC_IDENTIFIERS.add("ComponentPeer");
        PUBLIC_IDENTIFIERS.add("ComponentSampleModel");
        PUBLIC_IDENTIFIERS.add("ComponentUI");
        PUBLIC_IDENTIFIERS.add("ComponentView");
        PUBLIC_IDENTIFIERS.add("Composite");
        PUBLIC_IDENTIFIERS.add("CompositeContext");
        PUBLIC_IDENTIFIERS.add("CompositeData");
        PUBLIC_IDENTIFIERS.add("CompositeDataInvocationHandler");
        PUBLIC_IDENTIFIERS.add("CompositeDataSupport");
        PUBLIC_IDENTIFIERS.add("CompositeDataView");
        PUBLIC_IDENTIFIERS.add("CompositeFontDescriptor");
        PUBLIC_IDENTIFIERS.add("CompositeGlyphMapper");
        PUBLIC_IDENTIFIERS.add("CompositeGuardingDynamicLinker");
        PUBLIC_IDENTIFIERS.add("CompositeName");
        PUBLIC_IDENTIFIERS.add("CompositePipe");
        PUBLIC_IDENTIFIERS.add("CompositeType");
        PUBLIC_IDENTIFIERS.add("CompositeTypeBasedGuardingDynamicLinker");
        PUBLIC_IDENTIFIERS.add("CompositeView");
        PUBLIC_IDENTIFIERS.add("Compound");
        PUBLIC_IDENTIFIERS.add("CompoundAssignmentTree");
        PUBLIC_IDENTIFIERS.add("CompoundBorder");
        PUBLIC_IDENTIFIERS.add("CompoundBorderUIResource");
        PUBLIC_IDENTIFIERS.add("CompoundControl");
        PUBLIC_IDENTIFIERS.add("CompoundEdit");
        PUBLIC_IDENTIFIERS.add("CompoundInstruction");
        PUBLIC_IDENTIFIERS.add("CompoundName");
        PUBLIC_IDENTIFIERS.add("CompoundScope");
        PUBLIC_IDENTIFIERS.add("CompoundType");
        PUBLIC_IDENTIFIERS.add("CompoundTypeKind");
        PUBLIC_IDENTIFIERS.add("CompoundWrap");
        PUBLIC_IDENTIFIERS.add("CompressedKlassPointers");
        PUBLIC_IDENTIFIERS.add("CompressedLineNumberReadStream");
        PUBLIC_IDENTIFIERS.add("CompressedOops");
        PUBLIC_IDENTIFIERS.add("CompressedReadStream");
        PUBLIC_IDENTIFIERS.add("CompressedStream");
        PUBLIC_IDENTIFIERS.add("CompressedWriteStream");
        PUBLIC_IDENTIFIERS.add("CompressExpandOperation");
        PUBLIC_IDENTIFIERS.add("CompressIndexes");
        PUBLIC_IDENTIFIERS.add("Compression");
        PUBLIC_IDENTIFIERS.add("ComputeAnnotationSet");
        PUBLIC_IDENTIFIERS.add("Conclude");
        PUBLIC_IDENTIFIERS.add("ConcreteMethodImpl");
        PUBLIC_IDENTIFIERS.add("ConcurrentHashMap");
        PUBLIC_IDENTIFIERS.add("ConcurrentLinkedDeque");
        PUBLIC_IDENTIFIERS.add("ConcurrentLinkedQueue");
        PUBLIC_IDENTIFIERS.add("ConcurrentLocksPrinter");
        PUBLIC_IDENTIFIERS.add("ConcurrentMap");
        PUBLIC_IDENTIFIERS.add("ConcurrentModificationException");
        PUBLIC_IDENTIFIERS.add("ConcurrentNavigableMap");
        PUBLIC_IDENTIFIERS.add("ConcurrentSkipListMap");
        PUBLIC_IDENTIFIERS.add("ConcurrentSkipListSet");
        PUBLIC_IDENTIFIERS.add("Condition");
        PUBLIC_IDENTIFIERS.add("ConditionalExpressionTree");
        PUBLIC_IDENTIFIERS.add("Config");
        PUBLIC_IDENTIFIERS.add("ConfigEntry");
        PUBLIC_IDENTIFIERS.add("ConfigException");
        PUBLIC_IDENTIFIERS.add("ConfigFile");
        PUBLIC_IDENTIFIERS.add("ConfigurableBridge");
        PUBLIC_IDENTIFIERS.add("Configuration");
        PUBLIC_IDENTIFIERS.add("ConfigurationException");
        PUBLIC_IDENTIFIERS.add("ConfigurationSpi");
        PUBLIC_IDENTIFIERS.add("ConfirmationCallback");
        PUBLIC_IDENTIFIERS.add("ConnectDialog");
        PUBLIC_IDENTIFIERS.add("ConnectException");
        PUBLIC_IDENTIFIERS.add("ConnectIOException");
        PUBLIC_IDENTIFIERS.add("Connection");
        PUBLIC_IDENTIFIERS.add("ConnectionBuilder");
        PUBLIC_IDENTIFIERS.add("ConnectionEvent");
        PUBLIC_IDENTIFIERS.add("ConnectionEventListener");
        PUBLIC_IDENTIFIERS.add("ConnectionPendingException");
        PUBLIC_IDENTIFIERS.add("ConnectionPoolDataSource");
        PUBLIC_IDENTIFIERS.add("ConnectionResetException");
        PUBLIC_IDENTIFIERS.add("Connector");
        PUBLIC_IDENTIFIERS.add("ConnectorAddressLink");
        PUBLIC_IDENTIFIERS.add("Console");
        PUBLIC_IDENTIFIERS.add("ConsoleCallbackHandler");
        PUBLIC_IDENTIFIERS.add("ConsoleHandler");
        PUBLIC_IDENTIFIERS.add("ConsoleIOContextTestSupport");
        PUBLIC_IDENTIFIERS.add("Constable");
        PUBLIC_IDENTIFIERS.add("Constant");
        PUBLIC_IDENTIFIERS.add("CONSTANT_Class_info");
        PUBLIC_IDENTIFIERS.add("CONSTANT_Double_info");
        PUBLIC_IDENTIFIERS.add("CONSTANT_Dynamic_info");
        PUBLIC_IDENTIFIERS.add("CONSTANT_Fieldref_info");
        PUBLIC_IDENTIFIERS.add("CONSTANT_Float_info");
        PUBLIC_IDENTIFIERS.add("CONSTANT_Integer_info");
        PUBLIC_IDENTIFIERS.add("CONSTANT_InterfaceMethodref_info");
        PUBLIC_IDENTIFIERS.add("CONSTANT_InvokeDynamic_info");
        PUBLIC_IDENTIFIERS.add("CONSTANT_Long_info");
        PUBLIC_IDENTIFIERS.add("CONSTANT_MethodHandle_info");
        PUBLIC_IDENTIFIERS.add("CONSTANT_Methodref_info");
        PUBLIC_IDENTIFIERS.add("CONSTANT_MethodType_info");
        PUBLIC_IDENTIFIERS.add("CONSTANT_Module_info");
        PUBLIC_IDENTIFIERS.add("CONSTANT_NameAndType_info");
        PUBLIC_IDENTIFIERS.add("CONSTANT_Package_info");
        PUBLIC_IDENTIFIERS.add("CONSTANT_String_info");
        PUBLIC_IDENTIFIERS.add("CONSTANT_Utf8_info");
        PUBLIC_IDENTIFIERS.add("ConstantCallSite");
        PUBLIC_IDENTIFIERS.add("ConstantCaseLabelTree");
        PUBLIC_IDENTIFIERS.add("ConstantCP");
        PUBLIC_IDENTIFIERS.add("ConstantDesc");
        PUBLIC_IDENTIFIERS.add("ConstantDoubleValue");
        PUBLIC_IDENTIFIERS.add("ConstantIntValue");
        PUBLIC_IDENTIFIERS.add("ConstantLongValue");
        PUBLIC_IDENTIFIERS.add("ConstantObject");
        PUBLIC_IDENTIFIERS.add("ConstantOopReadValue");
        PUBLIC_IDENTIFIERS.add("ConstantPool");
        PUBLIC_IDENTIFIERS.add("ConstantPoolCache");
        PUBLIC_IDENTIFIERS.add("ConstantPoolCacheEntry");
        PUBLIC_IDENTIFIERS.add("ConstantPoolException");
        PUBLIC_IDENTIFIERS.add("ConstantPoolGen");
        PUBLIC_IDENTIFIERS.add("ConstantPoolWriter");
        PUBLIC_IDENTIFIERS.add("ConstantPushInstruction");
        PUBLIC_IDENTIFIERS.add("ConstantReflectionProvider");
        PUBLIC_IDENTIFIERS.add("Constants");
        PUBLIC_IDENTIFIERS.add("ConstantsSummaryBuilder");
        PUBLIC_IDENTIFIERS.add("ConstantsSummaryWriter");
        PUBLIC_IDENTIFIERS.add("ConstantsSummaryWriterImpl");
        PUBLIC_IDENTIFIERS.add("ConstantTag");
        PUBLIC_IDENTIFIERS.add("ConstantValue_attribute");
        PUBLIC_IDENTIFIERS.add("ConstantWriter");
        PUBLIC_IDENTIFIERS.add("ConstIterator");
        PUBLIC_IDENTIFIERS.add("ConstMethod");
        PUBLIC_IDENTIFIERS.add("ConstrainableGraphics");
        PUBLIC_IDENTIFIERS.add("Constraints");
        PUBLIC_IDENTIFIERS.add("ConstraintsParameters");
        PUBLIC_IDENTIFIERS.add("ConstructionException");
        PUBLIC_IDENTIFIERS.add("ConstructorAccessor");
        PUBLIC_IDENTIFIERS.add("ConstructorBuilder");
        PUBLIC_IDENTIFIERS.add("ConstructorParameters");
        PUBLIC_IDENTIFIERS.add("ConstructorProperties");
        PUBLIC_IDENTIFIERS.add("ConstructorRepository");
        PUBLIC_IDENTIFIERS.add("ConstructorScope");
        PUBLIC_IDENTIFIERS.add("ConstructorWriter");
        PUBLIC_IDENTIFIERS.add("ConstructorWriterImpl");
        PUBLIC_IDENTIFIERS.add("Consumer");
        PUBLIC_IDENTIFIERS.add("ConsumerSubscriber");
        PUBLIC_IDENTIFIERS.add("Container");
        PUBLIC_IDENTIFIERS.add("ContainerAccessor");
        PUBLIC_IDENTIFIERS.add("ContainerAdapter");
        PUBLIC_IDENTIFIERS.add("ContainerCPUThrottlingEvent");
        PUBLIC_IDENTIFIERS.add("ContainerCPUUsageEvent");
        PUBLIC_IDENTIFIERS.add("ContainerEvent");
        PUBLIC_IDENTIFIERS.add("ContainerIOUsageEvent");
        PUBLIC_IDENTIFIERS.add("ContainerListener");
        PUBLIC_IDENTIFIERS.add("ContainerOrderFocusTraversalPolicy");
        PUBLIC_IDENTIFIERS.add("ContainerPeer");
        PUBLIC_IDENTIFIERS.add("ContainmentFilter");
        PUBLIC_IDENTIFIERS.add("Contended");
        PUBLIC_IDENTIFIERS.add("Content");
        PUBLIC_IDENTIFIERS.add("ContentBuilder");
        PUBLIC_IDENTIFIERS.add("ContentCacheEntry");
        PUBLIC_IDENTIFIERS.add("ContentDriver");
        PUBLIC_IDENTIFIERS.add("ContentHandler");
        PUBLIC_IDENTIFIERS.add("ContentHandlerFactory");
        PUBLIC_IDENTIFIERS.add("ContentInfo");
        PUBLIC_IDENTIFIERS.add("ContentModelValidator");
        PUBLIC_IDENTIFIERS.add("Contents");
        PUBLIC_IDENTIFIERS.add("ContentSigner");
        PUBLIC_IDENTIFIERS.add("ContentSignerParameters");
        PUBLIC_IDENTIFIERS.add("ContentType");
        PUBLIC_IDENTIFIERS.add("Context");
        PUBLIC_IDENTIFIERS.add("ContextCapabilities");
        PUBLIC_IDENTIFIERS.add("ContextEnumerator");
        PUBLIC_IDENTIFIERS.add("ContextMatchStepPattern");
        PUBLIC_IDENTIFIERS.add("ContextNodeList");
        PUBLIC_IDENTIFIERS.add("ContextNotEmptyException");
        PUBLIC_IDENTIFIERS.add("ContextualRenderedImageFactory");
        PUBLIC_IDENTIFIERS.add("ContiguousSpace");
        PUBLIC_IDENTIFIERS.add("Continuation");
        PUBLIC_IDENTIFIERS.add("ContinuationFrame");
        PUBLIC_IDENTIFIERS.add("ContinuationScope");
        PUBLIC_IDENTIFIERS.add("ContinuationSupport");
        PUBLIC_IDENTIFIERS.add("Continue");
        PUBLIC_IDENTIFIERS.add("ContinueTree");
        PUBLIC_IDENTIFIERS.add("Control");
        PUBLIC_IDENTIFIERS.add("ControlFactory");
        PUBLIC_IDENTIFIERS.add("ControllerEventListener");
        PUBLIC_IDENTIFIERS.add("Conversion");
        PUBLIC_IDENTIFIERS.add("ConversionComparator");
        PUBLIC_IDENTIFIERS.add("ConversionInstruction");
        PUBLIC_IDENTIFIERS.add("Convert");
        PUBLIC_IDENTIFIERS.add("ConvolveOp");
        PUBLIC_IDENTIFIERS.add("CookieExtension");
        PUBLIC_IDENTIFIERS.add("CookieHandler");
        PUBLIC_IDENTIFIERS.add("CookieManager");
        PUBLIC_IDENTIFIERS.add("CookiePolicy");
        PUBLIC_IDENTIFIERS.add("CookieStore");
        PUBLIC_IDENTIFIERS.add("Copier");
        PUBLIC_IDENTIFIERS.add("CopyAction");
        PUBLIC_IDENTIFIERS.add("CopyKeyAdapter");
        PUBLIC_IDENTIFIERS.add("CopyOnWriteArrayList");
        PUBLIC_IDENTIFIERS.add("CopyOnWriteArraySet");
        PUBLIC_IDENTIFIERS.add("CopyOption");
        PUBLIC_IDENTIFIERS.add("CoreDocumentImpl");
        PUBLIC_IDENTIFIERS.add("CoreDOMImplementationImpl");
        PUBLIC_IDENTIFIERS.add("CoreReflectionFactory");
        PUBLIC_IDENTIFIERS.add("CoroutineManager");
        PUBLIC_IDENTIFIERS.add("CoroutineParser");
        PUBLIC_IDENTIFIERS.add("CountDownLatch");
        PUBLIC_IDENTIFIERS.add("CountdownTracker");
        PUBLIC_IDENTIFIERS.add("CountedCompleter");
        PUBLIC_IDENTIFIERS.add("CountedTimerTask");
        PUBLIC_IDENTIFIERS.add("CountedTimerTaskUtils");
        PUBLIC_IDENTIFIERS.add("Counter");
        PUBLIC_IDENTIFIERS.add("CounterData");
        PUBLIC_IDENTIFIERS.add("CounterMonitor");
        PUBLIC_IDENTIFIERS.add("CounterMonitorMBean");
        PUBLIC_IDENTIFIERS.add("CountingPrintable");
        PUBLIC_IDENTIFIERS.add("CPInfo");
        PUBLIC_IDENTIFIERS.add("CPInstruction");
        PUBLIC_IDENTIFIERS.add("CPPExpressions");
        PUBLIC_IDENTIFIERS.add("CPRefInfo");
        PUBLIC_IDENTIFIERS.add("CPUEvent");
        PUBLIC_IDENTIFIERS.add("CPUFeatureName");
        PUBLIC_IDENTIFIERS.add("CRC32");
        PUBLIC_IDENTIFIERS.add("Crc32CksumType");
        PUBLIC_IDENTIFIERS.add("CreatedFontTracker");
        PUBLIC_IDENTIFIERS.add("CreateMBeanDialog");
        PUBLIC_IDENTIFIERS.add("CredentialException");
        PUBLIC_IDENTIFIERS.add("CredentialExpiredException");
        PUBLIC_IDENTIFIERS.add("CredentialNotFoundException");
        PUBLIC_IDENTIFIERS.add("Credentials");
        PUBLIC_IDENTIFIERS.add("CredentialsCache");
        PUBLIC_IDENTIFIERS.add("CredentialsUtil");
        PUBLIC_IDENTIFIERS.add("Criterion");
        PUBLIC_IDENTIFIERS.add("CRL");
        PUBLIC_IDENTIFIERS.add("CRLDistributionPointsExtension");
        PUBLIC_IDENTIFIERS.add("CRLException");
        PUBLIC_IDENTIFIERS.add("CRLExtensions");
        PUBLIC_IDENTIFIERS.add("CRLNumberExtension");
        PUBLIC_IDENTIFIERS.add("CRLReasonCodeExtension");
        PUBLIC_IDENTIFIERS.add("CRLSelector");
        PUBLIC_IDENTIFIERS.add("CropImageFilter");
        PUBLIC_IDENTIFIERS.add("Crossings");
        PUBLIC_IDENTIFIERS.add("CRTable");
        PUBLIC_IDENTIFIERS.add("CRTFlags");
        PUBLIC_IDENTIFIERS.add("CSS");
        PUBLIC_IDENTIFIERS.add("CSS2Properties");
        PUBLIC_IDENTIFIERS.add("CSSCharsetRule");
        PUBLIC_IDENTIFIERS.add("CSSFontFaceRule");
        PUBLIC_IDENTIFIERS.add("CSSImportRule");
        PUBLIC_IDENTIFIERS.add("CSSMediaRule");
        PUBLIC_IDENTIFIERS.add("CSSPageRule");
        PUBLIC_IDENTIFIERS.add("CSSPrimitiveValue");
        PUBLIC_IDENTIFIERS.add("CSSRule");
        PUBLIC_IDENTIFIERS.add("CSSRuleList");
        PUBLIC_IDENTIFIERS.add("CSSStyleDeclaration");
        PUBLIC_IDENTIFIERS.add("CSSStyleRule");
        PUBLIC_IDENTIFIERS.add("CSSStyleSheet");
        PUBLIC_IDENTIFIERS.add("CSSUnknownRule");
        PUBLIC_IDENTIFIERS.add("CSSValue");
        PUBLIC_IDENTIFIERS.add("CSSValueList");
        PUBLIC_IDENTIFIERS.add("CStringTreeNodeAdapter");
        PUBLIC_IDENTIFIERS.add("CStringUtilities");
        PUBLIC_IDENTIFIERS.add("CSV");
        PUBLIC_IDENTIFIERS.add("CSVParseException");
        PUBLIC_IDENTIFIERS.add("CtrDrbg");
        PUBLIC_IDENTIFIERS.add("CtSym");
        PUBLIC_IDENTIFIERS.add("CTypeTreeNodeAdapter");
        PUBLIC_IDENTIFIERS.add("CubicCurve2D");
        PUBLIC_IDENTIFIERS.add("CUPSPrinter");
        PUBLIC_IDENTIFIERS.add("CurrencyNameProvider");
        PUBLIC_IDENTIFIERS.add("CurrencyNameProviderImpl");
        PUBLIC_IDENTIFIERS.add("CurrencyNames");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_af");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_af_NA");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_agq");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ak");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_am");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ann");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ar");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ar_DJ");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ar_ER");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ar_KM");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ar_LB");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ar_SO");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ar_SS");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_as");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_asa");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ast");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_az");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_az_Cyrl");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_bas");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_be");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_bem");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_bez");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_bg");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_bm");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_bn");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_bn_IN");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_bo");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_bo_IN");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_br");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_brx");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_bs");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_bs_Cyrl");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ca");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ca_FR");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ccp");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ce");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ceb");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_cgg");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_chr");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ckb");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_cs");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_cv");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_cy");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_da");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_dav");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_de");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_de_CH");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_de_LI");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_de_LU");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_dje");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_doi");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_dsb");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_dyo");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_dz");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ebu");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ee");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_el");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_001");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_AE");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_AG");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_AI");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_AU");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_BB");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_BI");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_BM");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_BS");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_BW");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_BZ");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_CA");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_CC");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_CK");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_CX");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_DK");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_DM");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_ER");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_FJ");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_FK");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_GD");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_GG");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_GH");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_GI");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_GM");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_GY");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_IM");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_IN");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_JE");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_JM");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_KE");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_KI");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_KN");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_KY");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_LC");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_LR");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_LS");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_MG");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_MO");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_MS");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_MT");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_MU");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_MV");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_MW");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_MY");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_NA");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_NF");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_NG");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_NR");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_NU");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_NZ");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_PG");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_PK");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_PN");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_RW");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_SB");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_SC");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_SE");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_SG");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_SH");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_SL");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_SS");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_SX");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_SZ");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_TK");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_TO");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_TT");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_TV");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_TZ");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_UG");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_VC");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_VU");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_WS");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_ZA");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_en_ZM");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_419");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_AR");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_BO");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_BR");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_BZ");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_CL");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_CO");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_CR");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_CU");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_DO");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_EC");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_GQ");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_GT");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_HN");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_MX");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_NI");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_PA");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_PE");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_PH");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_PR");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_PY");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_SV");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_US");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_UY");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_es_VE");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_et");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_eu");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ewo");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fa");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fa_AF");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff_Adlm");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff_Adlm_BF");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff_Adlm_CM");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff_Adlm_GH");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff_Adlm_GM");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff_Adlm_GW");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff_Adlm_LR");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff_Adlm_MR");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff_Adlm_NE");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff_Adlm_NG");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff_Adlm_SL");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff_Adlm_SN");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff_Latn_GH");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff_Latn_GM");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff_Latn_GN");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff_Latn_LR");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff_Latn_MR");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff_Latn_NG");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ff_Latn_SL");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fi");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fil");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fo");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fo_DK");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fr");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fr_BI");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fr_CA");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fr_CD");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fr_DJ");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fr_DZ");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fr_GN");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fr_HT");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fr_KM");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fr_LU");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fr_MG");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fr_MR");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fr_MU");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fr_RW");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fr_SC");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fr_SY");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fr_TN");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fr_VU");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fur");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_fy");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ga");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_gd");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_gl");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_gsw");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_gu");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_guz");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ha");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ha_GH");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_haw");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_he");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_hi");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_hi_Latn");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_hr");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_hr_BA");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_hsb");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_hu");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_hy");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ia");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_id");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ig");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ii");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_is");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_it");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ja");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_jgo");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_jmc");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_jv");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ka");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_kab");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_kam");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_kde");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_kea");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_kgp");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_khq");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ki");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_kk");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_kkj");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_kl");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_kln");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_km");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_kn");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ko");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_kok");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ks");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ks_Deva");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ksb");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ksf");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ksh");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ku");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ky");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_lag");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_lb");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_lg");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_lkt");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ln");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ln_AO");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_lo");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_lrc");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_lt");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_lu");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_luo");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_luy");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_lv");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_mai");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_mas");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_mas_TZ");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_mdf");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_mer");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_mfe");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_mg");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_mgh");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_mgo");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_mi");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_mk");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ml");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_mn");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_mni");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_mr");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ms");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ms_BN");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ms_ID");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ms_SG");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_mt");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_mua");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_my");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_mzn");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_naq");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_nd");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ne");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_nl");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_nl_AW");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_nl_BQ");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_nl_CW");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_nl_SR");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_nl_SX");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_nmg");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_nn");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_nnh");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_no");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_nus");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_nyn");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_om");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_om_KE");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_or");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_os");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_os_RU");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_pa");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_pa_Arab");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_pcm");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_pis");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_pl");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ps");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ps_PK");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_pt");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_pt_AO");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_pt_CV");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_pt_LU");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_pt_MO");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_pt_MZ");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_pt_PT");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_pt_ST");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_qu");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_qu_BO");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_qu_EC");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_rm");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_rn");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ro");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ro_MD");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_rof");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ru");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ru_BY");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ru_KG");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ru_KZ");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ru_MD");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_rw");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_rwk");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sa");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sah");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_saq");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sat");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sbp");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sc");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sd");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sd_Deva");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_se");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_se_SE");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_seh");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ses");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sg");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_shi");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_shi_Latn");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_si");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sk");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sl");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_smn");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sn");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_so");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_so_DJ");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_so_ET");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_so_KE");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sq");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sq_MK");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sr");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sr_Cyrl_BA");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sr_Latn");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sr_Latn_BA");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_su");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sv");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sw");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sw_CD");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sw_KE");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_sw_UG");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ta");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ta_LK");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ta_MY");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ta_SG");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_te");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_teo");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_teo_KE");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_tg");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_th");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ti");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ti_ER");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_tk");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_to");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_tr");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_tt");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_twq");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_tzm");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ug");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_uk");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ur");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_ur_IN");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_uz");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_uz_Arab");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_uz_Cyrl");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_vai");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_vai_Latn");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_vi");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_vun");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_wae");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_wo");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_xh");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_xog");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_yav");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_yi");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_yo");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_yo_BJ");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_yrl");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_yrl_CO");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_yrl_VE");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_yue");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_yue_Hans");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_zgh");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_zh");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_zh_Hans_HK");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_zh_Hans_MO");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_zh_Hans_SG");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_zh_Hant");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_zh_Hant_HK");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_zh_Hant_MO");
        PUBLIC_IDENTIFIERS.add("CurrencyNames_zu");
        PUBLIC_IDENTIFIERS.add("CurrencyNamesProvider");
        PUBLIC_IDENTIFIERS.add("CurrentNodeListFilter");
        PUBLIC_IDENTIFIERS.add("Cursor");
        PUBLIC_IDENTIFIERS.add("CursorAccessor");
        PUBLIC_IDENTIFIERS.add("CursorSupport");
        PUBLIC_IDENTIFIERS.add("Curve");
        PUBLIC_IDENTIFIERS.add("CurveDB");
        PUBLIC_IDENTIFIERS.add("CustomCursor");
        PUBLIC_IDENTIFIERS.add("Customizer");
        PUBLIC_IDENTIFIERS.add("CustomMediaTray");
        PUBLIC_IDENTIFIERS.add("CustomStringPool");
        PUBLIC_IDENTIFIERS.add("CutAction");
        PUBLIC_IDENTIFIERS.add("Cutoff");
        PUBLIC_IDENTIFIERS.add("CVAttributes");
        PUBLIC_IDENTIFIERS.add("CyclicBarrier");
    }

    private static void addPublicIdentifiersDE() {
        PUBLIC_IDENTIFIERS.add("D2F");
        PUBLIC_IDENTIFIERS.add("D2I");
        PUBLIC_IDENTIFIERS.add("D2L");
        PUBLIC_IDENTIFIERS.add("DADD");
        PUBLIC_IDENTIFIERS.add("DALOAD");
        PUBLIC_IDENTIFIERS.add("DASTORE");
        PUBLIC_IDENTIFIERS.add("Data");
        PUBLIC_IDENTIFIERS.add("DataAmount");
        PUBLIC_IDENTIFIERS.add("DatabaseMetaData");
        PUBLIC_IDENTIFIERS.add("DataBuffer");
        PUBLIC_IDENTIFIERS.add("DataBufferNative");
        PUBLIC_IDENTIFIERS.add("DataDirectory");
        PUBLIC_IDENTIFIERS.add("DataFlavor");
        PUBLIC_IDENTIFIERS.add("DataFlavorUtil");
        PUBLIC_IDENTIFIERS.add("DataFormatException");
        PUBLIC_IDENTIFIERS.add("DataFrame");
        PUBLIC_IDENTIFIERS.add("DatagramChannel");
        PUBLIC_IDENTIFIERS.add("DatagramSocket");
        PUBLIC_IDENTIFIERS.add("DatagramSocketAdaptor");
        PUBLIC_IDENTIFIERS.add("DatagramSocketImpl");
        PUBLIC_IDENTIFIERS.add("DatagramSocketImplFactory");
        PUBLIC_IDENTIFIERS.add("DataInput");
        PUBLIC_IDENTIFIERS.add("DataInputStream");
        PUBLIC_IDENTIFIERS.add("DataLayout");
        PUBLIC_IDENTIFIERS.add("DataLine");
        PUBLIC_IDENTIFIERS.add("DataManipulate");
        PUBLIC_IDENTIFIERS.add("DataOutput");
        PUBLIC_IDENTIFIERS.add("DataOutputStream");
        PUBLIC_IDENTIFIERS.add("DataSource");
        PUBLIC_IDENTIFIERS.add("DataStealer");
        PUBLIC_IDENTIFIERS.add("DataTransferer");
        PUBLIC_IDENTIFIERS.add("DataTruncation");
        PUBLIC_IDENTIFIERS.add("DatatypeConfigurationException");
        PUBLIC_IDENTIFIERS.add("DatatypeException");
        PUBLIC_IDENTIFIERS.add("DatatypeFactory");
        PUBLIC_IDENTIFIERS.add("DatatypeFactoryImpl");
        PUBLIC_IDENTIFIERS.add("DatatypeMessageFormatter");
        PUBLIC_IDENTIFIERS.add("DatatypeValidator");
        PUBLIC_IDENTIFIERS.add("Date");
        PUBLIC_IDENTIFIERS.add("DateConverter");
        PUBLIC_IDENTIFIERS.add("DateDV");
        PUBLIC_IDENTIFIERS.add("DateEditor");
        PUBLIC_IDENTIFIERS.add("DateFormat");
        PUBLIC_IDENTIFIERS.add("DateFormatProvider");
        PUBLIC_IDENTIFIERS.add("DateFormatProviderImpl");
        PUBLIC_IDENTIFIERS.add("DateFormatSymbols");
        PUBLIC_IDENTIFIERS.add("DateFormatSymbolsProvider");
        PUBLIC_IDENTIFIERS.add("DateFormatSymbolsProviderImpl");
        PUBLIC_IDENTIFIERS.add("DateFormatter");
        PUBLIC_IDENTIFIERS.add("DateTimeDV");
        PUBLIC_IDENTIFIERS.add("DateTimeException");
        PUBLIC_IDENTIFIERS.add("DateTimeParseException");
        PUBLIC_IDENTIFIERS.add("DateTimeSyntax");
        PUBLIC_IDENTIFIERS.add("DayDV");
        PUBLIC_IDENTIFIERS.add("DCAttribute");
        PUBLIC_IDENTIFIERS.add("DCAuthor");
        PUBLIC_IDENTIFIERS.add("DCBlockTag");
        PUBLIC_IDENTIFIERS.add("DCComment");
        PUBLIC_IDENTIFIERS.add("DCDeprecated");
        PUBLIC_IDENTIFIERS.add("DCDocComment");
        PUBLIC_IDENTIFIERS.add("DCDocRoot");
        PUBLIC_IDENTIFIERS.add("DCDocType");
        PUBLIC_IDENTIFIERS.add("DCEndElement");
        PUBLIC_IDENTIFIERS.add("DCEndPosTree");
        PUBLIC_IDENTIFIERS.add("DCEntity");
        PUBLIC_IDENTIFIERS.add("DCErroneous");
        PUBLIC_IDENTIFIERS.add("DCHidden");
        PUBLIC_IDENTIFIERS.add("DCIdentifier");
        PUBLIC_IDENTIFIERS.add("DCIndex");
        PUBLIC_IDENTIFIERS.add("DCInheritDoc");
        PUBLIC_IDENTIFIERS.add("DCInlineTag");
        PUBLIC_IDENTIFIERS.add("DCLink");
        PUBLIC_IDENTIFIERS.add("DCLiteral");
        PUBLIC_IDENTIFIERS.add("DCMPG");
        PUBLIC_IDENTIFIERS.add("DCMPL");
        PUBLIC_IDENTIFIERS.add("DCONST");
        PUBLIC_IDENTIFIERS.add("DCParam");
        PUBLIC_IDENTIFIERS.add("DCProvides");
        PUBLIC_IDENTIFIERS.add("DCReference");
        PUBLIC_IDENTIFIERS.add("DCReturn");
        PUBLIC_IDENTIFIERS.add("DCSee");
        PUBLIC_IDENTIFIERS.add("DCSerial");
        PUBLIC_IDENTIFIERS.add("DCSerialData");
        PUBLIC_IDENTIFIERS.add("DCSerialField");
        PUBLIC_IDENTIFIERS.add("DCSince");
        PUBLIC_IDENTIFIERS.add("DCSnippet");
        PUBLIC_IDENTIFIERS.add("DCSpec");
        PUBLIC_IDENTIFIERS.add("DCStartElement");
        PUBLIC_IDENTIFIERS.add("DCSummary");
        PUBLIC_IDENTIFIERS.add("DCSystemProperty");
        PUBLIC_IDENTIFIERS.add("DCText");
        PUBLIC_IDENTIFIERS.add("DCThrows");
        PUBLIC_IDENTIFIERS.add("DCTree");
        PUBLIC_IDENTIFIERS.add("DCUnknownBlockTag");
        PUBLIC_IDENTIFIERS.add("DCUnknownInlineTag");
        PUBLIC_IDENTIFIERS.add("DCUses");
        PUBLIC_IDENTIFIERS.add("DCValue");
        PUBLIC_IDENTIFIERS.add("DCVersion");
        PUBLIC_IDENTIFIERS.add("DDIV");
        PUBLIC_IDENTIFIERS.add("DeadlockDetectionPanel");
        PUBLIC_IDENTIFIERS.add("DeadlockDetector");
        PUBLIC_IDENTIFIERS.add("Debug");
        PUBLIC_IDENTIFIERS.add("DebugDirectory");
        PUBLIC_IDENTIFIERS.add("DebugDirectoryEntry");
        PUBLIC_IDENTIFIERS.add("DebugEvent");
        PUBLIC_IDENTIFIERS.add("Debugger");
        PUBLIC_IDENTIFIERS.add("DebuggerBase");
        PUBLIC_IDENTIFIERS.add("DebuggerConsolePanel");
        PUBLIC_IDENTIFIERS.add("DebuggerException");
        PUBLIC_IDENTIFIERS.add("DebuggerInterface");
        PUBLIC_IDENTIFIERS.add("DebuggerUtilities");
        PUBLIC_IDENTIFIERS.add("DebugGraphics");
        PUBLIC_IDENTIFIERS.add("DebugInfoReadStream");
        PUBLIC_IDENTIFIERS.add("DebugInformationRecorder");
        PUBLIC_IDENTIFIERS.add("DebugServer");
        PUBLIC_IDENTIFIERS.add("DebugTypes");
        PUBLIC_IDENTIFIERS.add("DebugVC50");
        PUBLIC_IDENTIFIERS.add("DebugVC50MemberAttributes");
        PUBLIC_IDENTIFIERS.add("DebugVC50ReservedTypes");
        PUBLIC_IDENTIFIERS.add("DebugVC50SegDesc");
        PUBLIC_IDENTIFIERS.add("DebugVC50SegDescEnums");
        PUBLIC_IDENTIFIERS.add("DebugVC50SegInfo");
        PUBLIC_IDENTIFIERS.add("DebugVC50SrcModFileDesc");
        PUBLIC_IDENTIFIERS.add("DebugVC50SrcModLineNumberMap");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSAlignSym");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSFileIndex");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSGlobalPub");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSGlobalSym");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSGlobalTypes");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSLibraries");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSModule");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSMPC");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSOffsetMap16");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSOffsetMap32");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSPreComp");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSPublic");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSPublicSym");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSSegMap");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSSegName");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSSrcLnSeg");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSSrcModule");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSStaticSym");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSSymbolBase");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSSymbols");
        PUBLIC_IDENTIFIERS.add("DebugVC50SSTypes");
        PUBLIC_IDENTIFIERS.add("DebugVC50Subsection");
        PUBLIC_IDENTIFIERS.add("DebugVC50SubsectionDirectory");
        PUBLIC_IDENTIFIERS.add("DebugVC50SubsectionTypes");
        PUBLIC_IDENTIFIERS.add("DebugVC50SymbolEnums");
        PUBLIC_IDENTIFIERS.add("DebugVC50SymbolIterator");
        PUBLIC_IDENTIFIERS.add("DebugVC50SymbolTypes");
        PUBLIC_IDENTIFIERS.add("DebugVC50TypeEnums");
        PUBLIC_IDENTIFIERS.add("DebugVC50TypeIterator");
        PUBLIC_IDENTIFIERS.add("DebugVC50TypeLeafIndices");
        PUBLIC_IDENTIFIERS.add("DebugVC50WrongNumericTypeException");
        PUBLIC_IDENTIFIERS.add("DebugVC50X86RegisterEnums");
        PUBLIC_IDENTIFIERS.add("DecimalDV");
        PUBLIC_IDENTIFIERS.add("DecimalFormat");
        PUBLIC_IDENTIFIERS.add("DecimalFormatSymbols");
        PUBLIC_IDENTIFIERS.add("DecimalFormatSymbolsProvider");
        PUBLIC_IDENTIFIERS.add("DecimalFormatSymbolsProviderImpl");
        PUBLIC_IDENTIFIERS.add("DeclarationSnippet");
        PUBLIC_IDENTIFIERS.add("DeclaredType");
        PUBLIC_IDENTIFIERS.add("DeclHandler");
        PUBLIC_IDENTIFIERS.add("Decoder");
        PUBLIC_IDENTIFIERS.add("Decoder_DBCSONLY");
        PUBLIC_IDENTIFIERS.add("Decoder_EBCDIC");
        PUBLIC_IDENTIFIERS.add("Decoder_EUC_SIM");
        PUBLIC_IDENTIFIERS.add("DecodingCallback");
        PUBLIC_IDENTIFIERS.add("DeconstructionPatternTree");
        PUBLIC_IDENTIFIERS.add("Decoration");
        PUBLIC_IDENTIFIERS.add("DeepNodeListImpl");
        PUBLIC_IDENTIFIERS.add("DefaultAsynchronousChannelProvider");
        PUBLIC_IDENTIFIERS.add("DefaultAttribute");
        PUBLIC_IDENTIFIERS.add("DefaultBoundedRangeModel");
        PUBLIC_IDENTIFIERS.add("DefaultButtonModel");
        PUBLIC_IDENTIFIERS.add("DefaultCaret");
        PUBLIC_IDENTIFIERS.add("DefaultCaseLabelTree");
        PUBLIC_IDENTIFIERS.add("DefaultCellEditor");
        PUBLIC_IDENTIFIERS.add("DefaultColorSelectionModel");
        PUBLIC_IDENTIFIERS.add("DefaultComboBoxModel");
        PUBLIC_IDENTIFIERS.add("DefaultDesktopManager");
        PUBLIC_IDENTIFIERS.add("DefaultDisposerRecord");
        PUBLIC_IDENTIFIERS.add("DefaultDocument");
        PUBLIC_IDENTIFIERS.add("DefaultDocumentEvent");
        PUBLIC_IDENTIFIERS.add("DefaultEditor");
        PUBLIC_IDENTIFIERS.add("DefaultEditorKit");
        PUBLIC_IDENTIFIERS.add("DefaultElement");
        PUBLIC_IDENTIFIERS.add("DefaultErrorHandler");
        PUBLIC_IDENTIFIERS.add("DefaultExpander");
        PUBLIC_IDENTIFIERS.add("DefaultFileSystemProvider");
        PUBLIC_IDENTIFIERS.add("DefaultFileTypeDetector");
        PUBLIC_IDENTIFIERS.add("DefaultFocusManager");
        PUBLIC_IDENTIFIERS.add("DefaultFocusTraversalPolicy");
        PUBLIC_IDENTIFIERS.add("DefaultFormatter");
        PUBLIC_IDENTIFIERS.add("DefaultFormatterFactory");
        PUBLIC_IDENTIFIERS.add("DefaultFtpClientProvider");
        PUBLIC_IDENTIFIERS.add("DefaultHandler");
        PUBLIC_IDENTIFIERS.add("DefaultHandler2");
        PUBLIC_IDENTIFIERS.add("DefaultHeapVisitor");
        PUBLIC_IDENTIFIERS.add("DefaultHighlighter");
        PUBLIC_IDENTIFIERS.add("DefaultHighlightPainter");
        PUBLIC_IDENTIFIERS.add("DefaultHistory");
        PUBLIC_IDENTIFIERS.add("DefaultHttpServerProvider");
        PUBLIC_IDENTIFIERS.add("DefaultInternalObjectFilter");
        PUBLIC_IDENTIFIERS.add("DefaultKeyboardFocusManager");
        PUBLIC_IDENTIFIERS.add("DefaultKeyboardFocusManagerAccessor");
        PUBLIC_IDENTIFIERS.add("DefaultKeyTypedAction");
        PUBLIC_IDENTIFIERS.add("DefaultLayoutStyle");
        PUBLIC_IDENTIFIERS.add("DefaultLdapDnsProvider");
        PUBLIC_IDENTIFIERS.add("DefaultListCellRenderer");
        PUBLIC_IDENTIFIERS.add("DefaultListModel");
        PUBLIC_IDENTIFIERS.add("DefaultListSelectionModel");
        PUBLIC_IDENTIFIERS.add("DefaultLoaderRepository");
        PUBLIC_IDENTIFIERS.add("DefaultLoggerFinder");
        PUBLIC_IDENTIFIERS.add("DefaultLookup");
        PUBLIC_IDENTIFIERS.add("DefaultMBeanServerInterceptor");
        PUBLIC_IDENTIFIERS.add("DefaultMenuLayout");
        PUBLIC_IDENTIFIERS.add("DefaultMetadataVisitor");
        PUBLIC_IDENTIFIERS.add("DefaultMetalTheme");
        PUBLIC_IDENTIFIERS.add("DefaultMutableTreeNode");
        PUBLIC_IDENTIFIERS.add("DefaultMXBeanMappingFactory");
        PUBLIC_IDENTIFIERS.add("DefaultNode");
        PUBLIC_IDENTIFIERS.add("DefaultObjectVisitor");
        PUBLIC_IDENTIFIERS.add("DefaultOopVisitor");
        PUBLIC_IDENTIFIERS.add("DefaultParser");
        PUBLIC_IDENTIFIERS.add("DefaultPersistenceDelegate");
        PUBLIC_IDENTIFIERS.add("DefaultProxySelector");
        PUBLIC_IDENTIFIERS.add("DefaultRefMapFormatter");
        PUBLIC_IDENTIFIERS.add("DefaultResponseControlFactory");
        PUBLIC_IDENTIFIERS.add("DefaultRowSorter");
        PUBLIC_IDENTIFIERS.add("Defaults");
        PUBLIC_IDENTIFIERS.add("DefaultSelectorProvider");
        PUBLIC_IDENTIFIERS.add("DefaultSingleSelectionModel");
        PUBLIC_IDENTIFIERS.add("DefaultStyledDocument");
        PUBLIC_IDENTIFIERS.add("DefaultSymbolVisitor");
        PUBLIC_IDENTIFIERS.add("DefaultSynthStyle");
        PUBLIC_IDENTIFIERS.add("DefaultTableCellHeaderRenderer");
        PUBLIC_IDENTIFIERS.add("DefaultTableCellRenderer");
        PUBLIC_IDENTIFIERS.add("DefaultTableColumnModel");
        PUBLIC_IDENTIFIERS.add("DefaultTableModel");
        PUBLIC_IDENTIFIERS.add("DefaultText");
        PUBLIC_IDENTIFIERS.add("DefaultTextField");
        PUBLIC_IDENTIFIERS.add("DefaultTextUI");
        PUBLIC_IDENTIFIERS.add("DefaultTreeCellEditor");
        PUBLIC_IDENTIFIERS.add("DefaultTreeCellRenderer");
        PUBLIC_IDENTIFIERS.add("DefaultTreeModel");
        PUBLIC_IDENTIFIERS.add("DefaultTreeSelectionModel");
        PUBLIC_IDENTIFIERS.add("DefaultTypeVisitor");
        PUBLIC_IDENTIFIERS.add("DefaultValues");
        PUBLIC_IDENTIFIERS.add("DefaultXMLDocumentHandler");
        PUBLIC_IDENTIFIERS.add("DeferredAttr");
        PUBLIC_IDENTIFIERS.add("DeferredCDATASectionImpl");
        PUBLIC_IDENTIFIERS.add("DeferredCommentImpl");
        PUBLIC_IDENTIFIERS.add("DeferredCompleter");
        PUBLIC_IDENTIFIERS.add("DeferredCompletionFailureHandler");
        PUBLIC_IDENTIFIERS.add("DeferredDiagnosticHandler");
        PUBLIC_IDENTIFIERS.add("DeferredDocumentImpl");
        PUBLIC_IDENTIFIERS.add("DeferredDocumentTypeImpl");
        PUBLIC_IDENTIFIERS.add("DeferredDOMImplementationImpl");
        PUBLIC_IDENTIFIERS.add("DeferredElementDefinitionImpl");
        PUBLIC_IDENTIFIERS.add("DeferredElementImpl");
        PUBLIC_IDENTIFIERS.add("DeferredElementNSImpl");
        PUBLIC_IDENTIFIERS.add("DeferredEntityImpl");
        PUBLIC_IDENTIFIERS.add("DeferredEntityReferenceImpl");
        PUBLIC_IDENTIFIERS.add("DeferredLintHandler");
        PUBLIC_IDENTIFIERS.add("DeferredNode");
        PUBLIC_IDENTIFIERS.add("DeferredNotationImpl");
        PUBLIC_IDENTIFIERS.add("DeferredProcessingInstructionImpl");
        PUBLIC_IDENTIFIERS.add("DeferredTextImpl");
        PUBLIC_IDENTIFIERS.add("DeferredType");
        PUBLIC_IDENTIFIERS.add("DefinedBy");
        PUBLIC_IDENTIFIERS.add("Deflater");
        PUBLIC_IDENTIFIERS.add("DeflaterInputStream");
        PUBLIC_IDENTIFIERS.add("DeflaterOutputStream");
        PUBLIC_IDENTIFIERS.add("DefNewGeneration");
        PUBLIC_IDENTIFIERS.add("Delayed");
        PUBLIC_IDENTIFIERS.add("DelayQueue");
        PUBLIC_IDENTIFIERS.add("DelegatableDecoder");
        PUBLIC_IDENTIFIERS.add("DelegateAction");
        PUBLIC_IDENTIFIERS.add("DelegatedSymbol");
        PUBLIC_IDENTIFIERS.add("DelegatedType");
        PUBLIC_IDENTIFIERS.add("DelegateHttpsURLConnection");
        PUBLIC_IDENTIFIERS.add("DelegatingHttpExchange");
        PUBLIC_IDENTIFIERS.add("DelegatingJavaFileManager");
        PUBLIC_IDENTIFIERS.add("DeltaCRLIndicatorExtension");
        PUBLIC_IDENTIFIERS.add("DeoptimizationBlob");
        PUBLIC_IDENTIFIERS.add("Dependencies");
        PUBLIC_IDENTIFIERS.add("Dependency");
        PUBLIC_IDENTIFIERS.add("DependencyKind");
        PUBLIC_IDENTIFIERS.add("DeployParams");
        PUBLIC_IDENTIFIERS.add("DeprData");
        PUBLIC_IDENTIFIERS.add("DeprDB");
        PUBLIC_IDENTIFIERS.add("Deprecated");
        PUBLIC_IDENTIFIERS.add("Deprecated_attribute");
        PUBLIC_IDENTIFIERS.add("DeprecatedAPIListBuilder");
        PUBLIC_IDENTIFIERS.add("DeprecatedListWriter");
        PUBLIC_IDENTIFIERS.add("DeprecatedTaglet");
        PUBLIC_IDENTIFIERS.add("DeprecatedTree");
        PUBLIC_IDENTIFIERS.add("DepsAnalyzer");
        PUBLIC_IDENTIFIERS.add("Depth");
        PUBLIC_IDENTIFIERS.add("Deque");
        PUBLIC_IDENTIFIERS.add("Dereference");
        PUBLIC_IDENTIFIERS.add("DEREncodedKeyValue");
        PUBLIC_IDENTIFIERS.add("DEREncodedKeyValueResolver");
        PUBLIC_IDENTIFIERS.add("DerEncoder");
        PUBLIC_IDENTIFIERS.add("DerInputStream");
        PUBLIC_IDENTIFIERS.add("DerValue");
        PUBLIC_IDENTIFIERS.add("Des3");
        PUBLIC_IDENTIFIERS.add("Des3DkCrypto");
        PUBLIC_IDENTIFIERS.add("DesCbcCrcEType");
        PUBLIC_IDENTIFIERS.add("DescendantIterator");
        PUBLIC_IDENTIFIERS.add("DescendingVisitor");
        PUBLIC_IDENTIFIERS.add("Description");
        PUBLIC_IDENTIFIERS.add("Descriptor");
        PUBLIC_IDENTIFIERS.add("DescriptorAccess");
        PUBLIC_IDENTIFIERS.add("DescriptorCache");
        PUBLIC_IDENTIFIERS.add("DescriptorException");
        PUBLIC_IDENTIFIERS.add("DescriptorKey");
        PUBLIC_IDENTIFIERS.add("DescriptorRead");
        PUBLIC_IDENTIFIERS.add("DescriptorSupport");
        PUBLIC_IDENTIFIERS.add("DESedeCipher");
        PUBLIC_IDENTIFIERS.add("DESedeKeySpec");
        PUBLIC_IDENTIFIERS.add("DeserializationChecker");
        PUBLIC_IDENTIFIERS.add("DesignMode");
        PUBLIC_IDENTIFIERS.add("DESKeySpec");
        PUBLIC_IDENTIFIERS.add("Desktop");
        PUBLIC_IDENTIFIERS.add("DesktopDatatransferService");
        PUBLIC_IDENTIFIERS.add("DesktopDatatransferServiceImpl");
        PUBLIC_IDENTIFIERS.add("DesktopIconActionListener");
        PUBLIC_IDENTIFIERS.add("DesktopIconMouseListener");
        PUBLIC_IDENTIFIERS.add("DesktopIconUI");
        PUBLIC_IDENTIFIERS.add("DesktopManager");
        PUBLIC_IDENTIFIERS.add("DesktopPaneUI");
        PUBLIC_IDENTIFIERS.add("DesktopPeer");
        PUBLIC_IDENTIFIERS.add("DesktopProperty");
        PUBLIC_IDENTIFIERS.add("DesMacCksumType");
        PUBLIC_IDENTIFIERS.add("DesMacKCksumType");
        PUBLIC_IDENTIFIERS.add("Destroyable");
        PUBLIC_IDENTIFIERS.add("DestroyFailedException");
        PUBLIC_IDENTIFIERS.add("DestSurfaceProvider");
        PUBLIC_IDENTIFIERS.add("DFAContentModel");
        PUBLIC_IDENTIFIERS.add("DflCache");
        PUBLIC_IDENTIFIERS.add("DGC");
        PUBLIC_IDENTIFIERS.add("DGCAckHandler");
        PUBLIC_IDENTIFIERS.add("DHGenParameterSpec");
        PUBLIC_IDENTIFIERS.add("DHKey");
        PUBLIC_IDENTIFIERS.add("DHParameterSpec");
        PUBLIC_IDENTIFIERS.add("DHPrivateKey");
        PUBLIC_IDENTIFIERS.add("DHPrivateKeySpec");
        PUBLIC_IDENTIFIERS.add("DHPublicKey");
        PUBLIC_IDENTIFIERS.add("DHPublicKeySpec");
        PUBLIC_IDENTIFIERS.add("Diag");
        PUBLIC_IDENTIFIERS.add("Diagnostic");
        PUBLIC_IDENTIFIERS.add("DiagnosticCommandImpl");
        PUBLIC_IDENTIFIERS.add("DiagnosticCommandMBean");
        PUBLIC_IDENTIFIERS.add("DiagnosticFormatter");
        PUBLIC_IDENTIFIERS.add("DiagnosticHandler");
        PUBLIC_IDENTIFIERS.add("DiagnosticInfo");
        PUBLIC_IDENTIFIERS.add("DiagnosticListener");
        PUBLIC_IDENTIFIERS.add("DiagnosticPosition");
        PUBLIC_IDENTIFIERS.add("DiagnosticSource");
        PUBLIC_IDENTIFIERS.add("DiagnosticSourceUnwrapper");
        PUBLIC_IDENTIFIERS.add("Diags");
        PUBLIC_IDENTIFIERS.add("Dialog");
        PUBLIC_IDENTIFIERS.add("DialogOwnerAccessor");
        PUBLIC_IDENTIFIERS.add("DialogPeer");
        PUBLIC_IDENTIFIERS.add("Dictionary");
        PUBLIC_IDENTIFIERS.add("DictionaryBasedBreakIterator");
        PUBLIC_IDENTIFIERS.add("Diff");
        PUBLIC_IDENTIFIERS.add("DiffHelper");
        PUBLIC_IDENTIFIERS.add("Digester");
        PUBLIC_IDENTIFIERS.add("DigesterFactory");
        PUBLIC_IDENTIFIERS.add("DigesterOutputStream");
        PUBLIC_IDENTIFIERS.add("DigestException");
        PUBLIC_IDENTIFIERS.add("DigestInputStream");
        PUBLIC_IDENTIFIERS.add("DigestMethod");
        PUBLIC_IDENTIFIERS.add("DigestMethodParameterSpec");
        PUBLIC_IDENTIFIERS.add("DigestOutputStream");
        PUBLIC_IDENTIFIERS.add("Dimension");
        PUBLIC_IDENTIFIERS.add("Dimension2D");
        PUBLIC_IDENTIFIERS.add("DimensionUIResource");
        PUBLIC_IDENTIFIERS.add("DirArchive");
        PUBLIC_IDENTIFIERS.add("DirContext");
        PUBLIC_IDENTIFIERS.add("DirectBuffer");
        PUBLIC_IDENTIFIERS.add("DirectColorModel");
        PUBLIC_IDENTIFIERS.add("DirectExecutionControl");
        PUBLIC_IDENTIFIERS.add("Directive");
        PUBLIC_IDENTIFIERS.add("DirectiveTree");
        PUBLIC_IDENTIFIERS.add("DirectiveVisitor");
        PUBLIC_IDENTIFIERS.add("DirectMethodHandleDesc");
        PUBLIC_IDENTIFIERS.add("DirectoryCellRenderer");
        PUBLIC_IDENTIFIERS.add("DirectoryComboBoxAction");
        PUBLIC_IDENTIFIERS.add("DirectoryComboBoxModel");
        PUBLIC_IDENTIFIERS.add("DirectoryManager");
        PUBLIC_IDENTIFIERS.add("DirectoryNotEmptyException");
        PUBLIC_IDENTIFIERS.add("DirectoryStream");
        PUBLIC_IDENTIFIERS.add("DirObjectFactory");
        PUBLIC_IDENTIFIERS.add("DirSearch");
        PUBLIC_IDENTIFIERS.add("DirStateFactory");
        PUBLIC_IDENTIFIERS.add("DirtyRegion");
        PUBLIC_IDENTIFIERS.add("DisabledAlgorithmConstraints");
        PUBLIC_IDENTIFIERS.add("Disassembler");
        PUBLIC_IDENTIFIERS.add("DisassemblerTask");
        PUBLIC_IDENTIFIERS.add("DisassemblerTool");
        PUBLIC_IDENTIFIERS.add("DiscardDiagnosticHandler");
        PUBLIC_IDENTIFIERS.add("DiscardOldestPolicy");
        PUBLIC_IDENTIFIERS.add("DiscardPolicy");
        PUBLIC_IDENTIFIERS.add("Dispatcher");
        PUBLIC_IDENTIFIERS.add("DispatcherWrapper");
        PUBLIC_IDENTIFIERS.add("Display");
        PUBLIC_IDENTIFIERS.add("DisplayChangedListener");
        PUBLIC_IDENTIFIERS.add("Disposer");
        PUBLIC_IDENTIFIERS.add("DisposerRecord");
        PUBLIC_IDENTIFIERS.add("DisposerTarget");
        PUBLIC_IDENTIFIERS.add("DistributionPoint");
        PUBLIC_IDENTIFIERS.add("DistributionPointFetcher");
        PUBLIC_IDENTIFIERS.add("DistributionPointName");
        PUBLIC_IDENTIFIERS.add("Div");
        PUBLIC_IDENTIFIERS.add("DividerLayout");
        PUBLIC_IDENTIFIERS.add("DkCrypto");
        PUBLIC_IDENTIFIERS.add("DLL");
        PUBLIC_IDENTIFIERS.add("DLLCharacteristics");
        PUBLIC_IDENTIFIERS.add("DLOAD");
        PUBLIC_IDENTIFIERS.add("DMUL");
        PUBLIC_IDENTIFIERS.add("DNEG");
        PUBLIC_IDENTIFIERS.add("DnsClient");
        PUBLIC_IDENTIFIERS.add("DnsContext");
        PUBLIC_IDENTIFIERS.add("DnsContextFactory");
        PUBLIC_IDENTIFIERS.add("DNSName");
        PUBLIC_IDENTIFIERS.add("DnsUrl");
        PUBLIC_IDENTIFIERS.add("dnsURLContext");
        PUBLIC_IDENTIFIERS.add("dnsURLContextFactory");
        PUBLIC_IDENTIFIERS.add("Doc");
        PUBLIC_IDENTIFIERS.add("DocAttribute");
        PUBLIC_IDENTIFIERS.add("DocAttributeSet");
        PUBLIC_IDENTIFIERS.add("DocCommentInfo");
        PUBLIC_IDENTIFIERS.add("DocCommentParser");
        PUBLIC_IDENTIFIERS.add("DocCommentTable");
        PUBLIC_IDENTIFIERS.add("DocCommentTree");
        PUBLIC_IDENTIFIERS.add("DocEnvImpl");
        PUBLIC_IDENTIFIERS.add("DocFile");
        PUBLIC_IDENTIFIERS.add("DocFileElement");
        PUBLIC_IDENTIFIERS.add("DocFileFactory");
        PUBLIC_IDENTIFIERS.add("DocFileIOException");
        PUBLIC_IDENTIFIERS.add("DocFilesHandler");
        PUBLIC_IDENTIFIERS.add("DocFilesHandlerImpl");
        PUBLIC_IDENTIFIERS.add("DocFinder");
        PUBLIC_IDENTIFIERS.add("DocFlavor");
        PUBLIC_IDENTIFIERS.add("DockingListener");
        PUBLIC_IDENTIFIERS.add("Doclet");
        PUBLIC_IDENTIFIERS.add("DocletConstants");
        PUBLIC_IDENTIFIERS.add("DocletElement");
        PUBLIC_IDENTIFIERS.add("DocletEnvironment");
        PUBLIC_IDENTIFIERS.add("DocletException");
        PUBLIC_IDENTIFIERS.add("DocLink");
        PUBLIC_IDENTIFIERS.add("DocLint");
        PUBLIC_IDENTIFIERS.add("DocPath");
        PUBLIC_IDENTIFIERS.add("DocPaths");
        PUBLIC_IDENTIFIERS.add("DocPretty");
        PUBLIC_IDENTIFIERS.add("DocPrintJob");
        PUBLIC_IDENTIFIERS.add("DocRootTaglet");
        PUBLIC_IDENTIFIERS.add("DocRootTree");
        PUBLIC_IDENTIFIERS.add("DocSourcePositions");
        PUBLIC_IDENTIFIERS.add("DocTree");
        PUBLIC_IDENTIFIERS.add("DocTreeFactory");
        PUBLIC_IDENTIFIERS.add("DocTreeMaker");
        PUBLIC_IDENTIFIERS.add("DocTreePath");
        PUBLIC_IDENTIFIERS.add("DocTreePathScanner");
        PUBLIC_IDENTIFIERS.add("DocTrees");
        PUBLIC_IDENTIFIERS.add("DocTreeScanner");
        PUBLIC_IDENTIFIERS.add("DocTreeVisitor");
        PUBLIC_IDENTIFIERS.add("DocTypeTree");
        PUBLIC_IDENTIFIERS.add("Document");
        PUBLIC_IDENTIFIERS.add("Documentation");
        PUBLIC_IDENTIFIERS.add("DocumentationTask");
        PUBLIC_IDENTIFIERS.add("DocumentationTool");
        PUBLIC_IDENTIFIERS.add("DocumentBuilder");
        PUBLIC_IDENTIFIERS.add("DocumentBuilderFactory");
        PUBLIC_IDENTIFIERS.add("DocumentBuilderFactoryImpl");
        PUBLIC_IDENTIFIERS.add("DocumentBuilderImpl");
        PUBLIC_IDENTIFIERS.add("DocumentCSS");
        PUBLIC_IDENTIFIERS.add("Documented");
        PUBLIC_IDENTIFIERS.add("DocumentEvent");
        PUBLIC_IDENTIFIERS.add("DocumentFilter");
        PUBLIC_IDENTIFIERS.add("DocumentFragment");
        PUBLIC_IDENTIFIERS.add("DocumentFragmentImpl");
        PUBLIC_IDENTIFIERS.add("DocumentHandler");
        PUBLIC_IDENTIFIERS.add("DocumentImpl");
        PUBLIC_IDENTIFIERS.add("DocumentListener");
        PUBLIC_IDENTIFIERS.add("DocumentParser");
        PUBLIC_IDENTIFIERS.add("DocumentPropertiesUI");
        PUBLIC_IDENTIFIERS.add("DocumentRange");
        PUBLIC_IDENTIFIERS.add("DocumentStyle");
        PUBLIC_IDENTIFIERS.add("DocumentTraversal");
        PUBLIC_IDENTIFIERS.add("DocumentType");
        PUBLIC_IDENTIFIERS.add("DocumentTypeImpl");
        PUBLIC_IDENTIFIERS.add("DocumentView");
        PUBLIC_IDENTIFIERS.add("DOM");
        PUBLIC_IDENTIFIERS.add("DOM2DTM");
        PUBLIC_IDENTIFIERS.add("DOM2DTMdefaultNamespaceDeclarationNode");
        PUBLIC_IDENTIFIERS.add("DOM2SAX");
        PUBLIC_IDENTIFIERS.add("DOM2TO");
        PUBLIC_IDENTIFIERS.add("DOM3Serializer");
        PUBLIC_IDENTIFIERS.add("DomainCombiner");
        PUBLIC_IDENTIFIERS.add("DOMBuilder");
        PUBLIC_IDENTIFIERS.add("DOMCache");
        PUBLIC_IDENTIFIERS.add("DOMCanonicalizationMethod");
        PUBLIC_IDENTIFIERS.add("DOMConfiguration");
        PUBLIC_IDENTIFIERS.add("DOMConfigurationImpl");
        PUBLIC_IDENTIFIERS.add("DOMCryptoContext");
        PUBLIC_IDENTIFIERS.add("DOMDigestMethod");
        PUBLIC_IDENTIFIERS.add("DOMEnhancedForDTM");
        PUBLIC_IDENTIFIERS.add("DOMEntityResolverWrapper");
        PUBLIC_IDENTIFIERS.add("DOMError");
        PUBLIC_IDENTIFIERS.add("DOMErrorHandler");
        PUBLIC_IDENTIFIERS.add("DOMErrorHandlerWrapper");
        PUBLIC_IDENTIFIERS.add("DOMErrorImpl");
        PUBLIC_IDENTIFIERS.add("DOMException");
        PUBLIC_IDENTIFIERS.add("DOMHMACSignatureMethod");
        PUBLIC_IDENTIFIERS.add("DOMImplementation");
        PUBLIC_IDENTIFIERS.add("DOMImplementationCSS");
        PUBLIC_IDENTIFIERS.add("DOMImplementationImpl");
        PUBLIC_IDENTIFIERS.add("DOMImplementationList");
        PUBLIC_IDENTIFIERS.add("DOMImplementationListImpl");
        PUBLIC_IDENTIFIERS.add("DOMImplementationLS");
        PUBLIC_IDENTIFIERS.add("DOMImplementationSource");
        PUBLIC_IDENTIFIERS.add("DOMImplementationSourceImpl");
        PUBLIC_IDENTIFIERS.add("DOMInputImpl");
        PUBLIC_IDENTIFIERS.add("DOMKeyValue");
        PUBLIC_IDENTIFIERS.add("DOMLocator");
        PUBLIC_IDENTIFIERS.add("DOMLocatorImpl");
        PUBLIC_IDENTIFIERS.add("DOMMessageFormatter");
        PUBLIC_IDENTIFIERS.add("DOMNamespaceContext");
        PUBLIC_IDENTIFIERS.add("DOMNormalizer");
        PUBLIC_IDENTIFIERS.add("DOMOutputImpl");
        PUBLIC_IDENTIFIERS.add("DOMParser");
        PUBLIC_IDENTIFIERS.add("DOMParserImpl");
        PUBLIC_IDENTIFIERS.add("DOMResult");
        PUBLIC_IDENTIFIERS.add("DOMRSAPSSSignatureMethod");
        PUBLIC_IDENTIFIERS.add("DOMSerializer");
        PUBLIC_IDENTIFIERS.add("DOMSerializerImpl");
        PUBLIC_IDENTIFIERS.add("DOMSignatureMethod");
        PUBLIC_IDENTIFIERS.add("DOMSignatureValue");
        PUBLIC_IDENTIFIERS.add("DOMSignContext");
        PUBLIC_IDENTIFIERS.add("DOMSource");
        PUBLIC_IDENTIFIERS.add("DOMStringList");
        PUBLIC_IDENTIFIERS.add("DOMStringListImpl");
        PUBLIC_IDENTIFIERS.add("DOMStructure");
        PUBLIC_IDENTIFIERS.add("DOMSubTreeData");
        PUBLIC_IDENTIFIERS.add("DOMTransform");
        PUBLIC_IDENTIFIERS.add("DOMURIReference");
        PUBLIC_IDENTIFIERS.add("DOMUtil");
        PUBLIC_IDENTIFIERS.add("DOMValidateContext");
        PUBLIC_IDENTIFIERS.add("DOMWSFilter");
        PUBLIC_IDENTIFIERS.add("DOMXSImplementationSourceImpl");
        PUBLIC_IDENTIFIERS.add("DontInline");
        PUBLIC_IDENTIFIERS.add("DosFileAttributes");
        PUBLIC_IDENTIFIERS.add("DosFileAttributeView");
        PUBLIC_IDENTIFIERS.add("DotGraphAttributes");
        PUBLIC_IDENTIFIERS.add("DottableNode");
        PUBLIC_IDENTIFIERS.add("DotVisitor");
        PUBLIC_IDENTIFIERS.add("Double");
        PUBLIC_IDENTIFIERS.add("DoubleAccumulator");
        PUBLIC_IDENTIFIERS.add("DoubleAdder");
        PUBLIC_IDENTIFIERS.add("DoubleBinaryOperator");
        PUBLIC_IDENTIFIERS.add("DoubleBuffer");
        PUBLIC_IDENTIFIERS.add("DoubleByte");
        PUBLIC_IDENTIFIERS.add("DoubleByteEncoder");
        PUBLIC_IDENTIFIERS.add("DoubleClickListener");
        PUBLIC_IDENTIFIERS.add("DoubleConsts");
        PUBLIC_IDENTIFIERS.add("DoubleConsumer");
        PUBLIC_IDENTIFIERS.add("DoubleData");
        PUBLIC_IDENTIFIERS.add("DoubleDV");
        PUBLIC_IDENTIFIERS.add("DoubleEditor");
        PUBLIC_IDENTIFIERS.add("DoubleField");
        PUBLIC_IDENTIFIERS.add("DoubleFunction");
        PUBLIC_IDENTIFIERS.add("DoubleMapMultiConsumer");
        PUBLIC_IDENTIFIERS.add("DoublePredicate");
        PUBLIC_IDENTIFIERS.add("DoubleSignature");
        PUBLIC_IDENTIFIERS.add("DoubleStream");
        PUBLIC_IDENTIFIERS.add("DoubleSummaryStatistics");
        PUBLIC_IDENTIFIERS.add("DoubleSupplier");
        PUBLIC_IDENTIFIERS.add("DoubleToIntFunction");
        PUBLIC_IDENTIFIERS.add("DoubleToLongFunction");
        PUBLIC_IDENTIFIERS.add("DoubleTreeNodeAdapter");
        PUBLIC_IDENTIFIERS.add("DoubleType");
        PUBLIC_IDENTIFIERS.add("DoubleTypeImpl");
        PUBLIC_IDENTIFIERS.add("DoubleUnaryOperator");
        PUBLIC_IDENTIFIERS.add("DoubleValue");
        PUBLIC_IDENTIFIERS.add("DoubleValueImpl");
        PUBLIC_IDENTIFIERS.add("DoubleVector");
        PUBLIC_IDENTIFIERS.add("DoWhileLoopTree");
        PUBLIC_IDENTIFIERS.add("DowncallLinker");
        PUBLIC_IDENTIFIERS.add("DPathConsumer2D");
        PUBLIC_IDENTIFIERS.add("DraconianErrorHandler");
        PUBLIC_IDENTIFIERS.add("DragController");
        PUBLIC_IDENTIFIERS.add("DragGestureEvent");
        PUBLIC_IDENTIFIERS.add("DragGestureListener");
        PUBLIC_IDENTIFIERS.add("DragGestureRecognizer");
        PUBLIC_IDENTIFIERS.add("DragSource");
        PUBLIC_IDENTIFIERS.add("DragSourceAdapter");
        PUBLIC_IDENTIFIERS.add("DragSourceContext");
        PUBLIC_IDENTIFIERS.add("DragSourceContextAccessor");
        PUBLIC_IDENTIFIERS.add("DragSourceContextPeer");
        PUBLIC_IDENTIFIERS.add("DragSourceContextWrapper");
        PUBLIC_IDENTIFIERS.add("DragSourceDragEvent");
        PUBLIC_IDENTIFIERS.add("DragSourceDropEvent");
        PUBLIC_IDENTIFIERS.add("DragSourceEvent");
        PUBLIC_IDENTIFIERS.add("DragSourceListener");
        PUBLIC_IDENTIFIERS.add("DragSourceMotionListener");
        PUBLIC_IDENTIFIERS.add("DragWindow");
        PUBLIC_IDENTIFIERS.add("DrawGlyphList");
        PUBLIC_IDENTIFIERS.add("DrawGlyphListAA");
        PUBLIC_IDENTIFIERS.add("DrawGlyphListColor");
        PUBLIC_IDENTIFIERS.add("DrawGlyphListLCD");
        PUBLIC_IDENTIFIERS.add("DrawHandler");
        PUBLIC_IDENTIFIERS.add("DrawImage");
        PUBLIC_IDENTIFIERS.add("DrawImagePipe");
        PUBLIC_IDENTIFIERS.add("DrawLine");
        PUBLIC_IDENTIFIERS.add("DrawParallelogram");
        PUBLIC_IDENTIFIERS.add("DrawPath");
        PUBLIC_IDENTIFIERS.add("DrawPolygons");
        PUBLIC_IDENTIFIERS.add("DrawRect");
        PUBLIC_IDENTIFIERS.add("DrbgParameters");
        PUBLIC_IDENTIFIERS.add("DREM");
        PUBLIC_IDENTIFIERS.add("DRETURN");
        PUBLIC_IDENTIFIERS.add("Driver");
        PUBLIC_IDENTIFIERS.add("DriverAction");
        PUBLIC_IDENTIFIERS.add("DriverManager");
        PUBLIC_IDENTIFIERS.add("DriverPropertyInfo");
        PUBLIC_IDENTIFIERS.add("DropLocation");
        PUBLIC_IDENTIFIERS.add("DropTarget");
        PUBLIC_IDENTIFIERS.add("DropTargetAdapter");
        PUBLIC_IDENTIFIERS.add("DropTargetAutoScroller");
        PUBLIC_IDENTIFIERS.add("DropTargetContextAccessor");
        PUBLIC_IDENTIFIERS.add("DropTargetContextPeer");
        PUBLIC_IDENTIFIERS.add("DropTargetContextWrapper");
        PUBLIC_IDENTIFIERS.add("DropTargetDragEvent");
        PUBLIC_IDENTIFIERS.add("DropTargetDropEvent");
        PUBLIC_IDENTIFIERS.add("DropTargetEvent");
        PUBLIC_IDENTIFIERS.add("DropTargetListener");
        PUBLIC_IDENTIFIERS.add("DropTargetPeer");
        PUBLIC_IDENTIFIERS.add("DSAKey");
        PUBLIC_IDENTIFIERS.add("DSAKeyFactory");
        PUBLIC_IDENTIFIERS.add("DSAKeyPairGenerator");
        PUBLIC_IDENTIFIERS.add("DSAKeyValue");
        PUBLIC_IDENTIFIERS.add("DSAKeyValueResolver");
        PUBLIC_IDENTIFIERS.add("DSAParameterGenerator");
        PUBLIC_IDENTIFIERS.add("DSAParameters");
        PUBLIC_IDENTIFIERS.add("DSAParameterSpec");
        PUBLIC_IDENTIFIERS.add("DSAParams");
        PUBLIC_IDENTIFIERS.add("DSAPrivateKey");
        PUBLIC_IDENTIFIERS.add("DSAPrivateKeySpec");
        PUBLIC_IDENTIFIERS.add("DSAPublicKey");
        PUBLIC_IDENTIFIERS.add("DSAPublicKeySpec");
        PUBLIC_IDENTIFIERS.add("DSO");
        PUBLIC_IDENTIFIERS.add("DSTORE");
        PUBLIC_IDENTIFIERS.add("DSUB");
        PUBLIC_IDENTIFIERS.add("DTD");
        PUBLIC_IDENTIFIERS.add("DTDConfiguration");
        PUBLIC_IDENTIFIERS.add("DTDConstants");
        PUBLIC_IDENTIFIERS.add("DTDDVFactory");
        PUBLIC_IDENTIFIERS.add("DTDDVFactoryImpl");
        PUBLIC_IDENTIFIERS.add("DTDEvent");
        PUBLIC_IDENTIFIERS.add("DTDGrammar");
        PUBLIC_IDENTIFIERS.add("DTDGrammarBucket");
        PUBLIC_IDENTIFIERS.add("DTDGrammarUtil");
        PUBLIC_IDENTIFIERS.add("DTDHandler");
        PUBLIC_IDENTIFIERS.add("DTDParser");
        PUBLIC_IDENTIFIERS.add("DTM");
        PUBLIC_IDENTIFIERS.add("DTMAxisIterator");
        PUBLIC_IDENTIFIERS.add("DTMAxisIteratorBase");
        PUBLIC_IDENTIFIERS.add("DTMAxisIterNodeList");
        PUBLIC_IDENTIFIERS.add("DTMAxisTraverser");
        PUBLIC_IDENTIFIERS.add("DTMChildIterNodeList");
        PUBLIC_IDENTIFIERS.add("DTMDefaultBase");
        PUBLIC_IDENTIFIERS.add("DTMDefaultBaseIterators");
        PUBLIC_IDENTIFIERS.add("DTMDefaultBaseTraversers");
        PUBLIC_IDENTIFIERS.add("DTMDocumentImpl");
        PUBLIC_IDENTIFIERS.add("DTMDOMException");
        PUBLIC_IDENTIFIERS.add("DTMException");
        PUBLIC_IDENTIFIERS.add("DTMFilter");
        PUBLIC_IDENTIFIERS.add("DTMIterator");
        PUBLIC_IDENTIFIERS.add("DTMManager");
        PUBLIC_IDENTIFIERS.add("DTMManagerDefault");
        PUBLIC_IDENTIFIERS.add("DTMNamedNodeMap");
        PUBLIC_IDENTIFIERS.add("DTMNodeIterator");
        PUBLIC_IDENTIFIERS.add("DTMNodeList");
        PUBLIC_IDENTIFIERS.add("DTMNodeListBase");
        PUBLIC_IDENTIFIERS.add("DTMNodeProxy");
        PUBLIC_IDENTIFIERS.add("DTMSafeStringPool");
        PUBLIC_IDENTIFIERS.add("DTMStringPool");
        PUBLIC_IDENTIFIERS.add("DTMTreeWalker");
        PUBLIC_IDENTIFIERS.add("DTMWSFilter");
        PUBLIC_IDENTIFIERS.add("DumbTerminal");
        PUBLIC_IDENTIFIERS.add("DummyDebugger");
        PUBLIC_IDENTIFIERS.add("DummyEvent");
        PUBLIC_IDENTIFIERS.add("DummyScope");
        PUBLIC_IDENTIFIERS.add("DummySymbolFinder");
        PUBLIC_IDENTIFIERS.add("DumpExports");
        PUBLIC_IDENTIFIERS.add("DUP");
        PUBLIC_IDENTIFIERS.add("DUP2");
        PUBLIC_IDENTIFIERS.add("DUP2_X1");
        PUBLIC_IDENTIFIERS.add("DUP2_X2");
        PUBLIC_IDENTIFIERS.add("DUP_X1");
        PUBLIC_IDENTIFIERS.add("DUP_X2");
        PUBLIC_IDENTIFIERS.add("DuplicateFormatFlagsException");
        PUBLIC_IDENTIFIERS.add("DuplicateRequestException");
        PUBLIC_IDENTIFIERS.add("Duration");
        PUBLIC_IDENTIFIERS.add("DurationDV");
        PUBLIC_IDENTIFIERS.add("DVFactoryException");
        PUBLIC_IDENTIFIERS.add("DwarfParser");
        PUBLIC_IDENTIFIERS.add("Dynamic");
        PUBLIC_IDENTIFIERS.add("DynamicCallSiteDesc");
        PUBLIC_IDENTIFIERS.add("DynamicConstantDesc");
        PUBLIC_IDENTIFIERS.add("DynamicMBean");
        PUBLIC_IDENTIFIERS.add("DynamicMBean2");
        PUBLIC_IDENTIFIERS.add("DynamicMethodSymbol");
        PUBLIC_IDENTIFIERS.add("DynamicUtilTreeNode");
        PUBLIC_IDENTIFIERS.add("DynamicVarSymbol");
        PUBLIC_IDENTIFIERS.add("ECCurveDefinition");
        PUBLIC_IDENTIFIERS.add("ECDSAOperations");
        PUBLIC_IDENTIFIERS.add("ECField");
        PUBLIC_IDENTIFIERS.add("ECFieldF2m");
        PUBLIC_IDENTIFIERS.add("ECFieldFp");
        PUBLIC_IDENTIFIERS.add("ECGenParameterSpec");
        PUBLIC_IDENTIFIERS.add("ECharacterCategory");
        PUBLIC_IDENTIFIERS.add("ECharacterDirection");
        PUBLIC_IDENTIFIERS.add("ECKey");
        PUBLIC_IDENTIFIERS.add("ECKeySizeParameterSpec");
        PUBLIC_IDENTIFIERS.add("ECKeyValue");
        PUBLIC_IDENTIFIERS.add("ECKeyValueResolver");
        PUBLIC_IDENTIFIERS.add("ECOperations");
        PUBLIC_IDENTIFIERS.add("ECParameterSpec");
        PUBLIC_IDENTIFIERS.add("ECPoint");
        PUBLIC_IDENTIFIERS.add("ECPrivateKey");
        PUBLIC_IDENTIFIERS.add("ECPrivateKeySpec");
        PUBLIC_IDENTIFIERS.add("ECPublicKey");
        PUBLIC_IDENTIFIERS.add("ECPublicKeySpec");
        PUBLIC_IDENTIFIERS.add("Ed25519");
        PUBLIC_IDENTIFIERS.add("Ed25519Operations");
        PUBLIC_IDENTIFIERS.add("Ed448");
        PUBLIC_IDENTIFIERS.add("Ed448Operations");
        PUBLIC_IDENTIFIERS.add("EdDSAAlgorithmParameters");
        PUBLIC_IDENTIFIERS.add("EdDSADigestAlgHolder");
        PUBLIC_IDENTIFIERS.add("EdDSAKeyFactory");
        PUBLIC_IDENTIFIERS.add("EdDSAKeyPairGenerator");
        PUBLIC_IDENTIFIERS.add("EdDSAOperations");
        PUBLIC_IDENTIFIERS.add("EdDSAParameters");
        PUBLIC_IDENTIFIERS.add("EdDSAParameterSpec");
        PUBLIC_IDENTIFIERS.add("EdDSASignature");
        PUBLIC_IDENTIFIERS.add("EdECKey");
        PUBLIC_IDENTIFIERS.add("EdECOperations");
        PUBLIC_IDENTIFIERS.add("EdECPrivateKey");
        PUBLIC_IDENTIFIERS.add("EdECPublicKey");
        PUBLIC_IDENTIFIERS.add("EDIPartyName");
        PUBLIC_IDENTIFIERS.add("EditableAtEndDocument");
        PUBLIC_IDENTIFIERS.add("EditFocusAdapter");
        PUBLIC_IDENTIFIERS.add("Editor");
        PUBLIC_IDENTIFIERS.add("EditorCommands");
        PUBLIC_IDENTIFIERS.add("EditorContainer");
        PUBLIC_IDENTIFIERS.add("EditorDelegate");
        PUBLIC_IDENTIFIERS.add("EditorFactory");
        PUBLIC_IDENTIFIERS.add("EditorKit");
        PUBLIC_IDENTIFIERS.add("EditPadProvider");
        PUBLIC_IDENTIFIERS.add("Element");
        PUBLIC_IDENTIFIERS.add("element_value");
        PUBLIC_IDENTIFIERS.add("element_value_pair");
        PUBLIC_IDENTIFIERS.add("ElementBuffer");
        PUBLIC_IDENTIFIERS.add("ElementChange");
        PUBLIC_IDENTIFIERS.add("ElementCSSInlineStyle");
        PUBLIC_IDENTIFIERS.add("ElementDefinitionImpl");
        PUBLIC_IDENTIFIERS.add("ElementEdit");
        PUBLIC_IDENTIFIERS.add("ElementFilter");
        PUBLIC_IDENTIFIERS.add("ElementHandler");
        PUBLIC_IDENTIFIERS.add("ElementImpl");
        PUBLIC_IDENTIFIERS.add("ElementIterator");
        PUBLIC_IDENTIFIERS.add("ElementKindVisitor14");
        PUBLIC_IDENTIFIERS.add("ElementKindVisitor6");
        PUBLIC_IDENTIFIERS.add("ElementKindVisitor7");
        PUBLIC_IDENTIFIERS.add("ElementKindVisitor8");
        PUBLIC_IDENTIFIERS.add("ElementKindVisitor9");
        PUBLIC_IDENTIFIERS.add("ElementListWriter");
        PUBLIC_IDENTIFIERS.add("ElementNSImpl");
        PUBLIC_IDENTIFIERS.add("ElementProxy");
        PUBLIC_IDENTIFIERS.add("ElementPSVI");
        PUBLIC_IDENTIFIERS.add("ElementPSVImpl");
        PUBLIC_IDENTIFIERS.add("Elements");
        PUBLIC_IDENTIFIERS.add("ElementScanner14");
        PUBLIC_IDENTIFIERS.add("ElementScanner6");
        PUBLIC_IDENTIFIERS.add("ElementScanner7");
        PUBLIC_IDENTIFIERS.add("ElementScanner8");
        PUBLIC_IDENTIFIERS.add("ElementScanner9");
        PUBLIC_IDENTIFIERS.add("ElementSpec");
        PUBLIC_IDENTIFIERS.add("ElementsTable");
        PUBLIC_IDENTIFIERS.add("ElementStack");
        PUBLIC_IDENTIFIERS.add("ElementStack2");
        PUBLIC_IDENTIFIERS.add("ElementState");
        PUBLIC_IDENTIFIERS.add("ElementTraversal");
        PUBLIC_IDENTIFIERS.add("ElementValue");
        PUBLIC_IDENTIFIERS.add("ElementValueGen");
        PUBLIC_IDENTIFIERS.add("ElementValuePair");
        PUBLIC_IDENTIFIERS.add("ElementValuePairGen");
        PUBLIC_IDENTIFIERS.add("ElementVisitor");
        PUBLIC_IDENTIFIERS.add("ELFException");
        PUBLIC_IDENTIFIERS.add("ELFFile");
        PUBLIC_IDENTIFIERS.add("ELFFileParser");
        PUBLIC_IDENTIFIERS.add("ELFHashTable");
        PUBLIC_IDENTIFIERS.add("ELFHeader");
        PUBLIC_IDENTIFIERS.add("ELFProgramHeader");
        PUBLIC_IDENTIFIERS.add("ELFSectionHeader");
        PUBLIC_IDENTIFIERS.add("ELFStringTable");
        PUBLIC_IDENTIFIERS.add("ELFSymbol");
        PUBLIC_IDENTIFIERS.add("Ellipse2D");
        PUBLIC_IDENTIFIERS.add("EllipticCurve");
        PUBLIC_IDENTIFIERS.add("EmbeddedFrame");
        PUBLIC_IDENTIFIERS.add("Empty");
        PUBLIC_IDENTIFIERS.add("EmptyBorder");
        PUBLIC_IDENTIFIERS.add("EmptyBorderUIResource");
        PUBLIC_IDENTIFIERS.add("EmptyEndPosTable");
        PUBLIC_IDENTIFIERS.add("EmptyPath");
        PUBLIC_IDENTIFIERS.add("EmptyPublisher");
        PUBLIC_IDENTIFIERS.add("EmptySelectionModel");
        PUBLIC_IDENTIFIERS.add("EmptySerializer");
        PUBLIC_IDENTIFIERS.add("EmptyStackException");
        PUBLIC_IDENTIFIERS.add("EmptyStatementTree");
        PUBLIC_IDENTIFIERS.add("EmptyVisitor");
        PUBLIC_IDENTIFIERS.add("Enabled");
        PUBLIC_IDENTIFIERS.add("EncAPRepPart");
        PUBLIC_IDENTIFIERS.add("EncASRepPart");
        PUBLIC_IDENTIFIERS.add("EncKDCRepPart");
        PUBLIC_IDENTIFIERS.add("EncKrbCredPart");
        PUBLIC_IDENTIFIERS.add("EnclosingMethod");
        PUBLIC_IDENTIFIERS.add("EnclosingMethod_attribute");
        PUBLIC_IDENTIFIERS.add("EncodedKeySpec");
        PUBLIC_IDENTIFIERS.add("EncodedSpeculationReason");
        PUBLIC_IDENTIFIERS.add("Encoder");
        PUBLIC_IDENTIFIERS.add("Encoder_DBCSONLY");
        PUBLIC_IDENTIFIERS.add("Encoder_EBCDIC");
        PUBLIC_IDENTIFIERS.add("Encoder_EUC_SIM");
        PUBLIC_IDENTIFIERS.add("Encoding");
        PUBLIC_IDENTIFIERS.add("EncodingInfo");
        PUBLIC_IDENTIFIERS.add("EncodingMap");
        PUBLIC_IDENTIFIERS.add("EncryptedData");
        PUBLIC_IDENTIFIERS.add("EncryptedPrivateKeyInfo");
        PUBLIC_IDENTIFIERS.add("EncryptionKey");
        PUBLIC_IDENTIFIERS.add("EncTGSRepPart");
        PUBLIC_IDENTIFIERS.add("EncTicketPart");
        PUBLIC_IDENTIFIERS.add("EndDocument");
        PUBLIC_IDENTIFIERS.add("EndDocumentEvent");
        PUBLIC_IDENTIFIERS.add("EndElement");
        PUBLIC_IDENTIFIERS.add("EndElementEvent");
        PUBLIC_IDENTIFIERS.add("EndElementTree");
        PUBLIC_IDENTIFIERS.add("EndOfFileException");
        PUBLIC_IDENTIFIERS.add("Endpoint");
        PUBLIC_IDENTIFIERS.add("EndPosTable");
        PUBLIC_IDENTIFIERS.add("EndSubPathHandler");
        PUBLIC_IDENTIFIERS.add("EngineTerminationException");
        PUBLIC_IDENTIFIERS.add("EnhancedForLoopTree");
        PUBLIC_IDENTIFIERS.add("Enter");
        PUBLIC_IDENTIFIERS.add("Entity");
        PUBLIC_IDENTIFIERS.add("ENTITYDatatypeValidator");
        PUBLIC_IDENTIFIERS.add("EntityDeclaration");
        PUBLIC_IDENTIFIERS.add("EntityDeclarationImpl");
        PUBLIC_IDENTIFIERS.add("EntityDV");
        PUBLIC_IDENTIFIERS.add("EntityImpl");
        PUBLIC_IDENTIFIERS.add("EntityReference");
        PUBLIC_IDENTIFIERS.add("EntityReferenceEvent");
        PUBLIC_IDENTIFIERS.add("EntityReferenceImpl");
        PUBLIC_IDENTIFIERS.add("EntityResolver");
        PUBLIC_IDENTIFIERS.add("EntityResolver2");
        PUBLIC_IDENTIFIERS.add("EntityResolver2Wrapper");
        PUBLIC_IDENTIFIERS.add("EntityResolverWrapper");
        PUBLIC_IDENTIFIERS.add("EntityState");
        PUBLIC_IDENTIFIERS.add("EntityTree");
        PUBLIC_IDENTIFIERS.add("EntropySource");
        PUBLIC_IDENTIFIERS.add("Entry");
        PUBLIC_IDENTIFIERS.add("EntryImpl");
        PUBLIC_IDENTIFIERS.add("EntryNotFound");
        PUBLIC_IDENTIFIERS.add("Enum");
        PUBLIC_IDENTIFIERS.add("Enum.EnumDesc");
        PUBLIC_IDENTIFIERS.add("Enum_element_value");
        PUBLIC_IDENTIFIERS.add("EnumCompleter");
        PUBLIC_IDENTIFIERS.add("EnumConstantBuilder");
        PUBLIC_IDENTIFIERS.add("EnumConstantNotPresentException");
        PUBLIC_IDENTIFIERS.add("EnumConstantNotPresentExceptionProxy");
        PUBLIC_IDENTIFIERS.add("EnumConstantWriter");
        PUBLIC_IDENTIFIERS.add("EnumConstantWriterImpl");
        PUBLIC_IDENTIFIERS.add("EnumControl");
        PUBLIC_IDENTIFIERS.add("EnumConverter");
        PUBLIC_IDENTIFIERS.add("EnumElementValue");
        PUBLIC_IDENTIFIERS.add("EnumElementValueGen");
        PUBLIC_IDENTIFIERS.add("Enumeration");
        PUBLIC_IDENTIFIERS.add("EnumKey");
        PUBLIC_IDENTIFIERS.add("EnumMap");
        PUBLIC_IDENTIFIERS.add("EnumSet");
        PUBLIC_IDENTIFIERS.add("EnumSyntax");
        PUBLIC_IDENTIFIERS.add("EnumType");
        PUBLIC_IDENTIFIERS.add("Env");
        PUBLIC_IDENTIFIERS.add("EnvHelp");
        PUBLIC_IDENTIFIERS.add("EnvUtils");
        PUBLIC_IDENTIFIERS.add("EOFError");
        PUBLIC_IDENTIFIERS.add("EOFException");
        PUBLIC_IDENTIFIERS.add("EOWindOp");
        PUBLIC_IDENTIFIERS.add("EPollSelectorProvider");
        PUBLIC_IDENTIFIERS.add("EpsilonHeap");
        PUBLIC_IDENTIFIERS.add("EPSPrinter");
        PUBLIC_IDENTIFIERS.add("EqualByteArray");
        PUBLIC_IDENTIFIERS.add("Equals");
        PUBLIC_IDENTIFIERS.add("Era");
        PUBLIC_IDENTIFIERS.add("ErasedClassType");
        PUBLIC_IDENTIFIERS.add("ErroneousSnippet");
        PUBLIC_IDENTIFIERS.add("ErroneousTree");
        PUBLIC_IDENTIFIERS.add("Error");
        PUBLIC_IDENTIFIERS.add("ErrorFrame");
        PUBLIC_IDENTIFIERS.add("ErrorHandler");
        PUBLIC_IDENTIFIERS.add("ErrorHandlerAdaptor");
        PUBLIC_IDENTIFIERS.add("ErrorHandlerProxy");
        PUBLIC_IDENTIFIERS.add("ErrorHandlerWrapper");
        PUBLIC_IDENTIFIERS.add("ErrorListener");
        PUBLIC_IDENTIFIERS.add("ErrorManager");
        PUBLIC_IDENTIFIERS.add("ErrorMessages");
        PUBLIC_IDENTIFIERS.add("ErrorMessages_de");
        PUBLIC_IDENTIFIERS.add("ErrorMessages_ja");
        PUBLIC_IDENTIFIERS.add("ErrorMessages_zh_CN");
        PUBLIC_IDENTIFIERS.add("Errors");
        PUBLIC_IDENTIFIERS.add("ErrorScope");
        PUBLIC_IDENTIFIERS.add("ErrorType");
        PUBLIC_IDENTIFIERS.add("EtchedBorder");
        PUBLIC_IDENTIFIERS.add("EtchedBorderUIResource");
        PUBLIC_IDENTIFIERS.add("EType");
        PUBLIC_IDENTIFIERS.add("ETypeInfo");
        PUBLIC_IDENTIFIERS.add("ETypeInfo2");
        PUBLIC_IDENTIFIERS.add("EUC_CN");
        PUBLIC_IDENTIFIERS.add("EUC_JP");
        PUBLIC_IDENTIFIERS.add("EUC_JP_LINUX");
        PUBLIC_IDENTIFIERS.add("EUC_JP_Open");
        PUBLIC_IDENTIFIERS.add("EUC_KR");
        PUBLIC_IDENTIFIERS.add("EUC_TW");
        PUBLIC_IDENTIFIERS.add("EvalException");
        PUBLIC_IDENTIFIERS.add("Event");
        PUBLIC_IDENTIFIERS.add("EventByteStream");
        PUBLIC_IDENTIFIERS.add("EventContext");
        PUBLIC_IDENTIFIERS.add("EventDirContext");
        PUBLIC_IDENTIFIERS.add("EventDispatcher");
        PUBLIC_IDENTIFIERS.add("EventException");
        PUBLIC_IDENTIFIERS.add("EventFilter");
        PUBLIC_IDENTIFIERS.add("EventFilterSupport");
        PUBLIC_IDENTIFIERS.add("EventHandler");
        PUBLIC_IDENTIFIERS.add("EventID");
        PUBLIC_IDENTIFIERS.add("EventImpl");
        PUBLIC_IDENTIFIERS.add("EventIterator");
        PUBLIC_IDENTIFIERS.add("EventListener");
        PUBLIC_IDENTIFIERS.add("EventListenerList");
        PUBLIC_IDENTIFIERS.add("EventListenerProxy");
        PUBLIC_IDENTIFIERS.add("EventObject");
        PUBLIC_IDENTIFIERS.add("EventProvider");
        PUBLIC_IDENTIFIERS.add("EventQueue");
        PUBLIC_IDENTIFIERS.add("EventQueueAccessor");
        PUBLIC_IDENTIFIERS.add("EventQueueImpl");
        PUBLIC_IDENTIFIERS.add("EventQueueItem");
        PUBLIC_IDENTIFIERS.add("EventQueueMonitor");
        PUBLIC_IDENTIFIERS.add("EventReaderDelegate");
        PUBLIC_IDENTIFIERS.add("EventRequest");
        PUBLIC_IDENTIFIERS.add("EventRequestManager");
        PUBLIC_IDENTIFIERS.add("EventSet");
        PUBLIC_IDENTIFIERS.add("EventSetDescriptor");
        PUBLIC_IDENTIFIERS.add("EventSetImpl");
        PUBLIC_IDENTIFIERS.add("EventSettings");
        PUBLIC_IDENTIFIERS.add("EventSettingsModifier");
        PUBLIC_IDENTIFIERS.add("EventStream");
        PUBLIC_IDENTIFIERS.add("EventTarget");
        PUBLIC_IDENTIFIERS.add("Example");
        PUBLIC_IDENTIFIERS.add("Exception");
        PUBLIC_IDENTIFIERS.add("Exception_data");
        PUBLIC_IDENTIFIERS.add("ExceptionalSupplier");
        PUBLIC_IDENTIFIERS.add("ExceptionBlob");
        PUBLIC_IDENTIFIERS.add("ExceptionEvent");
        PUBLIC_IDENTIFIERS.add("ExceptionInInitializerError");
        PUBLIC_IDENTIFIERS.add("ExceptionListener");
        PUBLIC_IDENTIFIERS.add("ExceptionProxy");
        PUBLIC_IDENTIFIERS.add("ExceptionRequest");
        PUBLIC_IDENTIFIERS.add("Exceptions_attribute");
        PUBLIC_IDENTIFIERS.add("ExceptionTableElement");
        PUBLIC_IDENTIFIERS.add("ExceptionThrower");
        PUBLIC_IDENTIFIERS.add("Exchanger");
        PUBLIC_IDENTIFIERS.add("ExecPty");
        PUBLIC_IDENTIFIERS.add("Executable");
        PUBLIC_IDENTIFIERS.add("ExecutableElement");
        PUBLIC_IDENTIFIERS.add("ExecutableImage");
        PUBLIC_IDENTIFIERS.add("ExecutableType");
        PUBLIC_IDENTIFIERS.add("ExecutionControl");
        PUBLIC_IDENTIFIERS.add("ExecutionControlException");
        PUBLIC_IDENTIFIERS.add("ExecutionControlProvider");
        PUBLIC_IDENTIFIERS.add("ExecutionEnv");
        PUBLIC_IDENTIFIERS.add("ExecutionException");
        PUBLIC_IDENTIFIERS.add("Executor");
        PUBLIC_IDENTIFIERS.add("ExecutorCompletionService");
        PUBLIC_IDENTIFIERS.add("Executors");
        PUBLIC_IDENTIFIERS.add("ExecutorService");
        PUBLIC_IDENTIFIERS.add("ExemptionMechanism");
        PUBLIC_IDENTIFIERS.add("ExemptionMechanismException");
        PUBLIC_IDENTIFIERS.add("ExemptionMechanismSpi");
        PUBLIC_IDENTIFIERS.add("ExitAction");
        PUBLIC_IDENTIFIERS.add("ExpandedNameTable");
        PUBLIC_IDENTIFIERS.add("Expander");
        PUBLIC_IDENTIFIERS.add("ExpandException");
        PUBLIC_IDENTIFIERS.add("ExpandVetoException");
        PUBLIC_IDENTIFIERS.add("Experimental");
        PUBLIC_IDENTIFIERS.add("ExportDirectoryTable");
        PUBLIC_IDENTIFIERS.add("ExportException");
        PUBLIC_IDENTIFIERS.add("ExportsDirective");
        PUBLIC_IDENTIFIERS.add("ExportsEntry");
        PUBLIC_IDENTIFIERS.add("ExportsTree");
        PUBLIC_IDENTIFIERS.add("Expression");
        PUBLIC_IDENTIFIERS.add("ExpressionContext");
        PUBLIC_IDENTIFIERS.add("ExpressionExecuter");
        PUBLIC_IDENTIFIERS.add("ExpressionInfo");
        PUBLIC_IDENTIFIERS.add("ExpressionNode");
        PUBLIC_IDENTIFIERS.add("ExpressionOwner");
        PUBLIC_IDENTIFIERS.add("ExpressionParser");
        PUBLIC_IDENTIFIERS.add("ExpressionParserConstants");
        PUBLIC_IDENTIFIERS.add("ExpressionParserTokenManager");
        PUBLIC_IDENTIFIERS.add("ExpressionResolver");
        PUBLIC_IDENTIFIERS.add("ExpressionSnippet");
        PUBLIC_IDENTIFIERS.add("ExpressionStatementTree");
        PUBLIC_IDENTIFIERS.add("ExpressionTree");
        PUBLIC_IDENTIFIERS.add("ExsltBase");
        PUBLIC_IDENTIFIERS.add("ExsltCommon");
        PUBLIC_IDENTIFIERS.add("ExsltDatetime");
        PUBLIC_IDENTIFIERS.add("ExsltDynamic");
        PUBLIC_IDENTIFIERS.add("ExsltMath");
        PUBLIC_IDENTIFIERS.add("ExsltSets");
        PUBLIC_IDENTIFIERS.add("ExsltStrings");
        PUBLIC_IDENTIFIERS.add("ExtendedBufferCapabilities");
        PUBLIC_IDENTIFIERS.add("ExtendedCharsets");
        PUBLIC_IDENTIFIERS.add("ExtendedGSSContext");
        PUBLIC_IDENTIFIERS.add("ExtendedGSSCredential");
        PUBLIC_IDENTIFIERS.add("ExtendedHomogeneousPoint");
        PUBLIC_IDENTIFIERS.add("ExtendedKeyCodes");
        PUBLIC_IDENTIFIERS.add("ExtendedKeyUsageExtension");
        PUBLIC_IDENTIFIERS.add("ExtendedMapMode");
        PUBLIC_IDENTIFIERS.add("ExtendedRequest");
        PUBLIC_IDENTIFIERS.add("ExtendedResponse");
        PUBLIC_IDENTIFIERS.add("ExtendedSAX");
        PUBLIC_IDENTIFIERS.add("ExtendedSchemaDVFactoryImpl");
        PUBLIC_IDENTIFIERS.add("ExtendedSocketOptions");
        PUBLIC_IDENTIFIERS.add("ExtendedSSLSession");
        PUBLIC_IDENTIFIERS.add("ExtendedTextLabel");
        PUBLIC_IDENTIFIERS.add("Extension");
        PUBLIC_IDENTIFIERS.add("Extensions");
        PUBLIC_IDENTIFIERS.add("ExtensionsProvider");
        PUBLIC_IDENTIFIERS.add("Extern");
        PUBLIC_IDENTIFIERS.add("ExternalEditor");
        PUBLIC_IDENTIFIERS.add("ExternalEntity");
        PUBLIC_IDENTIFIERS.add("Externalizable");
        PUBLIC_IDENTIFIERS.add("ExternalSpecsWriter");
        PUBLIC_IDENTIFIERS.add("ExternalSubsetResolver");
        PUBLIC_IDENTIFIERS.add("ExternalTerminal");
        PUBLIC_IDENTIFIERS.add("ExternalVFrame");
    }

    private static void addPublicIdentifiersFH() {
        PUBLIC_IDENTIFIERS.add("F2D");
        PUBLIC_IDENTIFIERS.add("F2I");
        PUBLIC_IDENTIFIERS.add("F2L");
        PUBLIC_IDENTIFIERS.add("Factory");
        PUBLIC_IDENTIFIERS.add("FactoryConfigurationError");
        PUBLIC_IDENTIFIERS.add("FADD");
        PUBLIC_IDENTIFIERS.add("FailedLoginException");
        PUBLIC_IDENTIFIERS.add("FailOverExecutionControlProvider");
        PUBLIC_IDENTIFIERS.add("Failure");
        PUBLIC_IDENTIFIERS.add("FallbackLocaleProviderAdapter");
        PUBLIC_IDENTIFIERS.add("FALOAD");
        PUBLIC_IDENTIFIERS.add("FamilyDescription");
        PUBLIC_IDENTIFIERS.add("Fast");
        PUBLIC_IDENTIFIERS.add("FASTORE");
        PUBLIC_IDENTIFIERS.add("FastStringBuffer");
        PUBLIC_IDENTIFIERS.add("FatalError");
        PUBLIC_IDENTIFIERS.add("Fault");
        PUBLIC_IDENTIFIERS.add("FcCompFont");
        PUBLIC_IDENTIFIERS.add("FcFontConfiguration");
        PUBLIC_IDENTIFIERS.add("FcFontManager");
        PUBLIC_IDENTIFIERS.add("FCMPG");
        PUBLIC_IDENTIFIERS.add("FCMPL");
        PUBLIC_IDENTIFIERS.add("FCONST");
        PUBLIC_IDENTIFIERS.add("FDBigInteger");
        PUBLIC_IDENTIFIERS.add("FDIV");
        PUBLIC_IDENTIFIERS.add("FeatureDescriptor");
        PUBLIC_IDENTIFIERS.add("FeaturePropertyBase");
        PUBLIC_IDENTIFIERS.add("FeatureState");
        PUBLIC_IDENTIFIERS.add("Field");
        PUBLIC_IDENTIFIERS.add("FieldAccessor");
        PUBLIC_IDENTIFIERS.add("FieldActivator");
        PUBLIC_IDENTIFIERS.add("FieldBorder");
        PUBLIC_IDENTIFIERS.add("FieldBuilder");
        PUBLIC_IDENTIFIERS.add("FieldGen");
        PUBLIC_IDENTIFIERS.add("FieldGenOrMethodGen");
        PUBLIC_IDENTIFIERS.add("FieldIdentifier");
        PUBLIC_IDENTIFIERS.add("FieldImpl");
        PUBLIC_IDENTIFIERS.add("FieldInsnNode");
        PUBLIC_IDENTIFIERS.add("FieldInstruction");
        PUBLIC_IDENTIFIERS.add("FieldNode");
        PUBLIC_IDENTIFIERS.add("FieldObserver");
        PUBLIC_IDENTIFIERS.add("FieldOrMethod");
        PUBLIC_IDENTIFIERS.add("FieldPosition");
        PUBLIC_IDENTIFIERS.add("FieldRemapper");
        PUBLIC_IDENTIFIERS.add("FieldRepository");
        PUBLIC_IDENTIFIERS.add("FieldSetter");
        PUBLIC_IDENTIFIERS.add("FieldTreeNodeAdapter");
        PUBLIC_IDENTIFIERS.add("FieldType");
        PUBLIC_IDENTIFIERS.add("FieldTypeSignature");
        PUBLIC_IDENTIFIERS.add("FieldView");
        PUBLIC_IDENTIFIERS.add("FieldVisitor");
        PUBLIC_IDENTIFIERS.add("FieldWriter");
        PUBLIC_IDENTIFIERS.add("FieldWriterImpl");
        PUBLIC_IDENTIFIERS.add("File");
        PUBLIC_IDENTIFIERS.add("FileAccess");
        PUBLIC_IDENTIFIERS.add("FileAlreadyExistsException");
        PUBLIC_IDENTIFIERS.add("FileAttribute");
        PUBLIC_IDENTIFIERS.add("FileAttributeView");
        PUBLIC_IDENTIFIERS.add("FileCacheImageInputStream");
        PUBLIC_IDENTIFIERS.add("FileCacheImageOutputStream");
        PUBLIC_IDENTIFIERS.add("FileCCacheConstants");
        PUBLIC_IDENTIFIERS.add("FileCellRenderer");
        PUBLIC_IDENTIFIERS.add("FileChannel");
        PUBLIC_IDENTIFIERS.add("FileChannelImpl");
        PUBLIC_IDENTIFIERS.add("FileChooserUI");
        PUBLIC_IDENTIFIERS.add("FileChooserUIAccessor");
        PUBLIC_IDENTIFIERS.add("FileCredentialsCache");
        PUBLIC_IDENTIFIERS.add("FileDialog");
        PUBLIC_IDENTIFIERS.add("FileDialogAccessor");
        PUBLIC_IDENTIFIERS.add("FileDialogPeer");
        PUBLIC_IDENTIFIERS.add("FileDownloadBodyHandler");
        PUBLIC_IDENTIFIERS.add("FileFilter");
        PUBLIC_IDENTIFIERS.add("FileFont");
        PUBLIC_IDENTIFIERS.add("FileFontStrike");
        PUBLIC_IDENTIFIERS.add("FileHandler");
        PUBLIC_IDENTIFIERS.add("FileIcon16");
        PUBLIC_IDENTIFIERS.add("FileImageInputStream");
        PUBLIC_IDENTIFIERS.add("FileImageInputStreamSpi");
        PUBLIC_IDENTIFIERS.add("FileImageOutputStream");
        PUBLIC_IDENTIFIERS.add("FileImageOutputStreamSpi");
        PUBLIC_IDENTIFIERS.add("FileImageSource");
        PUBLIC_IDENTIFIERS.add("FileInputStream");
        PUBLIC_IDENTIFIERS.add("FileKey");
        PUBLIC_IDENTIFIERS.add("FileLock");
        PUBLIC_IDENTIFIERS.add("FileLockImpl");
        PUBLIC_IDENTIFIERS.add("FileLockInterruptionException");
        PUBLIC_IDENTIFIERS.add("FileLoginModule");
        PUBLIC_IDENTIFIERS.add("FileMapHeader");
        PUBLIC_IDENTIFIERS.add("FileMapInfo");
        PUBLIC_IDENTIFIERS.add("FileMenu");
        PUBLIC_IDENTIFIERS.add("FileMonitoredVm");
        PUBLIC_IDENTIFIERS.add("FileNameCompleter");
        PUBLIC_IDENTIFIERS.add("FilenameFilter");
        PUBLIC_IDENTIFIERS.add("FileNameMap");
        PUBLIC_IDENTIFIERS.add("FileNotFoundException");
        PUBLIC_IDENTIFIERS.add("FileObject");
        PUBLIC_IDENTIFIERS.add("FileOutputStream");
        PUBLIC_IDENTIFIERS.add("FileOwnerAttributeView");
        PUBLIC_IDENTIFIERS.add("FilePane");
        PUBLIC_IDENTIFIERS.add("FilePaths");
        PUBLIC_IDENTIFIERS.add("FilePermCompat");
        PUBLIC_IDENTIFIERS.add("FilePublisher");
        PUBLIC_IDENTIFIERS.add("Filer");
        PUBLIC_IDENTIFIERS.add("FileReader");
        PUBLIC_IDENTIFIERS.add("FileRenderer");
        PUBLIC_IDENTIFIERS.add("FilerException");
        PUBLIC_IDENTIFIERS.add("FilesEvent");
        PUBLIC_IDENTIFIERS.add("FileStore");
        PUBLIC_IDENTIFIERS.add("FileStoreAttributeView");
        PUBLIC_IDENTIFIERS.add("FileSystem");
        PUBLIC_IDENTIFIERS.add("FileSystemAlreadyExistsException");
        PUBLIC_IDENTIFIERS.add("FileSystemException");
        PUBLIC_IDENTIFIERS.add("FileSystemImpl");
        PUBLIC_IDENTIFIERS.add("FileSystemLoopException");
        PUBLIC_IDENTIFIERS.add("FileSystemNotFoundException");
        PUBLIC_IDENTIFIERS.add("FileSystemProvider");
        PUBLIC_IDENTIFIERS.add("FileSystemView");
        PUBLIC_IDENTIFIERS.add("FileTypeDetector");
        PUBLIC_IDENTIFIERS.add("FileURLConnection");
        PUBLIC_IDENTIFIERS.add("FileURLMapper");
        PUBLIC_IDENTIFIERS.add("FileView");
        PUBLIC_IDENTIFIERS.add("FileVisitor");
        PUBLIC_IDENTIFIERS.add("FileWriter");
        PUBLIC_IDENTIFIERS.add("Filler");
        PUBLIC_IDENTIFIERS.add("FillParallelogram");
        PUBLIC_IDENTIFIERS.add("FillPath");
        PUBLIC_IDENTIFIERS.add("FillRect");
        PUBLIC_IDENTIFIERS.add("FillSpans");
        PUBLIC_IDENTIFIERS.add("Filter");
        PUBLIC_IDENTIFIERS.add("FilterBypass");
        PUBLIC_IDENTIFIERS.add("FilterComboBoxModel");
        PUBLIC_IDENTIFIERS.add("FilterComboBoxRenderer");
        PUBLIC_IDENTIFIERS.add("FilteredImageSource");
        PUBLIC_IDENTIFIERS.add("FilteredMemberList");
        PUBLIC_IDENTIFIERS.add("FilteredRowSet");
        PUBLIC_IDENTIFIERS.add("FilteredRowSetImpl");
        PUBLIC_IDENTIFIERS.add("FilterExprIterator");
        PUBLIC_IDENTIFIERS.add("FilterExprIteratorSimple");
        PUBLIC_IDENTIFIERS.add("FilterExprWalker");
        PUBLIC_IDENTIFIERS.add("FilterInfo");
        PUBLIC_IDENTIFIERS.add("FilterInputStream");
        PUBLIC_IDENTIFIERS.add("FilterOutputStream");
        PUBLIC_IDENTIFIERS.add("FilterReader");
        PUBLIC_IDENTIFIERS.add("Filters");
        PUBLIC_IDENTIFIERS.add("FilterWriter");
        PUBLIC_IDENTIFIERS.add("FinalizerInfo");
        PUBLIC_IDENTIFIERS.add("FindAction");
        PUBLIC_IDENTIFIERS.add("FindClassesAction");
        PUBLIC_IDENTIFIERS.add("FindCrashesAction");
        PUBLIC_IDENTIFIERS.add("Finder");
        PUBLIC_IDENTIFIERS.add("FindException");
        PUBLIC_IDENTIFIERS.add("FindInCodeCachePanel");
        PUBLIC_IDENTIFIERS.add("FindInHeapPanel");
        PUBLIC_IDENTIFIERS.add("FindObjectByType");
        PUBLIC_IDENTIFIERS.add("FindPanel");
        PUBLIC_IDENTIFIERS.add("FinishAction");
        PUBLIC_IDENTIFIERS.add("Finishings");
        PUBLIC_IDENTIFIERS.add("Fix");
        PUBLIC_IDENTIFIERS.add("FixComputer");
        PUBLIC_IDENTIFIERS.add("FixedHeightLayoutCache");
        PUBLIC_IDENTIFIERS.add("FixResult");
        PUBLIC_IDENTIFIERS.add("Flag");
        PUBLIC_IDENTIFIERS.add("FlagDumper");
        PUBLIC_IDENTIFIERS.add("Flags");
        PUBLIC_IDENTIFIERS.add("FlatteningPathIterator");
        PUBLIC_IDENTIFIERS.add("FlavorEvent");
        PUBLIC_IDENTIFIERS.add("FlavorException");
        PUBLIC_IDENTIFIERS.add("FlavorListener");
        PUBLIC_IDENTIFIERS.add("FlavorMap");
        PUBLIC_IDENTIFIERS.add("FlavorTable");
        PUBLIC_IDENTIFIERS.add("FlightRecorderListener");
        PUBLIC_IDENTIFIERS.add("FlightRecorderMXBean");
        PUBLIC_IDENTIFIERS.add("FlipBufferStrategy");
        PUBLIC_IDENTIFIERS.add("FLOAD");
        PUBLIC_IDENTIFIERS.add("Float");
        PUBLIC_IDENTIFIERS.add("FloatBuffer");
        PUBLIC_IDENTIFIERS.add("FloatConsts");
        PUBLIC_IDENTIFIERS.add("FloatControl");
        PUBLIC_IDENTIFIERS.add("FloatData");
        PUBLIC_IDENTIFIERS.add("FloatDV");
        PUBLIC_IDENTIFIERS.add("FloatEditor");
        PUBLIC_IDENTIFIERS.add("FloatField");
        PUBLIC_IDENTIFIERS.add("FloatingDecimal");
        PUBLIC_IDENTIFIERS.add("FloatSignature");
        PUBLIC_IDENTIFIERS.add("FloatTreeNodeAdapter");
        PUBLIC_IDENTIFIERS.add("FloatType");
        PUBLIC_IDENTIFIERS.add("FloatTypeImpl");
        PUBLIC_IDENTIFIERS.add("FloatValue");
        PUBLIC_IDENTIFIERS.add("FloatValueImpl");
        PUBLIC_IDENTIFIERS.add("FloatVector");
        PUBLIC_IDENTIFIERS.add("Flow");
        PUBLIC_IDENTIFIERS.add("FlowLayout");
        PUBLIC_IDENTIFIERS.add("FlowStrategy");
        PUBLIC_IDENTIFIERS.add("FlowTube");
        PUBLIC_IDENTIFIERS.add("FlowView");
        PUBLIC_IDENTIFIERS.add("Flush3DBorder");
        PUBLIC_IDENTIFIERS.add("Flushable");
        PUBLIC_IDENTIFIERS.add("FlushableCacheData");
        PUBLIC_IDENTIFIERS.add("FMUL");
        PUBLIC_IDENTIFIERS.add("FNEG");
        PUBLIC_IDENTIFIERS.add("FocusAdapter");
        PUBLIC_IDENTIFIERS.add("FocusBorder");
        PUBLIC_IDENTIFIERS.add("FocusEvent");
        PUBLIC_IDENTIFIERS.add("FocusHandler");
        PUBLIC_IDENTIFIERS.add("FocusListener");
        PUBLIC_IDENTIFIERS.add("FocusManager");
        PUBLIC_IDENTIFIERS.add("FocusTraversalPolicy");
        PUBLIC_IDENTIFIERS.add("FolderIcon16");
        PUBLIC_IDENTIFIERS.add("FollowingIterator");
        PUBLIC_IDENTIFIERS.add("FollowingSiblingIterator");
        PUBLIC_IDENTIFIERS.add("Font");
        PUBLIC_IDENTIFIERS.add("Font2D");
        PUBLIC_IDENTIFIERS.add("FontAccess");
        PUBLIC_IDENTIFIERS.add("FontAttribute");
        PUBLIC_IDENTIFIERS.add("FontConfigFont");
        PUBLIC_IDENTIFIERS.add("FontConfigInfo");
        PUBLIC_IDENTIFIERS.add("FontConfigManager");
        PUBLIC_IDENTIFIERS.add("FontConfiguration");
        PUBLIC_IDENTIFIERS.add("FontDescriptor");
        PUBLIC_IDENTIFIERS.add("FontEditor");
        PUBLIC_IDENTIFIERS.add("FontFamily");
        PUBLIC_IDENTIFIERS.add("FontFamilyAction");
        PUBLIC_IDENTIFIERS.add("FontFormatException");
        PUBLIC_IDENTIFIERS.add("FontInfo");
        PUBLIC_IDENTIFIERS.add("FontManager");
        PUBLIC_IDENTIFIERS.add("FontManagerForSGE");
        PUBLIC_IDENTIFIERS.add("FontManagerNativeLibrary");
        PUBLIC_IDENTIFIERS.add("FontMetrics");
        PUBLIC_IDENTIFIERS.add("FontPeer");
        PUBLIC_IDENTIFIERS.add("FontRenderContext");
        PUBLIC_IDENTIFIERS.add("FontScaler");
        PUBLIC_IDENTIFIERS.add("FontScalerException");
        PUBLIC_IDENTIFIERS.add("FontSizeAction");
        PUBLIC_IDENTIFIERS.add("FontStrike");
        PUBLIC_IDENTIFIERS.add("FontStrikeDesc");
        PUBLIC_IDENTIFIERS.add("FontSubstitution");
        PUBLIC_IDENTIFIERS.add("FontSupport");
        PUBLIC_IDENTIFIERS.add("FontUIResource");
        PUBLIC_IDENTIFIERS.add("ForAll");
        PUBLIC_IDENTIFIERS.add("ForceInline");
        PUBLIC_IDENTIFIERS.add("ForegroundAction");
        PUBLIC_IDENTIFIERS.add("ForkJoinPool");
        PUBLIC_IDENTIFIERS.add("ForkJoinTask");
        PUBLIC_IDENTIFIERS.add("ForkJoinWorkerThread");
        PUBLIC_IDENTIFIERS.add("ForkJoinWorkerThreadFactory");
        PUBLIC_IDENTIFIERS.add("ForLoopTree");
        PUBLIC_IDENTIFIERS.add("FormAction");
        PUBLIC_IDENTIFIERS.add("FormalTypeParameter");
        PUBLIC_IDENTIFIERS.add("Format");
        PUBLIC_IDENTIFIERS.add("FormatConversionProvider");
        PUBLIC_IDENTIFIERS.add("FormatData");
        PUBLIC_IDENTIFIERS.add("FormatData_af");
        PUBLIC_IDENTIFIERS.add("FormatData_af_NA");
        PUBLIC_IDENTIFIERS.add("FormatData_agq");
        PUBLIC_IDENTIFIERS.add("FormatData_ak");
        PUBLIC_IDENTIFIERS.add("FormatData_am");
        PUBLIC_IDENTIFIERS.add("FormatData_ar");
        PUBLIC_IDENTIFIERS.add("FormatData_ar_AE");
        PUBLIC_IDENTIFIERS.add("FormatData_ar_DZ");
        PUBLIC_IDENTIFIERS.add("FormatData_ar_EH");
        PUBLIC_IDENTIFIERS.add("FormatData_ar_IL");
        PUBLIC_IDENTIFIERS.add("FormatData_ar_IQ");
        PUBLIC_IDENTIFIERS.add("FormatData_ar_JO");
        PUBLIC_IDENTIFIERS.add("FormatData_ar_KM");
        PUBLIC_IDENTIFIERS.add("FormatData_ar_LB");
        PUBLIC_IDENTIFIERS.add("FormatData_ar_LY");
        PUBLIC_IDENTIFIERS.add("FormatData_ar_MA");
        PUBLIC_IDENTIFIERS.add("FormatData_ar_MR");
        PUBLIC_IDENTIFIERS.add("FormatData_ar_PS");
        PUBLIC_IDENTIFIERS.add("FormatData_ar_SA");
        PUBLIC_IDENTIFIERS.add("FormatData_ar_SO");
        PUBLIC_IDENTIFIERS.add("FormatData_ar_SY");
        PUBLIC_IDENTIFIERS.add("FormatData_ar_TN");
        PUBLIC_IDENTIFIERS.add("FormatData_as");
        PUBLIC_IDENTIFIERS.add("FormatData_asa");
        PUBLIC_IDENTIFIERS.add("FormatData_ast");
        PUBLIC_IDENTIFIERS.add("FormatData_az");
        PUBLIC_IDENTIFIERS.add("FormatData_az_Cyrl");
        PUBLIC_IDENTIFIERS.add("FormatData_bas");
        PUBLIC_IDENTIFIERS.add("FormatData_be");
        PUBLIC_IDENTIFIERS.add("FormatData_be_BY");
        PUBLIC_IDENTIFIERS.add("FormatData_be_TARASK");
        PUBLIC_IDENTIFIERS.add("FormatData_bem");
        PUBLIC_IDENTIFIERS.add("FormatData_bez");
        PUBLIC_IDENTIFIERS.add("FormatData_bg");
        PUBLIC_IDENTIFIERS.add("FormatData_bg_BG");
        PUBLIC_IDENTIFIERS.add("FormatData_bgc");
        PUBLIC_IDENTIFIERS.add("FormatData_bho");
        PUBLIC_IDENTIFIERS.add("FormatData_bm");
        PUBLIC_IDENTIFIERS.add("FormatData_bn");
        PUBLIC_IDENTIFIERS.add("FormatData_bn_IN");
        PUBLIC_IDENTIFIERS.add("FormatData_bo");
        PUBLIC_IDENTIFIERS.add("FormatData_bo_IN");
        PUBLIC_IDENTIFIERS.add("FormatData_br");
        PUBLIC_IDENTIFIERS.add("FormatData_brx");
        PUBLIC_IDENTIFIERS.add("FormatData_bs");
        PUBLIC_IDENTIFIERS.add("FormatData_bs_Cyrl");
        PUBLIC_IDENTIFIERS.add("FormatData_ca");
        PUBLIC_IDENTIFIERS.add("FormatData_ca_ES");
        PUBLIC_IDENTIFIERS.add("FormatData_ccp");
        PUBLIC_IDENTIFIERS.add("FormatData_ce");
        PUBLIC_IDENTIFIERS.add("FormatData_ceb");
        PUBLIC_IDENTIFIERS.add("FormatData_cgg");
        PUBLIC_IDENTIFIERS.add("FormatData_chr");
        PUBLIC_IDENTIFIERS.add("FormatData_ckb");
        PUBLIC_IDENTIFIERS.add("FormatData_ckb_IR");
        PUBLIC_IDENTIFIERS.add("FormatData_cs");
        PUBLIC_IDENTIFIERS.add("FormatData_cs_CZ");
        PUBLIC_IDENTIFIERS.add("FormatData_cv");
        PUBLIC_IDENTIFIERS.add("FormatData_cy");
        PUBLIC_IDENTIFIERS.add("FormatData_da");
        PUBLIC_IDENTIFIERS.add("FormatData_da_DK");
        PUBLIC_IDENTIFIERS.add("FormatData_dav");
        PUBLIC_IDENTIFIERS.add("FormatData_de");
        PUBLIC_IDENTIFIERS.add("FormatData_de_AT");
        PUBLIC_IDENTIFIERS.add("FormatData_de_CH");
        PUBLIC_IDENTIFIERS.add("FormatData_de_DE");
        PUBLIC_IDENTIFIERS.add("FormatData_de_IT");
        PUBLIC_IDENTIFIERS.add("FormatData_de_LI");
        PUBLIC_IDENTIFIERS.add("FormatData_de_LU");
        PUBLIC_IDENTIFIERS.add("FormatData_dje");
        PUBLIC_IDENTIFIERS.add("FormatData_doi");
        PUBLIC_IDENTIFIERS.add("FormatData_dsb");
        PUBLIC_IDENTIFIERS.add("FormatData_dua");
        PUBLIC_IDENTIFIERS.add("FormatData_dyo");
        PUBLIC_IDENTIFIERS.add("FormatData_dz");
        PUBLIC_IDENTIFIERS.add("FormatData_ebu");
        PUBLIC_IDENTIFIERS.add("FormatData_ee");
        PUBLIC_IDENTIFIERS.add("FormatData_ee_TG");
        PUBLIC_IDENTIFIERS.add("FormatData_el");
        PUBLIC_IDENTIFIERS.add("FormatData_el_CY");
        PUBLIC_IDENTIFIERS.add("FormatData_el_GR");
        PUBLIC_IDENTIFIERS.add("FormatData_en");
        PUBLIC_IDENTIFIERS.add("FormatData_en_001");
        PUBLIC_IDENTIFIERS.add("FormatData_en_150");
        PUBLIC_IDENTIFIERS.add("FormatData_en_AE");
        PUBLIC_IDENTIFIERS.add("FormatData_en_AI");
        PUBLIC_IDENTIFIERS.add("FormatData_en_AT");
        PUBLIC_IDENTIFIERS.add("FormatData_en_AU");
        PUBLIC_IDENTIFIERS.add("FormatData_en_BE");
        PUBLIC_IDENTIFIERS.add("FormatData_en_BI");
        PUBLIC_IDENTIFIERS.add("FormatData_en_BW");
        PUBLIC_IDENTIFIERS.add("FormatData_en_BZ");
        PUBLIC_IDENTIFIERS.add("FormatData_en_CA");
        PUBLIC_IDENTIFIERS.add("FormatData_en_CC");
        PUBLIC_IDENTIFIERS.add("FormatData_en_CH");
        PUBLIC_IDENTIFIERS.add("FormatData_en_CK");
        PUBLIC_IDENTIFIERS.add("FormatData_en_CM");
        PUBLIC_IDENTIFIERS.add("FormatData_en_CX");
        PUBLIC_IDENTIFIERS.add("FormatData_en_DE");
        PUBLIC_IDENTIFIERS.add("FormatData_en_DG");
        PUBLIC_IDENTIFIERS.add("FormatData_en_DK");
        PUBLIC_IDENTIFIERS.add("FormatData_en_FI");
        PUBLIC_IDENTIFIERS.add("FormatData_en_FK");
        PUBLIC_IDENTIFIERS.add("FormatData_en_GB");
        PUBLIC_IDENTIFIERS.add("FormatData_en_GG");
        PUBLIC_IDENTIFIERS.add("FormatData_en_GI");
        PUBLIC_IDENTIFIERS.add("FormatData_en_HK");
        PUBLIC_IDENTIFIERS.add("FormatData_en_IE");
        PUBLIC_IDENTIFIERS.add("FormatData_en_IL");
        PUBLIC_IDENTIFIERS.add("FormatData_en_IM");
        PUBLIC_IDENTIFIERS.add("FormatData_en_IN");
        PUBLIC_IDENTIFIERS.add("FormatData_en_IO");
        PUBLIC_IDENTIFIERS.add("FormatData_en_JE");
        PUBLIC_IDENTIFIERS.add("FormatData_en_KE");
        PUBLIC_IDENTIFIERS.add("FormatData_en_MG");
        PUBLIC_IDENTIFIERS.add("FormatData_en_MS");
        PUBLIC_IDENTIFIERS.add("FormatData_en_MT");
        PUBLIC_IDENTIFIERS.add("FormatData_en_MU");
        PUBLIC_IDENTIFIERS.add("FormatData_en_MV");
        PUBLIC_IDENTIFIERS.add("FormatData_en_NF");
        PUBLIC_IDENTIFIERS.add("FormatData_en_NG");
        PUBLIC_IDENTIFIERS.add("FormatData_en_NL");
        PUBLIC_IDENTIFIERS.add("FormatData_en_NR");
        PUBLIC_IDENTIFIERS.add("FormatData_en_NU");
        PUBLIC_IDENTIFIERS.add("FormatData_en_NZ");
        PUBLIC_IDENTIFIERS.add("FormatData_en_PH");
        PUBLIC_IDENTIFIERS.add("FormatData_en_PK");
        PUBLIC_IDENTIFIERS.add("FormatData_en_PN");
        PUBLIC_IDENTIFIERS.add("FormatData_en_RW");
        PUBLIC_IDENTIFIERS.add("FormatData_en_SC");
        PUBLIC_IDENTIFIERS.add("FormatData_en_SE");
        PUBLIC_IDENTIFIERS.add("FormatData_en_SG");
        PUBLIC_IDENTIFIERS.add("FormatData_en_SH");
        PUBLIC_IDENTIFIERS.add("FormatData_en_SI");
        PUBLIC_IDENTIFIERS.add("FormatData_en_SX");
        PUBLIC_IDENTIFIERS.add("FormatData_en_TK");
        PUBLIC_IDENTIFIERS.add("FormatData_en_TV");
        PUBLIC_IDENTIFIERS.add("FormatData_en_TZ");
        PUBLIC_IDENTIFIERS.add("FormatData_en_UG");
        PUBLIC_IDENTIFIERS.add("FormatData_en_US");
        PUBLIC_IDENTIFIERS.add("FormatData_en_US_POSIX");
        PUBLIC_IDENTIFIERS.add("FormatData_en_ZA");
        PUBLIC_IDENTIFIERS.add("FormatData_en_ZW");
        PUBLIC_IDENTIFIERS.add("FormatData_eo");
        PUBLIC_IDENTIFIERS.add("FormatData_es");
        PUBLIC_IDENTIFIERS.add("FormatData_es_419");
        PUBLIC_IDENTIFIERS.add("FormatData_es_AR");
        PUBLIC_IDENTIFIERS.add("FormatData_es_BO");
        PUBLIC_IDENTIFIERS.add("FormatData_es_CL");
        PUBLIC_IDENTIFIERS.add("FormatData_es_CO");
        PUBLIC_IDENTIFIERS.add("FormatData_es_CR");
        PUBLIC_IDENTIFIERS.add("FormatData_es_DO");
        PUBLIC_IDENTIFIERS.add("FormatData_es_EC");
        PUBLIC_IDENTIFIERS.add("FormatData_es_ES");
        PUBLIC_IDENTIFIERS.add("FormatData_es_GQ");
        PUBLIC_IDENTIFIERS.add("FormatData_es_GT");
        PUBLIC_IDENTIFIERS.add("FormatData_es_HN");
        PUBLIC_IDENTIFIERS.add("FormatData_es_MX");
        PUBLIC_IDENTIFIERS.add("FormatData_es_NI");
        PUBLIC_IDENTIFIERS.add("FormatData_es_PA");
        PUBLIC_IDENTIFIERS.add("FormatData_es_PE");
        PUBLIC_IDENTIFIERS.add("FormatData_es_PH");
        PUBLIC_IDENTIFIERS.add("FormatData_es_PR");
        PUBLIC_IDENTIFIERS.add("FormatData_es_PY");
        PUBLIC_IDENTIFIERS.add("FormatData_es_SV");
        PUBLIC_IDENTIFIERS.add("FormatData_es_US");
        PUBLIC_IDENTIFIERS.add("FormatData_es_UY");
        PUBLIC_IDENTIFIERS.add("FormatData_es_VE");
        PUBLIC_IDENTIFIERS.add("FormatData_et");
        PUBLIC_IDENTIFIERS.add("FormatData_et_EE");
        PUBLIC_IDENTIFIERS.add("FormatData_eu");
        PUBLIC_IDENTIFIERS.add("FormatData_ewo");
        PUBLIC_IDENTIFIERS.add("FormatData_fa");
        PUBLIC_IDENTIFIERS.add("FormatData_fa_AF");
        PUBLIC_IDENTIFIERS.add("FormatData_ff");
        PUBLIC_IDENTIFIERS.add("FormatData_ff_Adlm");
        PUBLIC_IDENTIFIERS.add("FormatData_ff_Adlm_GH");
        PUBLIC_IDENTIFIERS.add("FormatData_ff_Adlm_GM");
        PUBLIC_IDENTIFIERS.add("FormatData_ff_Adlm_LR");
        PUBLIC_IDENTIFIERS.add("FormatData_ff_Adlm_MR");
        PUBLIC_IDENTIFIERS.add("FormatData_ff_Adlm_SL");
        PUBLIC_IDENTIFIERS.add("FormatData_ff_Latn_GH");
        PUBLIC_IDENTIFIERS.add("FormatData_ff_Latn_GM");
        PUBLIC_IDENTIFIERS.add("FormatData_ff_Latn_LR");
        PUBLIC_IDENTIFIERS.add("FormatData_ff_Latn_MR");
        PUBLIC_IDENTIFIERS.add("FormatData_ff_Latn_SL");
        PUBLIC_IDENTIFIERS.add("FormatData_fi");
        PUBLIC_IDENTIFIERS.add("FormatData_fi_FI");
        PUBLIC_IDENTIFIERS.add("FormatData_fil");
        PUBLIC_IDENTIFIERS.add("FormatData_fo");
        PUBLIC_IDENTIFIERS.add("FormatData_fr");
        PUBLIC_IDENTIFIERS.add("FormatData_fr_BE");
        PUBLIC_IDENTIFIERS.add("FormatData_fr_CA");
        PUBLIC_IDENTIFIERS.add("FormatData_fr_CD");
        PUBLIC_IDENTIFIERS.add("FormatData_fr_CH");
        PUBLIC_IDENTIFIERS.add("FormatData_fr_CM");
        PUBLIC_IDENTIFIERS.add("FormatData_fr_DJ");
        PUBLIC_IDENTIFIERS.add("FormatData_fr_DZ");
        PUBLIC_IDENTIFIERS.add("FormatData_fr_FR");
        PUBLIC_IDENTIFIERS.add("FormatData_fr_HT");
        PUBLIC_IDENTIFIERS.add("FormatData_fr_LU");
        PUBLIC_IDENTIFIERS.add("FormatData_fr_MA");
        PUBLIC_IDENTIFIERS.add("FormatData_fr_ML");
        PUBLIC_IDENTIFIERS.add("FormatData_fr_MR");
        PUBLIC_IDENTIFIERS.add("FormatData_fr_RE");
        PUBLIC_IDENTIFIERS.add("FormatData_fr_SN");
        PUBLIC_IDENTIFIERS.add("FormatData_fr_SY");
        PUBLIC_IDENTIFIERS.add("FormatData_fr_TD");
        PUBLIC_IDENTIFIERS.add("FormatData_fr_TN");
        PUBLIC_IDENTIFIERS.add("FormatData_fr_VU");
        PUBLIC_IDENTIFIERS.add("FormatData_frr");
        PUBLIC_IDENTIFIERS.add("FormatData_fur");
        PUBLIC_IDENTIFIERS.add("FormatData_fy");
        PUBLIC_IDENTIFIERS.add("FormatData_ga");
        PUBLIC_IDENTIFIERS.add("FormatData_ga_IE");
        PUBLIC_IDENTIFIERS.add("FormatData_gd");
        PUBLIC_IDENTIFIERS.add("FormatData_gl");
        PUBLIC_IDENTIFIERS.add("FormatData_gsw");
        PUBLIC_IDENTIFIERS.add("FormatData_gu");
        PUBLIC_IDENTIFIERS.add("FormatData_guz");
        PUBLIC_IDENTIFIERS.add("FormatData_gv");
        PUBLIC_IDENTIFIERS.add("FormatData_ha");
        PUBLIC_IDENTIFIERS.add("FormatData_ha_GH");
        PUBLIC_IDENTIFIERS.add("FormatData_haw");
        PUBLIC_IDENTIFIERS.add("FormatData_he");
        PUBLIC_IDENTIFIERS.add("FormatData_he_IL");
        PUBLIC_IDENTIFIERS.add("FormatData_hi");
        PUBLIC_IDENTIFIERS.add("FormatData_hi_IN");
        PUBLIC_IDENTIFIERS.add("FormatData_hi_Latn");
        PUBLIC_IDENTIFIERS.add("FormatData_hr");
        PUBLIC_IDENTIFIERS.add("FormatData_hr_BA");
        PUBLIC_IDENTIFIERS.add("FormatData_hr_HR");
        PUBLIC_IDENTIFIERS.add("FormatData_hsb");
        PUBLIC_IDENTIFIERS.add("FormatData_hu");
        PUBLIC_IDENTIFIERS.add("FormatData_hu_HU");
        PUBLIC_IDENTIFIERS.add("FormatData_hy");
        PUBLIC_IDENTIFIERS.add("FormatData_ia");
        PUBLIC_IDENTIFIERS.add("FormatData_id");
        PUBLIC_IDENTIFIERS.add("FormatData_id_ID");
        PUBLIC_IDENTIFIERS.add("FormatData_ig");
        PUBLIC_IDENTIFIERS.add("FormatData_ii");
        PUBLIC_IDENTIFIERS.add("FormatData_is");
        PUBLIC_IDENTIFIERS.add("FormatData_is_IS");
        PUBLIC_IDENTIFIERS.add("FormatData_it");
        PUBLIC_IDENTIFIERS.add("FormatData_it_CH");
        PUBLIC_IDENTIFIERS.add("FormatData_it_IT");
        PUBLIC_IDENTIFIERS.add("FormatData_ja");
        PUBLIC_IDENTIFIERS.add("FormatData_ja_JP");
        PUBLIC_IDENTIFIERS.add("FormatData_jgo");
        PUBLIC_IDENTIFIERS.add("FormatData_jmc");
        PUBLIC_IDENTIFIERS.add("FormatData_jv");
        PUBLIC_IDENTIFIERS.add("FormatData_ka");
        PUBLIC_IDENTIFIERS.add("FormatData_kab");
        PUBLIC_IDENTIFIERS.add("FormatData_kam");
        PUBLIC_IDENTIFIERS.add("FormatData_kde");
        PUBLIC_IDENTIFIERS.add("FormatData_kea");
        PUBLIC_IDENTIFIERS.add("FormatData_kgp");
        PUBLIC_IDENTIFIERS.add("FormatData_khq");
        PUBLIC_IDENTIFIERS.add("FormatData_ki");
        PUBLIC_IDENTIFIERS.add("FormatData_kk");
        PUBLIC_IDENTIFIERS.add("FormatData_kkj");
        PUBLIC_IDENTIFIERS.add("FormatData_kl");
        PUBLIC_IDENTIFIERS.add("FormatData_kln");
        PUBLIC_IDENTIFIERS.add("FormatData_km");
        PUBLIC_IDENTIFIERS.add("FormatData_kn");
        PUBLIC_IDENTIFIERS.add("FormatData_ko");
        PUBLIC_IDENTIFIERS.add("FormatData_ko_KR");
        PUBLIC_IDENTIFIERS.add("FormatData_kok");
        PUBLIC_IDENTIFIERS.add("FormatData_ks");
        PUBLIC_IDENTIFIERS.add("FormatData_ks_Deva");
        PUBLIC_IDENTIFIERS.add("FormatData_ksb");
        PUBLIC_IDENTIFIERS.add("FormatData_ksf");
        PUBLIC_IDENTIFIERS.add("FormatData_ksh");
        PUBLIC_IDENTIFIERS.add("FormatData_ku");
        PUBLIC_IDENTIFIERS.add("FormatData_kw");
        PUBLIC_IDENTIFIERS.add("FormatData_ky");
        PUBLIC_IDENTIFIERS.add("FormatData_lag");
        PUBLIC_IDENTIFIERS.add("FormatData_lb");
        PUBLIC_IDENTIFIERS.add("FormatData_lg");
        PUBLIC_IDENTIFIERS.add("FormatData_lkt");
        PUBLIC_IDENTIFIERS.add("FormatData_ln");
        PUBLIC_IDENTIFIERS.add("FormatData_lo");
        PUBLIC_IDENTIFIERS.add("FormatData_lrc");
        PUBLIC_IDENTIFIERS.add("FormatData_lrc_IQ");
        PUBLIC_IDENTIFIERS.add("FormatData_lt");
        PUBLIC_IDENTIFIERS.add("FormatData_lt_LT");
        PUBLIC_IDENTIFIERS.add("FormatData_lu");
        PUBLIC_IDENTIFIERS.add("FormatData_luo");
        PUBLIC_IDENTIFIERS.add("FormatData_luy");
        PUBLIC_IDENTIFIERS.add("FormatData_lv");
        PUBLIC_IDENTIFIERS.add("FormatData_lv_LV");
        PUBLIC_IDENTIFIERS.add("FormatData_mai");
        PUBLIC_IDENTIFIERS.add("FormatData_mas");
        PUBLIC_IDENTIFIERS.add("FormatData_mer");
        PUBLIC_IDENTIFIERS.add("FormatData_mfe");
        PUBLIC_IDENTIFIERS.add("FormatData_mg");
        PUBLIC_IDENTIFIERS.add("FormatData_mgh");
        PUBLIC_IDENTIFIERS.add("FormatData_mgo");
        PUBLIC_IDENTIFIERS.add("FormatData_mi");
        PUBLIC_IDENTIFIERS.add("FormatData_mk");
        PUBLIC_IDENTIFIERS.add("FormatData_mk_MK");
        PUBLIC_IDENTIFIERS.add("FormatData_ml");
        PUBLIC_IDENTIFIERS.add("FormatData_mn");
        PUBLIC_IDENTIFIERS.add("FormatData_mni");
        PUBLIC_IDENTIFIERS.add("FormatData_mr");
        PUBLIC_IDENTIFIERS.add("FormatData_ms");
        PUBLIC_IDENTIFIERS.add("FormatData_ms_BN");
        PUBLIC_IDENTIFIERS.add("FormatData_ms_ID");
        PUBLIC_IDENTIFIERS.add("FormatData_ms_MY");
        PUBLIC_IDENTIFIERS.add("FormatData_mt");
        PUBLIC_IDENTIFIERS.add("FormatData_mt_MT");
        PUBLIC_IDENTIFIERS.add("FormatData_mua");
        PUBLIC_IDENTIFIERS.add("FormatData_my");
        PUBLIC_IDENTIFIERS.add("FormatData_mzn");
        PUBLIC_IDENTIFIERS.add("FormatData_naq");
        PUBLIC_IDENTIFIERS.add("FormatData_nb");
        PUBLIC_IDENTIFIERS.add("FormatData_nd");
        PUBLIC_IDENTIFIERS.add("FormatData_nds");
        PUBLIC_IDENTIFIERS.add("FormatData_ne");
        PUBLIC_IDENTIFIERS.add("FormatData_ne_IN");
        PUBLIC_IDENTIFIERS.add("FormatData_nl");
        PUBLIC_IDENTIFIERS.add("FormatData_nl_BE");
        PUBLIC_IDENTIFIERS.add("FormatData_nl_NL");
        PUBLIC_IDENTIFIERS.add("FormatData_nmg");
        PUBLIC_IDENTIFIERS.add("FormatData_nn");
        PUBLIC_IDENTIFIERS.add("FormatData_nnh");
        PUBLIC_IDENTIFIERS.add("FormatData_no");
        PUBLIC_IDENTIFIERS.add("FormatData_no_NO");
        PUBLIC_IDENTIFIERS.add("FormatData_no_NO_NY");
        PUBLIC_IDENTIFIERS.add("FormatData_nus");
        PUBLIC_IDENTIFIERS.add("FormatData_nyn");
        PUBLIC_IDENTIFIERS.add("FormatData_oc");
        PUBLIC_IDENTIFIERS.add("FormatData_oc_ES");
        PUBLIC_IDENTIFIERS.add("FormatData_om");
        PUBLIC_IDENTIFIERS.add("FormatData_om_KE");
        PUBLIC_IDENTIFIERS.add("FormatData_or");
        PUBLIC_IDENTIFIERS.add("FormatData_os");
        PUBLIC_IDENTIFIERS.add("FormatData_pa");
        PUBLIC_IDENTIFIERS.add("FormatData_pa_Arab");
        PUBLIC_IDENTIFIERS.add("FormatData_pcm");
        PUBLIC_IDENTIFIERS.add("FormatData_pis");
        PUBLIC_IDENTIFIERS.add("FormatData_pl");
        PUBLIC_IDENTIFIERS.add("FormatData_pl_PL");
        PUBLIC_IDENTIFIERS.add("FormatData_ps");
        PUBLIC_IDENTIFIERS.add("FormatData_ps_PK");
        PUBLIC_IDENTIFIERS.add("FormatData_pt");
        PUBLIC_IDENTIFIERS.add("FormatData_pt_BR");
        PUBLIC_IDENTIFIERS.add("FormatData_pt_MO");
        PUBLIC_IDENTIFIERS.add("FormatData_pt_PT");
        PUBLIC_IDENTIFIERS.add("FormatData_qu");
        PUBLIC_IDENTIFIERS.add("FormatData_qu_BO");
        PUBLIC_IDENTIFIERS.add("FormatData_raj");
        PUBLIC_IDENTIFIERS.add("FormatData_rm");
        PUBLIC_IDENTIFIERS.add("FormatData_rn");
        PUBLIC_IDENTIFIERS.add("FormatData_ro");
        PUBLIC_IDENTIFIERS.add("FormatData_ro_MD");
        PUBLIC_IDENTIFIERS.add("FormatData_ro_RO");
        PUBLIC_IDENTIFIERS.add("FormatData_rof");
        PUBLIC_IDENTIFIERS.add("FormatData_ru");
        PUBLIC_IDENTIFIERS.add("FormatData_ru_RU");
        PUBLIC_IDENTIFIERS.add("FormatData_ru_UA");
        PUBLIC_IDENTIFIERS.add("FormatData_rw");
        PUBLIC_IDENTIFIERS.add("FormatData_rwk");
        PUBLIC_IDENTIFIERS.add("FormatData_sa");
        PUBLIC_IDENTIFIERS.add("FormatData_sah");
        PUBLIC_IDENTIFIERS.add("FormatData_saq");
        PUBLIC_IDENTIFIERS.add("FormatData_sat");
        PUBLIC_IDENTIFIERS.add("FormatData_sbp");
        PUBLIC_IDENTIFIERS.add("FormatData_sc");
        PUBLIC_IDENTIFIERS.add("FormatData_sd");
        PUBLIC_IDENTIFIERS.add("FormatData_sd_Deva");
        PUBLIC_IDENTIFIERS.add("FormatData_se");
        PUBLIC_IDENTIFIERS.add("FormatData_se_FI");
        PUBLIC_IDENTIFIERS.add("FormatData_seh");
        PUBLIC_IDENTIFIERS.add("FormatData_ses");
        PUBLIC_IDENTIFIERS.add("FormatData_sg");
        PUBLIC_IDENTIFIERS.add("FormatData_shi");
        PUBLIC_IDENTIFIERS.add("FormatData_shi_Latn");
        PUBLIC_IDENTIFIERS.add("FormatData_si");
        PUBLIC_IDENTIFIERS.add("FormatData_sk");
        PUBLIC_IDENTIFIERS.add("FormatData_sk_SK");
        PUBLIC_IDENTIFIERS.add("FormatData_sl");
        PUBLIC_IDENTIFIERS.add("FormatData_sl_SI");
        PUBLIC_IDENTIFIERS.add("FormatData_smn");
        PUBLIC_IDENTIFIERS.add("FormatData_sms");
        PUBLIC_IDENTIFIERS.add("FormatData_sn");
        PUBLIC_IDENTIFIERS.add("FormatData_so");
        PUBLIC_IDENTIFIERS.add("FormatData_so_KE");
        PUBLIC_IDENTIFIERS.add("FormatData_sq");
        PUBLIC_IDENTIFIERS.add("FormatData_sq_AL");
        PUBLIC_IDENTIFIERS.add("FormatData_sq_MK");
        PUBLIC_IDENTIFIERS.add("FormatData_sq_XK");
        PUBLIC_IDENTIFIERS.add("FormatData_sr");
        PUBLIC_IDENTIFIERS.add("FormatData_sr_BA");
        PUBLIC_IDENTIFIERS.add("FormatData_sr_CS");
        PUBLIC_IDENTIFIERS.add("FormatData_sr_Cyrl_BA");
        PUBLIC_IDENTIFIERS.add("FormatData_sr_Cyrl_ME");
        PUBLIC_IDENTIFIERS.add("FormatData_sr_Cyrl_XK");
        PUBLIC_IDENTIFIERS.add("FormatData_sr_Latn");
        PUBLIC_IDENTIFIERS.add("FormatData_sr_Latn_BA");
        PUBLIC_IDENTIFIERS.add("FormatData_sr_Latn_ME");
        PUBLIC_IDENTIFIERS.add("FormatData_sr_Latn_XK");
        PUBLIC_IDENTIFIERS.add("FormatData_sr_ME");
        PUBLIC_IDENTIFIERS.add("FormatData_sr_RS");
        PUBLIC_IDENTIFIERS.add("FormatData_su");
        PUBLIC_IDENTIFIERS.add("FormatData_sv");
        PUBLIC_IDENTIFIERS.add("FormatData_sv_FI");
        PUBLIC_IDENTIFIERS.add("FormatData_sv_SE");
        PUBLIC_IDENTIFIERS.add("FormatData_sw");
        PUBLIC_IDENTIFIERS.add("FormatData_sw_CD");
        PUBLIC_IDENTIFIERS.add("FormatData_sw_KE");
        PUBLIC_IDENTIFIERS.add("FormatData_ta");
        PUBLIC_IDENTIFIERS.add("FormatData_ta_LK");
        PUBLIC_IDENTIFIERS.add("FormatData_ta_MY");
        PUBLIC_IDENTIFIERS.add("FormatData_ta_SG");
        PUBLIC_IDENTIFIERS.add("FormatData_te");
        PUBLIC_IDENTIFIERS.add("FormatData_teo");
        PUBLIC_IDENTIFIERS.add("FormatData_tg");
        PUBLIC_IDENTIFIERS.add("FormatData_th");
        PUBLIC_IDENTIFIERS.add("FormatData_th_TH");
        PUBLIC_IDENTIFIERS.add("FormatData_ti");
        PUBLIC_IDENTIFIERS.add("FormatData_ti_ER");
        PUBLIC_IDENTIFIERS.add("FormatData_tk");
        PUBLIC_IDENTIFIERS.add("FormatData_to");
        PUBLIC_IDENTIFIERS.add("FormatData_tr");
        PUBLIC_IDENTIFIERS.add("FormatData_tr_CY");
        PUBLIC_IDENTIFIERS.add("FormatData_tr_TR");
        PUBLIC_IDENTIFIERS.add("FormatData_tt");
        PUBLIC_IDENTIFIERS.add("FormatData_twq");
        PUBLIC_IDENTIFIERS.add("FormatData_tzm");
        PUBLIC_IDENTIFIERS.add("FormatData_ug");
        PUBLIC_IDENTIFIERS.add("FormatData_uk");
        PUBLIC_IDENTIFIERS.add("FormatData_uk_UA");
        PUBLIC_IDENTIFIERS.add("FormatData_ur");
        PUBLIC_IDENTIFIERS.add("FormatData_ur_IN");
        PUBLIC_IDENTIFIERS.add("FormatData_uz");
        PUBLIC_IDENTIFIERS.add("FormatData_uz_Arab");
        PUBLIC_IDENTIFIERS.add("FormatData_uz_Cyrl");
        PUBLIC_IDENTIFIERS.add("FormatData_vai");
        PUBLIC_IDENTIFIERS.add("FormatData_vai_Latn");
        PUBLIC_IDENTIFIERS.add("FormatData_vi");
        PUBLIC_IDENTIFIERS.add("FormatData_vi_VN");
        PUBLIC_IDENTIFIERS.add("FormatData_vun");
        PUBLIC_IDENTIFIERS.add("FormatData_wae");
        PUBLIC_IDENTIFIERS.add("FormatData_wo");
        PUBLIC_IDENTIFIERS.add("FormatData_xh");
        PUBLIC_IDENTIFIERS.add("FormatData_xog");
        PUBLIC_IDENTIFIERS.add("FormatData_yav");
        PUBLIC_IDENTIFIERS.add("FormatData_yi");
        PUBLIC_IDENTIFIERS.add("FormatData_yo");
        PUBLIC_IDENTIFIERS.add("FormatData_yo_BJ");
        PUBLIC_IDENTIFIERS.add("FormatData_yrl");
        PUBLIC_IDENTIFIERS.add("FormatData_yrl_CO");
        PUBLIC_IDENTIFIERS.add("FormatData_yrl_VE");
        PUBLIC_IDENTIFIERS.add("FormatData_yue");
        PUBLIC_IDENTIFIERS.add("FormatData_yue_Hans");
        PUBLIC_IDENTIFIERS.add("FormatData_zgh");
        PUBLIC_IDENTIFIERS.add("FormatData_zh");
        PUBLIC_IDENTIFIERS.add("FormatData_zh_CN");
        PUBLIC_IDENTIFIERS.add("FormatData_zh_Hans_HK");
        PUBLIC_IDENTIFIERS.add("FormatData_zh_Hans_MO");
        PUBLIC_IDENTIFIERS.add("FormatData_zh_Hans_SG");
        PUBLIC_IDENTIFIERS.add("FormatData_zh_Hant");
        PUBLIC_IDENTIFIERS.add("FormatData_zh_Hant_HK");
        PUBLIC_IDENTIFIERS.add("FormatData_zh_HK");
        PUBLIC_IDENTIFIERS.add("FormatData_zh_SG");
        PUBLIC_IDENTIFIERS.add("FormatData_zh_TW");
        PUBLIC_IDENTIFIERS.add("FormatData_zu");
        PUBLIC_IDENTIFIERS.add("FormatDataProvider");
        PUBLIC_IDENTIFIERS.add("FormatFlagsConversionMismatchException");
        PUBLIC_IDENTIFIERS.add("Formattable");
        PUBLIC_IDENTIFIERS.add("FormattableFlags");
        PUBLIC_IDENTIFIERS.add("FormattedFloatingDecimal");
        PUBLIC_IDENTIFIERS.add("Formatter");
        PUBLIC_IDENTIFIERS.add("FormatterClosedException");
        PUBLIC_IDENTIFIERS.add("FormatWithToString");
        PUBLIC_IDENTIFIERS.add("FormView");
        PUBLIC_IDENTIFIERS.add("ForwardingConfiguration");
        PUBLIC_IDENTIFIERS.add("ForwardingDiagnosticFormatter");
        PUBLIC_IDENTIFIERS.add("ForwardingFileObject");
        PUBLIC_IDENTIFIERS.add("ForwardingJavaFileManager");
        PUBLIC_IDENTIFIERS.add("ForwardingJavaFileObject");
        PUBLIC_IDENTIFIERS.add("FoundIndex");
        PUBLIC_IDENTIFIERS.add("FragmentContentDriver");
        PUBLIC_IDENTIFIERS.add("Fragments");
        PUBLIC_IDENTIFIERS.add("Frame");
        PUBLIC_IDENTIFIERS.add("FrameAccessor");
        PUBLIC_IDENTIFIERS.add("FrameBorder");
        PUBLIC_IDENTIFIERS.add("FrameEditorPaneTag");
        PUBLIC_IDENTIFIERS.add("FrameListener");
        PUBLIC_IDENTIFIERS.add("FrameNode");
        PUBLIC_IDENTIFIERS.add("FramePeer");
        PUBLIC_IDENTIFIERS.add("FrameProcessor");
        PUBLIC_IDENTIFIERS.add("FramesDecoder");
        PUBLIC_IDENTIFIERS.add("FramesEncoder");
        PUBLIC_IDENTIFIERS.add("FrameWrapper");
        PUBLIC_IDENTIFIERS.add("FREM");
        PUBLIC_IDENTIFIERS.add("Frequency");
        PUBLIC_IDENTIFIERS.add("FreshestCRLExtension");
        PUBLIC_IDENTIFIERS.add("FRETURN");
        PUBLIC_IDENTIFIERS.add("FromBitsCoercedOperation");
        PUBLIC_IDENTIFIERS.add("FSInfo");
        PUBLIC_IDENTIFIERS.add("FSTORE");
        PUBLIC_IDENTIFIERS.add("FSUB");
        PUBLIC_IDENTIFIERS.add("FtpClient");
        PUBLIC_IDENTIFIERS.add("FtpClientProvider");
        PUBLIC_IDENTIFIERS.add("FtpDirEntry");
        PUBLIC_IDENTIFIERS.add("FtpDirParser");
        PUBLIC_IDENTIFIERS.add("FtpInputStream");
        PUBLIC_IDENTIFIERS.add("FtpLoginException");
        PUBLIC_IDENTIFIERS.add("FtpOutputStream");
        PUBLIC_IDENTIFIERS.add("FtpProtocolException");
        PUBLIC_IDENTIFIERS.add("FtpURLConnection");
        PUBLIC_IDENTIFIERS.add("full_frame");
        PUBLIC_IDENTIFIERS.add("FullDVFactory");
        PUBLIC_IDENTIFIERS.add("FuncBoolean");
        PUBLIC_IDENTIFIERS.add("FuncCeiling");
        PUBLIC_IDENTIFIERS.add("FuncConcat");
        PUBLIC_IDENTIFIERS.add("FuncContains");
        PUBLIC_IDENTIFIERS.add("FuncCount");
        PUBLIC_IDENTIFIERS.add("FuncCurrent");
        PUBLIC_IDENTIFIERS.add("FuncDoclocation");
        PUBLIC_IDENTIFIERS.add("FuncExtElementAvailable");
        PUBLIC_IDENTIFIERS.add("FuncExtFunction");
        PUBLIC_IDENTIFIERS.add("FuncExtFunctionAvailable");
        PUBLIC_IDENTIFIERS.add("FuncFalse");
        PUBLIC_IDENTIFIERS.add("FuncFloor");
        PUBLIC_IDENTIFIERS.add("FuncGenerateId");
        PUBLIC_IDENTIFIERS.add("FuncHere");
        PUBLIC_IDENTIFIERS.add("FuncId");
        PUBLIC_IDENTIFIERS.add("FuncLang");
        PUBLIC_IDENTIFIERS.add("FuncLast");
        PUBLIC_IDENTIFIERS.add("FuncLoader");
        PUBLIC_IDENTIFIERS.add("FuncLocalPart");
        PUBLIC_IDENTIFIERS.add("FuncNamespace");
        PUBLIC_IDENTIFIERS.add("FuncNormalizeSpace");
        PUBLIC_IDENTIFIERS.add("FuncNot");
        PUBLIC_IDENTIFIERS.add("FuncNumber");
        PUBLIC_IDENTIFIERS.add("FuncPosition");
        PUBLIC_IDENTIFIERS.add("FuncQname");
        PUBLIC_IDENTIFIERS.add("FuncRound");
        PUBLIC_IDENTIFIERS.add("FuncStartsWith");
        PUBLIC_IDENTIFIERS.add("FuncString");
        PUBLIC_IDENTIFIERS.add("FuncStringLength");
        PUBLIC_IDENTIFIERS.add("FuncSubstring");
        PUBLIC_IDENTIFIERS.add("FuncSubstringAfter");
        PUBLIC_IDENTIFIERS.add("FuncSubstringBefore");
        PUBLIC_IDENTIFIERS.add("FuncSum");
        PUBLIC_IDENTIFIERS.add("FuncSystemProperty");
        PUBLIC_IDENTIFIERS.add("Function");
        PUBLIC_IDENTIFIERS.add("Function2Args");
        PUBLIC_IDENTIFIERS.add("Function3Args");
        PUBLIC_IDENTIFIERS.add("FunctionalInterface");
        PUBLIC_IDENTIFIERS.add("FunctionDef1Arg");
        PUBLIC_IDENTIFIERS.add("FunctionDescriptor");
        PUBLIC_IDENTIFIERS.add("FunctionDescriptorLookupError");
        PUBLIC_IDENTIFIERS.add("FunctionMultiArgs");
        PUBLIC_IDENTIFIERS.add("FunctionOneArg");
        PUBLIC_IDENTIFIERS.add("FunctionPattern");
        PUBLIC_IDENTIFIERS.add("Functions");
        PUBLIC_IDENTIFIERS.add("FunctionSym");
        PUBLIC_IDENTIFIERS.add("FunctionTable");
        PUBLIC_IDENTIFIERS.add("FunctionType");
        PUBLIC_IDENTIFIERS.add("FuncTranslate");
        PUBLIC_IDENTIFIERS.add("FuncTrue");
        PUBLIC_IDENTIFIERS.add("FuncUnparsedEntityURI");
        PUBLIC_IDENTIFIERS.add("Future");
        PUBLIC_IDENTIFIERS.add("FutureTask");
        PUBLIC_IDENTIFIERS.add("FwDispatcher");
        PUBLIC_IDENTIFIERS.add("G1CollectedHeap");
        PUBLIC_IDENTIFIERS.add("G1HeapRegionTable");
        PUBLIC_IDENTIFIERS.add("G1MonitoringSupport");
        PUBLIC_IDENTIFIERS.add("GapContent");
        PUBLIC_IDENTIFIERS.add("GarbageCollectionNotificationInfo");
        PUBLIC_IDENTIFIERS.add("GarbageCollectionNotifInfoCompositeData");
        PUBLIC_IDENTIFIERS.add("GarbageCollectorExtImpl");
        PUBLIC_IDENTIFIERS.add("GarbageCollectorImpl");
        PUBLIC_IDENTIFIERS.add("GarbageCollectorMXBean");
        PUBLIC_IDENTIFIERS.add("GatheringByteChannel");
        PUBLIC_IDENTIFIERS.add("GaugeMonitor");
        PUBLIC_IDENTIFIERS.add("GaugeMonitorMBean");
        PUBLIC_IDENTIFIERS.add("GB18030");
        PUBLIC_IDENTIFIERS.add("GBK");
        PUBLIC_IDENTIFIERS.add("GcInfo");
        PUBLIC_IDENTIFIERS.add("GcInfoBuilder");
        PUBLIC_IDENTIFIERS.add("GcInfoCompositeData");
        PUBLIC_IDENTIFIERS.add("GCM");
        PUBLIC_IDENTIFIERS.add("GCMOperation");
        PUBLIC_IDENTIFIERS.add("GCMParameterSpec");
        PUBLIC_IDENTIFIERS.add("Gen");
        PUBLIC_IDENTIFIERS.add("GenCollectedHeap");
        PUBLIC_IDENTIFIERS.add("General");
        PUBLIC_IDENTIFIERS.add("GeneralBinaryOp");
        PUBLIC_IDENTIFIERS.add("GeneralCompositePipe");
        PUBLIC_IDENTIFIERS.add("GeneralName");
        PUBLIC_IDENTIFIERS.add("GeneralNameInterface");
        PUBLIC_IDENTIFIERS.add("GeneralNames");
        PUBLIC_IDENTIFIERS.add("GeneralSecurityException");
        PUBLIC_IDENTIFIERS.add("GeneralSubtree");
        PUBLIC_IDENTIFIERS.add("GeneralSubtrees");
        PUBLIC_IDENTIFIERS.add("GeneralUnaryOp");
        PUBLIC_IDENTIFIERS.add("Generated");
        PUBLIC_IDENTIFIERS.add("GenerateOopMap");
        PUBLIC_IDENTIFIERS.add("Generation");
        PUBLIC_IDENTIFIERS.add("GenerationFactory");
        PUBLIC_IDENTIFIERS.add("GenerationSpec");
        PUBLIC_IDENTIFIERS.add("Generator");
        PUBLIC_IDENTIFIERS.add("GeneratorAdapter");
        PUBLIC_IDENTIFIERS.add("Generic");
        PUBLIC_IDENTIFIERS.add("GenericArray");
        PUBLIC_IDENTIFIERS.add("GenericArrayType");
        PUBLIC_IDENTIFIERS.add("GenericArrayTypeImpl");
        PUBLIC_IDENTIFIERS.add("GenericAttachingConnector");
        PUBLIC_IDENTIFIERS.add("GenericDeclaration");
        PUBLIC_IDENTIFIERS.add("GenericDeclRepository");
        PUBLIC_IDENTIFIERS.add("GenericGrowableArray");
        PUBLIC_IDENTIFIERS.add("GenericListeningConnector");
        PUBLIC_IDENTIFIERS.add("GenericsFactory");
        PUBLIC_IDENTIFIERS.add("GenericSignatureFormatError");
        PUBLIC_IDENTIFIERS.add("GenericURLContext");
        PUBLIC_IDENTIFIERS.add("GenericURLDirContext");
        PUBLIC_IDENTIFIERS.add("GetBooleanAction");
        PUBLIC_IDENTIFIERS.add("GetByte");
        PUBLIC_IDENTIFIERS.add("GETFIELD");
        PUBLIC_IDENTIFIERS.add("GetField");
        PUBLIC_IDENTIFIERS.add("GetFrame");
        PUBLIC_IDENTIFIERS.add("GetInstance");
        PUBLIC_IDENTIFIERS.add("GetInstanceAction");
        PUBLIC_IDENTIFIERS.add("GetIntegerAction");
        PUBLIC_IDENTIFIERS.add("GetLongAction");
        PUBLIC_IDENTIFIERS.add("GetPerfAction");
        PUBLIC_IDENTIFIERS.add("GetPropertyAction");
        PUBLIC_IDENTIFIERS.add("GETSTATIC");
        PUBLIC_IDENTIFIERS.add("gif");
        PUBLIC_IDENTIFIERS.add("GifImageDecoder");
        PUBLIC_IDENTIFIERS.add("GIFImageMetadata");
        PUBLIC_IDENTIFIERS.add("GIFImageMetadataFormat");
        PUBLIC_IDENTIFIERS.add("GIFImageMetadataFormatResources");
        PUBLIC_IDENTIFIERS.add("GIFImageReader");
        PUBLIC_IDENTIFIERS.add("GIFImageReaderSpi");
        PUBLIC_IDENTIFIERS.add("GIFImageWriter");
        PUBLIC_IDENTIFIERS.add("GIFImageWriterSpi");
        PUBLIC_IDENTIFIERS.add("GIFStreamMetadata");
        PUBLIC_IDENTIFIERS.add("GIFStreamMetadataFormat");
        PUBLIC_IDENTIFIERS.add("GIFStreamMetadataFormatResources");
        PUBLIC_IDENTIFIERS.add("GlassPaneDispatcher");
        PUBLIC_IDENTIFIERS.add("GlobalCursorManager");
        PUBLIC_IDENTIFIERS.add("GlobalSym");
        PUBLIC_IDENTIFIERS.add("Globs");
        PUBLIC_IDENTIFIERS.add("GLXOffScreenSurfaceData");
        PUBLIC_IDENTIFIERS.add("GLXSurfaceData");
        PUBLIC_IDENTIFIERS.add("GLXVolatileSurfaceManager");
        PUBLIC_IDENTIFIERS.add("GLXVSyncOffScreenSurfaceData");
        PUBLIC_IDENTIFIERS.add("GLXWindowSurfaceData");
        PUBLIC_IDENTIFIERS.add("GlyphDisposedListener");
        PUBLIC_IDENTIFIERS.add("GlyphListLoopPipe");
        PUBLIC_IDENTIFIERS.add("GlyphListPipe");
        PUBLIC_IDENTIFIERS.add("GlyphPainter");
        PUBLIC_IDENTIFIERS.add("GlyphVector");
        PUBLIC_IDENTIFIERS.add("GlyphView");
        PUBLIC_IDENTIFIERS.add("GoAwayFrame");
        PUBLIC_IDENTIFIERS.add("GoHomeAction");
        PUBLIC_IDENTIFIERS.add("GOTO");
        PUBLIC_IDENTIFIERS.add("GOTO_W");
        PUBLIC_IDENTIFIERS.add("GotoInstruction");
        PUBLIC_IDENTIFIERS.add("GradientPaint");
        PUBLIC_IDENTIFIERS.add("Grammar");
        PUBLIC_IDENTIFIERS.add("GrantEntry");
        PUBLIC_IDENTIFIERS.add("GraphDependencies");
        PUBLIC_IDENTIFIERS.add("GraphicAttribute");
        PUBLIC_IDENTIFIERS.add("Graphics");
        PUBLIC_IDENTIFIERS.add("Graphics2D");
        PUBLIC_IDENTIFIERS.add("GraphicsConfigTemplate");
        PUBLIC_IDENTIFIERS.add("GraphicsConfiguration");
        PUBLIC_IDENTIFIERS.add("GraphicsDevice");
        PUBLIC_IDENTIFIERS.add("GraphicsEnvironment");
        PUBLIC_IDENTIFIERS.add("GraphicsPrimitive");
        PUBLIC_IDENTIFIERS.add("GraphicsPrimitiveProxy");
        PUBLIC_IDENTIFIERS.add("GraphicsUtilities");
        PUBLIC_IDENTIFIERS.add("GraphStrategy");
        PUBLIC_IDENTIFIERS.add("GraphUtils");
        PUBLIC_IDENTIFIERS.add("GrayFilter");
        PUBLIC_IDENTIFIERS.add("Gregorian");
        PUBLIC_IDENTIFIERS.add("GregorianCalendar");
        PUBLIC_IDENTIFIERS.add("GridBagConstraints");
        PUBLIC_IDENTIFIERS.add("GridBagLayout");
        PUBLIC_IDENTIFIERS.add("GridLayout");
        PUBLIC_IDENTIFIERS.add("Group");
        PUBLIC_IDENTIFIERS.add("GroupLayout");
        PUBLIC_IDENTIFIERS.add("GroupPrincipal");
        PUBLIC_IDENTIFIERS.add("GrowableArray");
        PUBLIC_IDENTIFIERS.add("GrowableByteArray");
        PUBLIC_IDENTIFIERS.add("GrowableEltArray");
        PUBLIC_IDENTIFIERS.add("GrowableIntArray");
        PUBLIC_IDENTIFIERS.add("GrowablePointArray");
        PUBLIC_IDENTIFIERS.add("GrowableRectArray");
        PUBLIC_IDENTIFIERS.add("GrumpyHelper");
        PUBLIC_IDENTIFIERS.add("GSSCaller");
        PUBLIC_IDENTIFIERS.add("GSSContext");
        PUBLIC_IDENTIFIERS.add("GSSContextImpl");
        PUBLIC_IDENTIFIERS.add("GSSContextSpi");
        PUBLIC_IDENTIFIERS.add("GSSCredElement");
        PUBLIC_IDENTIFIERS.add("GSSCredential");
        PUBLIC_IDENTIFIERS.add("GSSCredentialImpl");
        PUBLIC_IDENTIFIERS.add("GSSCredentialSpi");
        PUBLIC_IDENTIFIERS.add("GSSException");
        PUBLIC_IDENTIFIERS.add("GSSExceptionImpl");
        PUBLIC_IDENTIFIERS.add("GSSHeader");
        PUBLIC_IDENTIFIERS.add("GSSManager");
        PUBLIC_IDENTIFIERS.add("GSSManagerImpl");
        PUBLIC_IDENTIFIERS.add("GSSName");
        PUBLIC_IDENTIFIERS.add("GSSNameElement");
        PUBLIC_IDENTIFIERS.add("GSSNameSpi");
        PUBLIC_IDENTIFIERS.add("GSSToken");
        PUBLIC_IDENTIFIERS.add("GSSUtil");
        PUBLIC_IDENTIFIERS.add("Gt");
        PUBLIC_IDENTIFIERS.add("Gte");
        PUBLIC_IDENTIFIERS.add("GTKColorType");
        PUBLIC_IDENTIFIERS.add("GTKConstants");
        PUBLIC_IDENTIFIERS.add("GTKDirectoryListModel");
        PUBLIC_IDENTIFIERS.add("GTKFileListModel");
        PUBLIC_IDENTIFIERS.add("GTKKeybindings");
        PUBLIC_IDENTIFIERS.add("GTKLookAndFeel");
        PUBLIC_IDENTIFIERS.add("Guard");
        PUBLIC_IDENTIFIERS.add("GuardedInvocation");
        PUBLIC_IDENTIFIERS.add("GuardedInvocationTransformer");
        PUBLIC_IDENTIFIERS.add("GuardedObject");
        PUBLIC_IDENTIFIERS.add("GuardingDynamicLinker");
        PUBLIC_IDENTIFIERS.add("GuardingDynamicLinkerExporter");
        PUBLIC_IDENTIFIERS.add("GuardingTypeConverterFactory");
        PUBLIC_IDENTIFIERS.add("GUIInitializedListener");
        PUBLIC_IDENTIFIERS.add("GZIPInputStream");
        PUBLIC_IDENTIFIERS.add("GZIPOutputStream");
        PUBLIC_IDENTIFIERS.add("HaltNode");
        PUBLIC_IDENTIFIERS.add("Handler");
        PUBLIC_IDENTIFIERS.add("HandlerBase");
        PUBLIC_IDENTIFIERS.add("HandshakeCompletedEvent");
        PUBLIC_IDENTIFIERS.add("HandshakeCompletedListener");
        PUBLIC_IDENTIFIERS.add("HandshakeOutStream");
        PUBLIC_IDENTIFIERS.add("HangulSyllableType");
        PUBLIC_IDENTIFIERS.add("HasControls");
        PUBLIC_IDENTIFIERS.add("HashAttributeSet");
        PUBLIC_IDENTIFIERS.add("HashDocAttributeSet");
        PUBLIC_IDENTIFIERS.add("HashDrbg");
        PUBLIC_IDENTIFIERS.add("HashMap");
        PUBLIC_IDENTIFIERS.add("HashPrintJobAttributeSet");
        PUBLIC_IDENTIFIERS.add("HashPrintRequestAttributeSet");
        PUBLIC_IDENTIFIERS.add("HashPrintServiceAttributeSet");
        PUBLIC_IDENTIFIERS.add("HashSet");
        PUBLIC_IDENTIFIERS.add("HashSupplier");
        PUBLIC_IDENTIFIERS.add("Hashtable");
        PUBLIC_IDENTIFIERS.add("HasPositionalPredChecker");
        PUBLIC_IDENTIFIERS.add("Head");
        PUBLIC_IDENTIFIERS.add("Header");
        PUBLIC_IDENTIFIERS.add("HeaderClosure");
        PUBLIC_IDENTIFIERS.add("HeaderFrame");
        PUBLIC_IDENTIFIERS.add("HeaderParser");
        PUBLIC_IDENTIFIERS.add("Headers");
        PUBLIC_IDENTIFIERS.add("HeadersFrame");
        PUBLIC_IDENTIFIERS.add("HeadlessException");
        PUBLIC_IDENTIFIERS.add("HeadTail");
        PUBLIC_IDENTIFIERS.add("HeapBlock");
        PUBLIC_IDENTIFIERS.add("HeapDumper");
        PUBLIC_IDENTIFIERS.add("HeapGraphWriter");
        PUBLIC_IDENTIFIERS.add("HeapGXLWriter");
        PUBLIC_IDENTIFIERS.add("HeapHprofBinWriter");
        PUBLIC_IDENTIFIERS.add("HeapMemorySegmentImpl");
        PUBLIC_IDENTIFIERS.add("HeapNode");
        PUBLIC_IDENTIFIERS.add("HeapParametersPanel");
        PUBLIC_IDENTIFIERS.add("HeapPrinter");
        PUBLIC_IDENTIFIERS.add("HeapProgressThunk");
        PUBLIC_IDENTIFIERS.add("HeapRegion");
        PUBLIC_IDENTIFIERS.add("HeapRegionClosure");
        PUBLIC_IDENTIFIERS.add("HeapRegionManager");
        PUBLIC_IDENTIFIERS.add("HeapRegionSetBase");
        PUBLIC_IDENTIFIERS.add("HeapRegionType");
        PUBLIC_IDENTIFIERS.add("HeapSummary");
        PUBLIC_IDENTIFIERS.add("HeapVisitor");
        PUBLIC_IDENTIFIERS.add("HelloWorld");
        PUBLIC_IDENTIFIERS.add("HelpAction");
        PUBLIC_IDENTIFIERS.add("HelperNodeList");
        PUBLIC_IDENTIFIERS.add("HelpFormatter");
        PUBLIC_IDENTIFIERS.add("HelpMenu");
        PUBLIC_IDENTIFIERS.add("HelpWriter");
        PUBLIC_IDENTIFIERS.add("HexBinaryDV");
        PUBLIC_IDENTIFIERS.add("HexDumpEncoder");
        PUBLIC_IDENTIFIERS.add("Hidden");
        PUBLIC_IDENTIFIERS.add("HiddenAction");
        PUBLIC_IDENTIFIERS.add("HiddenTree");
        PUBLIC_IDENTIFIERS.add("Hierarchy");
        PUBLIC_IDENTIFIERS.add("HierarchyBoundsAdapter");
        PUBLIC_IDENTIFIERS.add("HierarchyBoundsListener");
        PUBLIC_IDENTIFIERS.add("HierarchyEvent");
        PUBLIC_IDENTIFIERS.add("HierarchyListener");
        PUBLIC_IDENTIFIERS.add("HierContextEnumerator");
        PUBLIC_IDENTIFIERS.add("HierMemDirCtx");
        PUBLIC_IDENTIFIERS.add("Highlight");
        PUBLIC_IDENTIFIERS.add("Highlighter");
        PUBLIC_IDENTIFIERS.add("HighlightPainter");
        PUBLIC_IDENTIFIERS.add("HighPrecisionJScrollBar");
        PUBLIC_IDENTIFIERS.add("HistoricallyNamedCharset");
        PUBLIC_IDENTIFIERS.add("History");
        PUBLIC_IDENTIFIERS.add("HistoryComboBox");
        PUBLIC_IDENTIFIERS.add("HKSCS");
        PUBLIC_IDENTIFIERS.add("HKSCSMapping");
        PUBLIC_IDENTIFIERS.add("HmacDrbg");
        PUBLIC_IDENTIFIERS.add("HmacMd5ArcFourCksumType");
        PUBLIC_IDENTIFIERS.add("HmacSha1Aes128CksumType");
        PUBLIC_IDENTIFIERS.add("HmacSha1Aes256CksumType");
        PUBLIC_IDENTIFIERS.add("HmacSha1Des3KdCksumType");
        PUBLIC_IDENTIFIERS.add("HmacSha2Aes128CksumType");
        PUBLIC_IDENTIFIERS.add("HmacSha2Aes256CksumType");
        PUBLIC_IDENTIFIERS.add("HostAddress");
        PUBLIC_IDENTIFIERS.add("HostAddresses");
        PUBLIC_IDENTIFIERS.add("HostEvent");
        PUBLIC_IDENTIFIERS.add("HostIdentifier");
        PUBLIC_IDENTIFIERS.add("HostListener");
        PUBLIC_IDENTIFIERS.add("HostLocaleProviderAdapter");
        PUBLIC_IDENTIFIERS.add("HostnameChecker");
        PUBLIC_IDENTIFIERS.add("HostnameVerifier");
        PUBLIC_IDENTIFIERS.add("HotSpotAgent");
        PUBLIC_IDENTIFIERS.add("HotSpotAttachProvider");
        PUBLIC_IDENTIFIERS.add("HotspotClassLoadingMBean");
        PUBLIC_IDENTIFIERS.add("HotSpotCodeCacheProvider");
        PUBLIC_IDENTIFIERS.add("HotspotCompilationMBean");
        PUBLIC_IDENTIFIERS.add("HotSpotCompilationRequest");
        PUBLIC_IDENTIFIERS.add("HotSpotCompiledCode");
        PUBLIC_IDENTIFIERS.add("HotSpotConstant");
        PUBLIC_IDENTIFIERS.add("HotSpotConstantReflectionProvider");
        PUBLIC_IDENTIFIERS.add("HotSpotDiagnostic");
        PUBLIC_IDENTIFIERS.add("HotSpotDiagnosticMXBean");
        PUBLIC_IDENTIFIERS.add("HotSpotForeignCallTarget");
        PUBLIC_IDENTIFIERS.add("HotSpotInstalledCode");
        PUBLIC_IDENTIFIERS.add("HotspotInternal");
        PUBLIC_IDENTIFIERS.add("HotspotInternalMBean");
        PUBLIC_IDENTIFIERS.add("HotSpotJavaType");
        PUBLIC_IDENTIFIERS.add("HotSpotJVMCIBackendFactory");
        PUBLIC_IDENTIFIERS.add("HotSpotJVMCICompilerFactory");
        PUBLIC_IDENTIFIERS.add("HotSpotJVMCIUnsupportedOperationError");
        PUBLIC_IDENTIFIERS.add("HotSpotMemoryAccessProvider");
        PUBLIC_IDENTIFIERS.add("HotspotMemoryMBean");
        PUBLIC_IDENTIFIERS.add("HotSpotMetaAccessProvider");
        PUBLIC_IDENTIFIERS.add("HotSpotMetaspaceConstant");
        PUBLIC_IDENTIFIERS.add("HotSpotMethodHandleAccessProvider");
        PUBLIC_IDENTIFIERS.add("HotSpotModifiers");
        PUBLIC_IDENTIFIERS.add("HotSpotNmethod");
        PUBLIC_IDENTIFIERS.add("HotSpotObjectConstant");
        PUBLIC_IDENTIFIERS.add("HotSpotResolvedJavaField");
        PUBLIC_IDENTIFIERS.add("HotSpotResolvedJavaMethod");
        PUBLIC_IDENTIFIERS.add("HotSpotResolvedJavaType");
        PUBLIC_IDENTIFIERS.add("HotSpotResolvedObjectType");
        PUBLIC_IDENTIFIERS.add("HotspotRuntimeMBean");
        PUBLIC_IDENTIFIERS.add("HotSpotRuntimeStub");
        PUBLIC_IDENTIFIERS.add("HotSpotSignature");
        PUBLIC_IDENTIFIERS.add("HotSpotSpeculationLog");
        PUBLIC_IDENTIFIERS.add("HotSpotStackFrameReference");
        PUBLIC_IDENTIFIERS.add("HotSpotStackIntrospection");
        PUBLIC_IDENTIFIERS.add("HotSpotThreadImpl");
        PUBLIC_IDENTIFIERS.add("HotspotThreadMBean");
        PUBLIC_IDENTIFIERS.add("HotSpotTypeDataBase");
        PUBLIC_IDENTIFIERS.add("HotSpotVirtualMachine");
        PUBLIC_IDENTIFIERS.add("HotSpotVMConfigAccess");
        PUBLIC_IDENTIFIERS.add("HotSpotVMEventListener");
        PUBLIC_IDENTIFIERS.add("HSBChangeListener");
        PUBLIC_IDENTIFIERS.add("HSDB");
        PUBLIC_IDENTIFIERS.add("HSDBActionManager");
        PUBLIC_IDENTIFIERS.add("HTML");
        PUBLIC_IDENTIFIERS.add("HTMLAccessibleContext");
        PUBLIC_IDENTIFIERS.add("HTMLAnchorElement");
        PUBLIC_IDENTIFIERS.add("HTMLAppletElement");
        PUBLIC_IDENTIFIERS.add("HTMLAreaElement");
        PUBLIC_IDENTIFIERS.add("HTMLBaseElement");
        PUBLIC_IDENTIFIERS.add("HTMLBaseFontElement");
        PUBLIC_IDENTIFIERS.add("HTMLBodyElement");
        PUBLIC_IDENTIFIERS.add("HTMLBRElement");
        PUBLIC_IDENTIFIERS.add("HTMLButtonElement");
        PUBLIC_IDENTIFIERS.add("HTMLCollection");
        PUBLIC_IDENTIFIERS.add("HtmlConfiguration");
        PUBLIC_IDENTIFIERS.add("HTMLDirectoryElement");
        PUBLIC_IDENTIFIERS.add("HTMLDivElement");
        PUBLIC_IDENTIFIERS.add("HTMLDListElement");
        PUBLIC_IDENTIFIERS.add("HtmlDoclet");
        PUBLIC_IDENTIFIERS.add("HtmlDocletWriter");
        PUBLIC_IDENTIFIERS.add("HTMLDocument");
        PUBLIC_IDENTIFIERS.add("HtmlDocument");
        PUBLIC_IDENTIFIERS.add("HTMLDOMImplementation");
        PUBLIC_IDENTIFIERS.add("HTMLEditorKit");
        PUBLIC_IDENTIFIERS.add("HTMLElement");
        PUBLIC_IDENTIFIERS.add("HTMLFactory");
        PUBLIC_IDENTIFIERS.add("HTMLFieldSetElement");
        PUBLIC_IDENTIFIERS.add("HTMLFontElement");
        PUBLIC_IDENTIFIERS.add("HTMLFormElement");
        PUBLIC_IDENTIFIERS.add("HTMLFrameElement");
        PUBLIC_IDENTIFIERS.add("HTMLFrameHyperlinkEvent");
        PUBLIC_IDENTIFIERS.add("HTMLFrameSetElement");
        PUBLIC_IDENTIFIERS.add("HTMLGenerator");
        PUBLIC_IDENTIFIERS.add("HTMLHeadElement");
        PUBLIC_IDENTIFIERS.add("HTMLHeadingElement");
        PUBLIC_IDENTIFIERS.add("HTMLHRElement");
        PUBLIC_IDENTIFIERS.add("HTMLHtmlElement");
        PUBLIC_IDENTIFIERS.add("HtmlId");
        PUBLIC_IDENTIFIERS.add("HtmlIds");
        PUBLIC_IDENTIFIERS.add("HTMLIFrameElement");
        PUBLIC_IDENTIFIERS.add("HTMLImageElement");
        PUBLIC_IDENTIFIERS.add("HtmlIndexBuilder");
        PUBLIC_IDENTIFIERS.add("HTMLInputElement");
        PUBLIC_IDENTIFIERS.add("HTMLIsIndexElement");
        PUBLIC_IDENTIFIERS.add("HTMLLabelElement");
        PUBLIC_IDENTIFIERS.add("HTMLLegendElement");
        PUBLIC_IDENTIFIERS.add("HTMLLIElement");
        PUBLIC_IDENTIFIERS.add("HTMLLink");
        PUBLIC_IDENTIFIERS.add("HTMLLinkElement");
        PUBLIC_IDENTIFIERS.add("HtmlLinkFactory");
        PUBLIC_IDENTIFIERS.add("HtmlLinkInfo");
        PUBLIC_IDENTIFIERS.add("HTMLMapElement");
        PUBLIC_IDENTIFIERS.add("HTMLMenuElement");
        PUBLIC_IDENTIFIERS.add("HTMLMetaElement");
        PUBLIC_IDENTIFIERS.add("HTMLModElement");
        PUBLIC_IDENTIFIERS.add("HTMLObjectElement");
        PUBLIC_IDENTIFIERS.add("HTMLOListElement");
        PUBLIC_IDENTIFIERS.add("HTMLOptGroupElement");
        PUBLIC_IDENTIFIERS.add("HTMLOptionElement");
        PUBLIC_IDENTIFIERS.add("HtmlOptions");
        PUBLIC_IDENTIFIERS.add("HTMLPane");
        PUBLIC_IDENTIFIERS.add("HTMLParagraphElement");
        PUBLIC_IDENTIFIERS.add("HTMLParamElement");
        PUBLIC_IDENTIFIERS.add("HTMLPreElement");
        PUBLIC_IDENTIFIERS.add("HTMLQuoteElement");
        PUBLIC_IDENTIFIERS.add("HTMLReader");
        PUBLIC_IDENTIFIERS.add("HTMLScriptElement");
        PUBLIC_IDENTIFIERS.add("HTMLSelectElement");
        PUBLIC_IDENTIFIERS.add("HtmlSerialFieldWriter");
        PUBLIC_IDENTIFIERS.add("HTMLSerializer");
        PUBLIC_IDENTIFIERS.add("HtmlSerialMethodWriter");
        PUBLIC_IDENTIFIERS.add("HTMLStyleElement");
        PUBLIC_IDENTIFIERS.add("HTMLTableCaptionElement");
        PUBLIC_IDENTIFIERS.add("HTMLTableCellElement");
        PUBLIC_IDENTIFIERS.add("HTMLTableColElement");
        PUBLIC_IDENTIFIERS.add("HTMLTableElement");
        PUBLIC_IDENTIFIERS.add("HTMLTableRowElement");
        PUBLIC_IDENTIFIERS.add("HTMLTableSectionElement");
        PUBLIC_IDENTIFIERS.add("HtmlTag");
        PUBLIC_IDENTIFIERS.add("HTMLTextAction");
        PUBLIC_IDENTIFIERS.add("HTMLTextAreaElement");
        PUBLIC_IDENTIFIERS.add("HTMLTitleElement");
        PUBLIC_IDENTIFIERS.add("HtmlTree");
        PUBLIC_IDENTIFIERS.add("HTMLUListElement");
        PUBLIC_IDENTIFIERS.add("HTMLWriter");
        PUBLIC_IDENTIFIERS.add("Http2Frame");
        PUBLIC_IDENTIFIERS.add("HttpAuthenticator");
        PUBLIC_IDENTIFIERS.add("HttpBodySubscriberWrapper");
        PUBLIC_IDENTIFIERS.add("HttpCapture");
        PUBLIC_IDENTIFIERS.add("HttpCaptureInputStream");
        PUBLIC_IDENTIFIERS.add("HttpCaptureOutputStream");
        PUBLIC_IDENTIFIERS.add("HttpClient");
        PUBLIC_IDENTIFIERS.add("HttpClientBuilderImpl");
        PUBLIC_IDENTIFIERS.add("HttpConnectTimeoutException");
        PUBLIC_IDENTIFIERS.add("HttpContext");
        PUBLIC_IDENTIFIERS.add("HttpExchange");
        PUBLIC_IDENTIFIERS.add("HttpHandler");
        PUBLIC_IDENTIFIERS.add("HttpHeadersBuilder");
        PUBLIC_IDENTIFIERS.add("HttpLogFormatter");
        PUBLIC_IDENTIFIERS.add("HttpPrincipal");
        PUBLIC_IDENTIFIERS.add("HttpRequest");
        PUBLIC_IDENTIFIERS.add("HttpRequestBuilderImpl");
        PUBLIC_IDENTIFIERS.add("HttpRequestImpl");
        PUBLIC_IDENTIFIERS.add("HttpResponse");
        PUBLIC_IDENTIFIERS.add("HttpResponseInputStream");
        PUBLIC_IDENTIFIERS.add("HttpRetryException");
        PUBLIC_IDENTIFIERS.add("HttpsConfigurator");
        PUBLIC_IDENTIFIERS.add("HttpServer");
        PUBLIC_IDENTIFIERS.add("HttpServerImpl");
        PUBLIC_IDENTIFIERS.add("HttpServerProvider");
        PUBLIC_IDENTIFIERS.add("HttpsExchange");
        PUBLIC_IDENTIFIERS.add("HttpsParameters");
        PUBLIC_IDENTIFIERS.add("HttpsServer");
        PUBLIC_IDENTIFIERS.add("HttpsServerImpl");
        PUBLIC_IDENTIFIERS.add("HttpsURLConnection");
        PUBLIC_IDENTIFIERS.add("HttpsURLConnectionImpl");
        PUBLIC_IDENTIFIERS.add("HttpTimeoutException");
        PUBLIC_IDENTIFIERS.add("HttpTimestamper");
        PUBLIC_IDENTIFIERS.add("HttpURLConnection");
        PUBLIC_IDENTIFIERS.add("Hurryable");
        PUBLIC_IDENTIFIERS.add("HyperlinkEvent");
        PUBLIC_IDENTIFIERS.add("HyperlinkListener");
        PUBLIC_IDENTIFIERS.add("Hypot");
    }

    private static void addPublicIdentifiersIJ() {
        PUBLIC_IDENTIFIERS.add("I18NImpl");
        PUBLIC_IDENTIFIERS.add("I2B");
        PUBLIC_IDENTIFIERS.add("I2C");
        PUBLIC_IDENTIFIERS.add("I2D");
        PUBLIC_IDENTIFIERS.add("I2F");
        PUBLIC_IDENTIFIERS.add("I2L");
        PUBLIC_IDENTIFIERS.add("I2S");
        PUBLIC_IDENTIFIERS.add("IADD");
        PUBLIC_IDENTIFIERS.add("IALOAD");
        PUBLIC_IDENTIFIERS.add("IAND");
        PUBLIC_IDENTIFIERS.add("IASTORE");
        PUBLIC_IDENTIFIERS.add("IBM037");
        PUBLIC_IDENTIFIERS.add("IBM1006");
        PUBLIC_IDENTIFIERS.add("IBM1025");
        PUBLIC_IDENTIFIERS.add("IBM1026");
        PUBLIC_IDENTIFIERS.add("IBM1046");
        PUBLIC_IDENTIFIERS.add("IBM1047");
        PUBLIC_IDENTIFIERS.add("IBM1097");
        PUBLIC_IDENTIFIERS.add("IBM1098");
        PUBLIC_IDENTIFIERS.add("IBM1112");
        PUBLIC_IDENTIFIERS.add("IBM1122");
        PUBLIC_IDENTIFIERS.add("IBM1123");
        PUBLIC_IDENTIFIERS.add("IBM1124");
        PUBLIC_IDENTIFIERS.add("IBM1129");
        PUBLIC_IDENTIFIERS.add("IBM1140");
        PUBLIC_IDENTIFIERS.add("IBM1141");
        PUBLIC_IDENTIFIERS.add("IBM1142");
        PUBLIC_IDENTIFIERS.add("IBM1143");
        PUBLIC_IDENTIFIERS.add("IBM1144");
        PUBLIC_IDENTIFIERS.add("IBM1145");
        PUBLIC_IDENTIFIERS.add("IBM1146");
        PUBLIC_IDENTIFIERS.add("IBM1147");
        PUBLIC_IDENTIFIERS.add("IBM1148");
        PUBLIC_IDENTIFIERS.add("IBM1149");
        PUBLIC_IDENTIFIERS.add("IBM1166");
        PUBLIC_IDENTIFIERS.add("IBM1364");
        PUBLIC_IDENTIFIERS.add("IBM1381");
        PUBLIC_IDENTIFIERS.add("IBM1383");
        PUBLIC_IDENTIFIERS.add("IBM273");
        PUBLIC_IDENTIFIERS.add("IBM277");
        PUBLIC_IDENTIFIERS.add("IBM278");
        PUBLIC_IDENTIFIERS.add("IBM280");
        PUBLIC_IDENTIFIERS.add("IBM284");
        PUBLIC_IDENTIFIERS.add("IBM285");
        PUBLIC_IDENTIFIERS.add("IBM290");
        PUBLIC_IDENTIFIERS.add("IBM29626C");
        PUBLIC_IDENTIFIERS.add("IBM297");
        PUBLIC_IDENTIFIERS.add("IBM300");
        PUBLIC_IDENTIFIERS.add("IBM33722");
        PUBLIC_IDENTIFIERS.add("IBM420");
        PUBLIC_IDENTIFIERS.add("IBM424");
        PUBLIC_IDENTIFIERS.add("IBM437");
        PUBLIC_IDENTIFIERS.add("IBM500");
        PUBLIC_IDENTIFIERS.add("IBM737");
        PUBLIC_IDENTIFIERS.add("IBM775");
        PUBLIC_IDENTIFIERS.add("IBM833");
        PUBLIC_IDENTIFIERS.add("IBM834");
        PUBLIC_IDENTIFIERS.add("IBM838");
        PUBLIC_IDENTIFIERS.add("IBM850");
        PUBLIC_IDENTIFIERS.add("IBM852");
        PUBLIC_IDENTIFIERS.add("IBM855");
        PUBLIC_IDENTIFIERS.add("IBM856");
        PUBLIC_IDENTIFIERS.add("IBM857");
        PUBLIC_IDENTIFIERS.add("IBM858");
        PUBLIC_IDENTIFIERS.add("IBM860");
        PUBLIC_IDENTIFIERS.add("IBM861");
        PUBLIC_IDENTIFIERS.add("IBM862");
        PUBLIC_IDENTIFIERS.add("IBM863");
        PUBLIC_IDENTIFIERS.add("IBM864");
        PUBLIC_IDENTIFIERS.add("IBM865");
        PUBLIC_IDENTIFIERS.add("IBM866");
        PUBLIC_IDENTIFIERS.add("IBM868");
        PUBLIC_IDENTIFIERS.add("IBM869");
        PUBLIC_IDENTIFIERS.add("IBM870");
        PUBLIC_IDENTIFIERS.add("IBM871");
        PUBLIC_IDENTIFIERS.add("IBM874");
        PUBLIC_IDENTIFIERS.add("IBM875");
        PUBLIC_IDENTIFIERS.add("IBM918");
        PUBLIC_IDENTIFIERS.add("IBM921");
        PUBLIC_IDENTIFIERS.add("IBM922");
        PUBLIC_IDENTIFIERS.add("IBM930");
        PUBLIC_IDENTIFIERS.add("IBM933");
        PUBLIC_IDENTIFIERS.add("IBM935");
        PUBLIC_IDENTIFIERS.add("IBM937");
        PUBLIC_IDENTIFIERS.add("IBM939");
        PUBLIC_IDENTIFIERS.add("IBM942");
        PUBLIC_IDENTIFIERS.add("IBM942C");
        PUBLIC_IDENTIFIERS.add("IBM943");
        PUBLIC_IDENTIFIERS.add("IBM943C");
        PUBLIC_IDENTIFIERS.add("IBM948");
        PUBLIC_IDENTIFIERS.add("IBM949");
        PUBLIC_IDENTIFIERS.add("IBM949C");
        PUBLIC_IDENTIFIERS.add("IBM950");
        PUBLIC_IDENTIFIERS.add("IBM964");
        PUBLIC_IDENTIFIERS.add("IBM970");
        PUBLIC_IDENTIFIERS.add("ICC_ColorSpace");
        PUBLIC_IDENTIFIERS.add("ICC_Profile");
        PUBLIC_IDENTIFIERS.add("ICC_ProfileAccessor");
        PUBLIC_IDENTIFIERS.add("Icon");
        PUBLIC_IDENTIFIERS.add("IconAccessibleContext");
        PUBLIC_IDENTIFIERS.add("IconButton");
        PUBLIC_IDENTIFIERS.add("IconCanvas");
        PUBLIC_IDENTIFIERS.add("IconElementInfo");
        PUBLIC_IDENTIFIERS.add("IconifyAction");
        PUBLIC_IDENTIFIERS.add("IconInfo");
        PUBLIC_IDENTIFIERS.add("IconLabel");
        PUBLIC_IDENTIFIERS.add("IconManager");
        PUBLIC_IDENTIFIERS.add("ICONST");
        PUBLIC_IDENTIFIERS.add("IconUIResource");
        PUBLIC_IDENTIFIERS.add("IconView");
        PUBLIC_IDENTIFIERS.add("IDDatatypeValidator");
        PUBLIC_IDENTIFIERS.add("IDDV");
        PUBLIC_IDENTIFIERS.add("Identifier");
        PUBLIC_IDENTIFIERS.add("IdentifierTree");
        PUBLIC_IDENTIFIERS.add("Identity");
        PUBLIC_IDENTIFIERS.add("IdentityArrayList");
        PUBLIC_IDENTIFIERS.add("IdentityConstraint");
        PUBLIC_IDENTIFIERS.add("IdentityHashMap");
        PUBLIC_IDENTIFIERS.add("IdentityLinkedList");
        PUBLIC_IDENTIFIERS.add("IdentityScope");
        PUBLIC_IDENTIFIERS.add("IDIV");
        PUBLIC_IDENTIFIERS.add("IDREFDatatypeValidator");
        PUBLIC_IDENTIFIERS.add("IDREFDV");
        PUBLIC_IDENTIFIERS.add("IF_ACMPEQ");
        PUBLIC_IDENTIFIERS.add("IF_ACMPNE");
        PUBLIC_IDENTIFIERS.add("IF_ICMPEQ");
        PUBLIC_IDENTIFIERS.add("IF_ICMPGE");
        PUBLIC_IDENTIFIERS.add("IF_ICMPGT");
        PUBLIC_IDENTIFIERS.add("IF_ICMPLE");
        PUBLIC_IDENTIFIERS.add("IF_ICMPLT");
        PUBLIC_IDENTIFIERS.add("IF_ICMPNE");
        PUBLIC_IDENTIFIERS.add("IFEQ");
        PUBLIC_IDENTIFIERS.add("IFGE");
        PUBLIC_IDENTIFIERS.add("IFGT");
        PUBLIC_IDENTIFIERS.add("IfInstruction");
        PUBLIC_IDENTIFIERS.add("IFLE");
        PUBLIC_IDENTIFIERS.add("IFLT");
        PUBLIC_IDENTIFIERS.add("IFNE");
        PUBLIC_IDENTIFIERS.add("IFNONNULL");
        PUBLIC_IDENTIFIERS.add("IFNULL");
        PUBLIC_IDENTIFIERS.add("IfTree");
        PUBLIC_IDENTIFIERS.add("IgnoreAllErrorHandler");
        PUBLIC_IDENTIFIERS.add("IgnoreBadWindowHandler");
        PUBLIC_IDENTIFIERS.add("IgnorePaintEvent");
        PUBLIC_IDENTIFIERS.add("IINC");
        PUBLIC_IDENTIFIERS.add("IincInsnNode");
        PUBLIC_IDENTIFIERS.add("IIOByteBuffer");
        PUBLIC_IDENTIFIERS.add("IIOException");
        PUBLIC_IDENTIFIERS.add("IIOImage");
        PUBLIC_IDENTIFIERS.add("IIOInvalidTreeException");
        PUBLIC_IDENTIFIERS.add("IIOMetadata");
        PUBLIC_IDENTIFIERS.add("IIOMetadataController");
        PUBLIC_IDENTIFIERS.add("IIOMetadataFormat");
        PUBLIC_IDENTIFIERS.add("IIOMetadataFormatImpl");
        PUBLIC_IDENTIFIERS.add("IIOMetadataNode");
        PUBLIC_IDENTIFIERS.add("IIOParam");
        PUBLIC_IDENTIFIERS.add("IIOParamController");
        PUBLIC_IDENTIFIERS.add("IIOReadProgressListener");
        PUBLIC_IDENTIFIERS.add("IIOReadUpdateListener");
        PUBLIC_IDENTIFIERS.add("IIOReadWarningListener");
        PUBLIC_IDENTIFIERS.add("IIOServiceProvider");
        PUBLIC_IDENTIFIERS.add("IIOWriteProgressListener");
        PUBLIC_IDENTIFIERS.add("IIOWriteWarningListener");
        PUBLIC_IDENTIFIERS.add("IllegalAccessError");
        PUBLIC_IDENTIFIERS.add("IllegalAccessException");
        PUBLIC_IDENTIFIERS.add("IllegalArgumentException");
        PUBLIC_IDENTIFIERS.add("IllegalBlockingModeException");
        PUBLIC_IDENTIFIERS.add("IllegalBlockSizeException");
        PUBLIC_IDENTIFIERS.add("IllegalCallerException");
        PUBLIC_IDENTIFIERS.add("IllegalChannelGroupException");
        PUBLIC_IDENTIFIERS.add("IllegalCharsetNameException");
        PUBLIC_IDENTIFIERS.add("IllegalClassFormatException");
        PUBLIC_IDENTIFIERS.add("IllegalComponentStateException");
        PUBLIC_IDENTIFIERS.add("IllegalConnectorArgumentsException");
        PUBLIC_IDENTIFIERS.add("IllegalFormatCodePointException");
        PUBLIC_IDENTIFIERS.add("IllegalFormatConversionException");
        PUBLIC_IDENTIFIERS.add("IllegalFormatException");
        PUBLIC_IDENTIFIERS.add("IllegalFormatFlagsException");
        PUBLIC_IDENTIFIERS.add("IllegalFormatPrecisionException");
        PUBLIC_IDENTIFIERS.add("IllegalFormatWidthException");
        PUBLIC_IDENTIFIERS.add("IllegalMonitorStateException");
        PUBLIC_IDENTIFIERS.add("IllegalPathStateException");
        PUBLIC_IDENTIFIERS.add("IllegalReceiveException");
        PUBLIC_IDENTIFIERS.add("IllegalSelectorException");
        PUBLIC_IDENTIFIERS.add("IllegalStateException");
        PUBLIC_IDENTIFIERS.add("IllegalThreadException");
        PUBLIC_IDENTIFIERS.add("IllegalThreadStateException");
        PUBLIC_IDENTIFIERS.add("IllegalUnbindException");
        PUBLIC_IDENTIFIERS.add("IllformedLocaleException");
        PUBLIC_IDENTIFIERS.add("ILOAD");
        PUBLIC_IDENTIFIERS.add("Image");
        PUBLIC_IDENTIFIERS.add("ImageAccessException");
        PUBLIC_IDENTIFIERS.add("ImageAccessor");
        PUBLIC_IDENTIFIERS.add("ImageBuilder");
        PUBLIC_IDENTIFIERS.add("ImageCache");
        PUBLIC_IDENTIFIERS.add("ImageCapabilities");
        PUBLIC_IDENTIFIERS.add("ImageConsumer");
        PUBLIC_IDENTIFIERS.add("ImageDecoder");
        PUBLIC_IDENTIFIERS.add("ImageFetchable");
        PUBLIC_IDENTIFIERS.add("ImageFilter");
        PUBLIC_IDENTIFIERS.add("ImageFormatException");
        PUBLIC_IDENTIFIERS.add("ImageIcon");
        PUBLIC_IDENTIFIERS.add("ImageIconUIResource");
        PUBLIC_IDENTIFIERS.add("ImageInputStream");
        PUBLIC_IDENTIFIERS.add("ImageInputStreamImpl");
        PUBLIC_IDENTIFIERS.add("ImageInputStreamSpi");
        PUBLIC_IDENTIFIERS.add("ImageLayoutException");
        PUBLIC_IDENTIFIERS.add("ImageLocation");
        PUBLIC_IDENTIFIERS.add("ImageObserver");
        PUBLIC_IDENTIFIERS.add("ImageOutputStream");
        PUBLIC_IDENTIFIERS.add("ImageOutputStreamImpl");
        PUBLIC_IDENTIFIERS.add("ImageOutputStreamSpi");
        PUBLIC_IDENTIFIERS.add("ImageProducer");
        PUBLIC_IDENTIFIERS.add("ImageProvider");
        PUBLIC_IDENTIFIERS.add("ImageReader");
        PUBLIC_IDENTIFIERS.add("ImageReaderFactory");
        PUBLIC_IDENTIFIERS.add("ImageReaderSpi");
        PUBLIC_IDENTIFIERS.add("ImageReaderWriterSpi");
        PUBLIC_IDENTIFIERS.add("ImageReadParam");
        PUBLIC_IDENTIFIERS.add("ImageRepresentation");
        PUBLIC_IDENTIFIERS.add("ImageStream");
        PUBLIC_IDENTIFIERS.add("ImageStrings");
        PUBLIC_IDENTIFIERS.add("ImageStringsReader");
        PUBLIC_IDENTIFIERS.add("ImageTranscoder");
        PUBLIC_IDENTIFIERS.add("ImageTranscoderSpi");
        PUBLIC_IDENTIFIERS.add("ImageTypeSpecifier");
        PUBLIC_IDENTIFIERS.add("ImageUtil");
        PUBLIC_IDENTIFIERS.add("ImageView");
        PUBLIC_IDENTIFIERS.add("ImageWaiter");
        PUBLIC_IDENTIFIERS.add("ImageWatched");
        PUBLIC_IDENTIFIERS.add("ImageWriteParam");
        PUBLIC_IDENTIFIERS.add("ImageWriter");
        PUBLIC_IDENTIFIERS.add("ImageWriterSpi");
        PUBLIC_IDENTIFIERS.add("ImagingLib");
        PUBLIC_IDENTIFIERS.add("ImagingOpException");
        PUBLIC_IDENTIFIERS.add("ImmediateOrRegister");
        PUBLIC_IDENTIFIERS.add("Immutable");
        PUBLIC_IDENTIFIERS.add("ImmutableDescriptor");
        PUBLIC_IDENTIFIERS.add("ImmutableExtendedSSLSession");
        PUBLIC_IDENTIFIERS.add("ImmutableIntegerModuloP");
        PUBLIC_IDENTIFIERS.add("ImmutableOopMap");
        PUBLIC_IDENTIFIERS.add("ImmutableOopMapPair");
        PUBLIC_IDENTIFIERS.add("ImmutableOopMapSet");
        PUBLIC_IDENTIFIERS.add("ImmutablePoint");
        PUBLIC_IDENTIFIERS.add("ImmutableSSLSession");
        PUBLIC_IDENTIFIERS.add("IMPDEP1");
        PUBLIC_IDENTIFIERS.add("IMPDEP2");
        PUBLIC_IDENTIFIERS.add("ImportFilter");
        PUBLIC_IDENTIFIERS.add("ImportScope");
        PUBLIC_IDENTIFIERS.add("ImportSnippet");
        PUBLIC_IDENTIFIERS.add("ImportTree");
        PUBLIC_IDENTIFIERS.add("IMUL");
        PUBLIC_IDENTIFIERS.add("InaccessibleObjectException");
        PUBLIC_IDENTIFIERS.add("InapplicableMethodException");
        PUBLIC_IDENTIFIERS.add("InclusiveNamespaces");
        PUBLIC_IDENTIFIERS.add("IncompatibleClassChangeError");
        PUBLIC_IDENTIFIERS.add("IncompatibleThreadStateException");
        PUBLIC_IDENTIFIERS.add("IncompleteAnnotationException");
        PUBLIC_IDENTIFIERS.add("InconsistentDebugInfoException");
        PUBLIC_IDENTIFIERS.add("IncorporationAction");
        PUBLIC_IDENTIFIERS.add("IncrementalSAXSource");
        PUBLIC_IDENTIFIERS.add("IncrementalSAXSource_Xerces");
        PUBLIC_IDENTIFIERS.add("IndentPrinter");
        PUBLIC_IDENTIFIERS.add("IndexableFieldIdentifier");
        PUBLIC_IDENTIFIERS.add("IndexBuilder");
        PUBLIC_IDENTIFIERS.add("IndexColorModel");
        PUBLIC_IDENTIFIERS.add("IndexedCollectionCertStore");
        PUBLIC_IDENTIFIERS.add("IndexedInstruction");
        PUBLIC_IDENTIFIERS.add("IndexedPropertyChangeEvent");
        PUBLIC_IDENTIFIERS.add("IndexedPropertyDescriptor");
        PUBLIC_IDENTIFIERS.add("IndexItem");
        PUBLIC_IDENTIFIERS.add("IndexOperation");
        PUBLIC_IDENTIFIERS.add("IndexOutOfBoundsException");
        PUBLIC_IDENTIFIERS.add("IndexRedirectWriter");
        PUBLIC_IDENTIFIERS.add("IndexTaglet");
        PUBLIC_IDENTIFIERS.add("IndexTree");
        PUBLIC_IDENTIFIERS.add("IndexWriter");
        PUBLIC_IDENTIFIERS.add("INEG");
        PUBLIC_IDENTIFIERS.add("InetAddress");
        PUBLIC_IDENTIFIERS.add("InetAddressConverter");
        PUBLIC_IDENTIFIERS.add("InetAddressResolver");
        PUBLIC_IDENTIFIERS.add("InetAddressResolverProvider");
        PUBLIC_IDENTIFIERS.add("InetSocketAddress");
        PUBLIC_IDENTIFIERS.add("Infer");
        PUBLIC_IDENTIFIERS.add("InferenceBound");
        PUBLIC_IDENTIFIERS.add("InferenceContext");
        PUBLIC_IDENTIFIERS.add("InferenceException");
        PUBLIC_IDENTIFIERS.add("Inflater");
        PUBLIC_IDENTIFIERS.add("InflaterInputStream");
        PUBLIC_IDENTIFIERS.add("InflaterOutputStream");
        PUBLIC_IDENTIFIERS.add("Info");
        PUBLIC_IDENTIFIERS.add("Infopoint");
        PUBLIC_IDENTIFIERS.add("InfoWindow");
        PUBLIC_IDENTIFIERS.add("InheritableTaglet");
        PUBLIC_IDENTIFIERS.add("InheritableThreadLocal");
        PUBLIC_IDENTIFIERS.add("InheritDocTaglet");
        PUBLIC_IDENTIFIERS.add("InheritDocTree");
        PUBLIC_IDENTIFIERS.add("Inherited");
        PUBLIC_IDENTIFIERS.add("InheritedDatagramChannelImpl");
        PUBLIC_IDENTIFIERS.add("InheritedServerSocketChannelImpl");
        PUBLIC_IDENTIFIERS.add("InheritedSocketChannelImpl");
        PUBLIC_IDENTIFIERS.add("InhibitAnyPolicyExtension");
        PUBLIC_IDENTIFIERS.add("Init");
        PUBLIC_IDENTIFIERS.add("InitialContext");
        PUBLIC_IDENTIFIERS.add("InitialContextFactory");
        PUBLIC_IDENTIFIERS.add("InitialContextFactoryBuilder");
        PUBLIC_IDENTIFIERS.add("InitialDirContext");
        PUBLIC_IDENTIFIERS.add("InitialFileParser");
        PUBLIC_IDENTIFIERS.add("InitialFileParserIntf");
        PUBLIC_IDENTIFIERS.add("InitialLdapContext");
        PUBLIC_IDENTIFIERS.add("InitMaxStreams");
        PUBLIC_IDENTIFIERS.add("InlineTagTree");
        PUBLIC_IDENTIFIERS.add("InlineTree");
        PUBLIC_IDENTIFIERS.add("InlineView");
        PUBLIC_IDENTIFIERS.add("InnerClasses_attribute");
        PUBLIC_IDENTIFIERS.add("InnerClassNode");
        PUBLIC_IDENTIFIERS.add("Input");
        PUBLIC_IDENTIFIERS.add("INPUT_STREAM");
        PUBLIC_IDENTIFIERS.add("InputContext");
        PUBLIC_IDENTIFIERS.add("InputEvent");
        PUBLIC_IDENTIFIERS.add("InputEventAccessor");
        PUBLIC_IDENTIFIERS.add("InputLexer");
        PUBLIC_IDENTIFIERS.add("InputMap");
        PUBLIC_IDENTIFIERS.add("InputMapUIResource");
        PUBLIC_IDENTIFIERS.add("InputMethod");
        PUBLIC_IDENTIFIERS.add("InputMethodAdapter");
        PUBLIC_IDENTIFIERS.add("InputMethodContext");
        PUBLIC_IDENTIFIERS.add("InputMethodDescriptor");
        PUBLIC_IDENTIFIERS.add("InputMethodEvent");
        PUBLIC_IDENTIFIERS.add("InputMethodHighlight");
        PUBLIC_IDENTIFIERS.add("InputMethodJFrame");
        PUBLIC_IDENTIFIERS.add("InputMethodListener");
        PUBLIC_IDENTIFIERS.add("InputMethodManager");
        PUBLIC_IDENTIFIERS.add("InputMethodRequests");
        PUBLIC_IDENTIFIERS.add("InputMethodSupport");
        PUBLIC_IDENTIFIERS.add("InputMethodWindow");
        PUBLIC_IDENTIFIERS.add("InputMismatchException");
        PUBLIC_IDENTIFIERS.add("InputSource");
        PUBLIC_IDENTIFIERS.add("InputStream");
        PUBLIC_IDENTIFIERS.add("InputStreamAdapter");
        PUBLIC_IDENTIFIERS.add("InputStreamImageInputStreamSpi");
        PUBLIC_IDENTIFIERS.add("InputStreamImageSource");
        PUBLIC_IDENTIFIERS.add("InputStreamPublisher");
        PUBLIC_IDENTIFIERS.add("InputStreamReader");
        PUBLIC_IDENTIFIERS.add("InputVerifier");
        PUBLIC_IDENTIFIERS.add("InsertBreakAction");
        PUBLIC_IDENTIFIERS.add("InsertContentAction");
        PUBLIC_IDENTIFIERS.add("InsertHTMLTextAction");
        PUBLIC_IDENTIFIERS.add("InsertRow");
        PUBLIC_IDENTIFIERS.add("InsertTabAction");
        PUBLIC_IDENTIFIERS.add("Insets");
        PUBLIC_IDENTIFIERS.add("InsetsUIResource");
        PUBLIC_IDENTIFIERS.add("insideTimerAction");
        PUBLIC_IDENTIFIERS.add("InsnList");
        PUBLIC_IDENTIFIERS.add("InsnNode");
        PUBLIC_IDENTIFIERS.add("InspectAction");
        PUBLIC_IDENTIFIERS.add("InspectedFrame");
        PUBLIC_IDENTIFIERS.add("InspectedFrameVisitor");
        PUBLIC_IDENTIFIERS.add("Inspector");
        PUBLIC_IDENTIFIERS.add("InstalledCode");
        PUBLIC_IDENTIFIERS.add("Instance");
        PUBLIC_IDENTIFIERS.add("InstanceAlreadyExistsException");
        PUBLIC_IDENTIFIERS.add("InstanceClassLoaderKlass");
        PUBLIC_IDENTIFIERS.add("InstanceConstructor");
        PUBLIC_IDENTIFIERS.add("InstanceKlass");
        PUBLIC_IDENTIFIERS.add("InstanceMirrorKlass");
        PUBLIC_IDENTIFIERS.add("InstanceNotFoundException");
        PUBLIC_IDENTIFIERS.add("INSTANCEOF");
        PUBLIC_IDENTIFIERS.add("InstanceOfTree");
        PUBLIC_IDENTIFIERS.add("InstanceRefKlass");
        PUBLIC_IDENTIFIERS.add("InstanceStackChunkKlass");
        PUBLIC_IDENTIFIERS.add("InstantEvent");
        PUBLIC_IDENTIFIERS.add("InstantiationError");
        PUBLIC_IDENTIFIERS.add("InstantiationException");
        PUBLIC_IDENTIFIERS.add("InstantSource");
        PUBLIC_IDENTIFIERS.add("Instruction");
        PUBLIC_IDENTIFIERS.add("InstructionAdapter");
        PUBLIC_IDENTIFIERS.add("InstructionComparator");
        PUBLIC_IDENTIFIERS.add("InstructionDetailWriter");
        PUBLIC_IDENTIFIERS.add("InstructionFactory");
        PUBLIC_IDENTIFIERS.add("InstructionFinder");
        PUBLIC_IDENTIFIERS.add("InstructionHandle");
        PUBLIC_IDENTIFIERS.add("InstructionList");
        PUBLIC_IDENTIFIERS.add("InstructionListObserver");
        PUBLIC_IDENTIFIERS.add("InstructionTargeter");
        PUBLIC_IDENTIFIERS.add("InstructionVisitor");
        PUBLIC_IDENTIFIERS.add("Instrument");
        PUBLIC_IDENTIFIERS.add("Instrumentation");
        PUBLIC_IDENTIFIERS.add("InstrumentationException");
        PUBLIC_IDENTIFIERS.add("InstrumentationImpl");
        PUBLIC_IDENTIFIERS.add("InsufficientResourcesException");
        PUBLIC_IDENTIFIERS.add("IntArray");
        PUBLIC_IDENTIFIERS.add("IntArrayWrapper");
        PUBLIC_IDENTIFIERS.add("IntBinaryOperator");
        PUBLIC_IDENTIFIERS.add("IntBuffer");
        PUBLIC_IDENTIFIERS.add("IntConsumer");
        PUBLIC_IDENTIFIERS.add("IntData");
        PUBLIC_IDENTIFIERS.add("Integer");
        PUBLIC_IDENTIFIERS.add("IntegerArgument");
        PUBLIC_IDENTIFIERS.add("IntegerComponentRaster");
        PUBLIC_IDENTIFIERS.add("IntegerDV");
        PUBLIC_IDENTIFIERS.add("IntegerEditor");
        PUBLIC_IDENTIFIERS.add("IntegerEnum");
        PUBLIC_IDENTIFIERS.add("IntegerFieldModuloP");
        PUBLIC_IDENTIFIERS.add("IntegerInterleavedRaster");
        PUBLIC_IDENTIFIERS.add("IntegerModuloP");
        PUBLIC_IDENTIFIERS.add("IntegerMonitor");
        PUBLIC_IDENTIFIERS.add("IntegerPolynomial");
        PUBLIC_IDENTIFIERS.add("IntegerPolynomialModBinP");
        PUBLIC_IDENTIFIERS.add("IntegerSyntax");
        PUBLIC_IDENTIFIERS.add("IntegerType");
        PUBLIC_IDENTIFIERS.add("IntegerTypeImpl");
        PUBLIC_IDENTIFIERS.add("IntegerValue");
        PUBLIC_IDENTIFIERS.add("IntegerValueImpl");
        PUBLIC_IDENTIFIERS.add("IntegratedParserConfiguration");
        PUBLIC_IDENTIFIERS.add("IntegrityHmac");
        PUBLIC_IDENTIFIERS.add("IntegrityHmacMD5");
        PUBLIC_IDENTIFIERS.add("IntegrityHmacRIPEMD160");
        PUBLIC_IDENTIFIERS.add("IntegrityHmacSHA1");
        PUBLIC_IDENTIFIERS.add("IntegrityHmacSHA224");
        PUBLIC_IDENTIFIERS.add("IntegrityHmacSHA256");
        PUBLIC_IDENTIFIERS.add("IntegrityHmacSHA384");
        PUBLIC_IDENTIFIERS.add("IntegrityHmacSHA512");
        PUBLIC_IDENTIFIERS.add("InterfaceAddress");
        PUBLIC_IDENTIFIERS.add("InterfaceType");
        PUBLIC_IDENTIFIERS.add("InternalAxisIteratorBase");
        PUBLIC_IDENTIFIERS.add("InternalDebugControl");
        PUBLIC_IDENTIFIERS.add("InternalDialog");
        PUBLIC_IDENTIFIERS.add("InternalEntity");
        PUBLIC_IDENTIFIERS.add("InternalError");
        PUBLIC_IDENTIFIERS.add("InternalEventHandler");
        PUBLIC_IDENTIFIERS.add("InternalException");
        PUBLIC_IDENTIFIERS.add("InternalFrameAdapter");
        PUBLIC_IDENTIFIERS.add("InternalFrameBorder");
        PUBLIC_IDENTIFIERS.add("InternalFrameEvent");
        PUBLIC_IDENTIFIERS.add("InternalFrameFocusTraversalPolicy");
        PUBLIC_IDENTIFIERS.add("InternalFrameLayout");
        PUBLIC_IDENTIFIERS.add("InternalFrameListener");
        PUBLIC_IDENTIFIERS.add("InternalFramePropertyChangeListener");
        PUBLIC_IDENTIFIERS.add("InternalFrameUI");
        PUBLIC_IDENTIFIERS.add("InternalLock");
        PUBLIC_IDENTIFIERS.add("InternalRuntimeError");
        PUBLIC_IDENTIFIERS.add("InternalTypeUtilities");
        PUBLIC_IDENTIFIERS.add("InternationalFormatter");
        PUBLIC_IDENTIFIERS.add("InteropProvider");
        PUBLIC_IDENTIFIERS.add("InteropProviderImpl");
        PUBLIC_IDENTIFIERS.add("InterpretedVFrame");
        PUBLIC_IDENTIFIERS.add("Interpreter");
        PUBLIC_IDENTIFIERS.add("InterpreterCodelet");
        PUBLIC_IDENTIFIERS.add("InterruptedByTimeoutException");
        PUBLIC_IDENTIFIERS.add("InterruptedException");
        PUBLIC_IDENTIFIERS.add("InterruptedIOException");
        PUBLIC_IDENTIFIERS.add("InterruptedNamingException");
        PUBLIC_IDENTIFIERS.add("Interruptible");
        PUBLIC_IDENTIFIERS.add("InterruptibleChannel");
        PUBLIC_IDENTIFIERS.add("IntersectionClassType");
        PUBLIC_IDENTIFIERS.add("IntersectionType");
        PUBLIC_IDENTIFIERS.add("IntersectionTypeTree");
        PUBLIC_IDENTIFIERS.add("Interval");
        PUBLIC_IDENTIFIERS.add("IntervalNode");
        PUBLIC_IDENTIFIERS.add("IntervalTree");
        PUBLIC_IDENTIFIERS.add("IntField");
        PUBLIC_IDENTIFIERS.add("IntFunction");
        PUBLIC_IDENTIFIERS.add("IntHashTable");
        PUBLIC_IDENTIFIERS.add("IntInsnNode");
        PUBLIC_IDENTIFIERS.add("IntMapMultiConsumer");
        PUBLIC_IDENTIFIERS.add("IntOp");
        PUBLIC_IDENTIFIERS.add("IntPredicate");
        PUBLIC_IDENTIFIERS.add("IntrinsicCandidate");
        PUBLIC_IDENTIFIERS.add("IntrospectionException");
        PUBLIC_IDENTIFIERS.add("Introspector");
        PUBLIC_IDENTIFIERS.add("IntSignature");
        PUBLIC_IDENTIFIERS.add("IntStack");
        PUBLIC_IDENTIFIERS.add("IntStream");
        PUBLIC_IDENTIFIERS.add("IntSummaryStatistics");
        PUBLIC_IDENTIFIERS.add("IntSupplier");
        PUBLIC_IDENTIFIERS.add("IntToDoubleFunction");
        PUBLIC_IDENTIFIERS.add("IntToLongFunction");
        PUBLIC_IDENTIFIERS.add("IntType");
        PUBLIC_IDENTIFIERS.add("IntUnaryOperator");
        PUBLIC_IDENTIFIERS.add("IntVector");
        PUBLIC_IDENTIFIERS.add("InvalidAlgorithmParameterException");
        PUBLIC_IDENTIFIERS.add("InvalidApplicationException");
        PUBLIC_IDENTIFIERS.add("InvalidAttributeIdentifierException");
        PUBLIC_IDENTIFIERS.add("InvalidAttributesException");
        PUBLIC_IDENTIFIERS.add("InvalidAttributeValueException");
        PUBLIC_IDENTIFIERS.add("InvalidCanonicalizerException");
        PUBLIC_IDENTIFIERS.add("InvalidClassException");
        PUBLIC_IDENTIFIERS.add("InvalidCodeIndexException");
        PUBLIC_IDENTIFIERS.add("InvalidDataException");
        PUBLIC_IDENTIFIERS.add("InvalidDatatypeFacetException");
        PUBLIC_IDENTIFIERS.add("InvalidDatatypeValueException");
        PUBLIC_IDENTIFIERS.add("InvalidDescriptor");
        PUBLIC_IDENTIFIERS.add("InvalidDigestValueException");
        PUBLIC_IDENTIFIERS.add("InvalidDnDOperationException");
        PUBLIC_IDENTIFIERS.add("InvalidEntry");
        PUBLIC_IDENTIFIERS.add("InvalidFormatException");
        PUBLIC_IDENTIFIERS.add("InvalidIndex");
        PUBLIC_IDENTIFIERS.add("InvalidityDateExtension");
        PUBLIC_IDENTIFIERS.add("InvalidJarIndexError");
        PUBLIC_IDENTIFIERS.add("InvalidKeyException");
        PUBLIC_IDENTIFIERS.add("InvalidKeySpecException");
        PUBLIC_IDENTIFIERS.add("InvalidLineNumberException");
        PUBLIC_IDENTIFIERS.add("InvalidMarkException");
        PUBLIC_IDENTIFIERS.add("InvalidMidiDataException");
        PUBLIC_IDENTIFIERS.add("InvalidModuleDescriptorException");
        PUBLIC_IDENTIFIERS.add("InvalidModuleException");
        PUBLIC_IDENTIFIERS.add("InvalidNameException");
        PUBLIC_IDENTIFIERS.add("InvalidObjectException");
        PUBLIC_IDENTIFIERS.add("InvalidOpenTypeException");
        PUBLIC_IDENTIFIERS.add("InvalidParameterException");
        PUBLIC_IDENTIFIERS.add("InvalidParameterSpecException");
        PUBLIC_IDENTIFIERS.add("InvalidPathException");
        PUBLIC_IDENTIFIERS.add("InvalidPipeException");
        PUBLIC_IDENTIFIERS.add("InvalidPreferencesFormatException");
        PUBLIC_IDENTIFIERS.add("InvalidPropertiesFormatException");
        PUBLIC_IDENTIFIERS.add("InvalidRelationIdException");
        PUBLIC_IDENTIFIERS.add("InvalidRelationServiceException");
        PUBLIC_IDENTIFIERS.add("InvalidRelationTypeException");
        PUBLIC_IDENTIFIERS.add("InvalidRequestStateException");
        PUBLIC_IDENTIFIERS.add("InvalidRoleInfoException");
        PUBLIC_IDENTIFIERS.add("InvalidRoleValueException");
        PUBLIC_IDENTIFIERS.add("InvalidSearchControlsException");
        PUBLIC_IDENTIFIERS.add("InvalidSearchFilterException");
        PUBLIC_IDENTIFIERS.add("InvalidSignatureException");
        PUBLIC_IDENTIFIERS.add("InvalidSignatureValueException");
        PUBLIC_IDENTIFIERS.add("InvalidStackFrameException");
        PUBLIC_IDENTIFIERS.add("InvalidStreamException");
        PUBLIC_IDENTIFIERS.add("InvalidTargetObjectTypeException");
        PUBLIC_IDENTIFIERS.add("InvalidTransformException");
        PUBLIC_IDENTIFIERS.add("InvalidTypeException");
        PUBLIC_IDENTIFIERS.add("InvalidValueException");
        PUBLIC_IDENTIFIERS.add("InverseDepsAnalyzer");
        PUBLIC_IDENTIFIERS.add("Invocable");
        PUBLIC_IDENTIFIERS.add("InvocationEvent");
        PUBLIC_IDENTIFIERS.add("InvocationEventAccessor");
        PUBLIC_IDENTIFIERS.add("InvocationException");
        PUBLIC_IDENTIFIERS.add("InvocationHandler");
        PUBLIC_IDENTIFIERS.add("InvocationKeyHandler");
        PUBLIC_IDENTIFIERS.add("InvocationMouseHandler");
        PUBLIC_IDENTIFIERS.add("InvocationMouseMotionHandler");
        PUBLIC_IDENTIFIERS.add("InvocationTargetException");
        PUBLIC_IDENTIFIERS.add("INVOKEDYNAMIC");
        PUBLIC_IDENTIFIERS.add("InvokeDynamicInsnNode");
        PUBLIC_IDENTIFIERS.add("InvokeInstruction");
        PUBLIC_IDENTIFIERS.add("Invoker");
        PUBLIC_IDENTIFIERS.add("INVOKESPECIAL");
        PUBLIC_IDENTIFIERS.add("INVOKESTATIC");
        PUBLIC_IDENTIFIERS.add("InvokeTarget");
        PUBLIC_IDENTIFIERS.add("INVOKEVIRTUAL");
        PUBLIC_IDENTIFIERS.add("IOError");
        PUBLIC_IDENTIFIERS.add("IOException");
        PUBLIC_IDENTIFIERS.add("IOR");
        PUBLIC_IDENTIFIERS.add("IOUtils");
        PUBLIC_IDENTIFIERS.add("IPAddressName");
        PUBLIC_IDENTIFIERS.add("IPAddressUtil");
        PUBLIC_IDENTIFIERS.add("IPPPrintService");
        PUBLIC_IDENTIFIERS.add("IREM");
        PUBLIC_IDENTIFIERS.add("IRETURN");
        PUBLIC_IDENTIFIERS.add("ISCII91");
        PUBLIC_IDENTIFIERS.add("ISHL");
        PUBLIC_IDENTIFIERS.add("ISHR");
        PUBLIC_IDENTIFIERS.add("IsindexAction");
        PUBLIC_IDENTIFIERS.add("ISO2022_CN");
        PUBLIC_IDENTIFIERS.add("ISO2022_CN_CNS");
        PUBLIC_IDENTIFIERS.add("ISO2022_CN_GB");
        PUBLIC_IDENTIFIERS.add("ISO2022_JP");
        PUBLIC_IDENTIFIERS.add("ISO2022_JP_2");
        PUBLIC_IDENTIFIERS.add("ISO2022_KR");
        PUBLIC_IDENTIFIERS.add("ISO_8859_1");
        PUBLIC_IDENTIFIERS.add("ISO_8859_11");
        PUBLIC_IDENTIFIERS.add("ISO_8859_13");
        PUBLIC_IDENTIFIERS.add("ISO_8859_15");
        PUBLIC_IDENTIFIERS.add("ISO_8859_16");
        PUBLIC_IDENTIFIERS.add("ISO_8859_2");
        PUBLIC_IDENTIFIERS.add("ISO_8859_3");
        PUBLIC_IDENTIFIERS.add("ISO_8859_4");
        PUBLIC_IDENTIFIERS.add("ISO_8859_5");
        PUBLIC_IDENTIFIERS.add("ISO_8859_6");
        PUBLIC_IDENTIFIERS.add("ISO_8859_7");
        PUBLIC_IDENTIFIERS.add("ISO_8859_8");
        PUBLIC_IDENTIFIERS.add("ISO_8859_9");
        PUBLIC_IDENTIFIERS.add("IsoCountryCode");
        PUBLIC_IDENTIFIERS.add("IssuerAlternativeNameExtension");
        PUBLIC_IDENTIFIERS.add("IssuingDistributionPointExtension");
        PUBLIC_IDENTIFIERS.add("ISTORE");
        PUBLIC_IDENTIFIERS.add("ISUB");
        PUBLIC_IDENTIFIERS.add("ItalicAction");
        PUBLIC_IDENTIFIERS.add("ItemEvent");
        PUBLIC_IDENTIFIERS.add("ItemHandler");
        PUBLIC_IDENTIFIERS.add("ItemListener");
        PUBLIC_IDENTIFIERS.add("ItemPSVI");
        PUBLIC_IDENTIFIERS.add("Items");
        PUBLIC_IDENTIFIERS.add("ItemSelectable");
        PUBLIC_IDENTIFIERS.add("Iterable");
        PUBLIC_IDENTIFIERS.add("IterablePublisher");
        PUBLIC_IDENTIFIERS.add("Iterator");
        PUBLIC_IDENTIFIERS.add("Iterators");
        PUBLIC_IDENTIFIERS.add("Itr");
        PUBLIC_IDENTIFIERS.add("IUSHR");
        PUBLIC_IDENTIFIERS.add("IvParameterSpec");
        PUBLIC_IDENTIFIERS.add("IXOR");
        PUBLIC_IDENTIFIERS.add("JansiSupport");
        PUBLIC_IDENTIFIERS.add("JApplet");
        PUBLIC_IDENTIFIERS.add("JarArchive");
        PUBLIC_IDENTIFIERS.add("JarConstraintsParameters");
        PUBLIC_IDENTIFIERS.add("JarEntry");
        PUBLIC_IDENTIFIERS.add("JarException");
        PUBLIC_IDENTIFIERS.add("JarFile");
        PUBLIC_IDENTIFIERS.add("JarIndex");
        PUBLIC_IDENTIFIERS.add("JarInputStream");
        PUBLIC_IDENTIFIERS.add("JarOutputStream");
        PUBLIC_IDENTIFIERS.add("JarSignerException");
        PUBLIC_IDENTIFIERS.add("JarToolProvider");
        PUBLIC_IDENTIFIERS.add("JarURLConnection");
        PUBLIC_IDENTIFIERS.add("java_lang_Class");
        PUBLIC_IDENTIFIERS.add("JavaAWTAccess");
        PUBLIC_IDENTIFIERS.add("JavaAWTFontAccess");
        PUBLIC_IDENTIFIERS.add("JavaBean");
        PUBLIC_IDENTIFIERS.add("JavaBeansAccess");
        PUBLIC_IDENTIFIERS.add("JavaCallWrapper");
        PUBLIC_IDENTIFIERS.add("JavacElements");
        PUBLIC_IDENTIFIERS.add("JavacFileManager");
        PUBLIC_IDENTIFIERS.add("JavacFiler");
        PUBLIC_IDENTIFIERS.add("JavaCharStream");
        PUBLIC_IDENTIFIERS.add("JavaClass");
        PUBLIC_IDENTIFIERS.add("JavacMessager");
        PUBLIC_IDENTIFIERS.add("JavacMessages");
        PUBLIC_IDENTIFIERS.add("JavaCompiler");
        PUBLIC_IDENTIFIERS.add("JavaConstant");
        PUBLIC_IDENTIFIERS.add("JavacParser");
        PUBLIC_IDENTIFIERS.add("JavacProcessingEnvironment");
        PUBLIC_IDENTIFIERS.add("JavacRoundEnvironment");
        PUBLIC_IDENTIFIERS.add("JavacScope");
        PUBLIC_IDENTIFIERS.add("JavacTask");
        PUBLIC_IDENTIFIERS.add("JavacTaskImpl");
        PUBLIC_IDENTIFIERS.add("JavacTaskPool");
        PUBLIC_IDENTIFIERS.add("JavacToolProvider");
        PUBLIC_IDENTIFIERS.add("JavacTrees");
        PUBLIC_IDENTIFIERS.add("JavacTypes");
        PUBLIC_IDENTIFIERS.add("JavadocClassFinder");
        PUBLIC_IDENTIFIERS.add("JavadocComment");
        PUBLIC_IDENTIFIERS.add("JavadocEnter");
        PUBLIC_IDENTIFIERS.add("JavadocFormatter");
        PUBLIC_IDENTIFIERS.add("JavadocHelper");
        PUBLIC_IDENTIFIERS.add("JavadocLog");
        PUBLIC_IDENTIFIERS.add("JavadocMemberEnter");
        PUBLIC_IDENTIFIERS.add("JavadocTaskImpl");
        PUBLIC_IDENTIFIERS.add("JavadocTodo");
        PUBLIC_IDENTIFIERS.add("JavadocTokenizer");
        PUBLIC_IDENTIFIERS.add("JavadocTool");
        PUBLIC_IDENTIFIERS.add("JavadocToolProvider");
        PUBLIC_IDENTIFIERS.add("JavaField");
        PUBLIC_IDENTIFIERS.add("JavaFileManager");
        PUBLIC_IDENTIFIERS.add("JavaFileObject");
        PUBLIC_IDENTIFIERS.add("JavaIOAccess");
        PUBLIC_IDENTIFIERS.add("JavaIOFileDescriptorAccess");
        PUBLIC_IDENTIFIERS.add("JavaIOFilePermissionAccess");
        PUBLIC_IDENTIFIERS.add("JavaIOPrintStreamAccess");
        PUBLIC_IDENTIFIERS.add("JavaIOPrintWriterAccess");
        PUBLIC_IDENTIFIERS.add("JavaIORandomAccessFileAccess");
        PUBLIC_IDENTIFIERS.add("JavaKeyStore");
        PUBLIC_IDENTIFIERS.add("JavaLangAccess");
        PUBLIC_IDENTIFIERS.add("JavaLangInvokeAccess");
        PUBLIC_IDENTIFIERS.add("JavaLangModuleAccess");
        PUBLIC_IDENTIFIERS.add("JavaLangRefAccess");
        PUBLIC_IDENTIFIERS.add("JavaLangReflectAccess");
        PUBLIC_IDENTIFIERS.add("JavaMethod");
        PUBLIC_IDENTIFIERS.add("JavaNetHttpCookieAccess");
        PUBLIC_IDENTIFIERS.add("JavaNetInetAddressAccess");
        PUBLIC_IDENTIFIERS.add("JavaNetUriAccess");
        PUBLIC_IDENTIFIERS.add("JavaNetURLAccess");
        PUBLIC_IDENTIFIERS.add("JavaNioAccess");
        PUBLIC_IDENTIFIERS.add("JavaObjectInputFilterAccess");
        PUBLIC_IDENTIFIERS.add("JavaObjectInputStreamAccess");
        PUBLIC_IDENTIFIERS.add("JavaObjectInputStreamReadString");
        PUBLIC_IDENTIFIERS.add("JavapFileManager");
        PUBLIC_IDENTIFIERS.add("JavapTask");
        PUBLIC_IDENTIFIERS.add("JavapToolProvider");
        PUBLIC_IDENTIFIERS.add("JavaRuntimeURLConnection");
        PUBLIC_IDENTIFIERS.add("JavaScriptScanner");
        PUBLIC_IDENTIFIERS.add("JavaSecurityAccess");
        PUBLIC_IDENTIFIERS.add("JavaSecurityPropertiesAccess");
        PUBLIC_IDENTIFIERS.add("JavaSecuritySignatureAccess");
        PUBLIC_IDENTIFIERS.add("JavaSecuritySpecAccess");
        PUBLIC_IDENTIFIERS.add("JavaShellToolBuilder");
        PUBLIC_IDENTIFIERS.add("JavaStackTraceAction");
        PUBLIC_IDENTIFIERS.add("JavaStackTracePanel");
        PUBLIC_IDENTIFIERS.add("JavaThread");
        PUBLIC_IDENTIFIERS.add("JavaThreadFactory");
        PUBLIC_IDENTIFIERS.add("JavaThreadPDAccess");
        PUBLIC_IDENTIFIERS.add("JavaThreadsPanel");
        PUBLIC_IDENTIFIERS.add("JavaThreadState");
        PUBLIC_IDENTIFIERS.add("JavaTimeDateTimePatternImpl");
        PUBLIC_IDENTIFIERS.add("JavaTimeDateTimePatternProvider");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_ar");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_ar_JO");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_ar_LB");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_ar_SY");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_be");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_bg");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_ca");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_cs");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_da");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_de");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_de_AT");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_el");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_en");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_en_AU");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_en_CA");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_en_GB");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_en_IE");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_en_IN");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_en_MT");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_en_NZ");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_en_SG");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_en_ZA");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_es");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_es_AR");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_es_BO");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_es_CL");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_es_CO");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_es_CR");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_es_DO");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_es_EC");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_es_GT");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_es_HN");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_es_MX");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_es_PA");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_es_PE");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_es_PR");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_es_PY");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_es_US");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_es_UY");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_es_VE");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_et");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_fi");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_fr");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_fr_BE");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_fr_CA");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_fr_CH");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_ga");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_he");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_hi_IN");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_hr");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_hu");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_id");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_is");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_it");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_it_CH");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_ja");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_ko");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_lt");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_lv");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_mk");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_ms");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_mt");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_nl");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_nl_BE");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_no");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_pl");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_pt");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_pt_PT");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_ro");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_ru");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_sk");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_sl");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_sq");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_sr");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_sr_BA");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_sr_Latn");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_sv");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_th");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_tr");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_uk");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_vi");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_zh");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_zh_HK");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_zh_SG");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementary_zh_TW");
        PUBLIC_IDENTIFIERS.add("JavaTimeSupplementaryProvider");
        PUBLIC_IDENTIFIERS.add("JavaTokenizer");
        PUBLIC_IDENTIFIERS.add("JavaType");
        PUBLIC_IDENTIFIERS.add("JavaUtilCollectionAccess");
        PUBLIC_IDENTIFIERS.add("JavaUtilConcurrentFJPAccess");
        PUBLIC_IDENTIFIERS.add("JavaUtilConcurrentTLRAccess");
        PUBLIC_IDENTIFIERS.add("JavaUtilJarAccess");
        PUBLIC_IDENTIFIERS.add("JavaUtilResourceBundleAccess");
        PUBLIC_IDENTIFIERS.add("JavaUtilZipFileAccess");
        PUBLIC_IDENTIFIERS.add("JavaValue");
        PUBLIC_IDENTIFIERS.add("JavaVFrame");
        PUBLIC_IDENTIFIERS.add("JavaxCryptoSealedObjectAccess");
        PUBLIC_IDENTIFIERS.add("JavaxCryptoSpecAccess");
        PUBLIC_IDENTIFIERS.add("JavaxSecurityAuthKerberosAccess");
        PUBLIC_IDENTIFIERS.add("JAXPConstants");
        PUBLIC_IDENTIFIERS.add("JAXPExtensionsProvider");
        PUBLIC_IDENTIFIERS.add("JAXPPrefixResolver");
        PUBLIC_IDENTIFIERS.add("JAXPSAXParser");
        PUBLIC_IDENTIFIERS.add("JAXPVariableStack");
        PUBLIC_IDENTIFIERS.add("JBooleanField");
        PUBLIC_IDENTIFIERS.add("JButton");
        PUBLIC_IDENTIFIERS.add("JByteField");
        PUBLIC_IDENTIFIERS.add("JCAnnotatedType");
        PUBLIC_IDENTIFIERS.add("JCAnnotation");
        PUBLIC_IDENTIFIERS.add("JCArrayAccess");
        PUBLIC_IDENTIFIERS.add("JCArrayTypeTree");
        PUBLIC_IDENTIFIERS.add("JCAssert");
        PUBLIC_IDENTIFIERS.add("JCAssign");
        PUBLIC_IDENTIFIERS.add("JCAssignOp");
        PUBLIC_IDENTIFIERS.add("JCBinary");
        PUBLIC_IDENTIFIERS.add("JCBindingPattern");
        PUBLIC_IDENTIFIERS.add("JCBlock");
        PUBLIC_IDENTIFIERS.add("JCBreak");
        PUBLIC_IDENTIFIERS.add("JCCase");
        PUBLIC_IDENTIFIERS.add("JCCaseLabel");
        PUBLIC_IDENTIFIERS.add("JCCatch");
        PUBLIC_IDENTIFIERS.add("JCClassDecl");
        PUBLIC_IDENTIFIERS.add("JCCompilationUnit");
        PUBLIC_IDENTIFIERS.add("JCConditional");
        PUBLIC_IDENTIFIERS.add("JCConstantCaseLabel");
        PUBLIC_IDENTIFIERS.add("JCContinue");
        PUBLIC_IDENTIFIERS.add("JCDefaultCaseLabel");
        PUBLIC_IDENTIFIERS.add("JCDiagnostic");
        PUBLIC_IDENTIFIERS.add("JCDirective");
        PUBLIC_IDENTIFIERS.add("JCDoWhileLoop");
        PUBLIC_IDENTIFIERS.add("JCEMapper");
        PUBLIC_IDENTIFIERS.add("JCEnhancedForLoop");
        PUBLIC_IDENTIFIERS.add("JCErroneous");
        PUBLIC_IDENTIFIERS.add("JCExports");
        PUBLIC_IDENTIFIERS.add("JCExpression");
        PUBLIC_IDENTIFIERS.add("JCExpressionStatement");
        PUBLIC_IDENTIFIERS.add("JCFieldAccess");
        PUBLIC_IDENTIFIERS.add("JCForLoop");
        PUBLIC_IDENTIFIERS.add("JCFunctionalExpression");
        PUBLIC_IDENTIFIERS.add("JCharField");
        PUBLIC_IDENTIFIERS.add("JCheckBox");
        PUBLIC_IDENTIFIERS.add("JCheckBoxMenuItem");
        PUBLIC_IDENTIFIERS.add("JCIdent");
        PUBLIC_IDENTIFIERS.add("JCIf");
        PUBLIC_IDENTIFIERS.add("JCImport");
        PUBLIC_IDENTIFIERS.add("JCInstanceOf");
        PUBLIC_IDENTIFIERS.add("JCLabeledStatement");
        PUBLIC_IDENTIFIERS.add("JCLambda");
        PUBLIC_IDENTIFIERS.add("JCLiteral");
        PUBLIC_IDENTIFIERS.add("JCmd");
        PUBLIC_IDENTIFIERS.add("JCMemberReference");
        PUBLIC_IDENTIFIERS.add("JCMethodDecl");
        PUBLIC_IDENTIFIERS.add("JCMethodInvocation");
        PUBLIC_IDENTIFIERS.add("JCModifiers");
        PUBLIC_IDENTIFIERS.add("JCModuleDecl");
        PUBLIC_IDENTIFIERS.add("JCNewArray");
        PUBLIC_IDENTIFIERS.add("JCNewClass");
        PUBLIC_IDENTIFIERS.add("JCNoType");
        PUBLIC_IDENTIFIERS.add("JColorChooser");
        PUBLIC_IDENTIFIERS.add("JComboBox");
        PUBLIC_IDENTIFIERS.add("JComponent");
        PUBLIC_IDENTIFIERS.add("JComponentAccessor");
        PUBLIC_IDENTIFIERS.add("JConsole");
        PUBLIC_IDENTIFIERS.add("JConsoleContext");
        PUBLIC_IDENTIFIERS.add("JConsolePlugin");
        PUBLIC_IDENTIFIERS.add("JCOpens");
        PUBLIC_IDENTIFIERS.add("JCOperatorExpression");
        PUBLIC_IDENTIFIERS.add("JCPackageDecl");
        PUBLIC_IDENTIFIERS.add("JCParens");
        PUBLIC_IDENTIFIERS.add("JCParenthesizedPattern");
        PUBLIC_IDENTIFIERS.add("JCPattern");
        PUBLIC_IDENTIFIERS.add("JCPatternCaseLabel");
        PUBLIC_IDENTIFIERS.add("JCPolyExpression");
        PUBLIC_IDENTIFIERS.add("JCPrimitiveType");
        PUBLIC_IDENTIFIERS.add("JCPrimitiveTypeTree");
        PUBLIC_IDENTIFIERS.add("JCProvides");
        PUBLIC_IDENTIFIERS.add("JCRecordPattern");
        PUBLIC_IDENTIFIERS.add("JCRequires");
        PUBLIC_IDENTIFIERS.add("JCReturn");
        PUBLIC_IDENTIFIERS.add("JCSkip");
        PUBLIC_IDENTIFIERS.add("JCStatement");
        PUBLIC_IDENTIFIERS.add("JCSwitch");
        PUBLIC_IDENTIFIERS.add("JCSwitchExpression");
        PUBLIC_IDENTIFIERS.add("JCSynchronized");
        PUBLIC_IDENTIFIERS.add("JCThrow");
        PUBLIC_IDENTIFIERS.add("JCTree");
        PUBLIC_IDENTIFIERS.add("JCTry");
        PUBLIC_IDENTIFIERS.add("JCTypeApply");
        PUBLIC_IDENTIFIERS.add("JCTypeCast");
        PUBLIC_IDENTIFIERS.add("JCTypeIntersection");
        PUBLIC_IDENTIFIERS.add("JCTypeParameter");
        PUBLIC_IDENTIFIERS.add("JCTypeUnion");
        PUBLIC_IDENTIFIERS.add("JCUnary");
        PUBLIC_IDENTIFIERS.add("JCUses");
        PUBLIC_IDENTIFIERS.add("JCVariableDecl");
        PUBLIC_IDENTIFIERS.add("JCVoidType");
        PUBLIC_IDENTIFIERS.add("JCWhileLoop");
        PUBLIC_IDENTIFIERS.add("JCWildcard");
        PUBLIC_IDENTIFIERS.add("JCYield");
        PUBLIC_IDENTIFIERS.add("JdbcRowSet");
        PUBLIC_IDENTIFIERS.add("JdbcRowSetImpl");
        PUBLIC_IDENTIFIERS.add("JdbcRowSetResourceBundle");
        PUBLIC_IDENTIFIERS.add("JdepsConfiguration");
        PUBLIC_IDENTIFIERS.add("JdepsFilter");
        PUBLIC_IDENTIFIERS.add("JDepsToolProvider");
        PUBLIC_IDENTIFIERS.add("JdepsWriter");
        PUBLIC_IDENTIFIERS.add("JDesktopIcon");
        PUBLIC_IDENTIFIERS.add("JDesktopPane");
        PUBLIC_IDENTIFIERS.add("JDialog");
        PUBLIC_IDENTIFIERS.add("JdiDefaultExecutionControl");
        PUBLIC_IDENTIFIERS.add("JdiExecutionControl");
        PUBLIC_IDENTIFIERS.add("JdiExecutionControlProvider");
        PUBLIC_IDENTIFIERS.add("JdiInitiator");
        PUBLIC_IDENTIFIERS.add("JdkConsole");
        PUBLIC_IDENTIFIERS.add("JdkConsoleProvider");
        PUBLIC_IDENTIFIERS.add("JdkConsoleProviderImpl");
        PUBLIC_IDENTIFIERS.add("JdkJfrConsumer");
        PUBLIC_IDENTIFIERS.add("JDKPlatformProvider");
        PUBLIC_IDENTIFIERS.add("JDKSettingControl");
        PUBLIC_IDENTIFIERS.add("JdkXmlFeatures");
        PUBLIC_IDENTIFIERS.add("JdkXmlUtils");
        PUBLIC_IDENTIFIERS.add("JDKXPathFactory");
        PUBLIC_IDENTIFIERS.add("JDoubleField");
        PUBLIC_IDENTIFIERS.add("JdpGenericPacket");
        PUBLIC_IDENTIFIERS.add("JdpPacket");
        PUBLIC_IDENTIFIERS.add("JEditorPane");
        PUBLIC_IDENTIFIERS.add("JEditorPaneAccessibleHypertextSupport");
        PUBLIC_IDENTIFIERS.add("JEP");
        PUBLIC_IDENTIFIERS.add("JFileChooser");
        PUBLIC_IDENTIFIERS.add("JFloatField");
        PUBLIC_IDENTIFIERS.add("JFormattedTextField");
        PUBLIC_IDENTIFIERS.add("JFrame");
        PUBLIC_IDENTIFIERS.add("JFrameWrapper");
        PUBLIC_IDENTIFIERS.add("JgssExtender");
        PUBLIC_IDENTIFIERS.add("JInfo");
        PUBLIC_IDENTIFIERS.add("JInternalFrame");
        PUBLIC_IDENTIFIERS.add("JInternalFrameWrapper");
        PUBLIC_IDENTIFIERS.add("JIntField");
        PUBLIC_IDENTIFIERS.add("JIS_X_0201");
        PUBLIC_IDENTIFIERS.add("JIS_X_0208");
        PUBLIC_IDENTIFIERS.add("JIS_X_0208_MS5022X");
        PUBLIC_IDENTIFIERS.add("JIS_X_0208_MS932");
        PUBLIC_IDENTIFIERS.add("JIS_X_0208_Solaris");
        PUBLIC_IDENTIFIERS.add("JIS_X_0212");
        PUBLIC_IDENTIFIERS.add("JIS_X_0212_MS5022X");
        PUBLIC_IDENTIFIERS.add("JIS_X_0212_Solaris");
        PUBLIC_IDENTIFIERS.add("JISAutoDetect");
        PUBLIC_IDENTIFIERS.add("JLabel");
        PUBLIC_IDENTIFIERS.add("JLayeredPane");
        PUBLIC_IDENTIFIERS.add("JLightweightFrameAccessor");
        PUBLIC_IDENTIFIERS.add("JlinkTask");
        PUBLIC_IDENTIFIERS.add("JlinkToolProvider");
        PUBLIC_IDENTIFIERS.add("JList");
        PUBLIC_IDENTIFIERS.add("JLongField");
        PUBLIC_IDENTIFIERS.add("JMap");
        PUBLIC_IDENTIFIERS.add("JMenu");
        PUBLIC_IDENTIFIERS.add("JMenuBar");
        PUBLIC_IDENTIFIERS.add("JMenuItem");
        PUBLIC_IDENTIFIERS.add("JMException");
        PUBLIC_IDENTIFIERS.add("JmodArchive");
        PUBLIC_IDENTIFIERS.add("JmodEntry");
        PUBLIC_IDENTIFIERS.add("JmodFile");
        PUBLIC_IDENTIFIERS.add("JmodTask");
        PUBLIC_IDENTIFIERS.add("JmodToolProvider");
        PUBLIC_IDENTIFIERS.add("JMRuntimeException");
        PUBLIC_IDENTIFIERS.add("JMX");
        PUBLIC_IDENTIFIERS.add("JMXAddressable");
        PUBLIC_IDENTIFIERS.add("JMXAuthenticator");
        PUBLIC_IDENTIFIERS.add("JMXConnectionNotification");
        PUBLIC_IDENTIFIERS.add("JMXConnector");
        PUBLIC_IDENTIFIERS.add("JMXConnectorFactory");
        PUBLIC_IDENTIFIERS.add("JMXConnectorProvider");
        PUBLIC_IDENTIFIERS.add("JMXConnectorServer");
        PUBLIC_IDENTIFIERS.add("JMXConnectorServerFactory");
        PUBLIC_IDENTIFIERS.add("JMXConnectorServerMBean");
        PUBLIC_IDENTIFIERS.add("JMXConnectorServerProvider");
        PUBLIC_IDENTIFIERS.add("JmxMBeanServerBuilder");
        PUBLIC_IDENTIFIERS.add("JMXPrincipal");
        PUBLIC_IDENTIFIERS.add("JmxProperties");
        PUBLIC_IDENTIFIERS.add("JMXProviderException");
        PUBLIC_IDENTIFIERS.add("JMXServerErrorException");
        PUBLIC_IDENTIFIERS.add("JMXServiceURL");
        PUBLIC_IDENTIFIERS.add("JMXSubjectDomainCombiner");
        PUBLIC_IDENTIFIERS.add("JnaSupport");
        PUBLIC_IDENTIFIERS.add("JndiLoginModule");
        PUBLIC_IDENTIFIERS.add("JNIHandleBlock");
        PUBLIC_IDENTIFIERS.add("JNIHandles");
        PUBLIC_IDENTIFIERS.add("JNITypeParser");
        PUBLIC_IDENTIFIERS.add("JNIWriter");
        PUBLIC_IDENTIFIERS.add("JobMediaSheets");
        PUBLIC_IDENTIFIERS.add("JobSheets");
        PUBLIC_IDENTIFIERS.add("JobState");
        PUBLIC_IDENTIFIERS.add("JobStateReason");
        PUBLIC_IDENTIFIERS.add("Johab");
        PUBLIC_IDENTIFIERS.add("Joinable");
        PUBLIC_IDENTIFIERS.add("JoiningGroup");
        PUBLIC_IDENTIFIERS.add("JoinRowSet");
        PUBLIC_IDENTIFIERS.add("JoinRowSetImpl");
        PUBLIC_IDENTIFIERS.add("JOptionPane");
        PUBLIC_IDENTIFIERS.add("JPackageToolProvider");
        PUBLIC_IDENTIFIERS.add("JPanel");
        PUBLIC_IDENTIFIERS.add("JPasswordField");
        PUBLIC_IDENTIFIERS.add("JPEG");
        PUBLIC_IDENTIFIERS.add("jpeg");
        PUBLIC_IDENTIFIERS.add("JPEGHuffmanTable");
        PUBLIC_IDENTIFIERS.add("JPEGImageDecoder");
        PUBLIC_IDENTIFIERS.add("JPEGImageMetadataFormat");
        PUBLIC_IDENTIFIERS.add("JPEGImageMetadataFormatResources");
        PUBLIC_IDENTIFIERS.add("JPEGImageReader");
        PUBLIC_IDENTIFIERS.add("JPEGImageReaderResources");
        PUBLIC_IDENTIFIERS.add("JPEGImageReaderSpi");
        PUBLIC_IDENTIFIERS.add("JPEGImageReadParam");
        PUBLIC_IDENTIFIERS.add("JPEGImageWriteParam");
        PUBLIC_IDENTIFIERS.add("JPEGImageWriter");
        PUBLIC_IDENTIFIERS.add("JPEGImageWriterResources");
        PUBLIC_IDENTIFIERS.add("JPEGImageWriterSpi");
        PUBLIC_IDENTIFIERS.add("JPEGMetadata");
        PUBLIC_IDENTIFIERS.add("JPEGQTable");
        PUBLIC_IDENTIFIERS.add("JPEGStreamMetadataFormat");
        PUBLIC_IDENTIFIERS.add("JPEGStreamMetadataFormatResources");
        PUBLIC_IDENTIFIERS.add("JPopupMenu");
        PUBLIC_IDENTIFIERS.add("JProgressBar");
        PUBLIC_IDENTIFIERS.add("Jps");
        PUBLIC_IDENTIFIERS.add("JRadioButton");
        PUBLIC_IDENTIFIERS.add("JRadioButtonMenuItem");
        PUBLIC_IDENTIFIERS.add("JRELocaleConstants");
        PUBLIC_IDENTIFIERS.add("JRELocaleProviderAdapter");
        PUBLIC_IDENTIFIERS.add("JRootPane");
        PUBLIC_IDENTIFIERS.add("JRTIndex");
        PUBLIC_IDENTIFIERS.add("JScrollBar");
        PUBLIC_IDENTIFIERS.add("JScrollPane");
        PUBLIC_IDENTIFIERS.add("JSeparator");
        PUBLIC_IDENTIFIERS.add("JSException");
        PUBLIC_IDENTIFIERS.add("JShell");
        PUBLIC_IDENTIFIERS.add("JShellException");
        PUBLIC_IDENTIFIERS.add("JShellTool");
        PUBLIC_IDENTIFIERS.add("JShellToolBuilder");
        PUBLIC_IDENTIFIERS.add("JShellToolProvider");
        PUBLIC_IDENTIFIERS.add("JShortField");
        PUBLIC_IDENTIFIERS.add("JSlider");
        PUBLIC_IDENTIFIERS.add("JSnap");
        PUBLIC_IDENTIFIERS.add("JSObject");
        PUBLIC_IDENTIFIERS.add("JSpinner");
        PUBLIC_IDENTIFIERS.add("JSplitPane");
        PUBLIC_IDENTIFIERS.add("JSR");
        PUBLIC_IDENTIFIERS.add("JSR_W");
        PUBLIC_IDENTIFIERS.add("JSRInlinerAdapter");
        PUBLIC_IDENTIFIERS.add("JsrInstruction");
        PUBLIC_IDENTIFIERS.add("JStack");
        PUBLIC_IDENTIFIERS.add("Jstat");
        PUBLIC_IDENTIFIERS.add("Jstatd");
        PUBLIC_IDENTIFIERS.add("JStatLogger");
        PUBLIC_IDENTIFIERS.add("JTabbedPane");
        PUBLIC_IDENTIFIERS.add("JTable");
        PUBLIC_IDENTIFIERS.add("JTableHeader");
        PUBLIC_IDENTIFIERS.add("JTextArea");
        PUBLIC_IDENTIFIERS.add("JTextComponent");
        PUBLIC_IDENTIFIERS.add("JTextComponentAccessor");
        PUBLIC_IDENTIFIERS.add("JTextField");
        PUBLIC_IDENTIFIERS.add("JTextPane");
        PUBLIC_IDENTIFIERS.add("JToggleButton");
        PUBLIC_IDENTIFIERS.add("JToolBar");
        PUBLIC_IDENTIFIERS.add("JToolTip");
        PUBLIC_IDENTIFIERS.add("JTree");
        PUBLIC_IDENTIFIERS.add("JTreeTable");
        PUBLIC_IDENTIFIERS.add("JulianCalendar");
        PUBLIC_IDENTIFIERS.add("JumpableGenerator");
        PUBLIC_IDENTIFIERS.add("JumpData");
        PUBLIC_IDENTIFIERS.add("JumpInsnNode");
        PUBLIC_IDENTIFIERS.add("JViewport");
        PUBLIC_IDENTIFIERS.add("JVMCI");
        PUBLIC_IDENTIFIERS.add("JVMCIBackend");
        PUBLIC_IDENTIFIERS.add("JVMCICompiler");
        PUBLIC_IDENTIFIERS.add("JVMCICompilerFactory");
        PUBLIC_IDENTIFIERS.add("JVMCIError");
        PUBLIC_IDENTIFIERS.add("JVMCIPermission");
        PUBLIC_IDENTIFIERS.add("JVMCIRuntime");
        PUBLIC_IDENTIFIERS.add("JVMCIServiceLocator");
        PUBLIC_IDENTIFIERS.add("JVMDebugger");
        PUBLIC_IDENTIFIERS.add("JVMDIClassStatus");
        PUBLIC_IDENTIFIERS.add("JVMState");
        PUBLIC_IDENTIFIERS.add("JvmtiAgentThread");
        PUBLIC_IDENTIFIERS.add("JvmtiExport");
        PUBLIC_IDENTIFIERS.add("JvmtiMountTransition");
        PUBLIC_IDENTIFIERS.add("JWebServer");
        PUBLIC_IDENTIFIERS.add("JWindow");
    }

    private static void addPublicIdentifiersKM() {
        PUBLIC_IDENTIFIERS.add("KAKeyDerivation");
        PUBLIC_IDENTIFIERS.add("KdcErrException");
        PUBLIC_IDENTIFIERS.add("KDCOptions");
        PUBLIC_IDENTIFIERS.add("KDCRep");
        PUBLIC_IDENTIFIERS.add("KDCReq");
        PUBLIC_IDENTIFIERS.add("KDCReqBody");
        PUBLIC_IDENTIFIERS.add("KeepAliveCache");
        PUBLIC_IDENTIFIERS.add("KeepAliveStream");
        PUBLIC_IDENTIFIERS.add("KerberosFlags");
        PUBLIC_IDENTIFIERS.add("KerberosKey");
        PUBLIC_IDENTIFIERS.add("KerberosSecrets");
        PUBLIC_IDENTIFIERS.add("KerberosTicket");
        PUBLIC_IDENTIFIERS.add("KerberosTime");
        PUBLIC_IDENTIFIERS.add("Kernel");
        PUBLIC_IDENTIFIERS.add("Key");
        PUBLIC_IDENTIFIERS.add("KeyAdapter");
        PUBLIC_IDENTIFIERS.add("KeyAgreement");
        PUBLIC_IDENTIFIERS.add("KeyAgreementSpi");
        PUBLIC_IDENTIFIERS.add("KeyAlreadyExistsException");
        PUBLIC_IDENTIFIERS.add("KeyBinding");
        PUBLIC_IDENTIFIERS.add("KeyboardDownRightHandler");
        PUBLIC_IDENTIFIERS.add("KeyboardEndHandler");
        PUBLIC_IDENTIFIERS.add("KeyboardFocusManager");
        PUBLIC_IDENTIFIERS.add("KeyboardFocusManagerAccessor");
        PUBLIC_IDENTIFIERS.add("KeyboardFocusManagerPeer");
        PUBLIC_IDENTIFIERS.add("KeyboardFocusManagerPeerImpl");
        PUBLIC_IDENTIFIERS.add("KeyboardFocusManagerPeerProvider");
        PUBLIC_IDENTIFIERS.add("KeyboardHomeHandler");
        PUBLIC_IDENTIFIERS.add("KeyboardResizeToggleHandler");
        PUBLIC_IDENTIFIERS.add("KeyboardUpLeftHandler");
        PUBLIC_IDENTIFIERS.add("KeyEvent");
        PUBLIC_IDENTIFIERS.add("KeyEventAccessor");
        PUBLIC_IDENTIFIERS.add("KeyEventDispatcher");
        PUBLIC_IDENTIFIERS.add("KeyEventPostProcessor");
        PUBLIC_IDENTIFIERS.add("KeyException");
        PUBLIC_IDENTIFIERS.add("KeyFactory");
        PUBLIC_IDENTIFIERS.add("KeyFactorySpi");
        PUBLIC_IDENTIFIERS.add("KeyGenerator");
        PUBLIC_IDENTIFIERS.add("KeyGeneratorSpi");
        PUBLIC_IDENTIFIERS.add("KeyHandler");
        PUBLIC_IDENTIFIERS.add("KeyIdentifier");
        PUBLIC_IDENTIFIERS.add("KeyIndex");
        PUBLIC_IDENTIFIERS.add("KeyIndexHeapNode");
        PUBLIC_IDENTIFIERS.add("KeyIndexIterator");
        PUBLIC_IDENTIFIERS.add("KeyInfo");
        PUBLIC_IDENTIFIERS.add("KeyInfoContent");
        PUBLIC_IDENTIFIERS.add("KeyInfoFactory");
        PUBLIC_IDENTIFIERS.add("KeyInfoReference");
        PUBLIC_IDENTIFIERS.add("KeyInfoReferenceResolver");
        PUBLIC_IDENTIFIERS.add("KeyListener");
        PUBLIC_IDENTIFIERS.add("KeyManagementException");
        PUBLIC_IDENTIFIERS.add("KeyManager");
        PUBLIC_IDENTIFIERS.add("KeyManagerFactory");
        PUBLIC_IDENTIFIERS.add("KeyManagerFactorySpi");
        PUBLIC_IDENTIFIERS.add("KeyMap");
        PUBLIC_IDENTIFIERS.add("Keymap");
        PUBLIC_IDENTIFIERS.add("KeyName");
        PUBLIC_IDENTIFIERS.add("KeyPairGenerator");
        PUBLIC_IDENTIFIERS.add("KeyPairGeneratorSpi");
        PUBLIC_IDENTIFIERS.add("KeyRef");
        PUBLIC_IDENTIFIERS.add("KeyRefValueStore");
        PUBLIC_IDENTIFIERS.add("KeyRep");
        PUBLIC_IDENTIFIERS.add("KeyResolver");
        PUBLIC_IDENTIFIERS.add("KeyResolverException");
        PUBLIC_IDENTIFIERS.add("KeyResolverSpi");
        PUBLIC_IDENTIFIERS.add("KeySelectionManager");
        PUBLIC_IDENTIFIERS.add("KeySelector");
        PUBLIC_IDENTIFIERS.add("KeySelectorException");
        PUBLIC_IDENTIFIERS.add("KeySelectorResult");
        PUBLIC_IDENTIFIERS.add("KeySpec");
        PUBLIC_IDENTIFIERS.add("KeyStore");
        PUBLIC_IDENTIFIERS.add("KeyStoreBuilderParameters");
        PUBLIC_IDENTIFIERS.add("KeyStoreDelegator");
        PUBLIC_IDENTIFIERS.add("KeyStoreException");
        PUBLIC_IDENTIFIERS.add("KeyStoreLoginModule");
        PUBLIC_IDENTIFIERS.add("KeyStoreResolver");
        PUBLIC_IDENTIFIERS.add("KeyStoreSpi");
        PUBLIC_IDENTIFIERS.add("KeyStoreUtil");
        PUBLIC_IDENTIFIERS.add("KeyStroke");
        PUBLIC_IDENTIFIERS.add("KeyStrokeAccessor");
        PUBLIC_IDENTIFIERS.add("KeyTab");
        PUBLIC_IDENTIFIERS.add("KeyTabConstants");
        PUBLIC_IDENTIFIERS.add("KeyTabEntry");
        PUBLIC_IDENTIFIERS.add("KeyTabInputStream");
        PUBLIC_IDENTIFIERS.add("KeyTabOutputStream");
        PUBLIC_IDENTIFIERS.add("KeyUsage");
        PUBLIC_IDENTIFIERS.add("KeyUsageExtension");
        PUBLIC_IDENTIFIERS.add("KeyValue");
        PUBLIC_IDENTIFIERS.add("KeyValueContent");
        PUBLIC_IDENTIFIERS.add("KeyValueStore");
        PUBLIC_IDENTIFIERS.add("Keywords");
        PUBLIC_IDENTIFIERS.add("Kind");
        PUBLIC_IDENTIFIERS.add("Kinds");
        PUBLIC_IDENTIFIERS.add("KindSelector");
        PUBLIC_IDENTIFIERS.add("KindVisitor");
        PUBLIC_IDENTIFIERS.add("Klass");
        PUBLIC_IDENTIFIERS.add("KlassArray");
        PUBLIC_IDENTIFIERS.add("KnownOIDs");
        PUBLIC_IDENTIFIERS.add("KOI8_R");
        PUBLIC_IDENTIFIERS.add("KOI8_U");
        PUBLIC_IDENTIFIERS.add("Krb5");
        PUBLIC_IDENTIFIERS.add("Krb5AcceptCredential");
        PUBLIC_IDENTIFIERS.add("Krb5InitCredential");
        PUBLIC_IDENTIFIERS.add("Krb5LoginModule");
        PUBLIC_IDENTIFIERS.add("Krb5NameElement");
        PUBLIC_IDENTIFIERS.add("Krb5Util");
        PUBLIC_IDENTIFIERS.add("KrbApErrException");
        PUBLIC_IDENTIFIERS.add("KrbApRep");
        PUBLIC_IDENTIFIERS.add("KrbApReq");
        PUBLIC_IDENTIFIERS.add("KrbAsReq");
        PUBLIC_IDENTIFIERS.add("KRBCred");
        PUBLIC_IDENTIFIERS.add("KrbCred");
        PUBLIC_IDENTIFIERS.add("KrbCredInfo");
        PUBLIC_IDENTIFIERS.add("KrbCryptoException");
        PUBLIC_IDENTIFIERS.add("KrbDataInputStream");
        PUBLIC_IDENTIFIERS.add("KrbDataOutputStream");
        PUBLIC_IDENTIFIERS.add("KrbErrException");
        PUBLIC_IDENTIFIERS.add("KRBError");
        PUBLIC_IDENTIFIERS.add("KrbException");
        PUBLIC_IDENTIFIERS.add("KrbTgsReq");
        PUBLIC_IDENTIFIERS.add("L2D");
        PUBLIC_IDENTIFIERS.add("L2F");
        PUBLIC_IDENTIFIERS.add("L2I");
        PUBLIC_IDENTIFIERS.add("Label");
        PUBLIC_IDENTIFIERS.add("LabeledBorder");
        PUBLIC_IDENTIFIERS.add("LabeledComponent");
        PUBLIC_IDENTIFIERS.add("LabeledStatementTree");
        PUBLIC_IDENTIFIERS.add("LabelNode");
        PUBLIC_IDENTIFIERS.add("LabelPeer");
        PUBLIC_IDENTIFIERS.add("LabelTranslator");
        PUBLIC_IDENTIFIERS.add("LabelUI");
        PUBLIC_IDENTIFIERS.add("LabelView");
        PUBLIC_IDENTIFIERS.add("LADD");
        PUBLIC_IDENTIFIERS.add("LALOAD");
        PUBLIC_IDENTIFIERS.add("LambdaConversionException");
        PUBLIC_IDENTIFIERS.add("LambdaExpressionTree");
        PUBLIC_IDENTIFIERS.add("LambdaToMethod");
        PUBLIC_IDENTIFIERS.add("LAND");
        PUBLIC_IDENTIFIERS.add("LanguageCallback");
        PUBLIC_IDENTIFIERS.add("LanguageTag");
        PUBLIC_IDENTIFIERS.add("LASTORE");
        PUBLIC_IDENTIFIERS.add("LastReq");
        PUBLIC_IDENTIFIERS.add("LastReqEntry");
        PUBLIC_IDENTIFIERS.add("LatencyRequest");
        PUBLIC_IDENTIFIERS.add("LauncherProperties");
        PUBLIC_IDENTIFIERS.add("LaunchingConnector");
        PUBLIC_IDENTIFIERS.add("LayeredHighlighter");
        PUBLIC_IDENTIFIERS.add("LayerInstantiationException");
        PUBLIC_IDENTIFIERS.add("LayerPainter");
        PUBLIC_IDENTIFIERS.add("LayerUI");
        PUBLIC_IDENTIFIERS.add("LayoutCharacters");
        PUBLIC_IDENTIFIERS.add("LayoutEngine");
        PUBLIC_IDENTIFIERS.add("LayoutEngineFactory");
        PUBLIC_IDENTIFIERS.add("LayoutFocusTraversalPolicy");
        PUBLIC_IDENTIFIERS.add("LayoutManager");
        PUBLIC_IDENTIFIERS.add("LayoutManager2");
        PUBLIC_IDENTIFIERS.add("LayoutPath");
        PUBLIC_IDENTIFIERS.add("LayoutPathImpl");
        PUBLIC_IDENTIFIERS.add("LayoutQueue");
        PUBLIC_IDENTIFIERS.add("LayoutResult");
        PUBLIC_IDENTIFIERS.add("LayoutStyle");
        PUBLIC_IDENTIFIERS.add("LayoutVerifier");
        PUBLIC_IDENTIFIERS.add("LazyBlockSym");
        PUBLIC_IDENTIFIERS.add("LazyCompositeData");
        PUBLIC_IDENTIFIERS.add("LazyDocCommentTable");
        PUBLIC_IDENTIFIERS.add("LazyInputMap");
        PUBLIC_IDENTIFIERS.add("LazyPipe");
        PUBLIC_IDENTIFIERS.add("LazyReflectiveObjectGenerator");
        PUBLIC_IDENTIFIERS.add("LazyType");
        PUBLIC_IDENTIFIERS.add("LazyValue");
        PUBLIC_IDENTIFIERS.add("LCDContrastKey");
        PUBLIC_IDENTIFIERS.add("LCDTextRenderer");
        PUBLIC_IDENTIFIERS.add("LCMP");
        PUBLIC_IDENTIFIERS.add("LCONST");
        PUBLIC_IDENTIFIERS.add("LDAPCertStoreParameters");
        PUBLIC_IDENTIFIERS.add("LdapContext");
        PUBLIC_IDENTIFIERS.add("LdapDnsProvider");
        PUBLIC_IDENTIFIERS.add("LdapLoginModule");
        PUBLIC_IDENTIFIERS.add("LdapName");
        PUBLIC_IDENTIFIERS.add("LdapReferralException");
        PUBLIC_IDENTIFIERS.add("ldapURLContextFactory");
        PUBLIC_IDENTIFIERS.add("LDC");
        PUBLIC_IDENTIFIERS.add("LDC2_W");
        PUBLIC_IDENTIFIERS.add("LDC_W");
        PUBLIC_IDENTIFIERS.add("LdcInsnNode");
        PUBLIC_IDENTIFIERS.add("LDIV");
        PUBLIC_IDENTIFIERS.add("LeafElement");
        PUBLIC_IDENTIFIERS.add("LeapableGenerator");
        PUBLIC_IDENTIFIERS.add("LegacyAlgorithmConstraints");
        PUBLIC_IDENTIFIERS.add("Length");
        PUBLIC_IDENTIFIERS.add("LetExpr");
        PUBLIC_IDENTIFIERS.add("Level");
        PUBLIC_IDENTIFIERS.add("Levenshtein");
        PUBLIC_IDENTIFIERS.add("Lexer");
        PUBLIC_IDENTIFIERS.add("LexicalHandler");
        PUBLIC_IDENTIFIERS.add("LightweightContent");
        PUBLIC_IDENTIFIERS.add("LightweightContentWrapper");
        PUBLIC_IDENTIFIERS.add("LightweightFrame");
        PUBLIC_IDENTIFIERS.add("LightweightFrameWrapper");
        PUBLIC_IDENTIFIERS.add("LightweightPeer");
        PUBLIC_IDENTIFIERS.add("LimitExceededException");
        PUBLIC_IDENTIFIERS.add("Line");
        PUBLIC_IDENTIFIERS.add("Line2D");
        PUBLIC_IDENTIFIERS.add("LineBorder");
        PUBLIC_IDENTIFIERS.add("LineBorderUIResource");
        PUBLIC_IDENTIFIERS.add("LineDisciplineTerminal");
        PUBLIC_IDENTIFIERS.add("LineEvent");
        PUBLIC_IDENTIFIERS.add("LineListener");
        PUBLIC_IDENTIFIERS.add("LineMap");
        PUBLIC_IDENTIFIERS.add("LineMetrics");
        PUBLIC_IDENTIFIERS.add("LineNotFoundException");
        PUBLIC_IDENTIFIERS.add("LineNumberGen");
        PUBLIC_IDENTIFIERS.add("LineNumberInfo");
        PUBLIC_IDENTIFIERS.add("LineNumberInputStream");
        PUBLIC_IDENTIFIERS.add("LineNumberNode");
        PUBLIC_IDENTIFIERS.add("LineNumberReader");
        PUBLIC_IDENTIFIERS.add("LineNumberTable");
        PUBLIC_IDENTIFIERS.add("LineNumberTable_attribute");
        PUBLIC_IDENTIFIERS.add("LineNumberTableElement");
        PUBLIC_IDENTIFIERS.add("LineNumberVisitor");
        PUBLIC_IDENTIFIERS.add("LineReader");
        PUBLIC_IDENTIFIERS.add("LineReaderImpl");
        PUBLIC_IDENTIFIERS.add("LineTabMapImpl");
        PUBLIC_IDENTIFIERS.add("LineUnavailableException");
        PUBLIC_IDENTIFIERS.add("Link");
        PUBLIC_IDENTIFIERS.add("LinkageError");
        PUBLIC_IDENTIFIERS.add("LinkController");
        PUBLIC_IDENTIFIERS.add("LinkedBlockingDeque");
        PUBLIC_IDENTIFIERS.add("LinkedBlockingQueue");
        PUBLIC_IDENTIFIERS.add("LinkedHashMap");
        PUBLIC_IDENTIFIERS.add("LinkedHashSet");
        PUBLIC_IDENTIFIERS.add("LinkedList");
        PUBLIC_IDENTIFIERS.add("LinkedTransferQueue");
        PUBLIC_IDENTIFIERS.add("Linker");
        PUBLIC_IDENTIFIERS.add("LinkerOptionImpl");
        PUBLIC_IDENTIFIERS.add("LinkerOptions");
        PUBLIC_IDENTIFIERS.add("LinkerServices");
        PUBLIC_IDENTIFIERS.add("LinkException");
        PUBLIC_IDENTIFIERS.add("LinkFactory");
        PUBLIC_IDENTIFIERS.add("LinkInfo");
        PUBLIC_IDENTIFIERS.add("LinkLoopException");
        PUBLIC_IDENTIFIERS.add("LinkRef");
        PUBLIC_IDENTIFIERS.add("LinkRequest");
        PUBLIC_IDENTIFIERS.add("Links");
        PUBLIC_IDENTIFIERS.add("LinkStyle");
        PUBLIC_IDENTIFIERS.add("LinkTree");
        PUBLIC_IDENTIFIERS.add("Lint");
        PUBLIC_IDENTIFIERS.add("LintLogger");
        PUBLIC_IDENTIFIERS.add("LinuxAArch64CallArranger");
        PUBLIC_IDENTIFIERS.add("LinuxAARCH64JavaThreadPDAccess");
        PUBLIC_IDENTIFIERS.add("LinuxAARCH64ThreadContext");
        PUBLIC_IDENTIFIERS.add("LinuxAArch64VaList");
        PUBLIC_IDENTIFIERS.add("LinuxAddress");
        PUBLIC_IDENTIFIERS.add("LinuxAMD64JavaThreadPDAccess");
        PUBLIC_IDENTIFIERS.add("LinuxAMD64ThreadContext");
        PUBLIC_IDENTIFIERS.add("LinuxAppBundler");
        PUBLIC_IDENTIFIERS.add("LinuxAppImageBuilder");
        PUBLIC_IDENTIFIERS.add("LinuxAsynchronousChannelProvider");
        PUBLIC_IDENTIFIERS.add("LinuxDebBundler");
        PUBLIC_IDENTIFIERS.add("LinuxDebugger");
        PUBLIC_IDENTIFIERS.add("LinuxDebuggerLocal");
        PUBLIC_IDENTIFIERS.add("LinuxOopHandle");
        PUBLIC_IDENTIFIERS.add("LinuxPPC64JavaThreadPDAccess");
        PUBLIC_IDENTIFIERS.add("LinuxPPC64ThreadContext");
        PUBLIC_IDENTIFIERS.add("LinuxRISCV64JavaThreadPDAccess");
        PUBLIC_IDENTIFIERS.add("LinuxRISCV64ThreadContext");
        PUBLIC_IDENTIFIERS.add("LinuxRpmBundler");
        PUBLIC_IDENTIFIERS.add("LinuxSignals");
        PUBLIC_IDENTIFIERS.add("LinuxVtblAccess");
        PUBLIC_IDENTIFIERS.add("LinuxX86JavaThreadPDAccess");
        PUBLIC_IDENTIFIERS.add("LinuxX86ThreadContext");
        PUBLIC_IDENTIFIERS.add("List");
        PUBLIC_IDENTIFIERS.add("ListBuffer");
        PUBLIC_IDENTIFIERS.add("ListCellRenderer");
        PUBLIC_IDENTIFIERS.add("ListDataEvent");
        PUBLIC_IDENTIFIERS.add("ListDataHandler");
        PUBLIC_IDENTIFIERS.add("ListDataListener");
        PUBLIC_IDENTIFIERS.add("ListDatatypeValidator");
        PUBLIC_IDENTIFIERS.add("ListDV");
        PUBLIC_IDENTIFIERS.add("ListEditor");
        PUBLIC_IDENTIFIERS.add("Listener");
        PUBLIC_IDENTIFIERS.add("ListenerNotFoundException");
        PUBLIC_IDENTIFIERS.add("ListeningConnector");
        PUBLIC_IDENTIFIERS.add("ListenKey");
        PUBLIC_IDENTIFIERS.add("ListingErrorHandler");
        PUBLIC_IDENTIFIERS.add("ListIterator");
        PUBLIC_IDENTIFIERS.add("ListModel");
        PUBLIC_IDENTIFIERS.add("ListMouseHandler");
        PUBLIC_IDENTIFIERS.add("ListMouseMotionHandler");
        PUBLIC_IDENTIFIERS.add("ListPeer");
        PUBLIC_IDENTIFIERS.add("ListResourceBundle");
        PUBLIC_IDENTIFIERS.add("ListSelectionEvent");
        PUBLIC_IDENTIFIERS.add("ListSelectionHandler");
        PUBLIC_IDENTIFIERS.add("ListSelectionListener");
        PUBLIC_IDENTIFIERS.add("ListSelectionModel");
        PUBLIC_IDENTIFIERS.add("ListTranslator");
        PUBLIC_IDENTIFIERS.add("ListUI");
        PUBLIC_IDENTIFIERS.add("ListView");
        PUBLIC_IDENTIFIERS.add("Literal");
        PUBLIC_IDENTIFIERS.add("LiteralTaglet");
        PUBLIC_IDENTIFIERS.add("LiteralTree");
        PUBLIC_IDENTIFIERS.add("LiveArguments");
        PUBLIC_IDENTIFIERS.add("LivenessAnalysis");
        PUBLIC_IDENTIFIERS.add("LivenessPath");
        PUBLIC_IDENTIFIERS.add("LivenessPathElement");
        PUBLIC_IDENTIFIERS.add("LivenessPathList");
        PUBLIC_IDENTIFIERS.add("LiveRef");
        PUBLIC_IDENTIFIERS.add("LiveRegionsClosure");
        PUBLIC_IDENTIFIERS.add("LiveRegionsProvider");
        PUBLIC_IDENTIFIERS.add("LLOAD");
        PUBLIC_IDENTIFIERS.add("LMUL");
        PUBLIC_IDENTIFIERS.add("LNEG");
        PUBLIC_IDENTIFIERS.add("LoadableConstant");
        PUBLIC_IDENTIFIERS.add("LoadClass");
        PUBLIC_IDENTIFIERS.add("LoaderDelegate");
        PUBLIC_IDENTIFIERS.add("LoaderHandler");
        PUBLIC_IDENTIFIERS.add("LoadInstruction");
        PUBLIC_IDENTIFIERS.add("LoadObject");
        PUBLIC_IDENTIFIERS.add("LoadObjectComparator");
        PUBLIC_IDENTIFIERS.add("LoadOperation");
        PUBLIC_IDENTIFIERS.add("LoadProc");
        PUBLIC_IDENTIFIERS.add("LoadStoreParameter");
        PUBLIC_IDENTIFIERS.add("LoadVectorMaskedOperation");
        PUBLIC_IDENTIFIERS.add("LoadVectorOperationWithMap");
        PUBLIC_IDENTIFIERS.add("Local");
        PUBLIC_IDENTIFIERS.add("LocaleData");
        PUBLIC_IDENTIFIERS.add("LocaleDataMetaInfo");
        PUBLIC_IDENTIFIERS.add("LocaleDataProvider");
        PUBLIC_IDENTIFIERS.add("LocaleExtensions");
        PUBLIC_IDENTIFIERS.add("LocaleNameProvider");
        PUBLIC_IDENTIFIERS.add("LocaleNameProviderImpl");
        PUBLIC_IDENTIFIERS.add("LocaleNames");
        PUBLIC_IDENTIFIERS.add("LocaleNames_af");
        PUBLIC_IDENTIFIERS.add("LocaleNames_agq");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ak");
        PUBLIC_IDENTIFIERS.add("LocaleNames_am");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ann");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ar");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ar_EG");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ar_LY");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ar_SA");
        PUBLIC_IDENTIFIERS.add("LocaleNames_as");
        PUBLIC_IDENTIFIERS.add("LocaleNames_asa");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ast");
        PUBLIC_IDENTIFIERS.add("LocaleNames_az");
        PUBLIC_IDENTIFIERS.add("LocaleNames_az_Cyrl");
        PUBLIC_IDENTIFIERS.add("LocaleNames_bas");
        PUBLIC_IDENTIFIERS.add("LocaleNames_be");
        PUBLIC_IDENTIFIERS.add("LocaleNames_bem");
        PUBLIC_IDENTIFIERS.add("LocaleNames_bez");
        PUBLIC_IDENTIFIERS.add("LocaleNames_bg");
        PUBLIC_IDENTIFIERS.add("LocaleNames_bgc");
        PUBLIC_IDENTIFIERS.add("LocaleNames_bho");
        PUBLIC_IDENTIFIERS.add("LocaleNames_bm");
        PUBLIC_IDENTIFIERS.add("LocaleNames_bn");
        PUBLIC_IDENTIFIERS.add("LocaleNames_bn_IN");
        PUBLIC_IDENTIFIERS.add("LocaleNames_bo");
        PUBLIC_IDENTIFIERS.add("LocaleNames_bo_IN");
        PUBLIC_IDENTIFIERS.add("LocaleNames_br");
        PUBLIC_IDENTIFIERS.add("LocaleNames_brx");
        PUBLIC_IDENTIFIERS.add("LocaleNames_bs");
        PUBLIC_IDENTIFIERS.add("LocaleNames_bs_Cyrl");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ca");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ccp");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ce");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ceb");
        PUBLIC_IDENTIFIERS.add("LocaleNames_cgg");
        PUBLIC_IDENTIFIERS.add("LocaleNames_chr");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ckb");
        PUBLIC_IDENTIFIERS.add("LocaleNames_cs");
        PUBLIC_IDENTIFIERS.add("LocaleNames_cv");
        PUBLIC_IDENTIFIERS.add("LocaleNames_cy");
        PUBLIC_IDENTIFIERS.add("LocaleNames_da");
        PUBLIC_IDENTIFIERS.add("LocaleNames_dav");
        PUBLIC_IDENTIFIERS.add("LocaleNames_de");
        PUBLIC_IDENTIFIERS.add("LocaleNames_de_AT");
        PUBLIC_IDENTIFIERS.add("LocaleNames_de_CH");
        PUBLIC_IDENTIFIERS.add("LocaleNames_dje");
        PUBLIC_IDENTIFIERS.add("LocaleNames_doi");
        PUBLIC_IDENTIFIERS.add("LocaleNames_dsb");
        PUBLIC_IDENTIFIERS.add("LocaleNames_dua");
        PUBLIC_IDENTIFIERS.add("LocaleNames_dyo");
        PUBLIC_IDENTIFIERS.add("LocaleNames_dz");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ebu");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ee");
        PUBLIC_IDENTIFIERS.add("LocaleNames_el");
        PUBLIC_IDENTIFIERS.add("LocaleNames_en");
        PUBLIC_IDENTIFIERS.add("LocaleNames_en_001");
        PUBLIC_IDENTIFIERS.add("LocaleNames_en_AU");
        PUBLIC_IDENTIFIERS.add("LocaleNames_en_CA");
        PUBLIC_IDENTIFIERS.add("LocaleNames_en_GB");
        PUBLIC_IDENTIFIERS.add("LocaleNames_en_IN");
        PUBLIC_IDENTIFIERS.add("LocaleNames_eo");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es_419");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es_AR");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es_BO");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es_CL");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es_CO");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es_CR");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es_DO");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es_EC");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es_GT");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es_HN");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es_MX");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es_NI");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es_PA");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es_PE");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es_PR");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es_PY");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es_SV");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es_US");
        PUBLIC_IDENTIFIERS.add("LocaleNames_es_VE");
        PUBLIC_IDENTIFIERS.add("LocaleNames_et");
        PUBLIC_IDENTIFIERS.add("LocaleNames_eu");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ewo");
        PUBLIC_IDENTIFIERS.add("LocaleNames_fa");
        PUBLIC_IDENTIFIERS.add("LocaleNames_fa_AF");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ff");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ff_Adlm");
        PUBLIC_IDENTIFIERS.add("LocaleNames_fi");
        PUBLIC_IDENTIFIERS.add("LocaleNames_fil");
        PUBLIC_IDENTIFIERS.add("LocaleNames_fo");
        PUBLIC_IDENTIFIERS.add("LocaleNames_fr");
        PUBLIC_IDENTIFIERS.add("LocaleNames_fr_BE");
        PUBLIC_IDENTIFIERS.add("LocaleNames_fr_CA");
        PUBLIC_IDENTIFIERS.add("LocaleNames_fr_CH");
        PUBLIC_IDENTIFIERS.add("LocaleNames_frr");
        PUBLIC_IDENTIFIERS.add("LocaleNames_fur");
        PUBLIC_IDENTIFIERS.add("LocaleNames_fy");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ga");
        PUBLIC_IDENTIFIERS.add("LocaleNames_gd");
        PUBLIC_IDENTIFIERS.add("LocaleNames_gl");
        PUBLIC_IDENTIFIERS.add("LocaleNames_gsw");
        PUBLIC_IDENTIFIERS.add("LocaleNames_gu");
        PUBLIC_IDENTIFIERS.add("LocaleNames_guz");
        PUBLIC_IDENTIFIERS.add("LocaleNames_gv");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ha");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ha_NE");
        PUBLIC_IDENTIFIERS.add("LocaleNames_haw");
        PUBLIC_IDENTIFIERS.add("LocaleNames_he");
        PUBLIC_IDENTIFIERS.add("LocaleNames_hi");
        PUBLIC_IDENTIFIERS.add("LocaleNames_hi_Latn");
        PUBLIC_IDENTIFIERS.add("LocaleNames_hr");
        PUBLIC_IDENTIFIERS.add("LocaleNames_hsb");
        PUBLIC_IDENTIFIERS.add("LocaleNames_hu");
        PUBLIC_IDENTIFIERS.add("LocaleNames_hy");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ia");
        PUBLIC_IDENTIFIERS.add("LocaleNames_id");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ig");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ii");
        PUBLIC_IDENTIFIERS.add("LocaleNames_is");
        PUBLIC_IDENTIFIERS.add("LocaleNames_it");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ja");
        PUBLIC_IDENTIFIERS.add("LocaleNames_jgo");
        PUBLIC_IDENTIFIERS.add("LocaleNames_jmc");
        PUBLIC_IDENTIFIERS.add("LocaleNames_jv");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ka");
        PUBLIC_IDENTIFIERS.add("LocaleNames_kab");
        PUBLIC_IDENTIFIERS.add("LocaleNames_kam");
        PUBLIC_IDENTIFIERS.add("LocaleNames_kde");
        PUBLIC_IDENTIFIERS.add("LocaleNames_kea");
        PUBLIC_IDENTIFIERS.add("LocaleNames_kgp");
        PUBLIC_IDENTIFIERS.add("LocaleNames_khq");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ki");
        PUBLIC_IDENTIFIERS.add("LocaleNames_kk");
        PUBLIC_IDENTIFIERS.add("LocaleNames_kkj");
        PUBLIC_IDENTIFIERS.add("LocaleNames_kl");
        PUBLIC_IDENTIFIERS.add("LocaleNames_kln");
        PUBLIC_IDENTIFIERS.add("LocaleNames_km");
        PUBLIC_IDENTIFIERS.add("LocaleNames_kn");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ko");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ko_KP");
        PUBLIC_IDENTIFIERS.add("LocaleNames_kok");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ks");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ks_Deva");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ksb");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ksf");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ksh");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ku");
        PUBLIC_IDENTIFIERS.add("LocaleNames_kw");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ky");
        PUBLIC_IDENTIFIERS.add("LocaleNames_lag");
        PUBLIC_IDENTIFIERS.add("LocaleNames_lb");
        PUBLIC_IDENTIFIERS.add("LocaleNames_lg");
        PUBLIC_IDENTIFIERS.add("LocaleNames_lkt");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ln");
        PUBLIC_IDENTIFIERS.add("LocaleNames_lo");
        PUBLIC_IDENTIFIERS.add("LocaleNames_lrc");
        PUBLIC_IDENTIFIERS.add("LocaleNames_lt");
        PUBLIC_IDENTIFIERS.add("LocaleNames_lu");
        PUBLIC_IDENTIFIERS.add("LocaleNames_luo");
        PUBLIC_IDENTIFIERS.add("LocaleNames_luy");
        PUBLIC_IDENTIFIERS.add("LocaleNames_lv");
        PUBLIC_IDENTIFIERS.add("LocaleNames_mai");
        PUBLIC_IDENTIFIERS.add("LocaleNames_mas");
        PUBLIC_IDENTIFIERS.add("LocaleNames_mdf");
        PUBLIC_IDENTIFIERS.add("LocaleNames_mer");
        PUBLIC_IDENTIFIERS.add("LocaleNames_mfe");
        PUBLIC_IDENTIFIERS.add("LocaleNames_mg");
        PUBLIC_IDENTIFIERS.add("LocaleNames_mgh");
        PUBLIC_IDENTIFIERS.add("LocaleNames_mgo");
        PUBLIC_IDENTIFIERS.add("LocaleNames_mi");
        PUBLIC_IDENTIFIERS.add("LocaleNames_mk");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ml");
        PUBLIC_IDENTIFIERS.add("LocaleNames_mn");
        PUBLIC_IDENTIFIERS.add("LocaleNames_mni");
        PUBLIC_IDENTIFIERS.add("LocaleNames_mr");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ms");
        PUBLIC_IDENTIFIERS.add("LocaleNames_mt");
        PUBLIC_IDENTIFIERS.add("LocaleNames_mua");
        PUBLIC_IDENTIFIERS.add("LocaleNames_my");
        PUBLIC_IDENTIFIERS.add("LocaleNames_mzn");
        PUBLIC_IDENTIFIERS.add("LocaleNames_naq");
        PUBLIC_IDENTIFIERS.add("LocaleNames_nd");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ne");
        PUBLIC_IDENTIFIERS.add("LocaleNames_nl");
        PUBLIC_IDENTIFIERS.add("LocaleNames_nmg");
        PUBLIC_IDENTIFIERS.add("LocaleNames_nn");
        PUBLIC_IDENTIFIERS.add("LocaleNames_nnh");
        PUBLIC_IDENTIFIERS.add("LocaleNames_no");
        PUBLIC_IDENTIFIERS.add("LocaleNames_nus");
        PUBLIC_IDENTIFIERS.add("LocaleNames_nyn");
        PUBLIC_IDENTIFIERS.add("LocaleNames_oc");
        PUBLIC_IDENTIFIERS.add("LocaleNames_om");
        PUBLIC_IDENTIFIERS.add("LocaleNames_or");
        PUBLIC_IDENTIFIERS.add("LocaleNames_os");
        PUBLIC_IDENTIFIERS.add("LocaleNames_pa");
        PUBLIC_IDENTIFIERS.add("LocaleNames_pa_Arab");
        PUBLIC_IDENTIFIERS.add("LocaleNames_pcm");
        PUBLIC_IDENTIFIERS.add("LocaleNames_pis");
        PUBLIC_IDENTIFIERS.add("LocaleNames_pl");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ps");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ps_PK");
        PUBLIC_IDENTIFIERS.add("LocaleNames_pt");
        PUBLIC_IDENTIFIERS.add("LocaleNames_pt_PT");
        PUBLIC_IDENTIFIERS.add("LocaleNames_qu");
        PUBLIC_IDENTIFIERS.add("LocaleNames_raj");
        PUBLIC_IDENTIFIERS.add("LocaleNames_rm");
        PUBLIC_IDENTIFIERS.add("LocaleNames_rn");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ro");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ro_MD");
        PUBLIC_IDENTIFIERS.add("LocaleNames_rof");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ru");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ru_UA");
        PUBLIC_IDENTIFIERS.add("LocaleNames_rw");
        PUBLIC_IDENTIFIERS.add("LocaleNames_rwk");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sa");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sah");
        PUBLIC_IDENTIFIERS.add("LocaleNames_saq");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sat");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sbp");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sc");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sd");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sd_Deva");
        PUBLIC_IDENTIFIERS.add("LocaleNames_se");
        PUBLIC_IDENTIFIERS.add("LocaleNames_se_FI");
        PUBLIC_IDENTIFIERS.add("LocaleNames_seh");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ses");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sg");
        PUBLIC_IDENTIFIERS.add("LocaleNames_shi");
        PUBLIC_IDENTIFIERS.add("LocaleNames_shi_Latn");
        PUBLIC_IDENTIFIERS.add("LocaleNames_si");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sk");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sl");
        PUBLIC_IDENTIFIERS.add("LocaleNames_smn");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sn");
        PUBLIC_IDENTIFIERS.add("LocaleNames_so");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sq");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sr");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sr_Cyrl_BA");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sr_Cyrl_ME");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sr_Cyrl_XK");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sr_Latn");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sr_Latn_BA");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sr_Latn_ME");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sr_Latn_XK");
        PUBLIC_IDENTIFIERS.add("LocaleNames_su");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sv");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sv_FI");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sw");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sw_CD");
        PUBLIC_IDENTIFIERS.add("LocaleNames_sw_KE");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ta");
        PUBLIC_IDENTIFIERS.add("LocaleNames_te");
        PUBLIC_IDENTIFIERS.add("LocaleNames_teo");
        PUBLIC_IDENTIFIERS.add("LocaleNames_tg");
        PUBLIC_IDENTIFIERS.add("LocaleNames_th");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ti");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ti_ER");
        PUBLIC_IDENTIFIERS.add("LocaleNames_tk");
        PUBLIC_IDENTIFIERS.add("LocaleNames_to");
        PUBLIC_IDENTIFIERS.add("LocaleNames_tok");
        PUBLIC_IDENTIFIERS.add("LocaleNames_tr");
        PUBLIC_IDENTIFIERS.add("LocaleNames_tt");
        PUBLIC_IDENTIFIERS.add("LocaleNames_twq");
        PUBLIC_IDENTIFIERS.add("LocaleNames_tzm");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ug");
        PUBLIC_IDENTIFIERS.add("LocaleNames_uk");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ur");
        PUBLIC_IDENTIFIERS.add("LocaleNames_ur_IN");
        PUBLIC_IDENTIFIERS.add("LocaleNames_uz");
        PUBLIC_IDENTIFIERS.add("LocaleNames_uz_Arab");
        PUBLIC_IDENTIFIERS.add("LocaleNames_uz_Cyrl");
        PUBLIC_IDENTIFIERS.add("LocaleNames_vai");
        PUBLIC_IDENTIFIERS.add("LocaleNames_vai_Latn");
        PUBLIC_IDENTIFIERS.add("LocaleNames_vi");
        PUBLIC_IDENTIFIERS.add("LocaleNames_vun");
        PUBLIC_IDENTIFIERS.add("LocaleNames_wae");
        PUBLIC_IDENTIFIERS.add("LocaleNames_wo");
        PUBLIC_IDENTIFIERS.add("LocaleNames_xh");
        PUBLIC_IDENTIFIERS.add("LocaleNames_xog");
        PUBLIC_IDENTIFIERS.add("LocaleNames_yav");
        PUBLIC_IDENTIFIERS.add("LocaleNames_yi");
        PUBLIC_IDENTIFIERS.add("LocaleNames_yo");
        PUBLIC_IDENTIFIERS.add("LocaleNames_yo_BJ");
        PUBLIC_IDENTIFIERS.add("LocaleNames_yrl");
        PUBLIC_IDENTIFIERS.add("LocaleNames_yrl_CO");
        PUBLIC_IDENTIFIERS.add("LocaleNames_yrl_VE");
        PUBLIC_IDENTIFIERS.add("LocaleNames_yue");
        PUBLIC_IDENTIFIERS.add("LocaleNames_yue_Hans");
        PUBLIC_IDENTIFIERS.add("LocaleNames_zgh");
        PUBLIC_IDENTIFIERS.add("LocaleNames_zh");
        PUBLIC_IDENTIFIERS.add("LocaleNames_zh_Hant");
        PUBLIC_IDENTIFIERS.add("LocaleNames_zh_Hant_HK");
        PUBLIC_IDENTIFIERS.add("LocaleNames_zu");
        PUBLIC_IDENTIFIERS.add("LocaleNamesBundle");
        PUBLIC_IDENTIFIERS.add("LocaleNamesProvider");
        PUBLIC_IDENTIFIERS.add("LocaleObjectCache");
        PUBLIC_IDENTIFIERS.add("LocaleProviderAdapter");
        PUBLIC_IDENTIFIERS.add("LocaleResources");
        PUBLIC_IDENTIFIERS.add("LocaleServiceProvider");
        PUBLIC_IDENTIFIERS.add("LocaleSyntaxException");
        PUBLIC_IDENTIFIERS.add("LocaleUtility");
        PUBLIC_IDENTIFIERS.add("LocalEventTimer");
        PUBLIC_IDENTIFIERS.add("LocalExecutionControl");
        PUBLIC_IDENTIFIERS.add("LocalExecutionControlProvider");
        PUBLIC_IDENTIFIERS.add("LocalGregorianCalendar");
        PUBLIC_IDENTIFIERS.add("LocalizedLoggerWrapper");
        PUBLIC_IDENTIFIERS.add("LocalizedMessage");
        PUBLIC_IDENTIFIERS.add("LocalizedObjectGetter");
        PUBLIC_IDENTIFIERS.add("LocalizedString");
        PUBLIC_IDENTIFIERS.add("LocalMonitoredVm");
        PUBLIC_IDENTIFIERS.add("LocalSeqNumber");
        PUBLIC_IDENTIFIERS.add("LocalSym");
        PUBLIC_IDENTIFIERS.add("LocalVariable");
        PUBLIC_IDENTIFIERS.add("LocalVariableAnnotationNode");
        PUBLIC_IDENTIFIERS.add("LocalVariableGen");
        PUBLIC_IDENTIFIERS.add("LocalVariableImpl");
        PUBLIC_IDENTIFIERS.add("LocalVariableInstruction");
        PUBLIC_IDENTIFIERS.add("LocalVariableNode");
        PUBLIC_IDENTIFIERS.add("LocalVariableRegistry");
        PUBLIC_IDENTIFIERS.add("LocalVariablesSorter");
        PUBLIC_IDENTIFIERS.add("LocalVariableTable");
        PUBLIC_IDENTIFIERS.add("LocalVariableTable_attribute");
        PUBLIC_IDENTIFIERS.add("LocalVariableTableElement");
        PUBLIC_IDENTIFIERS.add("LocalVariableTableWriter");
        PUBLIC_IDENTIFIERS.add("LocalVariableTypeTable");
        PUBLIC_IDENTIFIERS.add("LocalVariableTypeTable_attribute");
        PUBLIC_IDENTIFIERS.add("LocalVariableTypeTableWriter");
        PUBLIC_IDENTIFIERS.add("LocalVirtualMachine");
        PUBLIC_IDENTIFIERS.add("LocalVmManager");
        PUBLIC_IDENTIFIERS.add("Locatable");
        PUBLIC_IDENTIFIERS.add("LocatableEvent");
        PUBLIC_IDENTIFIERS.add("Location");
        PUBLIC_IDENTIFIERS.add("LocationArray");
        PUBLIC_IDENTIFIERS.add("LocationHandler");
        PUBLIC_IDENTIFIERS.add("LocationImpl");
        PUBLIC_IDENTIFIERS.add("LocationPath");
        PUBLIC_IDENTIFIERS.add("LocationPathPattern");
        PUBLIC_IDENTIFIERS.add("Locations");
        PUBLIC_IDENTIFIERS.add("LocationValue");
        PUBLIC_IDENTIFIERS.add("Locator");
        PUBLIC_IDENTIFIERS.add("Locator2");
        PUBLIC_IDENTIFIERS.add("Locator2Impl");
        PUBLIC_IDENTIFIERS.add("LocatorImpl");
        PUBLIC_IDENTIFIERS.add("LocatorProxy");
        PUBLIC_IDENTIFIERS.add("LocatorWrapper");
        PUBLIC_IDENTIFIERS.add("Lock");
        PUBLIC_IDENTIFIERS.add("LockInfo");
        PUBLIC_IDENTIFIERS.add("LockInfoCompositeData");
        PUBLIC_IDENTIFIERS.add("LockObject");
        PUBLIC_IDENTIFIERS.add("LockSupport");
        PUBLIC_IDENTIFIERS.add("LocPathIterator");
        PUBLIC_IDENTIFIERS.add("Log");
        PUBLIC_IDENTIFIERS.add("LogFile");
        PUBLIC_IDENTIFIERS.add("Logger");
        PUBLIC_IDENTIFIERS.add("LoggerConfiguration");
        PUBLIC_IDENTIFIERS.add("LoggerFactory");
        PUBLIC_IDENTIFIERS.add("LoggerFinder");
        PUBLIC_IDENTIFIERS.add("LoggerWrapper");
        PUBLIC_IDENTIFIERS.add("LoggingMXBean");
        PUBLIC_IDENTIFIERS.add("LogHandler");
        PUBLIC_IDENTIFIERS.add("LoginConfigImpl");
        PUBLIC_IDENTIFIERS.add("LoginContext");
        PUBLIC_IDENTIFIERS.add("LoginException");
        PUBLIC_IDENTIFIERS.add("LoginModule");
        PUBLIC_IDENTIFIERS.add("LoginModuleControlFlag");
        PUBLIC_IDENTIFIERS.add("LoginOptions");
        PUBLIC_IDENTIFIERS.add("LogInputStream");
        PUBLIC_IDENTIFIERS.add("LogManager");
        PUBLIC_IDENTIFIERS.add("LogManagerAccess");
        PUBLIC_IDENTIFIERS.add("LogOutputStream");
        PUBLIC_IDENTIFIERS.add("LogRecord");
        PUBLIC_IDENTIFIERS.add("LogStream");
        PUBLIC_IDENTIFIERS.add("Long");
        PUBLIC_IDENTIFIERS.add("LongAccumulator");
        PUBLIC_IDENTIFIERS.add("LongAdder");
        PUBLIC_IDENTIFIERS.add("LongArrayCounter");
        PUBLIC_IDENTIFIERS.add("LongArrayWrapper");
        PUBLIC_IDENTIFIERS.add("LongBinaryOperator");
        PUBLIC_IDENTIFIERS.add("LongBuffer");
        PUBLIC_IDENTIFIERS.add("LongCellRenderer");
        PUBLIC_IDENTIFIERS.add("LongConsumer");
        PUBLIC_IDENTIFIERS.add("LongCounter");
        PUBLIC_IDENTIFIERS.add("LongEditor");
        PUBLIC_IDENTIFIERS.add("LongField");
        PUBLIC_IDENTIFIERS.add("LongFunction");
        PUBLIC_IDENTIFIERS.add("LongHashMap");
        PUBLIC_IDENTIFIERS.add("LongMapMultiConsumer");
        PUBLIC_IDENTIFIERS.add("LongMonitor");
        PUBLIC_IDENTIFIERS.add("LongPredicate");
        PUBLIC_IDENTIFIERS.add("LongSignature");
        PUBLIC_IDENTIFIERS.add("LongStream");
        PUBLIC_IDENTIFIERS.add("LongSummaryStatistics");
        PUBLIC_IDENTIFIERS.add("LongSupplier");
        PUBLIC_IDENTIFIERS.add("LongToDoubleFunction");
        PUBLIC_IDENTIFIERS.add("LongToIntFunction");
        PUBLIC_IDENTIFIERS.add("LongTreeNodeAdapter");
        PUBLIC_IDENTIFIERS.add("LongType");
        PUBLIC_IDENTIFIERS.add("LongTypeImpl");
        PUBLIC_IDENTIFIERS.add("LongUnaryOperator");
        PUBLIC_IDENTIFIERS.add("LongValue");
        PUBLIC_IDENTIFIERS.add("LongValueImpl");
        PUBLIC_IDENTIFIERS.add("LongVector");
        PUBLIC_IDENTIFIERS.add("LookAndFeel");
        PUBLIC_IDENTIFIERS.add("LookAndFeelInfo");
        PUBLIC_IDENTIFIERS.add("LookupOp");
        PUBLIC_IDENTIFIERS.add("LOOKUPSWITCH");
        PUBLIC_IDENTIFIERS.add("LookupSwitchInsnNode");
        PUBLIC_IDENTIFIERS.add("LookupswitchPair");
        PUBLIC_IDENTIFIERS.add("LookupTable");
        PUBLIC_IDENTIFIERS.add("LoopBasedPipe");
        PUBLIC_IDENTIFIERS.add("LoopNode");
        PUBLIC_IDENTIFIERS.add("LoopPipe");
        PUBLIC_IDENTIFIERS.add("LOR");
        PUBLIC_IDENTIFIERS.add("Lower");
        PUBLIC_IDENTIFIERS.add("lr_parser");
        PUBLIC_IDENTIFIERS.add("LREM");
        PUBLIC_IDENTIFIERS.add("LRETURN");
        PUBLIC_IDENTIFIERS.add("LSException");
        PUBLIC_IDENTIFIERS.add("LSHL");
        PUBLIC_IDENTIFIERS.add("LSHR");
        PUBLIC_IDENTIFIERS.add("LSInput");
        PUBLIC_IDENTIFIERS.add("LSInputList");
        PUBLIC_IDENTIFIERS.add("LSLoadEvent");
        PUBLIC_IDENTIFIERS.add("LSOutput");
        PUBLIC_IDENTIFIERS.add("LSParser");
        PUBLIC_IDENTIFIERS.add("LSParserFilter");
        PUBLIC_IDENTIFIERS.add("LSProgressEvent");
        PUBLIC_IDENTIFIERS.add("LSResourceResolver");
        PUBLIC_IDENTIFIERS.add("LSSerializer");
        PUBLIC_IDENTIFIERS.add("LSSerializerFilter");
        PUBLIC_IDENTIFIERS.add("LSTORE");
        PUBLIC_IDENTIFIERS.add("LSUB");
        PUBLIC_IDENTIFIERS.add("Lt");
        PUBLIC_IDENTIFIERS.add("Lte");
        PUBLIC_IDENTIFIERS.add("LUSHR");
        PUBLIC_IDENTIFIERS.add("LXOR");
        PUBLIC_IDENTIFIERS.add("LZWCompressor");
        PUBLIC_IDENTIFIERS.add("LZWStringTable");
        PUBLIC_IDENTIFIERS.add("Mac");
        PUBLIC_IDENTIFIERS.add("MacArabic");
        PUBLIC_IDENTIFIERS.add("MacCentralEurope");
        PUBLIC_IDENTIFIERS.add("MacCroatian");
        PUBLIC_IDENTIFIERS.add("MacCyrillic");
        PUBLIC_IDENTIFIERS.add("MacDingbat");
        PUBLIC_IDENTIFIERS.add("MacGreek");
        PUBLIC_IDENTIFIERS.add("MachCallJavaNode");
        PUBLIC_IDENTIFIERS.add("MachCallNode");
        PUBLIC_IDENTIFIERS.add("MachCallRuntimeNode");
        PUBLIC_IDENTIFIERS.add("MachCallStaticJavaNode");
        PUBLIC_IDENTIFIERS.add("MacHebrew");
        PUBLIC_IDENTIFIERS.add("MachIfNode");
        PUBLIC_IDENTIFIERS.add("MachineDescription");
        PUBLIC_IDENTIFIERS.add("MachineDescriptionAArch64");
        PUBLIC_IDENTIFIERS.add("MachineDescriptionAMD64");
        PUBLIC_IDENTIFIERS.add("MachineDescriptionIntelX86");
        PUBLIC_IDENTIFIERS.add("MachineDescriptionPPC64");
        PUBLIC_IDENTIFIERS.add("MachineDescriptionRISCV64");
        PUBLIC_IDENTIFIERS.add("MachineDescriptionTwosComplement");
        PUBLIC_IDENTIFIERS.add("MachineTypes");
        PUBLIC_IDENTIFIERS.add("MachNode");
        PUBLIC_IDENTIFIERS.add("MachReturnNode");
        PUBLIC_IDENTIFIERS.add("MachSafePointNode");
        PUBLIC_IDENTIFIERS.add("MacIceland");
        PUBLIC_IDENTIFIERS.add("MacOsAArch64CallArranger");
        PUBLIC_IDENTIFIERS.add("MacOsAArch64VaList");
        PUBLIC_IDENTIFIERS.add("MacOutputStream");
        PUBLIC_IDENTIFIERS.add("Macro");
        PUBLIC_IDENTIFIERS.add("MacRoman");
        PUBLIC_IDENTIFIERS.add("MacRomania");
        PUBLIC_IDENTIFIERS.add("MacSpi");
        PUBLIC_IDENTIFIERS.add("MacSymbol");
        PUBLIC_IDENTIFIERS.add("MacThai");
        PUBLIC_IDENTIFIERS.add("MacTurkish");
        PUBLIC_IDENTIFIERS.add("MacUkraine");
        PUBLIC_IDENTIFIERS.add("MailToURLConnection");
        PUBLIC_IDENTIFIERS.add("Main");
        PUBLIC_IDENTIFIERS.add("MalformedByteSequenceException");
        PUBLIC_IDENTIFIERS.add("MalformedFrame");
        PUBLIC_IDENTIFIERS.add("MalformedInputException");
        PUBLIC_IDENTIFIERS.add("MalformedLinkException");
        PUBLIC_IDENTIFIERS.add("MalformedObjectNameException");
        PUBLIC_IDENTIFIERS.add("MalformedParameterizedTypeException");
        PUBLIC_IDENTIFIERS.add("MalformedParametersException");
        PUBLIC_IDENTIFIERS.add("MalformedURIException");
        PUBLIC_IDENTIFIERS.add("MalformedURLException");
        PUBLIC_IDENTIFIERS.add("ManagedBlocker");
        PUBLIC_IDENTIFIERS.add("ManagementFactory");
        PUBLIC_IDENTIFIERS.add("ManagementFactoryHelper");
        PUBLIC_IDENTIFIERS.add("ManagerFactoryParameters");
        PUBLIC_IDENTIFIERS.add("MandatoryWarningHandler");
        PUBLIC_IDENTIFIERS.add("Manifest");
        PUBLIC_IDENTIFIERS.add("ManifestDigester");
        PUBLIC_IDENTIFIERS.add("ManifestEntryVerifier");
        PUBLIC_IDENTIFIERS.add("Map");
        PUBLIC_IDENTIFIERS.add("MapMode");
        PUBLIC_IDENTIFIERS.add("MappedByteBuffer");
        PUBLIC_IDENTIFIERS.add("MappedByteBufferDataSource");
        PUBLIC_IDENTIFIERS.add("MappedMemorySegmentImpl");
        PUBLIC_IDENTIFIERS.add("MappedMXBeanType");
        PUBLIC_IDENTIFIERS.add("MappingSubscriber");
        PUBLIC_IDENTIFIERS.add("MapVisitor");
        PUBLIC_IDENTIFIERS.add("MarginBorder");
        PUBLIC_IDENTIFIERS.add("Mark");
        PUBLIC_IDENTIFIERS.add("MarkBits");
        PUBLIC_IDENTIFIERS.add("MarkerComments");
        PUBLIC_IDENTIFIERS.add("MarshalException");
        PUBLIC_IDENTIFIERS.add("MarshalInputStream");
        PUBLIC_IDENTIFIERS.add("MarshalOutputStream");
        PUBLIC_IDENTIFIERS.add("MaskBlit");
        PUBLIC_IDENTIFIERS.add("MaskFill");
        PUBLIC_IDENTIFIERS.add("MaskFormatter");
        PUBLIC_IDENTIFIERS.add("MaskingCallback");
        PUBLIC_IDENTIFIERS.add("MaskTile");
        PUBLIC_IDENTIFIERS.add("MaskTileManager");
        PUBLIC_IDENTIFIERS.add("MastheadIcon");
        PUBLIC_IDENTIFIERS.add("Match");
        PUBLIC_IDENTIFIERS.add("MatchBindings");
        PUBLIC_IDENTIFIERS.add("MatchBindingsComputer");
        PUBLIC_IDENTIFIERS.add("Matcher");
        PUBLIC_IDENTIFIERS.add("MatchingUtils");
        PUBLIC_IDENTIFIERS.add("MatchPatternIterator");
        PUBLIC_IDENTIFIERS.add("MatchResult");
        PUBLIC_IDENTIFIERS.add("Math");
        PUBLIC_IDENTIFIERS.add("MatteBorder");
        PUBLIC_IDENTIFIERS.add("MatteBorderUIResource");
        PUBLIC_IDENTIFIERS.add("MaximizableInternalFrame");
        PUBLIC_IDENTIFIERS.add("MaximizeAction");
        PUBLIC_IDENTIFIERS.add("MBeanAttributeInfo");
        PUBLIC_IDENTIFIERS.add("MBeanConstructorInfo");
        PUBLIC_IDENTIFIERS.add("MBeanException");
        PUBLIC_IDENTIFIERS.add("MBeanFeatureInfo");
        PUBLIC_IDENTIFIERS.add("MBeanInfo");
        PUBLIC_IDENTIFIERS.add("MBeanInstantiator");
        PUBLIC_IDENTIFIERS.add("MBeanNotificationInfo");
        PUBLIC_IDENTIFIERS.add("MBeanOperationInfo");
        PUBLIC_IDENTIFIERS.add("MBeanParameterInfo");
        PUBLIC_IDENTIFIERS.add("MBeanPermission");
        PUBLIC_IDENTIFIERS.add("MBeanRegistration");
        PUBLIC_IDENTIFIERS.add("MBeanRegistrationException");
        PUBLIC_IDENTIFIERS.add("MBeanServer");
        PUBLIC_IDENTIFIERS.add("MBeanServerAccessController");
        PUBLIC_IDENTIFIERS.add("MBeanServerBuilder");
        PUBLIC_IDENTIFIERS.add("MBeanServerConnection");
        PUBLIC_IDENTIFIERS.add("MBeanServerDelegate");
        PUBLIC_IDENTIFIERS.add("MBeanServerDelegateMBean");
        PUBLIC_IDENTIFIERS.add("MBeanServerFactory");
        PUBLIC_IDENTIFIERS.add("MBeanServerFileAccessController");
        PUBLIC_IDENTIFIERS.add("MBeanServerForwarder");
        PUBLIC_IDENTIFIERS.add("MBeanServerInterceptor");
        PUBLIC_IDENTIFIERS.add("MBeanServerInvocationHandler");
        PUBLIC_IDENTIFIERS.add("MBeanServerNotification");
        PUBLIC_IDENTIFIERS.add("MBeanServerNotificationFilter");
        PUBLIC_IDENTIFIERS.add("MBeanServerPermission");
        PUBLIC_IDENTIFIERS.add("MBeansTab");
        PUBLIC_IDENTIFIERS.add("MBeanSupport");
        PUBLIC_IDENTIFIERS.add("MBeanTrustPermission");
        PUBLIC_IDENTIFIERS.add("MechanismFactory");
        PUBLIC_IDENTIFIERS.add("Media");
        PUBLIC_IDENTIFIERS.add("MediaList");
        PUBLIC_IDENTIFIERS.add("MediaName");
        PUBLIC_IDENTIFIERS.add("MediaSize");
        PUBLIC_IDENTIFIERS.add("MediaSizeName");
        PUBLIC_IDENTIFIERS.add("MediaTracker");
        PUBLIC_IDENTIFIERS.add("MediaTray");
        PUBLIC_IDENTIFIERS.add("Member");
        PUBLIC_IDENTIFIERS.add("MemberEnter");
        PUBLIC_IDENTIFIERS.add("MemberFunctionType");
        PUBLIC_IDENTIFIERS.add("MemberReferenceTree");
        PUBLIC_IDENTIFIERS.add("MemberSelectTree");
        PUBLIC_IDENTIFIERS.add("MembershipKey");
        PUBLIC_IDENTIFIERS.add("MemberSummaryBuilder");
        PUBLIC_IDENTIFIERS.add("MemberSummaryWriter");
        PUBLIC_IDENTIFIERS.add("MemberWriter");
        PUBLIC_IDENTIFIERS.add("MemoizedBoolean");
        PUBLIC_IDENTIFIERS.add("MemoizedByte");
        PUBLIC_IDENTIFIERS.add("MemoizedChar");
        PUBLIC_IDENTIFIERS.add("MemoizedDouble");
        PUBLIC_IDENTIFIERS.add("MemoizedFloat");
        PUBLIC_IDENTIFIERS.add("MemoizedInt");
        PUBLIC_IDENTIFIERS.add("MemoizedLong");
        PUBLIC_IDENTIFIERS.add("MemoizedObject");
        PUBLIC_IDENTIFIERS.add("MemoizedShort");
        PUBLIC_IDENTIFIERS.add("MemoryAccessProvider");
        PUBLIC_IDENTIFIERS.add("MemoryAction");
        PUBLIC_IDENTIFIERS.add("MemoryAddress");
        PUBLIC_IDENTIFIERS.add("MemoryBarriers");
        PUBLIC_IDENTIFIERS.add("MemoryCache");
        PUBLIC_IDENTIFIERS.add("MemoryCacheImageInputStream");
        PUBLIC_IDENTIFIERS.add("MemoryCacheImageOutputStream");
        PUBLIC_IDENTIFIERS.add("MemoryCredentialsCache");
        PUBLIC_IDENTIFIERS.add("MemoryHandler");
        PUBLIC_IDENTIFIERS.add("MemoryImageSource");
        PUBLIC_IDENTIFIERS.add("MemoryLayout");
        PUBLIC_IDENTIFIERS.add("MemoryManagerMXBean");
        PUBLIC_IDENTIFIERS.add("MemoryMXBean");
        PUBLIC_IDENTIFIERS.add("MemoryNotificationInfo");
        PUBLIC_IDENTIFIERS.add("MemoryNotifInfoCompositeData");
        PUBLIC_IDENTIFIERS.add("MemoryPanel");
        PUBLIC_IDENTIFIERS.add("MemoryPoolMXBean");
        PUBLIC_IDENTIFIERS.add("MemoryPoolProxy");
        PUBLIC_IDENTIFIERS.add("MemoryPoolStat");
        PUBLIC_IDENTIFIERS.add("MemorySegment");
        PUBLIC_IDENTIFIERS.add("MemorySessionImpl");
        PUBLIC_IDENTIFIERS.add("MemoryUsage");
        PUBLIC_IDENTIFIERS.add("MemoryUsageCompositeData");
        PUBLIC_IDENTIFIERS.add("MemoryViewer");
        PUBLIC_IDENTIFIERS.add("MemRegion");
        PUBLIC_IDENTIFIERS.add("Menu");
        PUBLIC_IDENTIFIERS.add("MenuAccessor");
        PUBLIC_IDENTIFIERS.add("MenuBar");
        PUBLIC_IDENTIFIERS.add("MenuBarAccessor");
        PUBLIC_IDENTIFIERS.add("MenuBarBorder");
        PUBLIC_IDENTIFIERS.add("MenuBarPeer");
        PUBLIC_IDENTIFIERS.add("MenuBarUI");
        PUBLIC_IDENTIFIERS.add("MenuComponent");
        PUBLIC_IDENTIFIERS.add("MenuComponentAccessor");
        PUBLIC_IDENTIFIERS.add("MenuComponentPeer");
        PUBLIC_IDENTIFIERS.add("MenuContainer");
        PUBLIC_IDENTIFIERS.add("MenuDragMouseEvent");
        PUBLIC_IDENTIFIERS.add("MenuDragMouseListener");
        PUBLIC_IDENTIFIERS.add("MenuElement");
        PUBLIC_IDENTIFIERS.add("MenuEvent");
        PUBLIC_IDENTIFIERS.add("MenuItem");
        PUBLIC_IDENTIFIERS.add("MenuItemAccessor");
        PUBLIC_IDENTIFIERS.add("MenuItemBorder");
        PUBLIC_IDENTIFIERS.add("MenuItemCheckIconFactory");
        PUBLIC_IDENTIFIERS.add("MenuItemLayoutHelper");
        PUBLIC_IDENTIFIERS.add("MenuItemPeer");
        PUBLIC_IDENTIFIERS.add("MenuItemUI");
        PUBLIC_IDENTIFIERS.add("MenuKeyEvent");
        PUBLIC_IDENTIFIERS.add("MenuKeyListener");
        PUBLIC_IDENTIFIERS.add("MenuListener");
        PUBLIC_IDENTIFIERS.add("MenuPeer");
        PUBLIC_IDENTIFIERS.add("MenuSelectionManager");
        PUBLIC_IDENTIFIERS.add("MenuShortcut");
        PUBLIC_IDENTIFIERS.add("MessageDigest");
        PUBLIC_IDENTIFIERS.add("MessageDigestSpi");
        PUBLIC_IDENTIFIERS.add("MessageDigestSpi2");
        PUBLIC_IDENTIFIERS.add("MessageEncoder");
        PUBLIC_IDENTIFIERS.add("MessageFormat");
        PUBLIC_IDENTIFIERS.add("MessageFormatter");
        PUBLIC_IDENTIFIERS.add("MessageHandler");
        PUBLIC_IDENTIFIERS.add("MessageHeader");
        PUBLIC_IDENTIFIERS.add("MessageInfo");
        PUBLIC_IDENTIFIERS.add("MessageInfoImpl");
        PUBLIC_IDENTIFIERS.add("MessageOutput");
        PUBLIC_IDENTIFIERS.add("MessageProp");
        PUBLIC_IDENTIFIERS.add("MessageQueue");
        PUBLIC_IDENTIFIERS.add("MessageQueueBackend");
        PUBLIC_IDENTIFIERS.add("Messager");
        PUBLIC_IDENTIFIERS.add("Messages");
        PUBLIC_IDENTIFIERS.add("MetaAccessProvider");
        PUBLIC_IDENTIFIERS.add("Metadata");
        PUBLIC_IDENTIFIERS.add("MetadataDefinition");
        PUBLIC_IDENTIFIERS.add("MetadataField");
        PUBLIC_IDENTIFIERS.add("MetadataTreeNodeAdapter");
        PUBLIC_IDENTIFIERS.add("MetadataVisitor");
        PUBLIC_IDENTIFIERS.add("MetaEventListener");
        PUBLIC_IDENTIFIERS.add("MetaKeywords");
        PUBLIC_IDENTIFIERS.add("MetalBorders");
        PUBLIC_IDENTIFIERS.add("MetalButtonUI");
        PUBLIC_IDENTIFIERS.add("MetalCheckBoxIcon");
        PUBLIC_IDENTIFIERS.add("MetalCheckBoxUI");
        PUBLIC_IDENTIFIERS.add("MetalComboBoxButton");
        PUBLIC_IDENTIFIERS.add("MetalComboBoxEditor");
        PUBLIC_IDENTIFIERS.add("MetalComboBoxIcon");
        PUBLIC_IDENTIFIERS.add("MetalComboBoxLayoutManager");
        PUBLIC_IDENTIFIERS.add("MetalComboBoxUI");
        PUBLIC_IDENTIFIERS.add("MetalComboPopup");
        PUBLIC_IDENTIFIERS.add("MetalContainerListener");
        PUBLIC_IDENTIFIERS.add("MetalDesktopIconUI");
        PUBLIC_IDENTIFIERS.add("MetalDividerLayout");
        PUBLIC_IDENTIFIERS.add("MetalDockingListener");
        PUBLIC_IDENTIFIERS.add("MetalFileChooserUI");
        PUBLIC_IDENTIFIERS.add("MetalIconFactory");
        PUBLIC_IDENTIFIERS.add("MetalInternalFrameTitlePane");
        PUBLIC_IDENTIFIERS.add("MetalInternalFrameUI");
        PUBLIC_IDENTIFIERS.add("MetalLabelUI");
        PUBLIC_IDENTIFIERS.add("MetalLookAndFeel");
        PUBLIC_IDENTIFIERS.add("MetalMenuBarUI");
        PUBLIC_IDENTIFIERS.add("MetalPopupMenuSeparatorUI");
        PUBLIC_IDENTIFIERS.add("MetalProgressBarUI");
        PUBLIC_IDENTIFIERS.add("MetalPropertyChangeListener");
        PUBLIC_IDENTIFIERS.add("MetalPropertyListener");
        PUBLIC_IDENTIFIERS.add("MetalRadioButtonUI");
        PUBLIC_IDENTIFIERS.add("MetalRolloverListener");
        PUBLIC_IDENTIFIERS.add("MetalRootPaneUI");
        PUBLIC_IDENTIFIERS.add("MetalScrollBarUI");
        PUBLIC_IDENTIFIERS.add("MetalScrollButton");
        PUBLIC_IDENTIFIERS.add("MetalScrollPaneUI");
        PUBLIC_IDENTIFIERS.add("MetalSeparatorUI");
        PUBLIC_IDENTIFIERS.add("MetalSliderUI");
        PUBLIC_IDENTIFIERS.add("MetalSplitPaneUI");
        PUBLIC_IDENTIFIERS.add("MetalTabbedPaneUI");
        PUBLIC_IDENTIFIERS.add("MetalTextFieldUI");
        PUBLIC_IDENTIFIERS.add("MetalTheme");
        PUBLIC_IDENTIFIERS.add("MetalToggleButtonUI");
        PUBLIC_IDENTIFIERS.add("MetalToolBarUI");
        PUBLIC_IDENTIFIERS.add("MetalToolTipUI");
        PUBLIC_IDENTIFIERS.add("MetalTreeUI");
        PUBLIC_IDENTIFIERS.add("MetaMessage");
        PUBLIC_IDENTIFIERS.add("MetaspaceObj");
        PUBLIC_IDENTIFIERS.add("MetaUtil");
        PUBLIC_IDENTIFIERS.add("MeteredStream");
        PUBLIC_IDENTIFIERS.add("Method");
        PUBLIC_IDENTIFIERS.add("MethodAccessor");
        PUBLIC_IDENTIFIERS.add("MethodArray");
        PUBLIC_IDENTIFIERS.add("MethodBuilder");
        PUBLIC_IDENTIFIERS.add("MethodCounters");
        PUBLIC_IDENTIFIERS.add("MethodData");
        PUBLIC_IDENTIFIERS.add("MethodDataInterface");
        PUBLIC_IDENTIFIERS.add("MethodDescriptor");
        PUBLIC_IDENTIFIERS.add("MethodEntryEvent");
        PUBLIC_IDENTIFIERS.add("MethodEntryRequest");
        PUBLIC_IDENTIFIERS.add("MethodExitEvent");
        PUBLIC_IDENTIFIERS.add("MethodExitRequest");
        PUBLIC_IDENTIFIERS.add("MethodGen");
        PUBLIC_IDENTIFIERS.add("MethodGenerator");
        PUBLIC_IDENTIFIERS.add("MethodHandle");
        PUBLIC_IDENTIFIERS.add("MethodHandleAccessProvider");
        PUBLIC_IDENTIFIERS.add("MethodHandleDesc");
        PUBLIC_IDENTIFIERS.add("MethodHandleInfo");
        PUBLIC_IDENTIFIERS.add("MethodHandleProxies");
        PUBLIC_IDENTIFIERS.add("MethodHandles");
        PUBLIC_IDENTIFIERS.add("MethodHandlesAdapterBlob");
        PUBLIC_IDENTIFIERS.add("MethodHandleSymbol");
        PUBLIC_IDENTIFIERS.add("MethodHandleTransformer");
        PUBLIC_IDENTIFIERS.add("MethodImpl");
        PUBLIC_IDENTIFIERS.add("MethodInfo");
        PUBLIC_IDENTIFIERS.add("MethodInsnNode");
        PUBLIC_IDENTIFIERS.add("MethodInvocationTree");
        PUBLIC_IDENTIFIERS.add("MethodNode");
        PUBLIC_IDENTIFIERS.add("MethodObserver");
        PUBLIC_IDENTIFIERS.add("MethodParameter");
        PUBLIC_IDENTIFIERS.add("MethodParameters");
        PUBLIC_IDENTIFIERS.add("MethodParameters_attribute");
        PUBLIC_IDENTIFIERS.add("MethodRemapper");
        PUBLIC_IDENTIFIERS.add("MethodRepository");
        PUBLIC_IDENTIFIERS.add("MethodScope");
        PUBLIC_IDENTIFIERS.add("MethodSig");
        PUBLIC_IDENTIFIERS.add("MethodSnippet");
        PUBLIC_IDENTIFIERS.add("MethodSymbol");
        PUBLIC_IDENTIFIERS.add("MethodTree");
        PUBLIC_IDENTIFIERS.add("MethodType");
        PUBLIC_IDENTIFIERS.add("MethodTypeConversionStrategy");
        PUBLIC_IDENTIFIERS.add("MethodTypeDesc");
        PUBLIC_IDENTIFIERS.add("MethodTypeSignature");
        PUBLIC_IDENTIFIERS.add("MethodVisitor");
        PUBLIC_IDENTIFIERS.add("MethodWriter");
        PUBLIC_IDENTIFIERS.add("MethodWriterImpl");
        PUBLIC_IDENTIFIERS.add("Metrics");
        PUBLIC_IDENTIFIERS.add("MFontConfiguration");
        PUBLIC_IDENTIFIERS.add("MGF1ParameterSpec");
        PUBLIC_IDENTIFIERS.add("MgmtData");
        PUBLIC_IDENTIFIERS.add("MidiChannel");
        PUBLIC_IDENTIFIERS.add("MidiDevice");
        PUBLIC_IDENTIFIERS.add("MidiDeviceProvider");
        PUBLIC_IDENTIFIERS.add("MidiDeviceReceiver");
        PUBLIC_IDENTIFIERS.add("MidiDeviceTransmitter");
        PUBLIC_IDENTIFIERS.add("MidiEvent");
        PUBLIC_IDENTIFIERS.add("MidiFileFormat");
        PUBLIC_IDENTIFIERS.add("MidiFileReader");
        PUBLIC_IDENTIFIERS.add("MidiFileWriter");
        PUBLIC_IDENTIFIERS.add("MidiMessage");
        PUBLIC_IDENTIFIERS.add("MidiSystem");
        PUBLIC_IDENTIFIERS.add("MidiUnavailableException");
        PUBLIC_IDENTIFIERS.add("MimeEntry");
        PUBLIC_IDENTIFIERS.add("MimeTable");
        PUBLIC_IDENTIFIERS.add("MimeTypeParseException");
        PUBLIC_IDENTIFIERS.add("MinimalHTMLWriter");
        PUBLIC_IDENTIFIERS.add("MinimizeAction");
        PUBLIC_IDENTIFIERS.add("Minus");
        PUBLIC_IDENTIFIERS.add("Mirror");
        PUBLIC_IDENTIFIERS.add("MirroredTypeException");
        PUBLIC_IDENTIFIERS.add("MirroredTypesException");
        PUBLIC_IDENTIFIERS.add("MirrorEvent");
        PUBLIC_IDENTIFIERS.add("MissingFormatArgumentException");
        PUBLIC_IDENTIFIERS.add("MissingFormatWidthException");
        PUBLIC_IDENTIFIERS.add("MissingInfoHandler");
        PUBLIC_IDENTIFIERS.add("MissingMemberHandlerFactory");
        PUBLIC_IDENTIFIERS.add("MissingResourceException");
        PUBLIC_IDENTIFIERS.add("MissingResourceFailureException");
        PUBLIC_IDENTIFIERS.add("MixedContentModel");
        PUBLIC_IDENTIFIERS.add("Mixer");
        PUBLIC_IDENTIFIERS.add("MixerProvider");
        PUBLIC_IDENTIFIERS.add("MLet");
        PUBLIC_IDENTIFIERS.add("MLetContent");
        PUBLIC_IDENTIFIERS.add("MLetMBean");
        PUBLIC_IDENTIFIERS.add("Mod");
        PUBLIC_IDENTIFIERS.add("ModalExclude");
        PUBLIC_IDENTIFIERS.add("ModalityEvent");
        PUBLIC_IDENTIFIERS.add("ModalityListener");
        PUBLIC_IDENTIFIERS.add("Mode");
        PUBLIC_IDENTIFIERS.add("ModelAbstractChannelMixer");
        PUBLIC_IDENTIFIERS.add("ModelAbstractOscillator");
        PUBLIC_IDENTIFIERS.add("ModelChannelMixer");
        PUBLIC_IDENTIFIERS.add("ModelDirectedPlayer");
        PUBLIC_IDENTIFIERS.add("ModelDirector");
        PUBLIC_IDENTIFIERS.add("ModelInstrument");
        PUBLIC_IDENTIFIERS.add("ModelListener");
        PUBLIC_IDENTIFIERS.add("ModelMBean");
        PUBLIC_IDENTIFIERS.add("ModelMBeanAttributeInfo");
        PUBLIC_IDENTIFIERS.add("ModelMBeanConstructorInfo");
        PUBLIC_IDENTIFIERS.add("ModelMBeanInfo");
        PUBLIC_IDENTIFIERS.add("ModelMBeanInfoSupport");
        PUBLIC_IDENTIFIERS.add("ModelMBeanNotificationBroadcaster");
        PUBLIC_IDENTIFIERS.add("ModelMBeanNotificationInfo");
        PUBLIC_IDENTIFIERS.add("ModelMBeanOperationInfo");
        PUBLIC_IDENTIFIERS.add("ModelOscillator");
        PUBLIC_IDENTIFIERS.add("ModelOscillatorStream");
        PUBLIC_IDENTIFIERS.add("ModelTransform");
        PUBLIC_IDENTIFIERS.add("ModelWavetable");
        PUBLIC_IDENTIFIERS.add("ModelWrapper");
        PUBLIC_IDENTIFIERS.add("ModifiableClassLoaderRepository");
        PUBLIC_IDENTIFIERS.add("ModificationItem");
        PUBLIC_IDENTIFIERS.add("ModificationWatchpointEvent");
        PUBLIC_IDENTIFIERS.add("ModificationWatchpointRequest");
        PUBLIC_IDENTIFIERS.add("Modifier");
        PUBLIC_IDENTIFIERS.add("ModifiersProvider");
        PUBLIC_IDENTIFIERS.add("ModifiersTree");
        PUBLIC_IDENTIFIERS.add("ModularJarArchive");
        PUBLIC_IDENTIFIERS.add("ModularRuntimeImage");
        PUBLIC_IDENTIFIERS.add("Module");
        PUBLIC_IDENTIFIERS.add("Module_attribute");
        PUBLIC_IDENTIFIERS.add("ModuleAnalyzer");
        PUBLIC_IDENTIFIERS.add("ModuleDescriptor");
        PUBLIC_IDENTIFIERS.add("ModuleDotGraph");
        PUBLIC_IDENTIFIERS.add("ModuleElement");
        PUBLIC_IDENTIFIERS.add("ModuleExportNode");
        PUBLIC_IDENTIFIERS.add("ModuleExportsAnalyzer");
        PUBLIC_IDENTIFIERS.add("ModuleFinder");
        PUBLIC_IDENTIFIERS.add("ModuleGraphBuilder");
        PUBLIC_IDENTIFIERS.add("ModuleHashes_attribute");
        PUBLIC_IDENTIFIERS.add("ModuleHashesBuilder");
        PUBLIC_IDENTIFIERS.add("ModuleHelper");
        PUBLIC_IDENTIFIERS.add("ModuleIndexWriter");
        PUBLIC_IDENTIFIERS.add("ModuleInfoBuilder");
        PUBLIC_IDENTIFIERS.add("ModuleLayer");
        PUBLIC_IDENTIFIERS.add("ModuleLayer.Controller");
        PUBLIC_IDENTIFIERS.add("ModuleMainClass_attribute");
        PUBLIC_IDENTIFIERS.add("ModuleNameFromSourceReader");
        PUBLIC_IDENTIFIERS.add("ModuleNameReader");
        PUBLIC_IDENTIFIERS.add("ModuleNode");
        PUBLIC_IDENTIFIERS.add("ModuleOpenNode");
        PUBLIC_IDENTIFIERS.add("ModulePackages_attribute");
        PUBLIC_IDENTIFIERS.add("ModulePath");
        PUBLIC_IDENTIFIERS.add("ModuleProvideNode");
        PUBLIC_IDENTIFIERS.add("ModuleReader");
        PUBLIC_IDENTIFIERS.add("ModuleReference");
        PUBLIC_IDENTIFIERS.add("ModuleReferenceImpl");
        PUBLIC_IDENTIFIERS.add("ModuleRemapper");
        PUBLIC_IDENTIFIERS.add("ModuleRequireNode");
        PUBLIC_IDENTIFIERS.add("ModuleResolution_attribute");
        PUBLIC_IDENTIFIERS.add("Modules");
        PUBLIC_IDENTIFIERS.add("ModuleSummaryBuilder");
        PUBLIC_IDENTIFIERS.add("ModuleSummaryWriter");
        PUBLIC_IDENTIFIERS.add("ModuleSymbol");
        PUBLIC_IDENTIFIERS.add("ModuleTarget_attribute");
        PUBLIC_IDENTIFIERS.add("ModuleTree");
        PUBLIC_IDENTIFIERS.add("ModuleType");
        PUBLIC_IDENTIFIERS.add("ModuleVisitor");
        PUBLIC_IDENTIFIERS.add("ModuleWriterImpl");
        PUBLIC_IDENTIFIERS.add("Monitor");
        PUBLIC_IDENTIFIERS.add("Monitorable");
        PUBLIC_IDENTIFIERS.add("MonitorCacheDumpPanel");
        PUBLIC_IDENTIFIERS.add("MonitorContendedEnteredEvent");
        PUBLIC_IDENTIFIERS.add("MonitorContendedEnteredRequest");
        PUBLIC_IDENTIFIERS.add("MonitorContendedEnterEvent");
        PUBLIC_IDENTIFIERS.add("MonitorContendedEnterRequest");
        PUBLIC_IDENTIFIERS.add("MonitorDataException");
        PUBLIC_IDENTIFIERS.add("MonitorDeflationThread");
        PUBLIC_IDENTIFIERS.add("MonitoredHost");
        PUBLIC_IDENTIFIERS.add("MonitoredHostProvider");
        PUBLIC_IDENTIFIERS.add("MonitoredHostService");
        PUBLIC_IDENTIFIERS.add("MonitoredVm");
        PUBLIC_IDENTIFIERS.add("MonitoredVmUtil");
        PUBLIC_IDENTIFIERS.add("MONITORENTER");
        PUBLIC_IDENTIFIERS.add("MonitorException");
        PUBLIC_IDENTIFIERS.add("MONITOREXIT");
        PUBLIC_IDENTIFIERS.add("MonitorInfo");
        PUBLIC_IDENTIFIERS.add("MonitorInfoCompositeData");
        PUBLIC_IDENTIFIERS.add("MonitorInfoImpl");
        PUBLIC_IDENTIFIERS.add("MonitorMBean");
        PUBLIC_IDENTIFIERS.add("MonitorNotification");
        PUBLIC_IDENTIFIERS.add("MonitorSettingException");
        PUBLIC_IDENTIFIERS.add("MonitorStatus");
        PUBLIC_IDENTIFIERS.add("MonitorStatusChangeEvent");
        PUBLIC_IDENTIFIERS.add("MonitorStructureException");
        PUBLIC_IDENTIFIERS.add("MonitorTypeException");
        PUBLIC_IDENTIFIERS.add("MonitorValue");
        PUBLIC_IDENTIFIERS.add("MonitorVersionException");
        PUBLIC_IDENTIFIERS.add("MonitorWaitedEvent");
        PUBLIC_IDENTIFIERS.add("MonitorWaitedRequest");
        PUBLIC_IDENTIFIERS.add("MonitorWaitEvent");
        PUBLIC_IDENTIFIERS.add("MonitorWaitRequest");
        PUBLIC_IDENTIFIERS.add("MonthDayDV");
        PUBLIC_IDENTIFIERS.add("MonthDV");
        PUBLIC_IDENTIFIERS.add("MoreDrbgParameters");
        PUBLIC_IDENTIFIERS.add("MostSpecificReturnCheck");
        PUBLIC_IDENTIFIERS.add("MotifBorders");
        PUBLIC_IDENTIFIERS.add("MotifButtonListener");
        PUBLIC_IDENTIFIERS.add("MotifButtonUI");
        PUBLIC_IDENTIFIERS.add("MotifCaret");
        PUBLIC_IDENTIFIERS.add("MotifChangeHandler");
        PUBLIC_IDENTIFIERS.add("MotifCheckBoxMenuItemUI");
        PUBLIC_IDENTIFIERS.add("MotifCheckBoxUI");
        PUBLIC_IDENTIFIERS.add("MotifCollapsedIcon");
        PUBLIC_IDENTIFIERS.add("MotifComboBoxUI");
        PUBLIC_IDENTIFIERS.add("MotifComboPopup");
        PUBLIC_IDENTIFIERS.add("MotifDesktopIconUI");
        PUBLIC_IDENTIFIERS.add("MotifDesktopPaneUI");
        PUBLIC_IDENTIFIERS.add("MotifDirectoryListModel");
        PUBLIC_IDENTIFIERS.add("MotifEditorPaneUI");
        PUBLIC_IDENTIFIERS.add("MotifExpandedIcon");
        PUBLIC_IDENTIFIERS.add("MotifFileChooserUI");
        PUBLIC_IDENTIFIERS.add("MotifFileListModel");
        PUBLIC_IDENTIFIERS.add("MotifIconFactory");
        PUBLIC_IDENTIFIERS.add("MotifInternalFrameTitlePane");
        PUBLIC_IDENTIFIERS.add("MotifInternalFrameUI");
        PUBLIC_IDENTIFIERS.add("MotifLabelUI");
        PUBLIC_IDENTIFIERS.add("MotifLookAndFeel");
        PUBLIC_IDENTIFIERS.add("MotifMenuBarUI");
        PUBLIC_IDENTIFIERS.add("MotifMenuItemUI");
        PUBLIC_IDENTIFIERS.add("MotifMenuUI");
        PUBLIC_IDENTIFIERS.add("MotifOptionPaneUI");
        PUBLIC_IDENTIFIERS.add("MotifPasswordFieldUI");
        PUBLIC_IDENTIFIERS.add("MotifPopupMenuBorder");
        PUBLIC_IDENTIFIERS.add("MotifPopupMenuSeparatorUI");
        PUBLIC_IDENTIFIERS.add("MotifPopupMenuUI");
        PUBLIC_IDENTIFIERS.add("MotifProgressBarUI");
        PUBLIC_IDENTIFIERS.add("MotifRadioButtonMenuItemUI");
        PUBLIC_IDENTIFIERS.add("MotifRadioButtonUI");
        PUBLIC_IDENTIFIERS.add("MotifScrollBarButton");
        PUBLIC_IDENTIFIERS.add("MotifScrollBarUI");
        PUBLIC_IDENTIFIERS.add("MotifScrollPaneUI");
        PUBLIC_IDENTIFIERS.add("MotifSeparatorUI");
        PUBLIC_IDENTIFIERS.add("MotifSliderUI");
        PUBLIC_IDENTIFIERS.add("MotifSplitPaneDivider");
        PUBLIC_IDENTIFIERS.add("MotifSplitPaneUI");
        PUBLIC_IDENTIFIERS.add("MotifTabbedPaneUI");
        PUBLIC_IDENTIFIERS.add("MotifTextAreaUI");
        PUBLIC_IDENTIFIERS.add("MotifTextFieldUI");
        PUBLIC_IDENTIFIERS.add("MotifTextPaneUI");
        PUBLIC_IDENTIFIERS.add("MotifTextUI");
        PUBLIC_IDENTIFIERS.add("MotifToggleButtonUI");
        PUBLIC_IDENTIFIERS.add("MotifTreeCellRenderer");
        PUBLIC_IDENTIFIERS.add("MotifTreeUI");
        PUBLIC_IDENTIFIERS.add("MouseAdapter");
        PUBLIC_IDENTIFIERS.add("MouseDragGestureRecognizer");
        PUBLIC_IDENTIFIERS.add("MouseEvent");
        PUBLIC_IDENTIFIERS.add("MouseEventAccessor");
        PUBLIC_IDENTIFIERS.add("MouseEventListener");
        PUBLIC_IDENTIFIERS.add("MouseHandler");
        PUBLIC_IDENTIFIERS.add("MouseInfo");
        PUBLIC_IDENTIFIERS.add("MouseInfoPeer");
        PUBLIC_IDENTIFIERS.add("MouseInputAdapter");
        PUBLIC_IDENTIFIERS.add("MouseInputHandler");
        PUBLIC_IDENTIFIERS.add("MouseInputListener");
        PUBLIC_IDENTIFIERS.add("MouseListener");
        PUBLIC_IDENTIFIERS.add("MouseMotionAdapter");
        PUBLIC_IDENTIFIERS.add("MouseMotionListener");
        PUBLIC_IDENTIFIERS.add("MouseSupport");
        PUBLIC_IDENTIFIERS.add("MouseWheelEvent");
        PUBLIC_IDENTIFIERS.add("MouseWheelHandler");
        PUBLIC_IDENTIFIERS.add("MouseWheelListener");
        PUBLIC_IDENTIFIERS.add("Move");
        PUBLIC_IDENTIFIERS.add("MoveAction");
        PUBLIC_IDENTIFIERS.add("MS1250");
        PUBLIC_IDENTIFIERS.add("MS1251");
        PUBLIC_IDENTIFIERS.add("MS1252");
        PUBLIC_IDENTIFIERS.add("MS1253");
        PUBLIC_IDENTIFIERS.add("MS1254");
        PUBLIC_IDENTIFIERS.add("MS1255");
        PUBLIC_IDENTIFIERS.add("MS1256");
        PUBLIC_IDENTIFIERS.add("MS1257");
        PUBLIC_IDENTIFIERS.add("MS1258");
        PUBLIC_IDENTIFIERS.add("MS50220");
        PUBLIC_IDENTIFIERS.add("MS50221");
        PUBLIC_IDENTIFIERS.add("MS874");
        PUBLIC_IDENTIFIERS.add("MS932");
        PUBLIC_IDENTIFIERS.add("MS932_0213");
        PUBLIC_IDENTIFIERS.add("MS936");
        PUBLIC_IDENTIFIERS.add("MS949");
        PUBLIC_IDENTIFIERS.add("MS950");
        PUBLIC_IDENTIFIERS.add("MS950_HKSCS");
        PUBLIC_IDENTIFIERS.add("MS950_HKSCS_XP");
        PUBLIC_IDENTIFIERS.add("MsgKey");
        PUBLIC_IDENTIFIERS.add("MSISO2022JP");
        PUBLIC_IDENTIFIERS.add("Mult");
        PUBLIC_IDENTIFIERS.add("MULTIANEWARRAY");
        PUBLIC_IDENTIFIERS.add("MultiANewArrayInsnNode");
        PUBLIC_IDENTIFIERS.add("MultiBranchData");
        PUBLIC_IDENTIFIERS.add("MultiButtonUI");
        PUBLIC_IDENTIFIERS.add("MulticastChannel");
        PUBLIC_IDENTIFIERS.add("MulticastSocket");
        PUBLIC_IDENTIFIERS.add("MultiColorChooserUI");
        PUBLIC_IDENTIFIERS.add("MultiComboBoxUI");
        PUBLIC_IDENTIFIERS.add("MultiDesktopIconUI");
        PUBLIC_IDENTIFIERS.add("MultiDesktopPaneUI");
        PUBLIC_IDENTIFIERS.add("MultiDoc");
        PUBLIC_IDENTIFIERS.add("MultiDocPrintJob");
        PUBLIC_IDENTIFIERS.add("MultiDocPrintService");
        PUBLIC_IDENTIFIERS.add("MultiFileChooserUI");
        PUBLIC_IDENTIFIERS.add("MultiInternalFrameUI");
        PUBLIC_IDENTIFIERS.add("MultiLabelUI");
        PUBLIC_IDENTIFIERS.add("MultilineDiagnostic");
        PUBLIC_IDENTIFIERS.add("MultiListUI");
        PUBLIC_IDENTIFIERS.add("MultiLookAndFeel");
        PUBLIC_IDENTIFIERS.add("MultiMenuBarUI");
        PUBLIC_IDENTIFIERS.add("MultiMenuItemUI");
        PUBLIC_IDENTIFIERS.add("MultiNode");
        PUBLIC_IDENTIFIERS.add("MultiOptionPaneUI");
        PUBLIC_IDENTIFIERS.add("MultiPanelUI");
        PUBLIC_IDENTIFIERS.add("MultiPixelPackedSampleModel");
        PUBLIC_IDENTIFIERS.add("MultipleDocumentHandling");
        PUBLIC_IDENTIFIERS.add("MultipleGradientPaint");
        PUBLIC_IDENTIFIERS.add("MultipleMaster");
        PUBLIC_IDENTIFIERS.add("MultipleNodeCounter");
        PUBLIC_IDENTIFIERS.add("MultipleScopeNamespaceSupport");
        PUBLIC_IDENTIFIERS.add("MultiplicativeInverser");
        PUBLIC_IDENTIFIERS.add("MultiPopupMenuUI");
        PUBLIC_IDENTIFIERS.add("MultiProgressBarUI");
        PUBLIC_IDENTIFIERS.add("MultiResolutionCachedImage");
        PUBLIC_IDENTIFIERS.add("MultiResolutionImage");
        PUBLIC_IDENTIFIERS.add("MultiResolutionToolkitImage");
        PUBLIC_IDENTIFIERS.add("MultiRootPaneUI");
        PUBLIC_IDENTIFIERS.add("MultiScrollBarUI");
        PUBLIC_IDENTIFIERS.add("MultiScrollPaneUI");
        PUBLIC_IDENTIFIERS.add("MultiSeparatorUI");
        PUBLIC_IDENTIFIERS.add("MultiSliderUI");
        PUBLIC_IDENTIFIERS.add("MultiSpinnerUI");
        PUBLIC_IDENTIFIERS.add("MultiSplitPaneUI");
        PUBLIC_IDENTIFIERS.add("MultiTabbedPaneUI");
        PUBLIC_IDENTIFIERS.add("MultiTableHeaderUI");
        PUBLIC_IDENTIFIERS.add("MultiTableUI");
        PUBLIC_IDENTIFIERS.add("MultiTaskListener");
        PUBLIC_IDENTIFIERS.add("MultiTextUI");
        PUBLIC_IDENTIFIERS.add("MultiToolBarUI");
        PUBLIC_IDENTIFIERS.add("MultiToolTipUI");
        PUBLIC_IDENTIFIERS.add("MultiTreeUI");
        PUBLIC_IDENTIFIERS.add("MultiValuedNodeHeapIterator");
        PUBLIC_IDENTIFIERS.add("MultiViewportUI");
        PUBLIC_IDENTIFIERS.add("Mutable");
        PUBLIC_IDENTIFIERS.add("MutableAttributeSet");
        PUBLIC_IDENTIFIERS.add("MutableAttrListImpl");
        PUBLIC_IDENTIFIERS.add("MutableCallSite");
        PUBLIC_IDENTIFIERS.add("MutableComboBoxModel");
        PUBLIC_IDENTIFIERS.add("MutableElement");
        PUBLIC_IDENTIFIERS.add("MutableInteger");
        PUBLIC_IDENTIFIERS.add("MutableIntegerModuloP");
        PUBLIC_IDENTIFIERS.add("MutablePoint");
        PUBLIC_IDENTIFIERS.add("MutableSpace");
        PUBLIC_IDENTIFIERS.add("MutableTreeNode");
        PUBLIC_IDENTIFIERS.add("MutationEvent");
        PUBLIC_IDENTIFIERS.add("MutationEventImpl");
        PUBLIC_IDENTIFIERS.add("MutationException");
        PUBLIC_IDENTIFIERS.add("Mutex");
        PUBLIC_IDENTIFIERS.add("MXBean");
        PUBLIC_IDENTIFIERS.add("MXBeanLookup");
        PUBLIC_IDENTIFIERS.add("MXBeanMapping");
        PUBLIC_IDENTIFIERS.add("MXBeanMappingFactory");
        PUBLIC_IDENTIFIERS.add("MXBeanProxy");
        PUBLIC_IDENTIFIERS.add("MXBeanSupport");
    }

    private static void addPublicIdentifiersNR() {
        PUBLIC_IDENTIFIERS.add("Name");
        PUBLIC_IDENTIFIERS.add("NameAlreadyBoundException");
        PUBLIC_IDENTIFIERS.add("NameCallback");
        PUBLIC_IDENTIFIERS.add("NameClassPair");
        PUBLIC_IDENTIFIERS.add("NameConstraintsExtension");
        PUBLIC_IDENTIFIERS.add("NamedAndTyped");
        PUBLIC_IDENTIFIERS.add("NamedEvent");
        PUBLIC_IDENTIFIERS.add("NamedFieldIdentifier");
        PUBLIC_IDENTIFIERS.add("NamedImportScope");
        PUBLIC_IDENTIFIERS.add("NamedNodeMap");
        PUBLIC_IDENTIFIERS.add("NamedNodeMapImpl");
        PUBLIC_IDENTIFIERS.add("NamedObject");
        PUBLIC_IDENTIFIERS.add("NamedParameterSpec");
        PUBLIC_IDENTIFIERS.add("NamedStyle");
        PUBLIC_IDENTIFIERS.add("NameFilter");
        PUBLIC_IDENTIFIERS.add("NameList");
        PUBLIC_IDENTIFIERS.add("NameMapper");
        PUBLIC_IDENTIFIERS.add("NameNotFoundException");
        PUBLIC_IDENTIFIERS.add("NameParser");
        PUBLIC_IDENTIFIERS.add("Names");
        PUBLIC_IDENTIFIERS.add("NameSignatureInstruction");
        PUBLIC_IDENTIFIERS.add("NameSpace");
        PUBLIC_IDENTIFIERS.add("Namespace");
        PUBLIC_IDENTIFIERS.add("NamespaceChangeListener");
        PUBLIC_IDENTIFIERS.add("NamespaceContext");
        PUBLIC_IDENTIFIERS.add("NamespaceContextWrapper");
        PUBLIC_IDENTIFIERS.add("NamespaceImpl");
        PUBLIC_IDENTIFIERS.add("NamespaceIterator");
        PUBLIC_IDENTIFIERS.add("NamespaceMappings");
        PUBLIC_IDENTIFIERS.add("NamespaceSupport");
        PUBLIC_IDENTIFIERS.add("NameSpaceSymbTable");
        PUBLIC_IDENTIFIERS.add("NamingEnumeration");
        PUBLIC_IDENTIFIERS.add("NamingEvent");
        PUBLIC_IDENTIFIERS.add("NamingException");
        PUBLIC_IDENTIFIERS.add("NamingExceptionEvent");
        PUBLIC_IDENTIFIERS.add("NamingListener");
        PUBLIC_IDENTIFIERS.add("NamingManager");
        PUBLIC_IDENTIFIERS.add("NamingManagerHelper");
        PUBLIC_IDENTIFIERS.add("NamingSecurityException");
        PUBLIC_IDENTIFIERS.add("NarrowKlassField");
        PUBLIC_IDENTIFIERS.add("NarrowOopField");
        PUBLIC_IDENTIFIERS.add("Native");
        PUBLIC_IDENTIFIERS.add("NativeEntryPoint");
        PUBLIC_IDENTIFIERS.add("NativeFont");
        PUBLIC_IDENTIFIERS.add("NativeGlyphMapper");
        PUBLIC_IDENTIFIERS.add("NativeImageReinitialize");
        PUBLIC_IDENTIFIERS.add("NativeLibrary");
        PUBLIC_IDENTIFIERS.add("NativeMemorySegmentImpl");
        PUBLIC_IDENTIFIERS.add("NativeMethodException");
        PUBLIC_IDENTIFIERS.add("NativePluginFactory");
        PUBLIC_IDENTIFIERS.add("NativeSignatureIterator");
        PUBLIC_IDENTIFIERS.add("NativeThread");
        PUBLIC_IDENTIFIERS.add("NavigableMap");
        PUBLIC_IDENTIFIERS.add("NavigableSet");
        PUBLIC_IDENTIFIERS.add("NavigateAction");
        PUBLIC_IDENTIFIERS.add("Navigation");
        PUBLIC_IDENTIFIERS.add("NavigationFilter");
        PUBLIC_IDENTIFIERS.add("NClob");
        PUBLIC_IDENTIFIERS.add("Neg");
        PUBLIC_IDENTIFIERS.add("NegativeArraySizeException");
        PUBLIC_IDENTIFIERS.add("NegotiateCallbackHandler");
        PUBLIC_IDENTIFIERS.add("Negotiator");
        PUBLIC_IDENTIFIERS.add("NegotiatorImpl");
        PUBLIC_IDENTIFIERS.add("NegTokenInit");
        PUBLIC_IDENTIFIERS.add("NegTokenTarg");
        PUBLIC_IDENTIFIERS.add("NestedClassWriter");
        PUBLIC_IDENTIFIERS.add("NestedClassWriterImpl");
        PUBLIC_IDENTIFIERS.add("NestHost_attribute");
        PUBLIC_IDENTIFIERS.add("NestMembers_attribute");
        PUBLIC_IDENTIFIERS.add("Net");
        PUBLIC_IDENTIFIERS.add("NetClient");
        PUBLIC_IDENTIFIERS.add("NetProperties");
        PUBLIC_IDENTIFIERS.add("NetscapeCertTypeExtension");
        PUBLIC_IDENTIFIERS.add("NetworkChannel");
        PUBLIC_IDENTIFIERS.add("NetworkClient");
        PUBLIC_IDENTIFIERS.add("NEW");
        PUBLIC_IDENTIFIERS.add("NewAction");
        PUBLIC_IDENTIFIERS.add("NewAPIBuilder");
        PUBLIC_IDENTIFIERS.add("NewAPIListWriter");
        PUBLIC_IDENTIFIERS.add("NEWARRAY");
        PUBLIC_IDENTIFIERS.add("NewArrayTree");
        PUBLIC_IDENTIFIERS.add("NewClassTree");
        PUBLIC_IDENTIFIERS.add("NewFolderAction");
        PUBLIC_IDENTIFIERS.add("NextAction");
        PUBLIC_IDENTIFIERS.add("NimbusLookAndFeel");
        PUBLIC_IDENTIFIERS.add("NMethod");
        PUBLIC_IDENTIFIERS.add("NMTOKENDatatypeValidator");
        PUBLIC_IDENTIFIERS.add("NoClassDefFoundError");
        PUBLIC_IDENTIFIERS.add("NoConnectionPendingException");
        PUBLIC_IDENTIFIERS.add("Node");
        PUBLIC_IDENTIFIERS.add("Node_Array");
        PUBLIC_IDENTIFIERS.add("Node_List");
        PUBLIC_IDENTIFIERS.add("NodeChangeEvent");
        PUBLIC_IDENTIFIERS.add("NodeChangeListener");
        PUBLIC_IDENTIFIERS.add("NodeConsumer");
        PUBLIC_IDENTIFIERS.add("NodeCounter");
        PUBLIC_IDENTIFIERS.add("NodeDimensions");
        PUBLIC_IDENTIFIERS.add("NodeDimensionsHandler");
        PUBLIC_IDENTIFIERS.add("NodeFilter");
        PUBLIC_IDENTIFIERS.add("NodeImpl");
        PUBLIC_IDENTIFIERS.add("NodeInfo");
        PUBLIC_IDENTIFIERS.add("NodeIterator");
        PUBLIC_IDENTIFIERS.add("NodeIteratorBase");
        PUBLIC_IDENTIFIERS.add("NodeIteratorImpl");
        PUBLIC_IDENTIFIERS.add("NodeList");
        PUBLIC_IDENTIFIERS.add("NodeLocator");
        PUBLIC_IDENTIFIERS.add("NodeNotFoundException");
        PUBLIC_IDENTIFIERS.add("NodeSequence");
        PUBLIC_IDENTIFIERS.add("NodeSet");
        PUBLIC_IDENTIFIERS.add("NodeSetData");
        PUBLIC_IDENTIFIERS.add("NodeSetDTM");
        PUBLIC_IDENTIFIERS.add("NodeSortRecord");
        PUBLIC_IDENTIFIERS.add("NodeSortRecordFactory");
        PUBLIC_IDENTIFIERS.add("NodeTest");
        PUBLIC_IDENTIFIERS.add("NodeTestFilter");
        PUBLIC_IDENTIFIERS.add("NodeVector");
        PUBLIC_IDENTIFIERS.add("NoInitialContextException");
        PUBLIC_IDENTIFIERS.add("NonBaseLocaleDataMetaInfo");
        PUBLIC_IDENTIFIERS.add("NonBlocking");
        PUBLIC_IDENTIFIERS.add("NonBlockingInputStream");
        PUBLIC_IDENTIFIERS.add("NonBlockingInputStreamImpl");
        PUBLIC_IDENTIFIERS.add("NonBlockingPumpInputStream");
        PUBLIC_IDENTIFIERS.add("NonBlockingPumpReader");
        PUBLIC_IDENTIFIERS.add("NonBlockingReader");
        PUBLIC_IDENTIFIERS.add("NonBlockingReaderImpl");
        PUBLIC_IDENTIFIERS.add("NonBootFilter");
        PUBLIC_IDENTIFIERS.add("Nonce");
        PUBLIC_IDENTIFIERS.add("NonConcreteMethodImpl");
        PUBLIC_IDENTIFIERS.add("NoninvertibleTransformException");
        PUBLIC_IDENTIFIERS.add("NonOptionArgumentSpec");
        PUBLIC_IDENTIFIERS.add("NonReadableChannelException");
        PUBLIC_IDENTIFIERS.add("NonValidatingConfiguration");
        PUBLIC_IDENTIFIERS.add("NonWritableChannelException");
        PUBLIC_IDENTIFIERS.add("NOP");
        PUBLIC_IDENTIFIERS.add("NoPermissionException");
        PUBLIC_IDENTIFIERS.add("NoPreview");
        PUBLIC_IDENTIFIERS.add("Normalizer2");
        PUBLIC_IDENTIFIERS.add("Normalizer2WithImpl");
        PUBLIC_IDENTIFIERS.add("NoRouteToHostException");
        PUBLIC_IDENTIFIERS.add("NoSuchAlgorithmException");
        PUBLIC_IDENTIFIERS.add("NoSuchAttributeException");
        PUBLIC_IDENTIFIERS.add("NoSuchDynamicMethodException");
        PUBLIC_IDENTIFIERS.add("NoSuchElementException");
        PUBLIC_IDENTIFIERS.add("NoSuchFieldError");
        PUBLIC_IDENTIFIERS.add("NoSuchFieldException");
        PUBLIC_IDENTIFIERS.add("NoSuchFileException");
        PUBLIC_IDENTIFIERS.add("NoSuchMechanismException");
        PUBLIC_IDENTIFIERS.add("NoSuchMethodError");
        PUBLIC_IDENTIFIERS.add("NoSuchMethodException");
        PUBLIC_IDENTIFIERS.add("NoSuchObjectException");
        PUBLIC_IDENTIFIERS.add("NoSuchPaddingException");
        PUBLIC_IDENTIFIERS.add("NoSuchProviderException");
        PUBLIC_IDENTIFIERS.add("NoSuchSymbolException");
        PUBLIC_IDENTIFIERS.add("NotActiveException");
        PUBLIC_IDENTIFIERS.add("Notation");
        PUBLIC_IDENTIFIERS.add("NOTATIONDatatypeValidator");
        PUBLIC_IDENTIFIERS.add("NotationDeclaration");
        PUBLIC_IDENTIFIERS.add("NotationDeclarationImpl");
        PUBLIC_IDENTIFIERS.add("NotationImpl");
        PUBLIC_IDENTIFIERS.add("NotBoundException");
        PUBLIC_IDENTIFIERS.add("NotCompliantMBeanException");
        PUBLIC_IDENTIFIERS.add("NotContextException");
        PUBLIC_IDENTIFIERS.add("NotDirectoryException");
        PUBLIC_IDENTIFIERS.add("Note");
        PUBLIC_IDENTIFIERS.add("NoteKind");
        PUBLIC_IDENTIFIERS.add("NotEquals");
        PUBLIC_IDENTIFIERS.add("Notes");
        PUBLIC_IDENTIFIERS.add("Notification");
        PUBLIC_IDENTIFIERS.add("NotificationAccessController");
        PUBLIC_IDENTIFIERS.add("NotificationBroadcaster");
        PUBLIC_IDENTIFIERS.add("NotificationBroadcasterSupport");
        PUBLIC_IDENTIFIERS.add("NotificationBuffer");
        PUBLIC_IDENTIFIERS.add("NotificationBufferFilter");
        PUBLIC_IDENTIFIERS.add("NotificationEmitter");
        PUBLIC_IDENTIFIERS.add("NotificationEmitterSupport");
        PUBLIC_IDENTIFIERS.add("NotificationFilter");
        PUBLIC_IDENTIFIERS.add("NotificationFilterSupport");
        PUBLIC_IDENTIFIERS.add("NotificationHandler");
        PUBLIC_IDENTIFIERS.add("NotificationListener");
        PUBLIC_IDENTIFIERS.add("NotificationResult");
        PUBLIC_IDENTIFIERS.add("NotificationThread");
        PUBLIC_IDENTIFIERS.add("NotImplementedException");
        PUBLIC_IDENTIFIERS.add("NotInHeapException");
        PUBLIC_IDENTIFIERS.add("NotLinkException");
        PUBLIC_IDENTIFIERS.add("NotSerializableException");
        PUBLIC_IDENTIFIERS.add("NotYetBoundException");
        PUBLIC_IDENTIFIERS.add("NotYetConnectedException");
        PUBLIC_IDENTIFIERS.add("NoType");
        PUBLIC_IDENTIFIERS.add("NSInfo");
        PUBLIC_IDENTIFIERS.add("NTDomainPrincipal");
        PUBLIC_IDENTIFIERS.add("NthDescendantIterator");
        PUBLIC_IDENTIFIERS.add("NTLMAuthentication");
        PUBLIC_IDENTIFIERS.add("NTLMAuthenticationCallback");
        PUBLIC_IDENTIFIERS.add("NTLoginModule");
        PUBLIC_IDENTIFIERS.add("NTNumericCredential");
        PUBLIC_IDENTIFIERS.add("NTSid");
        PUBLIC_IDENTIFIERS.add("NTSidDomainPrincipal");
        PUBLIC_IDENTIFIERS.add("NTSidGroupPrincipal");
        PUBLIC_IDENTIFIERS.add("NTSidPrimaryGroupPrincipal");
        PUBLIC_IDENTIFIERS.add("NTSidUserPrincipal");
        PUBLIC_IDENTIFIERS.add("NTSystem");
        PUBLIC_IDENTIFIERS.add("NTUserPrincipal");
        PUBLIC_IDENTIFIERS.add("NullCipher");
        PUBLIC_IDENTIFIERS.add("NullComponentPeer");
        PUBLIC_IDENTIFIERS.add("NullEType");
        PUBLIC_IDENTIFIERS.add("NullPipe");
        PUBLIC_IDENTIFIERS.add("NullPointerException");
        PUBLIC_IDENTIFIERS.add("NullSubscriber");
        PUBLIC_IDENTIFIERS.add("NullSurfaceData");
        PUBLIC_IDENTIFIERS.add("NullType");
        PUBLIC_IDENTIFIERS.add("Number");
        PUBLIC_IDENTIFIERS.add("NumberEditor");
        PUBLIC_IDENTIFIERS.add("NumberedRefMapFormatter");
        PUBLIC_IDENTIFIERS.add("NumberFormat");
        PUBLIC_IDENTIFIERS.add("NumberFormatException");
        PUBLIC_IDENTIFIERS.add("NumberFormatProvider");
        PUBLIC_IDENTIFIERS.add("NumberFormatProviderImpl");
        PUBLIC_IDENTIFIERS.add("NumberFormatter");
        PUBLIC_IDENTIFIERS.add("NumberType");
        PUBLIC_IDENTIFIERS.add("NumericClasses");
        PUBLIC_IDENTIFIERS.add("NumericType");
        PUBLIC_IDENTIFIERS.add("NZWindOp");
        PUBLIC_IDENTIFIERS.add("OAEPParameterSpec");
        PUBLIC_IDENTIFIERS.add("ObjArray");
        PUBLIC_IDENTIFIERS.add("ObjArrayKlass");
        PUBLIC_IDENTIFIERS.add("ObjCopyCmdBuilder");
        PUBLIC_IDENTIFIERS.add("ObjDoubleConsumer");
        PUBLIC_IDENTIFIERS.add("Object");
        PUBLIC_IDENTIFIERS.add("Object_variable_info");
        PUBLIC_IDENTIFIERS.add("ObjectChangeListener");
        PUBLIC_IDENTIFIERS.add("ObjectCollectedException");
        PUBLIC_IDENTIFIERS.add("ObjectContainer");
        PUBLIC_IDENTIFIERS.add("ObjectFactory");
        PUBLIC_IDENTIFIERS.add("ObjectFactoryBuilder");
        PUBLIC_IDENTIFIERS.add("ObjectFilter");
        PUBLIC_IDENTIFIERS.add("ObjectHeap");
        PUBLIC_IDENTIFIERS.add("ObjectHistogram");
        PUBLIC_IDENTIFIERS.add("ObjectHistogramElement");
        PUBLIC_IDENTIFIERS.add("ObjectHistogramPanel");
        PUBLIC_IDENTIFIERS.add("ObjectInput");
        PUBLIC_IDENTIFIERS.add("ObjectInputFilter");
        PUBLIC_IDENTIFIERS.add("ObjectInputStream");
        PUBLIC_IDENTIFIERS.add("ObjectInputValidation");
        PUBLIC_IDENTIFIERS.add("ObjectInstance");
        PUBLIC_IDENTIFIERS.add("ObjectList");
        PUBLIC_IDENTIFIERS.add("ObjectListPanel");
        PUBLIC_IDENTIFIERS.add("ObjectMethods");
        PUBLIC_IDENTIFIERS.add("ObjectMonitor");
        PUBLIC_IDENTIFIERS.add("ObjectName");
        PUBLIC_IDENTIFIERS.add("ObjectOutput");
        PUBLIC_IDENTIFIERS.add("ObjectOutputStream");
        PUBLIC_IDENTIFIERS.add("ObjectPool");
        PUBLIC_IDENTIFIERS.add("ObjectReader");
        PUBLIC_IDENTIFIERS.add("ObjectReference");
        PUBLIC_IDENTIFIERS.add("ObjectReferenceImpl");
        PUBLIC_IDENTIFIERS.add("ObjectStack");
        PUBLIC_IDENTIFIERS.add("ObjectStreamConstants");
        PUBLIC_IDENTIFIERS.add("ObjectStreamException");
        PUBLIC_IDENTIFIERS.add("ObjectStreamField");
        PUBLIC_IDENTIFIERS.add("ObjectSynchronizer");
        PUBLIC_IDENTIFIERS.add("ObjectType");
        PUBLIC_IDENTIFIERS.add("ObjectValue");
        PUBLIC_IDENTIFIERS.add("ObjectVector");
        PUBLIC_IDENTIFIERS.add("ObjectView");
        PUBLIC_IDENTIFIERS.add("ObjectVisitor");
        PUBLIC_IDENTIFIERS.add("ObjIntConsumer");
        PUBLIC_IDENTIFIERS.add("ObjLongConsumer");
        PUBLIC_IDENTIFIERS.add("Observable");
        PUBLIC_IDENTIFIERS.add("Observer");
        PUBLIC_IDENTIFIERS.add("ObsoleteMethodImpl");
        PUBLIC_IDENTIFIERS.add("OceanTheme");
        PUBLIC_IDENTIFIERS.add("OCSPNoCheckExtension");
        PUBLIC_IDENTIFIERS.add("OctetStreamData");
        PUBLIC_IDENTIFIERS.add("OfAddress");
        PUBLIC_IDENTIFIERS.add("OfBoolean");
        PUBLIC_IDENTIFIERS.add("OfByte");
        PUBLIC_IDENTIFIERS.add("OfChar");
        PUBLIC_IDENTIFIERS.add("OfDouble");
        PUBLIC_IDENTIFIERS.add("OfField");
        PUBLIC_IDENTIFIERS.add("OfFloat");
        PUBLIC_IDENTIFIERS.add("OffScreenImage");
        PUBLIC_IDENTIFIERS.add("OffScreenImageSource");
        PUBLIC_IDENTIFIERS.add("OffsetClosure");
        PUBLIC_IDENTIFIERS.add("OffsetTableContigSpace");
        PUBLIC_IDENTIFIERS.add("OfInt");
        PUBLIC_IDENTIFIERS.add("OfLong");
        PUBLIC_IDENTIFIERS.add("OfMethod");
        PUBLIC_IDENTIFIERS.add("OfPlatform");
        PUBLIC_IDENTIFIERS.add("OfPrimitive");
        PUBLIC_IDENTIFIERS.add("OfShort");
        PUBLIC_IDENTIFIERS.add("OfVirtual");
        PUBLIC_IDENTIFIERS.add("OGLDrawImage");
        PUBLIC_IDENTIFIERS.add("OGLRenderQueue");
        PUBLIC_IDENTIFIERS.add("OGLSurfaceData");
        PUBLIC_IDENTIFIERS.add("OGLSurfaceDataProxy");
        PUBLIC_IDENTIFIERS.add("Oid");
        PUBLIC_IDENTIFIERS.add("OIDMap");
        PUBLIC_IDENTIFIERS.add("OIDName");
        PUBLIC_IDENTIFIERS.add("OkAction");
        PUBLIC_IDENTIFIERS.add("OkCancelButtonPanel");
        PUBLIC_IDENTIFIERS.add("OkCancelDialog");
        PUBLIC_IDENTIFIERS.add("OneStepIterator");
        PUBLIC_IDENTIFIERS.add("OneStepIteratorForward");
        PUBLIC_IDENTIFIERS.add("Oop");
        PUBLIC_IDENTIFIERS.add("OopField");
        PUBLIC_IDENTIFIERS.add("OopHandle");
        PUBLIC_IDENTIFIERS.add("OopMapCacheEntry");
        PUBLIC_IDENTIFIERS.add("OopMapStream");
        PUBLIC_IDENTIFIERS.add("OopMapValue");
        PUBLIC_IDENTIFIERS.add("OopMapVisitor");
        PUBLIC_IDENTIFIERS.add("OopPrinter");
        PUBLIC_IDENTIFIERS.add("OopStorage");
        PUBLIC_IDENTIFIERS.add("OopTreeNodeAdapter");
        PUBLIC_IDENTIFIERS.add("OopTypes");
        PUBLIC_IDENTIFIERS.add("OopUtilities");
        PUBLIC_IDENTIFIERS.add("OopVisitor");
        PUBLIC_IDENTIFIERS.add("Opaque");
        PUBLIC_IDENTIFIERS.add("OpaqueFrameException");
        PUBLIC_IDENTIFIERS.add("OpCodes");
        PUBLIC_IDENTIFIERS.add("Opcodes");
        PUBLIC_IDENTIFIERS.add("OpenAction");
        PUBLIC_IDENTIFIERS.add("OpenDataException");
        PUBLIC_IDENTIFIERS.add("OpenFileInputStreamAction");
        PUBLIC_IDENTIFIERS.add("OpenFilesHandler");
        PUBLIC_IDENTIFIERS.add("OpeningHandshake");
        PUBLIC_IDENTIFIERS.add("OpenListResourceBundle");
        PUBLIC_IDENTIFIERS.add("OpenMBeanAttributeInfo");
        PUBLIC_IDENTIFIERS.add("OpenMBeanAttributeInfoSupport");
        PUBLIC_IDENTIFIERS.add("OpenMBeanConstructorInfo");
        PUBLIC_IDENTIFIERS.add("OpenMBeanConstructorInfoSupport");
        PUBLIC_IDENTIFIERS.add("OpenMBeanInfo");
        PUBLIC_IDENTIFIERS.add("OpenMBeanInfoSupport");
        PUBLIC_IDENTIFIERS.add("OpenMBeanOperationInfo");
        PUBLIC_IDENTIFIERS.add("OpenMBeanOperationInfoSupport");
        PUBLIC_IDENTIFIERS.add("OpenMBeanParameterInfo");
        PUBLIC_IDENTIFIERS.add("OpenMBeanParameterInfoSupport");
        PUBLIC_IDENTIFIERS.add("OpenOption");
        PUBLIC_IDENTIFIERS.add("OpensDirective");
        PUBLIC_IDENTIFIERS.add("OpensEntry");
        PUBLIC_IDENTIFIERS.add("OpensTree");
        PUBLIC_IDENTIFIERS.add("OpenType");
        PUBLIC_IDENTIFIERS.add("OpenURIHandler");
        PUBLIC_IDENTIFIERS.add("Operand");
        PUBLIC_IDENTIFIERS.add("OperatingSystemMXBean");
        PUBLIC_IDENTIFIERS.add("Operation");
        PUBLIC_IDENTIFIERS.add("OperationEntry");
        PUBLIC_IDENTIFIERS.add("OperationNotSupportedException");
        PUBLIC_IDENTIFIERS.add("OperationsException");
        PUBLIC_IDENTIFIERS.add("Operator");
        PUBLIC_IDENTIFIERS.add("Operators");
        PUBLIC_IDENTIFIERS.add("OperatorSymbol");
        PUBLIC_IDENTIFIERS.add("OpMap");
        PUBLIC_IDENTIFIERS.add("OpMapVector");
        PUBLIC_IDENTIFIERS.add("Option");
        PUBLIC_IDENTIFIERS.add("OptionalDataException");
        PUBLIC_IDENTIFIERS.add("OptionalHeader");
        PUBLIC_IDENTIFIERS.add("OptionalHeaderDataDirectories");
        PUBLIC_IDENTIFIERS.add("OptionalHeaderStandardFields");
        PUBLIC_IDENTIFIERS.add("OptionalHeaderWindowsSpecificFields");
        PUBLIC_IDENTIFIERS.add("OptionChecker");
        PUBLIC_IDENTIFIERS.add("OptionDeclarer");
        PUBLIC_IDENTIFIERS.add("OptionDescriptor");
        PUBLIC_IDENTIFIERS.add("OptionDialogBorder");
        PUBLIC_IDENTIFIERS.add("OptionException");
        PUBLIC_IDENTIFIERS.add("OptionFinder");
        PUBLIC_IDENTIFIERS.add("OptionFormat");
        PUBLIC_IDENTIFIERS.add("OptionHelper");
        PUBLIC_IDENTIFIERS.add("OptionLister");
        PUBLIC_IDENTIFIERS.add("OptionNameMap");
        PUBLIC_IDENTIFIERS.add("OptionOutputFormatter");
        PUBLIC_IDENTIFIERS.add("OptionPaneUI");
        PUBLIC_IDENTIFIERS.add("OptionParser");
        PUBLIC_IDENTIFIERS.add("Options");
        PUBLIC_IDENTIFIERS.add("OptionSet");
        PUBLIC_IDENTIFIERS.add("OptionSpec");
        PUBLIC_IDENTIFIERS.add("OptionSpecBuilder");
        PUBLIC_IDENTIFIERS.add("Or");
        PUBLIC_IDENTIFIERS.add("OrderClassLoaders");
        PUBLIC_IDENTIFIERS.add("OrderId");
        PUBLIC_IDENTIFIERS.add("OSEnvironment");
        PUBLIC_IDENTIFIERS.add("OSInfo");
        PUBLIC_IDENTIFIERS.add("OSThread");
        PUBLIC_IDENTIFIERS.add("OSUtils");
        PUBLIC_IDENTIFIERS.add("OtherName");
        PUBLIC_IDENTIFIERS.add("OutgoingHeaders");
        PUBLIC_IDENTIFIERS.add("OutlineTextRenderer");
        PUBLIC_IDENTIFIERS.add("OutOfMemoryError");
        PUBLIC_IDENTIFIERS.add("OutputFormat");
        PUBLIC_IDENTIFIERS.add("OutputFormatter");
        PUBLIC_IDENTIFIERS.add("OutputInt");
        PUBLIC_IDENTIFIERS.add("OutputKeys");
        PUBLIC_IDENTIFIERS.add("OutputStream");
        PUBLIC_IDENTIFIERS.add("OutputStreamImageOutputStreamSpi");
        PUBLIC_IDENTIFIERS.add("OutputStreamWriter");
        PUBLIC_IDENTIFIERS.add("OutputViewer");
        PUBLIC_IDENTIFIERS.add("outsideTimerAction");
        PUBLIC_IDENTIFIERS.add("OverlappingFileLockException");
        PUBLIC_IDENTIFIERS.add("OverlayLayout");
        PUBLIC_IDENTIFIERS.add("OverloadedChecksum");
        PUBLIC_IDENTIFIERS.add("Override");
        PUBLIC_IDENTIFIERS.add("OverrideNativeWindowHandle");
        PUBLIC_IDENTIFIERS.add("OverviewElement");
        PUBLIC_IDENTIFIERS.add("Package");
        PUBLIC_IDENTIFIERS.add("PackageElement");
        PUBLIC_IDENTIFIERS.add("PackageIndexWriter");
        PUBLIC_IDENTIFIERS.add("PackageLookup");
        PUBLIC_IDENTIFIERS.add("PackageNameFilter");
        PUBLIC_IDENTIFIERS.add("PackageNameFinder");
        PUBLIC_IDENTIFIERS.add("PackagerException");
        PUBLIC_IDENTIFIERS.add("PackageSummaryBuilder");
        PUBLIC_IDENTIFIERS.add("PackageSummaryWriter");
        PUBLIC_IDENTIFIERS.add("PackageSymbol");
        PUBLIC_IDENTIFIERS.add("PackageTree");
        PUBLIC_IDENTIFIERS.add("PackageTreeWriter");
        PUBLIC_IDENTIFIERS.add("PackageType");
        PUBLIC_IDENTIFIERS.add("PackageUseWriter");
        PUBLIC_IDENTIFIERS.add("PackageWriterImpl");
        PUBLIC_IDENTIFIERS.add("PackedColorModel");
        PUBLIC_IDENTIFIERS.add("Packet");
        PUBLIC_IDENTIFIERS.add("PAData");
        PUBLIC_IDENTIFIERS.add("PaddingLayout");
        PUBLIC_IDENTIFIERS.add("PAEncTSEnc");
        PUBLIC_IDENTIFIERS.add("PAForUserEnc");
        PUBLIC_IDENTIFIERS.add("Page");
        PUBLIC_IDENTIFIERS.add("Pageable");
        PUBLIC_IDENTIFIERS.add("PageableDoc");
        PUBLIC_IDENTIFIERS.add("PageCache");
        PUBLIC_IDENTIFIERS.add("PageFetcher");
        PUBLIC_IDENTIFIERS.add("PageFormat");
        PUBLIC_IDENTIFIERS.add("Paint");
        PUBLIC_IDENTIFIERS.add("Paint9Painter");
        PUBLIC_IDENTIFIERS.add("PaintContext");
        PUBLIC_IDENTIFIERS.add("Painter");
        PUBLIC_IDENTIFIERS.add("PaintEvent");
        PUBLIC_IDENTIFIERS.add("PaintEventDispatcher");
        PUBLIC_IDENTIFIERS.add("Pair");
        PUBLIC_IDENTIFIERS.add("PaletteBorder");
        PUBLIC_IDENTIFIERS.add("PaletteBuilder");
        PUBLIC_IDENTIFIERS.add("PaletteCloseIcon");
        PUBLIC_IDENTIFIERS.add("Panel");
        PUBLIC_IDENTIFIERS.add("PanelPeer");
        PUBLIC_IDENTIFIERS.add("PanelUI");
        PUBLIC_IDENTIFIERS.add("PaPacOptions");
        PUBLIC_IDENTIFIERS.add("Paper");
        PUBLIC_IDENTIFIERS.add("ParagraphAction");
        PUBLIC_IDENTIFIERS.add("ParagraphAttribute");
        PUBLIC_IDENTIFIERS.add("ParagraphView");
        PUBLIC_IDENTIFIERS.add("ParallelGroup");
        PUBLIC_IDENTIFIERS.add("ParallelListResourceBundle");
        PUBLIC_IDENTIFIERS.add("ParallelogramPipe");
        PUBLIC_IDENTIFIERS.add("ParallelScavengeHeap");
        PUBLIC_IDENTIFIERS.add("Parameter");
        PUBLIC_IDENTIFIERS.add("ParameterAnnotationEntry");
        PUBLIC_IDENTIFIERS.add("ParameterAnnotations");
        PUBLIC_IDENTIFIERS.add("ParameterBlock");
        PUBLIC_IDENTIFIERS.add("ParameterDescriptor");
        PUBLIC_IDENTIFIERS.add("Parameterizable");
        PUBLIC_IDENTIFIERS.add("ParameterizedType");
        PUBLIC_IDENTIFIERS.add("ParameterizedTypeImpl");
        PUBLIC_IDENTIFIERS.add("ParameterizedTypeTree");
        PUBLIC_IDENTIFIERS.add("ParameterMetaData");
        PUBLIC_IDENTIFIERS.add("ParameterNameProvider");
        PUBLIC_IDENTIFIERS.add("ParameterNode");
        PUBLIC_IDENTIFIERS.add("Parameters");
        PUBLIC_IDENTIFIERS.add("ParametersMap");
        PUBLIC_IDENTIFIERS.add("ParametersTypeData");
        PUBLIC_IDENTIFIERS.add("ParamSymbol");
        PUBLIC_IDENTIFIERS.add("ParamTaglet");
        PUBLIC_IDENTIFIERS.add("ParamTree");
        PUBLIC_IDENTIFIERS.add("ParenthesizedPatternTree");
        PUBLIC_IDENTIFIERS.add("ParenthesizedTree");
        PUBLIC_IDENTIFIERS.add("ParentNode");
        PUBLIC_IDENTIFIERS.add("ParsedLine");
        PUBLIC_IDENTIFIERS.add("ParseException");
        PUBLIC_IDENTIFIERS.add("ParsePosition");
        PUBLIC_IDENTIFIERS.add("Parser");
        PUBLIC_IDENTIFIERS.add("ParserAdapter");
        PUBLIC_IDENTIFIERS.add("ParserCallback");
        PUBLIC_IDENTIFIERS.add("ParserConfigurationException");
        PUBLIC_IDENTIFIERS.add("ParserConfigurationSettings");
        PUBLIC_IDENTIFIERS.add("ParserDelegator");
        PUBLIC_IDENTIFIERS.add("ParseResult");
        PUBLIC_IDENTIFIERS.add("ParserException");
        PUBLIC_IDENTIFIERS.add("ParserFactory");
        PUBLIC_IDENTIFIERS.add("ParseStatus");
        PUBLIC_IDENTIFIERS.add("ParsingException");
        PUBLIC_IDENTIFIERS.add("PartialCompositeContext");
        PUBLIC_IDENTIFIERS.add("PartialCompositeDirContext");
        PUBLIC_IDENTIFIERS.add("PartiallyInferredMethodType");
        PUBLIC_IDENTIFIERS.add("PartialResultException");
        PUBLIC_IDENTIFIERS.add("ParticipatesInPreview");
        PUBLIC_IDENTIFIERS.add("ParticleArray");
        PUBLIC_IDENTIFIERS.add("Password");
        PUBLIC_IDENTIFIERS.add("PasswordCallback");
        PUBLIC_IDENTIFIERS.add("PasswordProtection");
        PUBLIC_IDENTIFIERS.add("PasswordView");
        PUBLIC_IDENTIFIERS.add("PasteAction");
        PUBLIC_IDENTIFIERS.add("Patch");
        PUBLIC_IDENTIFIERS.add("PatchedModuleReader");
        PUBLIC_IDENTIFIERS.add("Path");
        PUBLIC_IDENTIFIERS.add("Path2D");
        PUBLIC_IDENTIFIERS.add("PathBodyHandler");
        PUBLIC_IDENTIFIERS.add("PathComponent");
        PUBLIC_IDENTIFIERS.add("PathConsumer2D");
        PUBLIC_IDENTIFIERS.add("PathConverter");
        PUBLIC_IDENTIFIERS.add("PathElement");
        PUBLIC_IDENTIFIERS.add("PathFactory");
        PUBLIC_IDENTIFIERS.add("PathFileObject");
        PUBLIC_IDENTIFIERS.add("PathGraphics");
        PUBLIC_IDENTIFIERS.add("PathIterator");
        PUBLIC_IDENTIFIERS.add("PathList");
        PUBLIC_IDENTIFIERS.add("PathMatcher");
        PUBLIC_IDENTIFIERS.add("PathProperties");
        PUBLIC_IDENTIFIERS.add("PathResourcePoolEntry");
        PUBLIC_IDENTIFIERS.add("PathSearchingVirtualMachine");
        PUBLIC_IDENTIFIERS.add("PathSubscriber");
        PUBLIC_IDENTIFIERS.add("Pattern");
        PUBLIC_IDENTIFIERS.add("PatternCaseLabelTree");
        PUBLIC_IDENTIFIERS.add("PatternSyntaxException");
        PUBLIC_IDENTIFIERS.add("PatternTree");
        PUBLIC_IDENTIFIERS.add("PBEKey");
        PUBLIC_IDENTIFIERS.add("PBEKeySpec");
        PUBLIC_IDENTIFIERS.add("PBEParameterSpec");
        PUBLIC_IDENTIFIERS.add("PCDesc");
        PUBLIC_IDENTIFIERS.add("PCK");
        PUBLIC_IDENTIFIERS.add("PCMM");
        PUBLIC_IDENTIFIERS.add("PDLOverrideSupported");
        PUBLIC_IDENTIFIERS.add("PeekGraphics");
        PUBLIC_IDENTIFIERS.add("PeekMetrics");
        PUBLIC_IDENTIFIERS.add("PeerAddrChange");
        PUBLIC_IDENTIFIERS.add("PeerAddressChangeNotification");
        PUBLIC_IDENTIFIERS.add("PeerEvent");
        PUBLIC_IDENTIFIERS.add("Pem");
        PUBLIC_IDENTIFIERS.add("PendingException");
        PUBLIC_IDENTIFIERS.add("Percentage");
        PUBLIC_IDENTIFIERS.add("PerfByteArrayCounter");
        PUBLIC_IDENTIFIERS.add("PerfByteArrayMonitor");
        PUBLIC_IDENTIFIERS.add("PerfCounter");
        PUBLIC_IDENTIFIERS.add("PerfDataBuffer");
        PUBLIC_IDENTIFIERS.add("PerfDataBufferImpl");
        PUBLIC_IDENTIFIERS.add("PerfDataBufferPrologue");
        PUBLIC_IDENTIFIERS.add("PerfDataEntry");
        PUBLIC_IDENTIFIERS.add("PerfDataEntryVisitor");
        PUBLIC_IDENTIFIERS.add("PerfDataFile");
        PUBLIC_IDENTIFIERS.add("PerfDataPrologue");
        PUBLIC_IDENTIFIERS.add("PerfectHashBuilder");
        PUBLIC_IDENTIFIERS.add("PerfInstrumentation");
        PUBLIC_IDENTIFIERS.add("PerfIntegerMonitor");
        PUBLIC_IDENTIFIERS.add("PerfLongArrayCounter");
        PUBLIC_IDENTIFIERS.add("PerfLongCounter");
        PUBLIC_IDENTIFIERS.add("PerfLongMonitor");
        PUBLIC_IDENTIFIERS.add("PerfMemory");
        PUBLIC_IDENTIFIERS.add("PerformanceLogger");
        PUBLIC_IDENTIFIERS.add("PerfStringConstantMonitor");
        PUBLIC_IDENTIFIERS.add("PerfStringCounter");
        PUBLIC_IDENTIFIERS.add("PerfStringMonitor");
        PUBLIC_IDENTIFIERS.add("PerfStringVariableMonitor");
        PUBLIC_IDENTIFIERS.add("Period");
        PUBLIC_IDENTIFIERS.add("Permission");
        PUBLIC_IDENTIFIERS.add("PermissionCollection");
        PUBLIC_IDENTIFIERS.add("PermissionEntry");
        PUBLIC_IDENTIFIERS.add("PermissionFactory");
        PUBLIC_IDENTIFIERS.add("PermittedSubclasses_attribute");
        PUBLIC_IDENTIFIERS.add("PersistenceDelegate");
        PUBLIC_IDENTIFIERS.add("PersistentMBean");
        PUBLIC_IDENTIFIERS.add("PersistentSnippet");
        PUBLIC_IDENTIFIERS.add("PGPData");
        PUBLIC_IDENTIFIERS.add("PhantomCleanable");
        PUBLIC_IDENTIFIERS.add("PhantomReference");
        PUBLIC_IDENTIFIERS.add("Phase");
        PUBLIC_IDENTIFIERS.add("PhaseCFG");
        PUBLIC_IDENTIFIERS.add("Phaser");
        PUBLIC_IDENTIFIERS.add("PhaseRegAlloc");
        PUBLIC_IDENTIFIERS.add("PhiNode");
        PUBLIC_IDENTIFIERS.add("PhysicalFont");
        PUBLIC_IDENTIFIERS.add("PhysicalStrike");
        PUBLIC_IDENTIFIERS.add("PingFrame");
        PUBLIC_IDENTIFIERS.add("Pipe");
        PUBLIC_IDENTIFIERS.add("PipedInputStream");
        PUBLIC_IDENTIFIERS.add("PipedOutputStream");
        PUBLIC_IDENTIFIERS.add("PipedReader");
        PUBLIC_IDENTIFIERS.add("PipedWriter");
        PUBLIC_IDENTIFIERS.add("PixelConverter");
        PUBLIC_IDENTIFIERS.add("PixelDrawPipe");
        PUBLIC_IDENTIFIERS.add("PixelFillPipe");
        PUBLIC_IDENTIFIERS.add("PixelGrabber");
        PUBLIC_IDENTIFIERS.add("PixelInterleavedSampleModel");
        PUBLIC_IDENTIFIERS.add("PixelsKey");
        PUBLIC_IDENTIFIERS.add("PixelToParallelogramConverter");
        PUBLIC_IDENTIFIERS.add("PixelToShapeConverter");
        PUBLIC_IDENTIFIERS.add("PKCS10");
        PUBLIC_IDENTIFIERS.add("PKCS10Attribute");
        PUBLIC_IDENTIFIERS.add("PKCS10Attributes");
        PUBLIC_IDENTIFIERS.add("PKCS11");
        PUBLIC_IDENTIFIERS.add("PKCS11Constants");
        PUBLIC_IDENTIFIERS.add("PKCS11Exception");
        PUBLIC_IDENTIFIERS.add("PKCS11RuntimeException");
        PUBLIC_IDENTIFIERS.add("PKCS7");
        PUBLIC_IDENTIFIERS.add("PKCS8EncodedKeySpec");
        PUBLIC_IDENTIFIERS.add("PKCS8Key");
        PUBLIC_IDENTIFIERS.add("PKCS9Attribute");
        PUBLIC_IDENTIFIERS.add("PKCS9Attributes");
        PUBLIC_IDENTIFIERS.add("PKIXBuilderParameters");
        PUBLIC_IDENTIFIERS.add("PKIXCertPathBuilderResult");
        PUBLIC_IDENTIFIERS.add("PKIXCertPathChecker");
        PUBLIC_IDENTIFIERS.add("PKIXCertPathValidatorResult");
        PUBLIC_IDENTIFIERS.add("PKIXExtendedParameters");
        PUBLIC_IDENTIFIERS.add("PKIXExtensions");
        PUBLIC_IDENTIFIERS.add("PKIXParameters");
        PUBLIC_IDENTIFIERS.add("PKIXRevocationChecker");
        PUBLIC_IDENTIFIERS.add("plain");
        PUBLIC_IDENTIFIERS.add("PlainDocument");
        PUBLIC_IDENTIFIERS.add("PlainTextInputStream");
        PUBLIC_IDENTIFIERS.add("PlainView");
        PUBLIC_IDENTIFIERS.add("PlatformComponent");
        PUBLIC_IDENTIFIERS.add("PlatformDescription");
        PUBLIC_IDENTIFIERS.add("PlatformFont");
        PUBLIC_IDENTIFIERS.add("PlatformGraphicsInfo");
        PUBLIC_IDENTIFIERS.add("PlatformInfo");
        PUBLIC_IDENTIFIERS.add("PlatformKind");
        PUBLIC_IDENTIFIERS.add("PlatformLogger");
        PUBLIC_IDENTIFIERS.add("PlatformLoggingMXBean");
        PUBLIC_IDENTIFIERS.add("PlatformManagedObject");
        PUBLIC_IDENTIFIERS.add("PlatformMBeanProvider");
        PUBLIC_IDENTIFIERS.add("PlatformNotSupported");
        PUBLIC_IDENTIFIERS.add("PlatformPrinterJobProxy");
        PUBLIC_IDENTIFIERS.add("PlatformProvider");
        PUBLIC_IDENTIFIERS.add("PlatformSocketImpl");
        PUBLIC_IDENTIFIERS.add("PlatformSupport");
        PUBLIC_IDENTIFIERS.add("PlatformSupportImpl");
        PUBLIC_IDENTIFIERS.add("PlatformUtils");
        PUBLIC_IDENTIFIERS.add("Plotter");
        PUBLIC_IDENTIFIERS.add("PlotterPanel");
        PUBLIC_IDENTIFIERS.add("Plugin");
        PUBLIC_IDENTIFIERS.add("PluginInfo");
        PUBLIC_IDENTIFIERS.add("PluginPrinter");
        PUBLIC_IDENTIFIERS.add("Plus");
        PUBLIC_IDENTIFIERS.add("PMap");
        PUBLIC_IDENTIFIERS.add("png");
        PUBLIC_IDENTIFIERS.add("PNGException");
        PUBLIC_IDENTIFIERS.add("PNGImageDecoder");
        PUBLIC_IDENTIFIERS.add("PNGImageReader");
        PUBLIC_IDENTIFIERS.add("PNGImageReaderSpi");
        PUBLIC_IDENTIFIERS.add("PNGImageWriterSpi");
        PUBLIC_IDENTIFIERS.add("PNGMetadata");
        PUBLIC_IDENTIFIERS.add("PNGMetadataFormat");
        PUBLIC_IDENTIFIERS.add("PNGMetadataFormatResources");
        PUBLIC_IDENTIFIERS.add("Point");
        PUBLIC_IDENTIFIERS.add("Point2D");
        PUBLIC_IDENTIFIERS.add("PointerFinder");
        PUBLIC_IDENTIFIERS.add("PointerLocation");
        PUBLIC_IDENTIFIERS.add("PointerType");
        PUBLIC_IDENTIFIERS.add("Policy");
        PUBLIC_IDENTIFIERS.add("PolicyConstraintsExtension");
        PUBLIC_IDENTIFIERS.add("PolicyFile");
        PUBLIC_IDENTIFIERS.add("PolicyInformation");
        PUBLIC_IDENTIFIERS.add("PolicyMappingsExtension");
        PUBLIC_IDENTIFIERS.add("PolicyNode");
        PUBLIC_IDENTIFIERS.add("PolicyParser");
        PUBLIC_IDENTIFIERS.add("PolicyQualifierInfo");
        PUBLIC_IDENTIFIERS.add("PolicySpi");
        PUBLIC_IDENTIFIERS.add("PolicyUtil");
        PUBLIC_IDENTIFIERS.add("PollDisposable");
        PUBLIC_IDENTIFIERS.add("Poller");
        PUBLIC_IDENTIFIERS.add("PollSelectorProvider");
        PUBLIC_IDENTIFIERS.add("Polygon");
        PUBLIC_IDENTIFIERS.add("PoolCallback");
        PUBLIC_IDENTIFIERS.add("PoolConstant");
        PUBLIC_IDENTIFIERS.add("PooledConnection");
        PUBLIC_IDENTIFIERS.add("PooledConnectionBuilder");
        PUBLIC_IDENTIFIERS.add("PooledConnectionFactory");
        PUBLIC_IDENTIFIERS.add("PoolOverflow");
        PUBLIC_IDENTIFIERS.add("PoolReader");
        PUBLIC_IDENTIFIERS.add("PoolWriter");
        PUBLIC_IDENTIFIERS.add("POP");
        PUBLIC_IDENTIFIERS.add("POP2");
        PUBLIC_IDENTIFIERS.add("PopInstruction");
        PUBLIC_IDENTIFIERS.add("Popup");
        PUBLIC_IDENTIFIERS.add("PopupFactory");
        PUBLIC_IDENTIFIERS.add("PopupFactoryAccessor");
        PUBLIC_IDENTIFIERS.add("PopupMenu");
        PUBLIC_IDENTIFIERS.add("PopupMenuAccessor");
        PUBLIC_IDENTIFIERS.add("PopupMenuBorder");
        PUBLIC_IDENTIFIERS.add("PopupMenuEvent");
        PUBLIC_IDENTIFIERS.add("PopupMenuListener");
        PUBLIC_IDENTIFIERS.add("PopupMenuPeer");
        PUBLIC_IDENTIFIERS.add("PopupMenuUI");
        PUBLIC_IDENTIFIERS.add("Port");
        PUBLIC_IDENTIFIERS.add("PortUnreachableException");
        PUBLIC_IDENTIFIERS.add("Position");
        PUBLIC_IDENTIFIERS.add("PosixFileAttributes");
        PUBLIC_IDENTIFIERS.add("PosixFileAttributeView");
        PUBLIC_IDENTIFIERS.add("PosixPtyTerminal");
        PUBLIC_IDENTIFIERS.add("POSIXSignals");
        PUBLIC_IDENTIFIERS.add("PosixSysTerminal");
        PUBLIC_IDENTIFIERS.add("PosterOutputStream");
        PUBLIC_IDENTIFIERS.add("PostProcessor");
        PUBLIC_IDENTIFIERS.add("PostResult");
        PUBLIC_IDENTIFIERS.add("PostVMInitHook");
        PUBLIC_IDENTIFIERS.add("Pow");
        PUBLIC_IDENTIFIERS.add("PPC64CurrentFrameGuess");
        PUBLIC_IDENTIFIERS.add("PPC64Frame");
        PUBLIC_IDENTIFIERS.add("PPC64JavaCallWrapper");
        PUBLIC_IDENTIFIERS.add("PPC64RegisterMap");
        PUBLIC_IDENTIFIERS.add("PPC64ThreadContext");
        PUBLIC_IDENTIFIERS.add("PreAction");
        PUBLIC_IDENTIFIERS.add("PrecedingIterator");
        PUBLIC_IDENTIFIERS.add("PrecedingSiblingIterator");
        PUBLIC_IDENTIFIERS.add("Preconditions");
        PUBLIC_IDENTIFIERS.add("Predicate");
        PUBLIC_IDENTIFIERS.add("PredicatedNodeTest");
        PUBLIC_IDENTIFIERS.add("PreferenceChangeEvent");
        PUBLIC_IDENTIFIERS.add("PreferenceChangeListener");
        PUBLIC_IDENTIFIERS.add("Preferences");
        PUBLIC_IDENTIFIERS.add("PreferencesFactory");
        PUBLIC_IDENTIFIERS.add("PreferencesHandler");
        PUBLIC_IDENTIFIERS.add("PrefixResolver");
        PUBLIC_IDENTIFIERS.add("PrefixResolverDefault");
        PUBLIC_IDENTIFIERS.add("PreHashedMap");
        PUBLIC_IDENTIFIERS.add("PreparedStatement");
        PUBLIC_IDENTIFIERS.add("Pretty");
        PUBLIC_IDENTIFIERS.add("Preview");
        PUBLIC_IDENTIFIERS.add("PreviewAPIListBuilder");
        PUBLIC_IDENTIFIERS.add("PreviewFeature");
        PUBLIC_IDENTIFIERS.add("PreviewFeatures");
        PUBLIC_IDENTIFIERS.add("PreviewFlagProvider");
        PUBLIC_IDENTIFIERS.add("PreviewListWriter");
        PUBLIC_IDENTIFIERS.add("Primitive_element_value");
        PUBLIC_IDENTIFIERS.add("PrimitiveConstant");
        PUBLIC_IDENTIFIERS.add("PrimitiveIterator");
        PUBLIC_IDENTIFIERS.add("PrimitiveSlot");
        PUBLIC_IDENTIFIERS.add("PrimitiveType");
        PUBLIC_IDENTIFIERS.add("PrimitiveTypeTree");
        PUBLIC_IDENTIFIERS.add("PrimitiveValue");
        PUBLIC_IDENTIFIERS.add("PrimitiveValueImpl");
        PUBLIC_IDENTIFIERS.add("Principal");
        PUBLIC_IDENTIFIERS.add("PrincipalComparator");
        PUBLIC_IDENTIFIERS.add("PrincipalEntry");
        PUBLIC_IDENTIFIERS.add("PrincipalName");
        PUBLIC_IDENTIFIERS.add("Printable");
        PUBLIC_IDENTIFIERS.add("PrintAboveWriter");
        PUBLIC_IDENTIFIERS.add("PrintColorUIResource");
        PUBLIC_IDENTIFIERS.add("Printer");
        PUBLIC_IDENTIFIERS.add("PrinterAbortException");
        PUBLIC_IDENTIFIERS.add("PrinterException");
        PUBLIC_IDENTIFIERS.add("PrinterGraphics");
        PUBLIC_IDENTIFIERS.add("PrinterIOException");
        PUBLIC_IDENTIFIERS.add("PrinterJob");
        PUBLIC_IDENTIFIERS.add("PrinterJobWrapper");
        PUBLIC_IDENTIFIERS.add("PrinterStateReason");
        PUBLIC_IDENTIFIERS.add("PrintEvent");
        PUBLIC_IDENTIFIERS.add("PrintException");
        PUBLIC_IDENTIFIERS.add("PrintFilesHandler");
        PUBLIC_IDENTIFIERS.add("PrintGraphics");
        PUBLIC_IDENTIFIERS.add("PrintingElementVisitor");
        PUBLIC_IDENTIFIERS.add("PrintingProcessor");
        PUBLIC_IDENTIFIERS.add("PrintingStatus");
        PUBLIC_IDENTIFIERS.add("PrintJob");
        PUBLIC_IDENTIFIERS.add("PrintJob2D");
        PUBLIC_IDENTIFIERS.add("PrintJobAdapter");
        PUBLIC_IDENTIFIERS.add("PrintJobAttribute");
        PUBLIC_IDENTIFIERS.add("PrintJobAttributeEvent");
        PUBLIC_IDENTIFIERS.add("PrintJobAttributeListener");
        PUBLIC_IDENTIFIERS.add("PrintJobAttributeSet");
        PUBLIC_IDENTIFIERS.add("PrintJobEvent");
        PUBLIC_IDENTIFIERS.add("PrintJobListener");
        PUBLIC_IDENTIFIERS.add("PrintQuality");
        PUBLIC_IDENTIFIERS.add("PrintRegionClosure");
        PUBLIC_IDENTIFIERS.add("PrintRequestAttribute");
        PUBLIC_IDENTIFIERS.add("PrintRequestAttributeSet");
        PUBLIC_IDENTIFIERS.add("PrintService");
        PUBLIC_IDENTIFIERS.add("PrintServiceAttribute");
        PUBLIC_IDENTIFIERS.add("PrintServiceAttributeEvent");
        PUBLIC_IDENTIFIERS.add("PrintServiceAttributeListener");
        PUBLIC_IDENTIFIERS.add("PrintServiceAttributeSet");
        PUBLIC_IDENTIFIERS.add("PrintServiceLookup");
        PUBLIC_IDENTIFIERS.add("PrintServiceLookupProvider");
        PUBLIC_IDENTIFIERS.add("PrintStream");
        PUBLIC_IDENTIFIERS.add("PrintStreamPrinter");
        PUBLIC_IDENTIFIERS.add("PrintWriter");
        PUBLIC_IDENTIFIERS.add("PriorityBlockingQueue");
        PUBLIC_IDENTIFIERS.add("PriorityFrame");
        PUBLIC_IDENTIFIERS.add("PriorityQueue");
        PUBLIC_IDENTIFIERS.add("PrivateAccess");
        PUBLIC_IDENTIFIERS.add("PrivateClassLoader");
        PUBLIC_IDENTIFIERS.add("PrivateKey");
        PUBLIC_IDENTIFIERS.add("PrivateKeyResolver");
        PUBLIC_IDENTIFIERS.add("PrivateKeyUsageExtension");
        PUBLIC_IDENTIFIERS.add("PrivateMLet");
        PUBLIC_IDENTIFIERS.add("PrivilegedAction");
        PUBLIC_IDENTIFIERS.add("PrivilegedActionException");
        PUBLIC_IDENTIFIERS.add("PrivilegedExceptionAction");
        PUBLIC_IDENTIFIERS.add("Process");
        PUBLIC_IDENTIFIERS.add("ProcessArgumentMatcher");
        PUBLIC_IDENTIFIERS.add("ProcessAttachingConnector");
        PUBLIC_IDENTIFIERS.add("ProcessBuilder");
        PUBLIC_IDENTIFIERS.add("ProcessBuilder.Redirect");
        PUBLIC_IDENTIFIERS.add("ProcessBuilder.Redirect.Type");
        PUBLIC_IDENTIFIERS.add("ProcessControl");
        PUBLIC_IDENTIFIERS.add("ProcessHandle");
        PUBLIC_IDENTIFIERS.add("ProcessHandle.Info");
        PUBLIC_IDENTIFIERS.add("ProcessHandler");
        PUBLIC_IDENTIFIERS.add("ProcessInfo");
        PUBLIC_IDENTIFIERS.add("Processing");
        PUBLIC_IDENTIFIERS.add("ProcessingEnvironment");
        PUBLIC_IDENTIFIERS.add("ProcessingInstruction");
        PUBLIC_IDENTIFIERS.add("ProcessingInstructionEvent");
        PUBLIC_IDENTIFIERS.add("ProcessingInstructionImpl");
        PUBLIC_IDENTIFIERS.add("ProcessListPanel");
        PUBLIC_IDENTIFIERS.add("Processor");
        PUBLIC_IDENTIFIERS.add("ProcessorVersion");
        PUBLIC_IDENTIFIERS.add("ProcessPath");
        PUBLIC_IDENTIFIERS.add("ProcImageClassLoader");
        PUBLIC_IDENTIFIERS.add("Profile");
        PUBLIC_IDENTIFIERS.add("ProfileData");
        PUBLIC_IDENTIFIERS.add("ProfileDataException");
        PUBLIC_IDENTIFIERS.add("ProfileDataVerifier");
        PUBLIC_IDENTIFIERS.add("ProfiledMethod");
        PUBLIC_IDENTIFIERS.add("ProfiledType");
        PUBLIC_IDENTIFIERS.add("ProfilingInfo");
        PUBLIC_IDENTIFIERS.add("ProgressBarPanel");
        PUBLIC_IDENTIFIERS.add("ProgressBarUI");
        PUBLIC_IDENTIFIERS.add("ProgressiveHeapVisitor");
        PUBLIC_IDENTIFIERS.add("ProgressMonitor");
        PUBLIC_IDENTIFIERS.add("ProgressMonitorInputStream");
        PUBLIC_IDENTIFIERS.add("ProjectivePoint");
        PUBLIC_IDENTIFIERS.add("ProjNode");
        PUBLIC_IDENTIFIERS.add("PropagatedException");
        PUBLIC_IDENTIFIERS.add("Properties");
        PUBLIC_IDENTIFIERS.add("PropertiesDefaultHandler");
        PUBLIC_IDENTIFIERS.add("PropertyBuilder");
        PUBLIC_IDENTIFIERS.add("PropertyChangeEvent");
        PUBLIC_IDENTIFIERS.add("PropertyChangeListener");
        PUBLIC_IDENTIFIERS.add("PropertyChangeListenerProxy");
        PUBLIC_IDENTIFIERS.add("PropertyChangeSupport");
        PUBLIC_IDENTIFIERS.add("PropertyDescriptor");
        PUBLIC_IDENTIFIERS.add("PropertyEditor");
        PUBLIC_IDENTIFIERS.add("PropertyEditorManager");
        PUBLIC_IDENTIFIERS.add("PropertyEditorSupport");
        PUBLIC_IDENTIFIERS.add("PropertyExpander");
        PUBLIC_IDENTIFIERS.add("PropertyHandler");
        PUBLIC_IDENTIFIERS.add("PropertyListener");
        PUBLIC_IDENTIFIERS.add("PropertyManager");
        PUBLIC_IDENTIFIERS.add("PropertyNames");
        PUBLIC_IDENTIFIERS.add("PropertyResourceBundle");
        PUBLIC_IDENTIFIERS.add("PropertyResourceBundleCharset");
        PUBLIC_IDENTIFIERS.add("PropertyState");
        PUBLIC_IDENTIFIERS.add("PropertyUtils");
        PUBLIC_IDENTIFIERS.add("PropertyVetoException");
        PUBLIC_IDENTIFIERS.add("PropertyWriter");
        PUBLIC_IDENTIFIERS.add("PropertyWriterImpl");
        PUBLIC_IDENTIFIERS.add("PropMwmHints");
        PUBLIC_IDENTIFIERS.add("ProtectionDomain");
        PUBLIC_IDENTIFIERS.add("ProtectionDomainCache");
        PUBLIC_IDENTIFIERS.add("ProtectionParameter");
        PUBLIC_IDENTIFIERS.add("ProtocolException");
        PUBLIC_IDENTIFIERS.add("ProtocolFamily");
        PUBLIC_IDENTIFIERS.add("Provider");
        PUBLIC_IDENTIFIERS.add("ProviderException");
        PUBLIC_IDENTIFIERS.add("ProviderMismatchException");
        PUBLIC_IDENTIFIERS.add("ProviderNotFoundException");
        PUBLIC_IDENTIFIERS.add("Providers");
        PUBLIC_IDENTIFIERS.add("ProvidesDirective");
        PUBLIC_IDENTIFIERS.add("ProvidesEntry");
        PUBLIC_IDENTIFIERS.add("ProvidesTree");
        PUBLIC_IDENTIFIERS.add("ProxiedGraphicsConfig");
        PUBLIC_IDENTIFIERS.add("Proxy");
        PUBLIC_IDENTIFIERS.add("ProxyClient");
        PUBLIC_IDENTIFIERS.add("ProxyGraphics");
        PUBLIC_IDENTIFIERS.add("ProxyGraphics2D");
        PUBLIC_IDENTIFIERS.add("ProxyLazyValue");
        PUBLIC_IDENTIFIERS.add("ProxyPrintGraphics");
        PUBLIC_IDENTIFIERS.add("ProxyRef");
        PUBLIC_IDENTIFIERS.add("ProxySelector");
        PUBLIC_IDENTIFIERS.add("PSOldGen");
        PUBLIC_IDENTIFIERS.add("PSource");
        PUBLIC_IDENTIFIERS.add("PSPrinterJob");
        PUBLIC_IDENTIFIERS.add("PSSParameterSpec");
        PUBLIC_IDENTIFIERS.add("PSStreamPrinterFactory");
        PUBLIC_IDENTIFIERS.add("PSStreamPrintJob");
        PUBLIC_IDENTIFIERS.add("PSStreamPrintService");
        PUBLIC_IDENTIFIERS.add("PStack");
        PUBLIC_IDENTIFIERS.add("PsuedoNames");
        PUBLIC_IDENTIFIERS.add("PSVIAttrNSImpl");
        PUBLIC_IDENTIFIERS.add("PSVIDocumentImpl");
        PUBLIC_IDENTIFIERS.add("PSVIDOMImplementationImpl");
        PUBLIC_IDENTIFIERS.add("PSVIElementNSImpl");
        PUBLIC_IDENTIFIERS.add("PSVIProvider");
        PUBLIC_IDENTIFIERS.add("PSYoungGen");
        PUBLIC_IDENTIFIERS.add("Pty");
        PUBLIC_IDENTIFIERS.add("PublicKey");
        PUBLIC_IDENTIFIERS.add("Publisher");
        PUBLIC_IDENTIFIERS.add("PumpReader");
        PUBLIC_IDENTIFIERS.add("Purpose");
        PUBLIC_IDENTIFIERS.add("PushbackInputStream");
        PUBLIC_IDENTIFIERS.add("PushbackReader");
        PUBLIC_IDENTIFIERS.add("PushInstruction");
        PUBLIC_IDENTIFIERS.add("PushPromiseFrame");
        PUBLIC_IDENTIFIERS.add("PushPromiseHandler");
        PUBLIC_IDENTIFIERS.add("PushPromisesHandlerWithMap");
        PUBLIC_IDENTIFIERS.add("PutAllAction");
        PUBLIC_IDENTIFIERS.add("PUTFIELD");
        PUBLIC_IDENTIFIERS.add("PutField");
        PUBLIC_IDENTIFIERS.add("PUTSTATIC");
        PUBLIC_IDENTIFIERS.add("QName");
        PUBLIC_IDENTIFIERS.add("QNameDV");
        PUBLIC_IDENTIFIERS.add("QuadCurve2D");
        PUBLIC_IDENTIFIERS.add("QualifiedNameable");
        PUBLIC_IDENTIFIERS.add("Query");
        PUBLIC_IDENTIFIERS.add("QueryEval");
        PUBLIC_IDENTIFIERS.add("QueryExp");
        PUBLIC_IDENTIFIERS.add("Queue");
        PUBLIC_IDENTIFIERS.add("QueueCallback");
        PUBLIC_IDENTIFIERS.add("QuitHandler");
        PUBLIC_IDENTIFIERS.add("QuitResponse");
        PUBLIC_IDENTIFIERS.add("Quo");
        PUBLIC_IDENTIFIERS.add("RadioButtonBorder");
        PUBLIC_IDENTIFIERS.add("RAFImageInputStreamSpi");
        PUBLIC_IDENTIFIERS.add("RAFImageOutputStreamSpi");
        PUBLIC_IDENTIFIERS.add("Random");
        PUBLIC_IDENTIFIERS.add("RandomAccess");
        PUBLIC_IDENTIFIERS.add("RandomAccessFile");
        PUBLIC_IDENTIFIERS.add("RandomAccessFileDataSource");
        PUBLIC_IDENTIFIERS.add("RandomDoublesSpliterator");
        PUBLIC_IDENTIFIERS.add("RandomGenerator");
        PUBLIC_IDENTIFIERS.add("RandomGeneratorProperties");
        PUBLIC_IDENTIFIERS.add("RandomIntsSpliterator");
        PUBLIC_IDENTIFIERS.add("RandomLongsSpliterator");
        PUBLIC_IDENTIFIERS.add("RandomSpliterator");
        PUBLIC_IDENTIFIERS.add("RandomSupport");
        PUBLIC_IDENTIFIERS.add("Range");
        PUBLIC_IDENTIFIERS.add("RangeException");
        PUBLIC_IDENTIFIERS.add("RangeExceptionImpl");
        PUBLIC_IDENTIFIERS.add("RangeImpl");
        PUBLIC_IDENTIFIERS.add("Raster");
        PUBLIC_IDENTIFIERS.add("RasterFormatException");
        PUBLIC_IDENTIFIERS.add("RasterOp");
        PUBLIC_IDENTIFIERS.add("RasterPrinterJob");
        PUBLIC_IDENTIFIERS.add("Raw");
        PUBLIC_IDENTIFIERS.add("RawChannel");
        PUBLIC_IDENTIFIERS.add("RawChannelTube");
        PUBLIC_IDENTIFIERS.add("RawCharacterHandler");
        PUBLIC_IDENTIFIERS.add("RawCommandLineLauncher");
        PUBLIC_IDENTIFIERS.add("RawConstant");
        PUBLIC_IDENTIFIERS.add("RawEvent");
        PUBLIC_IDENTIFIERS.add("RawHeapVisitor");
        PUBLIC_IDENTIFIERS.add("RawHtml");
        PUBLIC_IDENTIFIERS.add("RawOutputFormatter");
        PUBLIC_IDENTIFIERS.add("RBColor");
        PUBLIC_IDENTIFIERS.add("RBNode");
        PUBLIC_IDENTIFIERS.add("RBTree");
        PUBLIC_IDENTIFIERS.add("RC2Cipher");
        PUBLIC_IDENTIFIERS.add("RC2ParameterSpec");
        PUBLIC_IDENTIFIERS.add("RC5ParameterSpec");
        PUBLIC_IDENTIFIERS.add("RDN");
        PUBLIC_IDENTIFIERS.add("Rdn");
        PUBLIC_IDENTIFIERS.add("Readable");
        PUBLIC_IDENTIFIERS.add("ReadableByteChannel");
        PUBLIC_IDENTIFIERS.add("READER");
        PUBLIC_IDENTIFIERS.add("Reader");
        PUBLIC_IDENTIFIERS.add("ReaderUTF16");
        PUBLIC_IDENTIFIERS.add("ReaderUTF8");
        PUBLIC_IDENTIFIERS.add("ReaderUtil");
        PUBLIC_IDENTIFIERS.add("ReaderUtils");
        PUBLIC_IDENTIFIERS.add("ReadLock");
        PUBLIC_IDENTIFIERS.add("ReadOnlyBufferException");
        PUBLIC_IDENTIFIERS.add("ReadOnlyFileSystemException");
        PUBLIC_IDENTIFIERS.add("ReadOnlyIterator");
        PUBLIC_IDENTIFIERS.add("ReadOnlyTableCellEditor");
        PUBLIC_IDENTIFIERS.add("ReadPendingException");
        PUBLIC_IDENTIFIERS.add("ReadResult");
        PUBLIC_IDENTIFIERS.add("ReadWriteLock");
        PUBLIC_IDENTIFIERS.add("Realm");
        PUBLIC_IDENTIFIERS.add("RealmCallback");
        PUBLIC_IDENTIFIERS.add("RealmChoiceCallback");
        PUBLIC_IDENTIFIERS.add("RealmException");
        PUBLIC_IDENTIFIERS.add("Reason");
        PUBLIC_IDENTIFIERS.add("ReasonFlags");
        PUBLIC_IDENTIFIERS.add("Receiver");
        PUBLIC_IDENTIFIERS.add("ReceiverTypeData");
        PUBLIC_IDENTIFIERS.add("Record");
        PUBLIC_IDENTIFIERS.add("Record_attribute");
        PUBLIC_IDENTIFIERS.add("RecordComponent");
        PUBLIC_IDENTIFIERS.add("RecordComponentElement");
        PUBLIC_IDENTIFIERS.add("RecordComponentNode");
        PUBLIC_IDENTIFIERS.add("RecordComponentRemapper");
        PUBLIC_IDENTIFIERS.add("RecordComponentVisitor");
        PUBLIC_IDENTIFIERS.add("RecordedObject");
        PUBLIC_IDENTIFIERS.add("Recorder");
        PUBLIC_IDENTIFIERS.add("RecoveryDeferredTypeMap");
        PUBLIC_IDENTIFIERS.add("RecoveryLoadClass");
        PUBLIC_IDENTIFIERS.add("Rect");
        PUBLIC_IDENTIFIERS.add("Rectangle");
        PUBLIC_IDENTIFIERS.add("Rectangle2D");
        PUBLIC_IDENTIFIERS.add("RectangularShape");
        PUBLIC_IDENTIFIERS.add("RectSize");
        PUBLIC_IDENTIFIERS.add("RecursiveAction");
        PUBLIC_IDENTIFIERS.add("RecursiveTask");
        PUBLIC_IDENTIFIERS.add("Redirect");
        PUBLIC_IDENTIFIERS.add("ReductionOperation");
        PUBLIC_IDENTIFIERS.add("ReencodingInputStream");
        PUBLIC_IDENTIFIERS.add("ReentrantContext");
        PUBLIC_IDENTIFIERS.add("ReentrantContextProvider");
        PUBLIC_IDENTIFIERS.add("ReentrantContextProviderCLQ");
        PUBLIC_IDENTIFIERS.add("ReentrantContextProviderTL");
        PUBLIC_IDENTIFIERS.add("ReentrantLock");
        PUBLIC_IDENTIFIERS.add("ReentrantReadWriteLock");
        PUBLIC_IDENTIFIERS.add("Ref");
        PUBLIC_IDENTIFIERS.add("RefAddr");
        PUBLIC_IDENTIFIERS.add("Reference");
        PUBLIC_IDENTIFIERS.add("Referenceable");
        PUBLIC_IDENTIFIERS.add("ReferenceCountingDevice");
        PUBLIC_IDENTIFIERS.add("ReferenceData");
        PUBLIC_IDENTIFIERS.add("ReferenceMap");
        PUBLIC_IDENTIFIERS.add("ReferenceNodeSetData");
        PUBLIC_IDENTIFIERS.add("ReferenceNotInitializedException");
        PUBLIC_IDENTIFIERS.add("ReferenceOctetStreamData");
        PUBLIC_IDENTIFIERS.add("ReferenceParser");
        PUBLIC_IDENTIFIERS.add("ReferenceQueue");
        PUBLIC_IDENTIFIERS.add("ReferenceSubTreeData");
        PUBLIC_IDENTIFIERS.add("ReferenceTree");
        PUBLIC_IDENTIFIERS.add("ReferenceType");
        PUBLIC_IDENTIFIERS.add("ReferenceTypeImpl");
        PUBLIC_IDENTIFIERS.add("ReferenceUriSchemesSupported");
        PUBLIC_IDENTIFIERS.add("ReferenceWrapper");
        PUBLIC_IDENTIFIERS.add("ReferralException");
        PUBLIC_IDENTIFIERS.add("Reflection");
        PUBLIC_IDENTIFIERS.add("ReflectionException");
        PUBLIC_IDENTIFIERS.add("ReflectionFactory");
        PUBLIC_IDENTIFIERS.add("ReflectiveOperationException");
        PUBLIC_IDENTIFIERS.add("RefMapFormatter");
        PUBLIC_IDENTIFIERS.add("Refreshable");
        PUBLIC_IDENTIFIERS.add("RefreshFailedException");
        PUBLIC_IDENTIFIERS.add("RefType");
        PUBLIC_IDENTIFIERS.add("RegexMatcher");
        PUBLIC_IDENTIFIERS.add("Region");
        PUBLIC_IDENTIFIERS.add("RegionClipSpanIterator");
        PUBLIC_IDENTIFIERS.add("RegionIterator");
        PUBLIC_IDENTIFIERS.add("RegionNode");
        PUBLIC_IDENTIFIERS.add("RegionSpanIterator");
        PUBLIC_IDENTIFIERS.add("Register");
        PUBLIC_IDENTIFIERS.add("RegisterableService");
        PUBLIC_IDENTIFIERS.add("RegisterAttributes");
        PUBLIC_IDENTIFIERS.add("RegisterCategory");
        PUBLIC_IDENTIFIERS.add("RegisterConfig");
        PUBLIC_IDENTIFIERS.add("Registered");
        PUBLIC_IDENTIFIERS.add("RegisteredDomain");
        PUBLIC_IDENTIFIERS.add("RegisterMap");
        PUBLIC_IDENTIFIERS.add("RegistrationContext");
        PUBLIC_IDENTIFIERS.add("Registry");
        PUBLIC_IDENTIFIERS.add("RegistryContext");
        PUBLIC_IDENTIFIERS.add("RegistryContextFactory");
        PUBLIC_IDENTIFIERS.add("RegistryHandler");
        PUBLIC_IDENTIFIERS.add("RegistryImpl");
        PUBLIC_IDENTIFIERS.add("Regs");
        PUBLIC_IDENTIFIERS.add("RegularExpression");
        PUBLIC_IDENTIFIERS.add("Reifier");
        PUBLIC_IDENTIFIERS.add("RejectedExecutionException");
        PUBLIC_IDENTIFIERS.add("RejectedExecutionHandler");
        PUBLIC_IDENTIFIERS.add("Relation");
        PUBLIC_IDENTIFIERS.add("Relational");
        PUBLIC_IDENTIFIERS.add("RelationException");
        PUBLIC_IDENTIFIERS.add("RelationNotFoundException");
        PUBLIC_IDENTIFIERS.add("RelationNotification");
        PUBLIC_IDENTIFIERS.add("RelationService");
        PUBLIC_IDENTIFIERS.add("RelationServiceMBean");
        PUBLIC_IDENTIFIERS.add("RelationServiceNotRegisteredException");
        PUBLIC_IDENTIFIERS.add("RelationSupport");
        PUBLIC_IDENTIFIERS.add("RelationSupportMBean");
        PUBLIC_IDENTIFIERS.add("RelationType");
        PUBLIC_IDENTIFIERS.add("RelationTypeNotFoundException");
        PUBLIC_IDENTIFIERS.add("RelationTypeSupport");
        PUBLIC_IDENTIFIERS.add("RelativeDirectory");
        PUBLIC_IDENTIFIERS.add("RelativeFile");
        PUBLIC_IDENTIFIERS.add("RelativePath");
        PUBLIC_IDENTIFIERS.add("ReliableLog");
        PUBLIC_IDENTIFIERS.add("RelinkableCallSite");
        PUBLIC_IDENTIFIERS.add("Remapper");
        PUBLIC_IDENTIFIERS.add("RemappingAnnotationAdapter");
        PUBLIC_IDENTIFIERS.add("RemappingMethodAdapter");
        PUBLIC_IDENTIFIERS.add("Remote");
        PUBLIC_IDENTIFIERS.add("RemoteAARCH64Thread");
        PUBLIC_IDENTIFIERS.add("RemoteAARCH64ThreadContext");
        PUBLIC_IDENTIFIERS.add("RemoteAARCH64ThreadFactory");
        PUBLIC_IDENTIFIERS.add("RemoteAMD64Thread");
        PUBLIC_IDENTIFIERS.add("RemoteAMD64ThreadContext");
        PUBLIC_IDENTIFIERS.add("RemoteAMD64ThreadFactory");
        PUBLIC_IDENTIFIERS.add("RemoteCall");
        PUBLIC_IDENTIFIERS.add("RemoteDebugger");
        PUBLIC_IDENTIFIERS.add("RemoteDebuggerClient");
        PUBLIC_IDENTIFIERS.add("RemoteDebuggerServer");
        PUBLIC_IDENTIFIERS.add("RemoteException");
        PUBLIC_IDENTIFIERS.add("RemoteExecutionControl");
        PUBLIC_IDENTIFIERS.add("RemoteHost");
        PUBLIC_IDENTIFIERS.add("RemoteHostImpl");
        PUBLIC_IDENTIFIERS.add("RemoteMonitoredVm");
        PUBLIC_IDENTIFIERS.add("RemoteObject");
        PUBLIC_IDENTIFIERS.add("RemoteObjectInvocationHandler");
        PUBLIC_IDENTIFIERS.add("RemotePPC64Thread");
        PUBLIC_IDENTIFIERS.add("RemotePPC64ThreadContext");
        PUBLIC_IDENTIFIERS.add("RemotePPC64ThreadFactory");
        PUBLIC_IDENTIFIERS.add("RemoteRef");
        PUBLIC_IDENTIFIERS.add("RemoteReference");
        PUBLIC_IDENTIFIERS.add("RemoteRISCV64Thread");
        PUBLIC_IDENTIFIERS.add("RemoteRISCV64ThreadContext");
        PUBLIC_IDENTIFIERS.add("RemoteRISCV64ThreadFactory");
        PUBLIC_IDENTIFIERS.add("RemoteServer");
        PUBLIC_IDENTIFIERS.add("RemoteStub");
        PUBLIC_IDENTIFIERS.add("RemoteThread");
        PUBLIC_IDENTIFIERS.add("RemoteThreadFactory");
        PUBLIC_IDENTIFIERS.add("RemoteVm");
        PUBLIC_IDENTIFIERS.add("RemoteVmImpl");
        PUBLIC_IDENTIFIERS.add("RemoteVmManager");
        PUBLIC_IDENTIFIERS.add("RemoteX86Thread");
        PUBLIC_IDENTIFIERS.add("RemoteX86ThreadContext");
        PUBLIC_IDENTIFIERS.add("RemoteX86ThreadFactory");
        PUBLIC_IDENTIFIERS.add("RenderableImage");
        PUBLIC_IDENTIFIERS.add("RenderableImageOp");
        PUBLIC_IDENTIFIERS.add("RenderableImageProducer");
        PUBLIC_IDENTIFIERS.add("RenderBuffer");
        PUBLIC_IDENTIFIERS.add("RenderContext");
        PUBLIC_IDENTIFIERS.add("RenderedImage");
        PUBLIC_IDENTIFIERS.add("RenderedImageFactory");
        PUBLIC_IDENTIFIERS.add("Renderer");
        PUBLIC_IDENTIFIERS.add("RenderingEngine");
        PUBLIC_IDENTIFIERS.add("RenderingHints");
        PUBLIC_IDENTIFIERS.add("RenderLoops");
        PUBLIC_IDENTIFIERS.add("RenderQueue");
        PUBLIC_IDENTIFIERS.add("RepaintArea");
        PUBLIC_IDENTIFIERS.add("RepaintListener");
        PUBLIC_IDENTIFIERS.add("RepaintManager");
        PUBLIC_IDENTIFIERS.add("RepaintManagerAccessor");
        PUBLIC_IDENTIFIERS.add("Repeatable");
        PUBLIC_IDENTIFIERS.add("Replaceable");
        PUBLIC_IDENTIFIERS.add("ReplaceableString");
        PUBLIC_IDENTIFIERS.add("ReplaceableUCharacterIterator");
        PUBLIC_IDENTIFIERS.add("ReplayCache");
        PUBLIC_IDENTIFIERS.add("ReplicateScaleFilter");
        PUBLIC_IDENTIFIERS.add("Reporter");
        PUBLIC_IDENTIFIERS.add("Repository");
        PUBLIC_IDENTIFIERS.add("Request");
        PUBLIC_IDENTIFIERS.add("RequestFocusController");
        PUBLIC_IDENTIFIERS.add("RequiredModelMBean");
        PUBLIC_IDENTIFIERS.add("RequiresDirective");
        PUBLIC_IDENTIFIERS.add("RequiresEntry");
        PUBLIC_IDENTIFIERS.add("RequiresTree");
        PUBLIC_IDENTIFIERS.add("RescaleOp");
        PUBLIC_IDENTIFIERS.add("ReservedStackAccess");
        PUBLIC_IDENTIFIERS.add("ResetFrame");
        PUBLIC_IDENTIFIERS.add("ResolutionException");
        PUBLIC_IDENTIFIERS.add("ResolutionSyntax");
        PUBLIC_IDENTIFIERS.add("Resolve");
        PUBLIC_IDENTIFIERS.add("ResolvedJavaField");
        PUBLIC_IDENTIFIERS.add("ResolvedJavaMethod");
        PUBLIC_IDENTIFIERS.add("ResolvedJavaType");
        PUBLIC_IDENTIFIERS.add("ResolveListener");
        PUBLIC_IDENTIFIERS.add("Resolver");
        PUBLIC_IDENTIFIERS.add("ResolverAnonymous");
        PUBLIC_IDENTIFIERS.add("ResolverConfiguration");
        PUBLIC_IDENTIFIERS.add("ResolverDirectHTTP");
        PUBLIC_IDENTIFIERS.add("ResolveResult");
        PUBLIC_IDENTIFIERS.add("ResolverFragment");
        PUBLIC_IDENTIFIERS.add("ResolverLocalFilesystem");
        PUBLIC_IDENTIFIERS.add("ResolverXPointer");
        PUBLIC_IDENTIFIERS.add("Resource");
        PUBLIC_IDENTIFIERS.add("ResourceBundle");
        PUBLIC_IDENTIFIERS.add("ResourceBundleBasedAdapter");
        PUBLIC_IDENTIFIERS.add("ResourceBundleControlProvider");
        PUBLIC_IDENTIFIERS.add("ResourceBundleEnumeration");
        PUBLIC_IDENTIFIERS.add("ResourceBundleHelper");
        PUBLIC_IDENTIFIERS.add("ResourceBundleProvider");
        PUBLIC_IDENTIFIERS.add("ResourceCleanup");
        PUBLIC_IDENTIFIERS.add("ResourceDecompressor");
        PUBLIC_IDENTIFIERS.add("ResourceDecompressorFactory");
        PUBLIC_IDENTIFIERS.add("ResourceFilter");
        PUBLIC_IDENTIFIERS.add("ResourceIOException");
        PUBLIC_IDENTIFIERS.add("ResourceList");
        PUBLIC_IDENTIFIERS.add("ResourceLocator");
        PUBLIC_IDENTIFIERS.add("ResourceManager");
        PUBLIC_IDENTIFIERS.add("ResourcePool");
        PUBLIC_IDENTIFIERS.add("ResourcePoolBuilder");
        PUBLIC_IDENTIFIERS.add("ResourcePoolEntry");
        PUBLIC_IDENTIFIERS.add("ResourcePoolImpl");
        PUBLIC_IDENTIFIERS.add("ResourcePoolManager");
        PUBLIC_IDENTIFIERS.add("ResourcePoolModule");
        PUBLIC_IDENTIFIERS.add("ResourcePoolModuleView");
        PUBLIC_IDENTIFIERS.add("ResourcePrevisitor");
        PUBLIC_IDENTIFIERS.add("ResourceRecord");
        PUBLIC_IDENTIFIERS.add("ResourceResolver");
        PUBLIC_IDENTIFIERS.add("ResourceResolverContext");
        PUBLIC_IDENTIFIERS.add("ResourceResolverException");
        PUBLIC_IDENTIFIERS.add("ResourceResolverSpi");
        PUBLIC_IDENTIFIERS.add("Resources");
        PUBLIC_IDENTIFIERS.add("Resources_de");
        PUBLIC_IDENTIFIERS.add("Resources_ja");
        PUBLIC_IDENTIFIERS.add("Resources_zh_CN");
        PUBLIC_IDENTIFIERS.add("ResourcesMgr");
        PUBLIC_IDENTIFIERS.add("ResponseCache");
        PUBLIC_IDENTIFIERS.add("ResponseInfo");
        PUBLIC_IDENTIFIERS.add("ResponseSubscribers");
        PUBLIC_IDENTIFIERS.add("ResponseTimerEvent");
        PUBLIC_IDENTIFIERS.add("RestartableTask");
        PUBLIC_IDENTIFIERS.add("RestoreAction");
        PUBLIC_IDENTIFIERS.add("Result");
        PUBLIC_IDENTIFIERS.add("ResultContainer");
        PUBLIC_IDENTIFIERS.add("ResultSet");
        PUBLIC_IDENTIFIERS.add("ResultSetMetaData");
        PUBLIC_IDENTIFIERS.add("ResultTypeFinder");
        PUBLIC_IDENTIFIERS.add("RET");
        PUBLIC_IDENTIFIERS.add("RetData");
        PUBLIC_IDENTIFIERS.add("Retention");
        PUBLIC_IDENTIFIERS.add("RetrievalMethod");
        PUBLIC_IDENTIFIERS.add("RetrievalMethodResolver");
        PUBLIC_IDENTIFIERS.add("Retry");
        PUBLIC_IDENTIFIERS.add("RETURN");
        PUBLIC_IDENTIFIERS.add("ReturnaddressType");
        PUBLIC_IDENTIFIERS.add("ReturnInstruction");
        PUBLIC_IDENTIFIERS.add("ReturnTaglet");
        PUBLIC_IDENTIFIERS.add("ReturnTree");
        PUBLIC_IDENTIFIERS.add("ReturnType");
        PUBLIC_IDENTIFIERS.add("ReturnTypeEntry");
        PUBLIC_IDENTIFIERS.add("RevalidationHandler");
        PUBLIC_IDENTIFIERS.add("ReverbType");
        PUBLIC_IDENTIFIERS.add("ReverseAxesWalker");
        PUBLIC_IDENTIFIERS.add("ReversePtrs");
        PUBLIC_IDENTIFIERS.add("ReversePtrsAnalysis");
        PUBLIC_IDENTIFIERS.add("RevocationStatus");
        PUBLIC_IDENTIFIERS.add("RevPtrsTreeNodeAdapter");
        PUBLIC_IDENTIFIERS.add("RFC2253Parser");
        PUBLIC_IDENTIFIERS.add("RFC822Name");
        PUBLIC_IDENTIFIERS.add("Rgba");
        PUBLIC_IDENTIFIERS.add("RgbaPre");
        PUBLIC_IDENTIFIERS.add("RGBColor");
        PUBLIC_IDENTIFIERS.add("RGBImageFilter");
        PUBLIC_IDENTIFIERS.add("Rgbx");
        PUBLIC_IDENTIFIERS.add("RichConfiguration");
        PUBLIC_IDENTIFIERS.add("RichDiagnosticFormatter");
        PUBLIC_IDENTIFIERS.add("RichPrinter");
        PUBLIC_IDENTIFIERS.add("RISCV64");
        PUBLIC_IDENTIFIERS.add("RISCV64CurrentFrameGuess");
        PUBLIC_IDENTIFIERS.add("RISCV64Frame");
        PUBLIC_IDENTIFIERS.add("RISCV64HotSpotJVMCIBackendFactory");
        PUBLIC_IDENTIFIERS.add("RISCV64HotSpotRegisterConfig");
        PUBLIC_IDENTIFIERS.add("RISCV64JavaCallWrapper");
        PUBLIC_IDENTIFIERS.add("RISCV64RegisterMap");
        PUBLIC_IDENTIFIERS.add("RISCV64ThreadContext");
        PUBLIC_IDENTIFIERS.add("RMI");
        PUBLIC_IDENTIFIERS.add("RMIClassLoader");
        PUBLIC_IDENTIFIERS.add("RMIClassLoaderSpi");
        PUBLIC_IDENTIFIERS.add("RMIClientSocketFactory");
        PUBLIC_IDENTIFIERS.add("RMIConnection");
        PUBLIC_IDENTIFIERS.add("RMIConnectionImpl");
        PUBLIC_IDENTIFIERS.add("RMIConnector");
        PUBLIC_IDENTIFIERS.add("RMIConnectorServer");
        PUBLIC_IDENTIFIERS.add("RMIExporter");
        PUBLIC_IDENTIFIERS.add("RMIFailureHandler");
        PUBLIC_IDENTIFIERS.add("RMIHelper");
        PUBLIC_IDENTIFIERS.add("RMIIIOPServerImpl");
        PUBLIC_IDENTIFIERS.add("RMIJRMPServerImpl");
        PUBLIC_IDENTIFIERS.add("RMISecurityException");
        PUBLIC_IDENTIFIERS.add("RMISecurityManager");
        PUBLIC_IDENTIFIERS.add("RMIServer");
        PUBLIC_IDENTIFIERS.add("RMIServerImpl");
        PUBLIC_IDENTIFIERS.add("RMIServerSocketFactory");
        PUBLIC_IDENTIFIERS.add("RMISocketFactory");
        PUBLIC_IDENTIFIERS.add("rmiURLContext");
        PUBLIC_IDENTIFIERS.add("rmiURLContextFactory");
        PUBLIC_IDENTIFIERS.add("Robot");
        PUBLIC_IDENTIFIERS.add("RobotPeer");
        PUBLIC_IDENTIFIERS.add("RobustOopDeterminator");
        PUBLIC_IDENTIFIERS.add("Role");
        PUBLIC_IDENTIFIERS.add("RoleInfo");
        PUBLIC_IDENTIFIERS.add("RoleInfoNotFoundException");
        PUBLIC_IDENTIFIERS.add("RoleList");
        PUBLIC_IDENTIFIERS.add("RoleNotFoundException");
        PUBLIC_IDENTIFIERS.add("RoleResult");
        PUBLIC_IDENTIFIERS.add("RoleStatus");
        PUBLIC_IDENTIFIERS.add("RoleUnresolved");
        PUBLIC_IDENTIFIERS.add("RoleUnresolvedList");
        PUBLIC_IDENTIFIERS.add("RolloverButtonBorder");
        PUBLIC_IDENTIFIERS.add("RootIterator");
        PUBLIC_IDENTIFIERS.add("RootLayout");
        PUBLIC_IDENTIFIERS.add("RootNode");
        PUBLIC_IDENTIFIERS.add("RootPackageSymbol");
        PUBLIC_IDENTIFIERS.add("RootPaneContainer");
        PUBLIC_IDENTIFIERS.add("RootPaneUI");
        PUBLIC_IDENTIFIERS.add("RootTreeNodeAdapter");
        PUBLIC_IDENTIFIERS.add("RoundEnvironment");
        PUBLIC_IDENTIFIERS.add("RoundRectangle2D");
        PUBLIC_IDENTIFIERS.add("Row");
        PUBLIC_IDENTIFIERS.add("RowClosure");
        PUBLIC_IDENTIFIERS.add("RowFilter");
        PUBLIC_IDENTIFIERS.add("RowId");
        PUBLIC_IDENTIFIERS.add("RowMapper");
        PUBLIC_IDENTIFIERS.add("Rows");
        PUBLIC_IDENTIFIERS.add("RowSet");
        PUBLIC_IDENTIFIERS.add("RowSetEvent");
        PUBLIC_IDENTIFIERS.add("RowSetFactory");
        PUBLIC_IDENTIFIERS.add("RowSetInternal");
        PUBLIC_IDENTIFIERS.add("RowSetListener");
        PUBLIC_IDENTIFIERS.add("RowSetMetaData");
        PUBLIC_IDENTIFIERS.add("RowSetMetaDataImpl");
        PUBLIC_IDENTIFIERS.add("RowSetProvider");
        PUBLIC_IDENTIFIERS.add("RowSetReader");
        PUBLIC_IDENTIFIERS.add("RowSetWarning");
        PUBLIC_IDENTIFIERS.add("RowSetWriter");
        PUBLIC_IDENTIFIERS.add("RowSorter");
        PUBLIC_IDENTIFIERS.add("RowSorterEvent");
        PUBLIC_IDENTIFIERS.add("RowSorterListener");
        PUBLIC_IDENTIFIERS.add("RowView");
        PUBLIC_IDENTIFIERS.add("RSAKey");
        PUBLIC_IDENTIFIERS.add("RSAKeyFactory");
        PUBLIC_IDENTIFIERS.add("RSAKeyGenParameterSpec");
        PUBLIC_IDENTIFIERS.add("RSAKeyValue");
        PUBLIC_IDENTIFIERS.add("RSAKeyValueResolver");
        PUBLIC_IDENTIFIERS.add("RSAMultiPrimePrivateCrtKey");
        PUBLIC_IDENTIFIERS.add("RSAMultiPrimePrivateCrtKeySpec");
        PUBLIC_IDENTIFIERS.add("RSAOtherPrimeInfo");
        PUBLIC_IDENTIFIERS.add("RSAPrivateCrtKey");
        PUBLIC_IDENTIFIERS.add("RSAPrivateCrtKeySpec");
        PUBLIC_IDENTIFIERS.add("RSAPrivateKey");
        PUBLIC_IDENTIFIERS.add("RSAPrivateKeySpec");
        PUBLIC_IDENTIFIERS.add("RSAPSSSignature");
        PUBLIC_IDENTIFIERS.add("RSAPublicKey");
        PUBLIC_IDENTIFIERS.add("RSAPublicKeySpec");
        PUBLIC_IDENTIFIERS.add("RSAUtil");
        PUBLIC_IDENTIFIERS.add("RTFEditorKit");
        PUBLIC_IDENTIFIERS.add("RTFIterator");
        PUBLIC_IDENTIFIERS.add("RuleBasedBreakIterator");
        PUBLIC_IDENTIFIERS.add("RuleBasedCollator");
        PUBLIC_IDENTIFIERS.add("RunElement");
        PUBLIC_IDENTIFIERS.add("RunException");
        PUBLIC_IDENTIFIERS.add("Runnable");
        PUBLIC_IDENTIFIERS.add("RunnableFuture");
        PUBLIC_IDENTIFIERS.add("RunnableScheduledFuture");
        PUBLIC_IDENTIFIERS.add("Runtime");
        PUBLIC_IDENTIFIERS.add("Runtime.Version");
        PUBLIC_IDENTIFIERS.add("Runtime1");
        PUBLIC_IDENTIFIERS.add("RuntimeAnnotations_attribute");
        PUBLIC_IDENTIFIERS.add("RuntimeBlob");
        PUBLIC_IDENTIFIERS.add("RuntimeErrorException");
        PUBLIC_IDENTIFIERS.add("RuntimeException");
        PUBLIC_IDENTIFIERS.add("RuntimeInvisibleAnnotations");
        PUBLIC_IDENTIFIERS.add("RuntimeInvisibleAnnotations_attribute");
        PUBLIC_IDENTIFIERS.add("RuntimeInvisibleParameterAnnotations");
        PUBLIC_IDENTIFIERS.add("RuntimeInvisibleParameterAnnotations_attribute");
        PUBLIC_IDENTIFIERS.add("RuntimeInvisibleTypeAnnotations_attribute");
        PUBLIC_IDENTIFIERS.add("RuntimeMBeanException");
        PUBLIC_IDENTIFIERS.add("RuntimeMXBean");
        PUBLIC_IDENTIFIERS.add("RuntimeOperationsException");
        PUBLIC_IDENTIFIERS.add("RuntimeParameterAnnotations_attribute");
        PUBLIC_IDENTIFIERS.add("RuntimePermission");
        PUBLIC_IDENTIFIERS.add("RuntimeStub");
        PUBLIC_IDENTIFIERS.add("RuntimeTypeAnnotations_attribute");
        PUBLIC_IDENTIFIERS.add("RuntimeVisibleAnnotations");
        PUBLIC_IDENTIFIERS.add("RuntimeVisibleAnnotations_attribute");
        PUBLIC_IDENTIFIERS.add("RuntimeVisibleParameterAnnotations");
        PUBLIC_IDENTIFIERS.add("RuntimeVisibleParameterAnnotations_attribute");
        PUBLIC_IDENTIFIERS.add("RuntimeVisibleTypeAnnotations_attribute");
    }

    private static void addPublicIdentifiersSS() {
        PUBLIC_IDENTIFIERS.add("SAEditorPane");
        PUBLIC_IDENTIFIERS.add("SafepointBlob");
        PUBLIC_IDENTIFIERS.add("SafePointNode");
        PUBLIC_IDENTIFIERS.add("SafeThread");
        PUBLIC_IDENTIFIERS.add("SafeVarargs");
        PUBLIC_IDENTIFIERS.add("SAGetopt");
        PUBLIC_IDENTIFIERS.add("SAGetoptException");
        PUBLIC_IDENTIFIERS.add("SALauncher");
        PUBLIC_IDENTIFIERS.add("SALauncherLoader");
        PUBLIC_IDENTIFIERS.add("SAListener");
        PUBLIC_IDENTIFIERS.add("SALOAD");
        PUBLIC_IDENTIFIERS.add("SaltAndParams");
        PUBLIC_IDENTIFIERS.add("same_frame");
        PUBLIC_IDENTIFIERS.add("same_frame_extended");
        PUBLIC_IDENTIFIERS.add("same_locals_1_stack_item_frame");
        PUBLIC_IDENTIFIERS.add("same_locals_1_stack_item_frame_extended");
        PUBLIC_IDENTIFIERS.add("SampleModel");
        PUBLIC_IDENTIFIERS.add("SAPanel");
        PUBLIC_IDENTIFIERS.add("Sasl");
        PUBLIC_IDENTIFIERS.add("SaslClient");
        PUBLIC_IDENTIFIERS.add("SaslClientFactory");
        PUBLIC_IDENTIFIERS.add("SaslException");
        PUBLIC_IDENTIFIERS.add("SaslInputStream");
        PUBLIC_IDENTIFIERS.add("SaslServer");
        PUBLIC_IDENTIFIERS.add("SaslServerFactory");
        PUBLIC_IDENTIFIERS.add("SASTORE");
        PUBLIC_IDENTIFIERS.add("SaveAction");
        PUBLIC_IDENTIFIERS.add("SaveAsAction");
        PUBLIC_IDENTIFIERS.add("Savepoint");
        PUBLIC_IDENTIFIERS.add("SAX2DOM");
        PUBLIC_IDENTIFIERS.add("SAX2DTM");
        PUBLIC_IDENTIFIERS.add("SAX2DTM2");
        PUBLIC_IDENTIFIERS.add("SAX2RTFDTM");
        PUBLIC_IDENTIFIERS.add("SAX2StAXBaseWriter");
        PUBLIC_IDENTIFIERS.add("SAX2StAXEventWriter");
        PUBLIC_IDENTIFIERS.add("SAX2StAXStreamWriter");
        PUBLIC_IDENTIFIERS.add("SAX2XNI");
        PUBLIC_IDENTIFIERS.add("SAXException");
        PUBLIC_IDENTIFIERS.add("SAXMessageFormatter");
        PUBLIC_IDENTIFIERS.add("SAXNotRecognizedException");
        PUBLIC_IDENTIFIERS.add("SAXNotSupportedException");
        PUBLIC_IDENTIFIERS.add("SAXParseException");
        PUBLIC_IDENTIFIERS.add("SAXParser");
        PUBLIC_IDENTIFIERS.add("SAXParserFactory");
        PUBLIC_IDENTIFIERS.add("SAXParserFactoryImpl");
        PUBLIC_IDENTIFIERS.add("SAXParserImpl");
        PUBLIC_IDENTIFIERS.add("SAXResult");
        PUBLIC_IDENTIFIERS.add("SAXSource");
        PUBLIC_IDENTIFIERS.add("SAXSourceLocator");
        PUBLIC_IDENTIFIERS.add("SAXTransformerFactory");
        PUBLIC_IDENTIFIERS.add("Scale");
        PUBLIC_IDENTIFIERS.add("ScaledBlit");
        PUBLIC_IDENTIFIERS.add("Scan");
        PUBLIC_IDENTIFIERS.add("ScannedEntity");
        PUBLIC_IDENTIFIERS.add("Scanner");
        PUBLIC_IDENTIFIERS.add("ScannerFactory");
        PUBLIC_IDENTIFIERS.add("ScatteringByteChannel");
        PUBLIC_IDENTIFIERS.add("SCDynamicStoreConfig");
        PUBLIC_IDENTIFIERS.add("ScheduledExecutorService");
        PUBLIC_IDENTIFIERS.add("ScheduledFuture");
        PUBLIC_IDENTIFIERS.add("ScheduledThreadPoolExecutor");
        PUBLIC_IDENTIFIERS.add("Schema");
        PUBLIC_IDENTIFIERS.add("SchemaDateTimeException");
        PUBLIC_IDENTIFIERS.add("SchemaDOM");
        PUBLIC_IDENTIFIERS.add("SchemaDOMParser");
        PUBLIC_IDENTIFIERS.add("SchemaDVFactory");
        PUBLIC_IDENTIFIERS.add("SchemaDVFactoryImpl");
        PUBLIC_IDENTIFIERS.add("SchemaFactory");
        PUBLIC_IDENTIFIERS.add("SchemaFactoryLoader");
        PUBLIC_IDENTIFIERS.add("SchemaGrammar");
        PUBLIC_IDENTIFIERS.add("SchemaNamespaceSupport");
        PUBLIC_IDENTIFIERS.add("SchemaParsingConfig");
        PUBLIC_IDENTIFIERS.add("SchemaViolationException");
        PUBLIC_IDENTIFIERS.add("Scope");
        PUBLIC_IDENTIFIERS.add("ScopeDesc");
        PUBLIC_IDENTIFIERS.add("ScopedMemoryAccess");
        PUBLIC_IDENTIFIERS.add("ScopedValueContainer");
        PUBLIC_IDENTIFIERS.add("ScopeListener");
        PUBLIC_IDENTIFIERS.add("ScopeListenerList");
        PUBLIC_IDENTIFIERS.add("ScopeValue");
        PUBLIC_IDENTIFIERS.add("Screen");
        PUBLIC_IDENTIFIERS.add("ScreenSleepListener");
        PUBLIC_IDENTIFIERS.add("Script");
        PUBLIC_IDENTIFIERS.add("ScriptContext");
        PUBLIC_IDENTIFIERS.add("ScriptEngine");
        PUBLIC_IDENTIFIERS.add("ScriptEngineFactory");
        PUBLIC_IDENTIFIERS.add("ScriptEngineManager");
        PUBLIC_IDENTIFIERS.add("ScriptException");
        PUBLIC_IDENTIFIERS.add("Scrollable");
        PUBLIC_IDENTIFIERS.add("ScrollBar");
        PUBLIC_IDENTIFIERS.add("Scrollbar");
        PUBLIC_IDENTIFIERS.add("ScrollbarPeer");
        PUBLIC_IDENTIFIERS.add("ScrollBarUI");
        PUBLIC_IDENTIFIERS.add("ScrollListener");
        PUBLIC_IDENTIFIERS.add("ScrollPane");
        PUBLIC_IDENTIFIERS.add("ScrollPaneAdjustableAccessor");
        PUBLIC_IDENTIFIERS.add("ScrollPaneBorder");
        PUBLIC_IDENTIFIERS.add("ScrollPaneConstants");
        PUBLIC_IDENTIFIERS.add("ScrollPaneLayout");
        PUBLIC_IDENTIFIERS.add("ScrollPanePeer");
        PUBLIC_IDENTIFIERS.add("ScrollPaneUI");
        PUBLIC_IDENTIFIERS.add("ScrollPaneWheelScroller");
        PUBLIC_IDENTIFIERS.add("SctpChannel");
        PUBLIC_IDENTIFIERS.add("SctpChannelImpl");
        PUBLIC_IDENTIFIERS.add("SctpMultiChannel");
        PUBLIC_IDENTIFIERS.add("SctpMultiChannelImpl");
        PUBLIC_IDENTIFIERS.add("SctpNet");
        PUBLIC_IDENTIFIERS.add("SctpServerChannel");
        PUBLIC_IDENTIFIERS.add("SctpServerChannelImpl");
        PUBLIC_IDENTIFIERS.add("SctpSocketOption");
        PUBLIC_IDENTIFIERS.add("SctpStandardSocketOptions");
        PUBLIC_IDENTIFIERS.add("SctpStdSocketOption");
        PUBLIC_IDENTIFIERS.add("SdpProvider");
        PUBLIC_IDENTIFIERS.add("SealedObject");
        PUBLIC_IDENTIFIERS.add("SearchControls");
        PUBLIC_IDENTIFIERS.add("SearchFilter");
        PUBLIC_IDENTIFIERS.add("SearchResult");
        PUBLIC_IDENTIFIERS.add("SearchWriter");
        PUBLIC_IDENTIFIERS.add("SecondaryLoop");
        PUBLIC_IDENTIFIERS.add("SecretKey");
        PUBLIC_IDENTIFIERS.add("SecretKeyFactory");
        PUBLIC_IDENTIFIERS.add("SecretKeyFactorySpi");
        PUBLIC_IDENTIFIERS.add("SecretKeyResolver");
        PUBLIC_IDENTIFIERS.add("SecretKeySpec");
        PUBLIC_IDENTIFIERS.add("SectionElement");
        PUBLIC_IDENTIFIERS.add("SectionFlags");
        PUBLIC_IDENTIFIERS.add("SectionHeader");
        PUBLIC_IDENTIFIERS.add("SecureCacheResponse");
        PUBLIC_IDENTIFIERS.add("SecureClassLoader");
        PUBLIC_IDENTIFIERS.add("SecureDirectoryStream");
        PUBLIC_IDENTIFIERS.add("SecureLookupSupplier");
        PUBLIC_IDENTIFIERS.add("SecureRandom");
        PUBLIC_IDENTIFIERS.add("SecureRandomParameters");
        PUBLIC_IDENTIFIERS.add("SecureRandomSpi");
        PUBLIC_IDENTIFIERS.add("SecurityConfiguration");
        PUBLIC_IDENTIFIERS.add("SecurityException");
        PUBLIC_IDENTIFIERS.add("SecurityManager");
        PUBLIC_IDENTIFIERS.add("SecurityProperties");
        PUBLIC_IDENTIFIERS.add("SecuritySupport");
        PUBLIC_IDENTIFIERS.add("Seed");
        PUBLIC_IDENTIFIERS.add("SeekableByteChannel");
        PUBLIC_IDENTIFIERS.add("SeeTaglet");
        PUBLIC_IDENTIFIERS.add("SeeTree");
        PUBLIC_IDENTIFIERS.add("Segment");
        PUBLIC_IDENTIFIERS.add("SegmentAllocator");
        PUBLIC_IDENTIFIERS.add("SegmentScope");
        PUBLIC_IDENTIFIERS.add("SelChImpl");
        PUBLIC_IDENTIFIERS.add("Select");
        PUBLIC_IDENTIFIERS.add("SelectableChannel");
        PUBLIC_IDENTIFIERS.add("SelectableChannelCloser");
        PUBLIC_IDENTIFIERS.add("SelectedArgument");
        PUBLIC_IDENTIFIERS.add("SelectionKey");
        PUBLIC_IDENTIFIERS.add("SelectionListener");
        PUBLIC_IDENTIFIERS.add("SelectionModelPropertyChangeHandler");
        PUBLIC_IDENTIFIERS.add("Selector");
        PUBLIC_IDENTIFIERS.add("SelectorImpl");
        PUBLIC_IDENTIFIERS.add("SelectorProvider");
        PUBLIC_IDENTIFIERS.add("SelectorProviderImpl");
        PUBLIC_IDENTIFIERS.add("SelfIteratorNoPredicate");
        PUBLIC_IDENTIFIERS.add("Semaphore");
        PUBLIC_IDENTIFIERS.add("SendFailed");
        PUBLIC_IDENTIFIERS.add("SendFailedNotification");
        PUBLIC_IDENTIFIERS.add("Sensor");
        PUBLIC_IDENTIFIERS.add("Separator");
        PUBLIC_IDENTIFIERS.add("SeparatorUI");
        PUBLIC_IDENTIFIERS.add("SeqNumber");
        PUBLIC_IDENTIFIERS.add("Sequence");
        PUBLIC_IDENTIFIERS.add("SequencedEventAccessor");
        PUBLIC_IDENTIFIERS.add("SequenceInputStream");
        PUBLIC_IDENTIFIERS.add("SequenceLayout");
        PUBLIC_IDENTIFIERS.add("Sequencer");
        PUBLIC_IDENTIFIERS.add("Serial");
        PUBLIC_IDENTIFIERS.add("SerialArray");
        PUBLIC_IDENTIFIERS.add("SerialBlob");
        PUBLIC_IDENTIFIERS.add("SerialClob");
        PUBLIC_IDENTIFIERS.add("SerialDatalink");
        PUBLIC_IDENTIFIERS.add("SerialDataTree");
        PUBLIC_IDENTIFIERS.add("SerialException");
        PUBLIC_IDENTIFIERS.add("SerialFieldTree");
        PUBLIC_IDENTIFIERS.add("SerialFieldWriter");
        PUBLIC_IDENTIFIERS.add("SerialHeap");
        PUBLIC_IDENTIFIERS.add("Serializable");
        PUBLIC_IDENTIFIERS.add("SerializableConstant");
        PUBLIC_IDENTIFIERS.add("SerializableLocatorImpl");
        PUBLIC_IDENTIFIERS.add("SerializationHandler");
        PUBLIC_IDENTIFIERS.add("SerializedFormBuilder");
        PUBLIC_IDENTIFIERS.add("SerializedFormWriter");
        PUBLIC_IDENTIFIERS.add("SerializedFormWriterImpl");
        PUBLIC_IDENTIFIERS.add("Serializer");
        PUBLIC_IDENTIFIERS.add("SerializerBase");
        PUBLIC_IDENTIFIERS.add("SerializerFactory");
        PUBLIC_IDENTIFIERS.add("SerializerMessages");
        PUBLIC_IDENTIFIERS.add("SerializerMessages_de");
        PUBLIC_IDENTIFIERS.add("SerializerMessages_ja");
        PUBLIC_IDENTIFIERS.add("SerializerMessages_zh_CN");
        PUBLIC_IDENTIFIERS.add("SerializerTrace");
        PUBLIC_IDENTIFIERS.add("SerialJavaObject");
        PUBLIC_IDENTIFIERS.add("SerialMethodWriter");
        PUBLIC_IDENTIFIERS.add("SerialNumber");
        PUBLIC_IDENTIFIERS.add("SerialRef");
        PUBLIC_IDENTIFIERS.add("SerialStruct");
        PUBLIC_IDENTIFIERS.add("SerialTree");
        PUBLIC_IDENTIFIERS.add("SerialVer");
        PUBLIC_IDENTIFIERS.add("SerialVersionUIDAdder");
        PUBLIC_IDENTIFIERS.add("Server");
        PUBLIC_IDENTIFIERS.add("ServerCloneException");
        PUBLIC_IDENTIFIERS.add("ServerCommunicatorAdmin");
        PUBLIC_IDENTIFIERS.add("ServerError");
        PUBLIC_IDENTIFIERS.add("ServerException");
        PUBLIC_IDENTIFIERS.add("ServerName");
        PUBLIC_IDENTIFIERS.add("ServerNotActiveException");
        PUBLIC_IDENTIFIERS.add("ServerNotifForwarder");
        PUBLIC_IDENTIFIERS.add("ServerProvider");
        PUBLIC_IDENTIFIERS.add("ServerRef");
        PUBLIC_IDENTIFIERS.add("ServerRuntimeException");
        PUBLIC_IDENTIFIERS.add("ServerSocket");
        PUBLIC_IDENTIFIERS.add("ServerSocketChannel");
        PUBLIC_IDENTIFIERS.add("ServerSocketFactory");
        PUBLIC_IDENTIFIERS.add("Service");
        PUBLIC_IDENTIFIERS.add("SERVICE_FORMATTED");
        PUBLIC_IDENTIFIERS.add("ServiceConfigurationError");
        PUBLIC_IDENTIFIERS.add("ServiceDialog");
        PUBLIC_IDENTIFIERS.add("ServiceName");
        PUBLIC_IDENTIFIERS.add("ServiceNotFoundException");
        PUBLIC_IDENTIFIERS.add("ServiceRegistry");
        PUBLIC_IDENTIFIERS.add("ServiceThread");
        PUBLIC_IDENTIFIERS.add("ServiceUI");
        PUBLIC_IDENTIFIERS.add("ServiceUIFactory");
        PUBLIC_IDENTIFIERS.add("ServiceUnavailableException");
        PUBLIC_IDENTIFIERS.add("Set");
        PUBLIC_IDENTIFIERS.add("SetOfIntegerSyntax");
        PUBLIC_IDENTIFIERS.add("SetPosition");
        PUBLIC_IDENTIFIERS.add("SettingControl");
        PUBLIC_IDENTIFIERS.add("SettingDefinition");
        PUBLIC_IDENTIFIERS.add("SettingsFrame");
        PUBLIC_IDENTIFIERS.add("Severity");
        PUBLIC_IDENTIFIERS.add("SF2Region");
        PUBLIC_IDENTIFIERS.add("SHA256");
        PUBLIC_IDENTIFIERS.add("Shape");
        PUBLIC_IDENTIFIERS.add("ShapeDrawPipe");
        PUBLIC_IDENTIFIERS.add("ShardingKey");
        PUBLIC_IDENTIFIERS.add("ShardingKeyBuilder");
        PUBLIC_IDENTIFIERS.add("SharedHotSpotSpeculationLog");
        PUBLIC_IDENTIFIERS.add("SharedNameTable");
        PUBLIC_IDENTIFIERS.add("SharedSecrets");
        PUBLIC_IDENTIFIERS.add("SharedThreadContainer");
        PUBLIC_IDENTIFIERS.add("ShellFolder");
        PUBLIC_IDENTIFIERS.add("ShellFolderColumnInfo");
        PUBLIC_IDENTIFIERS.add("ShenandoahBitMap");
        PUBLIC_IDENTIFIERS.add("ShenandoahHeap");
        PUBLIC_IDENTIFIERS.add("ShenandoahHeapRegion");
        PUBLIC_IDENTIFIERS.add("Short");
        PUBLIC_IDENTIFIERS.add("ShortBandedRaster");
        PUBLIC_IDENTIFIERS.add("ShortBuffer");
        PUBLIC_IDENTIFIERS.add("ShortBufferException");
        PUBLIC_IDENTIFIERS.add("ShortComponentRaster");
        PUBLIC_IDENTIFIERS.add("ShortData");
        PUBLIC_IDENTIFIERS.add("ShortEditor");
        PUBLIC_IDENTIFIERS.add("ShortField");
        PUBLIC_IDENTIFIERS.add("ShortInterleavedRaster");
        PUBLIC_IDENTIFIERS.add("ShortList");
        PUBLIC_IDENTIFIERS.add("ShortLookupTable");
        PUBLIC_IDENTIFIERS.add("ShortMessage");
        PUBLIC_IDENTIFIERS.add("ShortSignature");
        PUBLIC_IDENTIFIERS.add("ShortType");
        PUBLIC_IDENTIFIERS.add("ShortTypeImpl");
        PUBLIC_IDENTIFIERS.add("ShortValue");
        PUBLIC_IDENTIFIERS.add("ShortValueImpl");
        PUBLIC_IDENTIFIERS.add("ShortVector");
        PUBLIC_IDENTIFIERS.add("ShowAction");
        PUBLIC_IDENTIFIERS.add("ShuffleIotaOperation");
        PUBLIC_IDENTIFIERS.add("ShuffleToVectorOperation");
        PUBLIC_IDENTIFIERS.add("Shutdown");
        PUBLIC_IDENTIFIERS.add("ShutdownChannelGroupException");
        PUBLIC_IDENTIFIERS.add("ShutdownNotification");
        PUBLIC_IDENTIFIERS.add("SignalHandler");
        PUBLIC_IDENTIFIERS.add("Signature");
        PUBLIC_IDENTIFIERS.add("Signature11ElementProxy");
        PUBLIC_IDENTIFIERS.add("Signature_attribute");
        PUBLIC_IDENTIFIERS.add("SignatureAlgorithm");
        PUBLIC_IDENTIFIERS.add("SignatureAlgorithmSpi");
        PUBLIC_IDENTIFIERS.add("SignatureBaseRSA");
        PUBLIC_IDENTIFIERS.add("SignatureConverter");
        PUBLIC_IDENTIFIERS.add("SignatureDSA");
        PUBLIC_IDENTIFIERS.add("SignatureECDSA");
        PUBLIC_IDENTIFIERS.add("SignatureECDSARIPEMD160");
        PUBLIC_IDENTIFIERS.add("SignatureECDSASHA1");
        PUBLIC_IDENTIFIERS.add("SignatureECDSASHA224");
        PUBLIC_IDENTIFIERS.add("SignatureECDSASHA256");
        PUBLIC_IDENTIFIERS.add("SignatureECDSASHA384");
        PUBLIC_IDENTIFIERS.add("SignatureECDSASHA512");
        PUBLIC_IDENTIFIERS.add("SignatureElementProxy");
        PUBLIC_IDENTIFIERS.add("SignatureException");
        PUBLIC_IDENTIFIERS.add("SignatureFileVerifier");
        PUBLIC_IDENTIFIERS.add("SignatureGenerator");
        PUBLIC_IDENTIFIERS.add("SignatureInfo");
        PUBLIC_IDENTIFIERS.add("SignatureIterator");
        PUBLIC_IDENTIFIERS.add("SignatureMethod");
        PUBLIC_IDENTIFIERS.add("SignatureMethodParameterSpec");
        PUBLIC_IDENTIFIERS.add("SignatureParser");
        PUBLIC_IDENTIFIERS.add("SignatureProperties");
        PUBLIC_IDENTIFIERS.add("SignatureProperty");
        PUBLIC_IDENTIFIERS.add("SignatureReader");
        PUBLIC_IDENTIFIERS.add("SignatureRemapper");
        PUBLIC_IDENTIFIERS.add("SignatureRSAMD5");
        PUBLIC_IDENTIFIERS.add("SignatureRSARIPEMD160");
        PUBLIC_IDENTIFIERS.add("SignatureRSASHA1");
        PUBLIC_IDENTIFIERS.add("SignatureRSASHA1MGF1");
        PUBLIC_IDENTIFIERS.add("SignatureRSASHA224");
        PUBLIC_IDENTIFIERS.add("SignatureRSASHA224MGF1");
        PUBLIC_IDENTIFIERS.add("SignatureRSASHA256");
        PUBLIC_IDENTIFIERS.add("SignatureRSASHA256MGF1");
        PUBLIC_IDENTIFIERS.add("SignatureRSASHA384");
        PUBLIC_IDENTIFIERS.add("SignatureRSASHA384MGF1");
        PUBLIC_IDENTIFIERS.add("SignatureRSASHA3_224MGF1");
        PUBLIC_IDENTIFIERS.add("SignatureRSASHA3_256MGF1");
        PUBLIC_IDENTIFIERS.add("SignatureRSASHA3_384MGF1");
        PUBLIC_IDENTIFIERS.add("SignatureRSASHA3_512MGF1");
        PUBLIC_IDENTIFIERS.add("SignatureRSASHA512");
        PUBLIC_IDENTIFIERS.add("SignatureRSASHA512MGF1");
        PUBLIC_IDENTIFIERS.add("SignatureRSASSAPSS");
        PUBLIC_IDENTIFIERS.add("Signatures");
        PUBLIC_IDENTIFIERS.add("SignatureSpi");
        PUBLIC_IDENTIFIERS.add("SignatureUtil");
        PUBLIC_IDENTIFIERS.add("SignatureValue");
        PUBLIC_IDENTIFIERS.add("SignatureVisitor");
        PUBLIC_IDENTIFIERS.add("SignatureWriter");
        PUBLIC_IDENTIFIERS.add("SignedInfo");
        PUBLIC_IDENTIFIERS.add("Signer");
        PUBLIC_IDENTIFIERS.add("SignerInfo");
        PUBLIC_IDENTIFIERS.add("SignerOutputStream");
        PUBLIC_IDENTIFIERS.add("Simple");
        PUBLIC_IDENTIFIERS.add("SimpleAnnotationValueVisitor14");
        PUBLIC_IDENTIFIERS.add("SimpleAnnotationValueVisitor6");
        PUBLIC_IDENTIFIERS.add("SimpleAnnotationValueVisitor7");
        PUBLIC_IDENTIFIERS.add("SimpleAnnotationValueVisitor8");
        PUBLIC_IDENTIFIERS.add("SimpleAnnotationValueVisitor9");
        PUBLIC_IDENTIFIERS.add("SimpleAsynchronousFileChannelImpl");
        PUBLIC_IDENTIFIERS.add("SimpleAttributeSet");
        PUBLIC_IDENTIFIERS.add("SimpleBeanInfo");
        PUBLIC_IDENTIFIERS.add("SimpleBindings");
        PUBLIC_IDENTIFIERS.add("SimpleClassTypeSignature");
        PUBLIC_IDENTIFIERS.add("SimpleConfiguration");
        PUBLIC_IDENTIFIERS.add("SimpleConsoleLogger");
        PUBLIC_IDENTIFIERS.add("SimpleContentModel");
        PUBLIC_IDENTIFIERS.add("SimpleDateFormat");
        PUBLIC_IDENTIFIERS.add("SimpleDiagnosticPosition");
        PUBLIC_IDENTIFIERS.add("SimpleDocletException");
        PUBLIC_IDENTIFIERS.add("SimpleDocTreeVisitor");
        PUBLIC_IDENTIFIERS.add("SimpleElementValue");
        PUBLIC_IDENTIFIERS.add("SimpleElementValueGen");
        PUBLIC_IDENTIFIERS.add("SimpleElementVisitor14");
        PUBLIC_IDENTIFIERS.add("SimpleElementVisitor6");
        PUBLIC_IDENTIFIERS.add("SimpleElementVisitor7");
        PUBLIC_IDENTIFIERS.add("SimpleElementVisitor8");
        PUBLIC_IDENTIFIERS.add("SimpleElementVisitor9");
        PUBLIC_IDENTIFIERS.add("SimpleEndPosTable");
        PUBLIC_IDENTIFIERS.add("SimpleEntry");
        PUBLIC_IDENTIFIERS.add("SimpleEUCEncoder");
        PUBLIC_IDENTIFIERS.add("SimpleFileVisitor");
        PUBLIC_IDENTIFIERS.add("SimpleFormatter");
        PUBLIC_IDENTIFIERS.add("SimpleImmutableEntry");
        PUBLIC_IDENTIFIERS.add("SimpleInputMethodWindow");
        PUBLIC_IDENTIFIERS.add("SimpleInstrument");
        PUBLIC_IDENTIFIERS.add("SimpleJavaFileObject");
        PUBLIC_IDENTIFIERS.add("SimpleLinkRequest");
        PUBLIC_IDENTIFIERS.add("SimpleLocator");
        PUBLIC_IDENTIFIERS.add("SimpleOptionNameMap");
        PUBLIC_IDENTIFIERS.add("SimpleRelinkableCallSite");
        PUBLIC_IDENTIFIERS.add("SimpleRemapper");
        PUBLIC_IDENTIFIERS.add("SimpleRenderedImage");
        PUBLIC_IDENTIFIERS.add("SimpleResultTreeImpl");
        PUBLIC_IDENTIFIERS.add("SimpleScriptContext");
        PUBLIC_IDENTIFIERS.add("SimpleSoundbank");
        PUBLIC_IDENTIFIERS.add("SimpleTaglet");
        PUBLIC_IDENTIFIERS.add("SimpleTimeZone");
        PUBLIC_IDENTIFIERS.add("SimpleTreeGroupNode");
        PUBLIC_IDENTIFIERS.add("SimpleTreeModel");
        PUBLIC_IDENTIFIERS.add("SimpleTreeNode");
        PUBLIC_IDENTIFIERS.add("SimpleTreeTableModel");
        PUBLIC_IDENTIFIERS.add("SimpleTreeVisitor");
        PUBLIC_IDENTIFIERS.add("SimpleType");
        PUBLIC_IDENTIFIERS.add("SimpleTypeVisitor14");
        PUBLIC_IDENTIFIERS.add("SimpleTypeVisitor6");
        PUBLIC_IDENTIFIERS.add("SimpleTypeVisitor7");
        PUBLIC_IDENTIFIERS.add("SimpleTypeVisitor8");
        PUBLIC_IDENTIFIERS.add("SimpleTypeVisitor9");
        PUBLIC_IDENTIFIERS.add("SimpleVerifier");
        PUBLIC_IDENTIFIERS.add("SimpleVisitor");
        PUBLIC_IDENTIFIERS.add("SinceTree");
        PUBLIC_IDENTIFIERS.add("SingleByte");
        PUBLIC_IDENTIFIERS.add("SingleCertificateResolver");
        PUBLIC_IDENTIFIERS.add("SingleClickListener");
        PUBLIC_IDENTIFIERS.add("SingleEntryRegistry");
        PUBLIC_IDENTIFIERS.add("SingleKeyResolver");
        PUBLIC_IDENTIFIERS.add("SingleNodeCounter");
        PUBLIC_IDENTIFIERS.add("SinglePixelPackedSampleModel");
        PUBLIC_IDENTIFIERS.add("SingleSelectionModel");
        PUBLIC_IDENTIFIERS.add("SingleTileRenderedImage");
        PUBLIC_IDENTIFIERS.add("SingletonBlob");
        PUBLIC_IDENTIFIERS.add("SingletonIterator");
        PUBLIC_IDENTIFIERS.add("SinkChannel");
        PUBLIC_IDENTIFIERS.add("SIPUSH");
        PUBLIC_IDENTIFIERS.add("Site");
        PUBLIC_IDENTIFIERS.add("Size");
        PUBLIC_IDENTIFIERS.add("Size2DSyntax");
        PUBLIC_IDENTIFIERS.add("SizeAction");
        PUBLIC_IDENTIFIERS.add("SizeLimitExceededException");
        PUBLIC_IDENTIFIERS.add("SizeRequirements");
        PUBLIC_IDENTIFIERS.add("SizeSequence");
        PUBLIC_IDENTIFIERS.add("SJIS");
        PUBLIC_IDENTIFIERS.add("SJIS_0213");
        PUBLIC_IDENTIFIERS.add("Skeleton");
        PUBLIC_IDENTIFIERS.add("SkeletonMismatchException");
        PUBLIC_IDENTIFIERS.add("SkeletonNotFoundException");
        PUBLIC_IDENTIFIERS.add("Skip");
        PUBLIC_IDENTIFIERS.add("SliderUI");
        PUBLIC_IDENTIFIERS.add("Small");
        PUBLIC_IDENTIFIERS.add("SmallAttributeSet");
        PUBLIC_IDENTIFIERS.add("SmallValue");
        PUBLIC_IDENTIFIERS.add("SmtpClient");
        PUBLIC_IDENTIFIERS.add("SmtpProtocolException");
        PUBLIC_IDENTIFIERS.add("Snapshot");
        PUBLIC_IDENTIFIERS.add("SnapshotMBeanServerConnection");
        PUBLIC_IDENTIFIERS.add("SNIMatcher");
        PUBLIC_IDENTIFIERS.add("Snippet");
        PUBLIC_IDENTIFIERS.add("SnippetEvent");
        PUBLIC_IDENTIFIERS.add("Snippets");
        PUBLIC_IDENTIFIERS.add("SnippetTaglet");
        PUBLIC_IDENTIFIERS.add("SnippetTree");
        PUBLIC_IDENTIFIERS.add("SnippetWrapper");
        PUBLIC_IDENTIFIERS.add("SNIServerName");
        PUBLIC_IDENTIFIERS.add("Socket");
        PUBLIC_IDENTIFIERS.add("SocketAddress");
        PUBLIC_IDENTIFIERS.add("SocketAttachingConnector");
        PUBLIC_IDENTIFIERS.add("SocketChannel");
        PUBLIC_IDENTIFIERS.add("SocketException");
        PUBLIC_IDENTIFIERS.add("SocketFactory");
        PUBLIC_IDENTIFIERS.add("SocketHandler");
        PUBLIC_IDENTIFIERS.add("SocketImpl");
        PUBLIC_IDENTIFIERS.add("SocketImplFactory");
        PUBLIC_IDENTIFIERS.add("SocketListeningConnector");
        PUBLIC_IDENTIFIERS.add("SocketOption");
        PUBLIC_IDENTIFIERS.add("SocketOptions");
        PUBLIC_IDENTIFIERS.add("Sockets");
        PUBLIC_IDENTIFIERS.add("SocketSecurityException");
        PUBLIC_IDENTIFIERS.add("SocketTimeoutException");
        PUBLIC_IDENTIFIERS.add("SocketTransportService");
        PUBLIC_IDENTIFIERS.add("SoftAbstractResampler");
        PUBLIC_IDENTIFIERS.add("SoftAudioProcessor");
        PUBLIC_IDENTIFIERS.add("SoftBevelBorder");
        PUBLIC_IDENTIFIERS.add("SoftCache");
        PUBLIC_IDENTIFIERS.add("SoftControl");
        PUBLIC_IDENTIFIERS.add("SoftMixingDataLine");
        PUBLIC_IDENTIFIERS.add("SoftProcess");
        PUBLIC_IDENTIFIERS.add("SoftReference");
        PUBLIC_IDENTIFIERS.add("SoftResampler");
        PUBLIC_IDENTIFIERS.add("SoftResamplerStreamer");
        PUBLIC_IDENTIFIERS.add("SolidTextRenderer");
        PUBLIC_IDENTIFIERS.add("SortableTableModel");
        PUBLIC_IDENTIFIERS.add("SortArrowIcon");
        PUBLIC_IDENTIFIERS.add("SortedMap");
        PUBLIC_IDENTIFIERS.add("SortedSet");
        PUBLIC_IDENTIFIERS.add("SortFiles");
        PUBLIC_IDENTIFIERS.add("SortHeaderCellRenderer");
        PUBLIC_IDENTIFIERS.add("SortHeaderMouseAdapter");
        PUBLIC_IDENTIFIERS.add("SortingFocusTraversalPolicy");
        PUBLIC_IDENTIFIERS.add("SortKey");
        PUBLIC_IDENTIFIERS.add("Soundbank");
        PUBLIC_IDENTIFIERS.add("SoundbankReader");
        PUBLIC_IDENTIFIERS.add("SoundbankResource");
        PUBLIC_IDENTIFIERS.add("Source");
        PUBLIC_IDENTIFIERS.add("SourceChannel");
        PUBLIC_IDENTIFIERS.add("SourceCodeAnalysis");
        PUBLIC_IDENTIFIERS.add("SourceCodePanel");
        PUBLIC_IDENTIFIERS.add("SourceDataLine");
        PUBLIC_IDENTIFIERS.add("SourceDebugExtension_attribute");
        PUBLIC_IDENTIFIERS.add("SourceFile_attribute");
        PUBLIC_IDENTIFIERS.add("SourceID_attribute");
        PUBLIC_IDENTIFIERS.add("SourceInterpreter");
        PUBLIC_IDENTIFIERS.add("SourceLoader");
        PUBLIC_IDENTIFIERS.add("SourceLocator");
        PUBLIC_IDENTIFIERS.add("SourcePositions");
        PUBLIC_IDENTIFIERS.add("SourceToHTMLConverter");
        PUBLIC_IDENTIFIERS.add("SourceTree");
        PUBLIC_IDENTIFIERS.add("SourceValue");
        PUBLIC_IDENTIFIERS.add("SourceWriter");
        PUBLIC_IDENTIFIERS.add("Space");
        PUBLIC_IDENTIFIERS.add("SpaceClosure");
        PUBLIC_IDENTIFIERS.add("SpanClipRenderer");
        PUBLIC_IDENTIFIERS.add("SpanIterator");
        PUBLIC_IDENTIFIERS.add("Spans");
        PUBLIC_IDENTIFIERS.add("SpanShapeRenderer");
        PUBLIC_IDENTIFIERS.add("SpecialAction");
        PUBLIC_IDENTIFIERS.add("SpeciesData");
        PUBLIC_IDENTIFIERS.add("SpecTaglet");
        PUBLIC_IDENTIFIERS.add("SpecTree");
        PUBLIC_IDENTIFIERS.add("Speculation");
        PUBLIC_IDENTIFIERS.add("SpeculationLog");
        PUBLIC_IDENTIFIERS.add("SpeculationReason");
        PUBLIC_IDENTIFIERS.add("SpeculationReasonEncoding");
        PUBLIC_IDENTIFIERS.add("SpeculativeTrapData");
        PUBLIC_IDENTIFIERS.add("SPILocaleProviderAdapter");
        PUBLIC_IDENTIFIERS.add("SpinnerDateModel");
        PUBLIC_IDENTIFIERS.add("SpinnerListModel");
        PUBLIC_IDENTIFIERS.add("SpinnerModel");
        PUBLIC_IDENTIFIERS.add("SpinnerNumberModel");
        PUBLIC_IDENTIFIERS.add("SpinnerUI");
        PUBLIC_IDENTIFIERS.add("SPIResolutionException");
        PUBLIC_IDENTIFIERS.add("SPKIData");
        PUBLIC_IDENTIFIERS.add("SplashScreen");
        PUBLIC_IDENTIFIERS.add("Spliterator");
        PUBLIC_IDENTIFIERS.add("SplitPaneBorder");
        PUBLIC_IDENTIFIERS.add("SplitPaneUI");
        PUBLIC_IDENTIFIERS.add("SplittableGenerator");
        PUBLIC_IDENTIFIERS.add("SpNegoContext");
        PUBLIC_IDENTIFIERS.add("SpNegoCredElement");
        PUBLIC_IDENTIFIERS.add("Spring");
        PUBLIC_IDENTIFIERS.add("SpringLayout");
        PUBLIC_IDENTIFIERS.add("SQLClientInfoException");
        PUBLIC_IDENTIFIERS.add("SQLData");
        PUBLIC_IDENTIFIERS.add("SQLDataException");
        PUBLIC_IDENTIFIERS.add("SQLException");
        PUBLIC_IDENTIFIERS.add("SQLFeatureNotSupportedException");
        PUBLIC_IDENTIFIERS.add("SQLInput");
        PUBLIC_IDENTIFIERS.add("SQLInputImpl");
        PUBLIC_IDENTIFIERS.add("SQLIntegrityConstraintViolationException");
        PUBLIC_IDENTIFIERS.add("SQLInvalidAuthorizationSpecException");
        PUBLIC_IDENTIFIERS.add("SQLNonTransientConnectionException");
        PUBLIC_IDENTIFIERS.add("SQLNonTransientException");
        PUBLIC_IDENTIFIERS.add("SQLOutput");
        PUBLIC_IDENTIFIERS.add("SQLOutputImpl");
        PUBLIC_IDENTIFIERS.add("SQLRecoverableException");
        PUBLIC_IDENTIFIERS.add("SQLSyntaxErrorException");
        PUBLIC_IDENTIFIERS.add("SQLTimeoutException");
        PUBLIC_IDENTIFIERS.add("SQLTransactionRollbackException");
        PUBLIC_IDENTIFIERS.add("SQLTransientConnectionException");
        PUBLIC_IDENTIFIERS.add("SQLTransientException");
        PUBLIC_IDENTIFIERS.add("SQLType");
        PUBLIC_IDENTIFIERS.add("SQLWarning");
        PUBLIC_IDENTIFIERS.add("SQLXML");
        PUBLIC_IDENTIFIERS.add("SSLContext");
        PUBLIC_IDENTIFIERS.add("SSLContextImpl");
        PUBLIC_IDENTIFIERS.add("SSLContextSpi");
        PUBLIC_IDENTIFIERS.add("SSLEngine");
        PUBLIC_IDENTIFIERS.add("SSLEngineResult");
        PUBLIC_IDENTIFIERS.add("SSLException");
        PUBLIC_IDENTIFIERS.add("SSLFlowDelegate");
        PUBLIC_IDENTIFIERS.add("SSLHandshakeException");
        PUBLIC_IDENTIFIERS.add("SSLKeyException");
        PUBLIC_IDENTIFIERS.add("SSLParameters");
        PUBLIC_IDENTIFIERS.add("SSLPeerUnverifiedException");
        PUBLIC_IDENTIFIERS.add("SSLProtocolException");
        PUBLIC_IDENTIFIERS.add("SslRMIClientSocketFactory");
        PUBLIC_IDENTIFIERS.add("SslRMIServerSocketFactory");
        PUBLIC_IDENTIFIERS.add("SSLServerSocket");
        PUBLIC_IDENTIFIERS.add("SSLServerSocketFactory");
        PUBLIC_IDENTIFIERS.add("SSLSession");
        PUBLIC_IDENTIFIERS.add("SSLSessionBindingEvent");
        PUBLIC_IDENTIFIERS.add("SSLSessionBindingListener");
        PUBLIC_IDENTIFIERS.add("SSLSessionContext");
        PUBLIC_IDENTIFIERS.add("SSLSocket");
        PUBLIC_IDENTIFIERS.add("SSLSocketFactory");
        PUBLIC_IDENTIFIERS.add("SSLTube");
        PUBLIC_IDENTIFIERS.add("Stable");
        PUBLIC_IDENTIFIERS.add("Stack");
        PUBLIC_IDENTIFIERS.add("stack_map_frame");
        PUBLIC_IDENTIFIERS.add("StackableScope");
        PUBLIC_IDENTIFIERS.add("StackConsumer");
        PUBLIC_IDENTIFIERS.add("StackFrame");
        PUBLIC_IDENTIFIERS.add("StackFrameImpl");
        PUBLIC_IDENTIFIERS.add("StackFrameInfo");
        PUBLIC_IDENTIFIERS.add("StackFrameStream");
        PUBLIC_IDENTIFIERS.add("StackInstruction");
        PUBLIC_IDENTIFIERS.add("StackIntrospection");
        PUBLIC_IDENTIFIERS.add("StackMap_attribute");
        PUBLIC_IDENTIFIERS.add("StackMapFormat");
        PUBLIC_IDENTIFIERS.add("StackMapTable_attribute");
        PUBLIC_IDENTIFIERS.add("StackMapTableWriter");
        PUBLIC_IDENTIFIERS.add("StackMapWriter");
        PUBLIC_IDENTIFIERS.add("StackOverflowError");
        PUBLIC_IDENTIFIERS.add("StackProducer");
        PUBLIC_IDENTIFIERS.add("StackTrace");
        PUBLIC_IDENTIFIERS.add("StackTraceElement");
        PUBLIC_IDENTIFIERS.add("StackTraceElementCompositeData");
        PUBLIC_IDENTIFIERS.add("StackValue");
        PUBLIC_IDENTIFIERS.add("StackValueCollection");
        PUBLIC_IDENTIFIERS.add("StackWalker");
        PUBLIC_IDENTIFIERS.add("StackWalker.Option");
        PUBLIC_IDENTIFIERS.add("StackWalker.StackFrame");
        PUBLIC_IDENTIFIERS.add("StampedLock");
        PUBLIC_IDENTIFIERS.add("StandardCharsets");
        PUBLIC_IDENTIFIERS.add("StandardDoclet");
        PUBLIC_IDENTIFIERS.add("StandardEmitterMBean");
        PUBLIC_IDENTIFIERS.add("StandardGlyphVector");
        PUBLIC_IDENTIFIERS.add("StandardJavaFileManager");
        PUBLIC_IDENTIFIERS.add("StandardMBean");
        PUBLIC_IDENTIFIERS.add("StandardMBeanSupport");
        PUBLIC_IDENTIFIERS.add("StandardMetadataFormat");
        PUBLIC_IDENTIFIERS.add("StandardMetadataFormatResources");
        PUBLIC_IDENTIFIERS.add("StandardParserConfiguration");
        PUBLIC_IDENTIFIERS.add("StarImportScope");
        PUBLIC_IDENTIFIERS.add("Start");
        PUBLIC_IDENTIFIERS.add("StartDocument");
        PUBLIC_IDENTIFIERS.add("StartDocumentEvent");
        PUBLIC_IDENTIFIERS.add("StartElement");
        PUBLIC_IDENTIFIERS.add("StartElementEvent");
        PUBLIC_IDENTIFIERS.add("StartElementTree");
        PUBLIC_IDENTIFIERS.add("StartTlsRequest");
        PUBLIC_IDENTIFIERS.add("StartTlsResponse");
        PUBLIC_IDENTIFIERS.add("State");
        PUBLIC_IDENTIFIERS.add("StateChangeAction");
        PUBLIC_IDENTIFIERS.add("StateEdit");
        PUBLIC_IDENTIFIERS.add("StateEditable");
        PUBLIC_IDENTIFIERS.add("StateFactory");
        PUBLIC_IDENTIFIERS.add("StateInfo");
        PUBLIC_IDENTIFIERS.add("Statement");
        PUBLIC_IDENTIFIERS.add("StatementEvent");
        PUBLIC_IDENTIFIERS.add("StatementEventListener");
        PUBLIC_IDENTIFIERS.add("StatementSnippet");
        PUBLIC_IDENTIFIERS.add("StatementTree");
        PUBLIC_IDENTIFIERS.add("StateTrackable");
        PUBLIC_IDENTIFIERS.add("StateTracker");
        PUBLIC_IDENTIFIERS.add("StaticBaseConstructor");
        PUBLIC_IDENTIFIERS.add("StaticField");
        PUBLIC_IDENTIFIERS.add("StaticFieldExpr");
        PUBLIC_IDENTIFIERS.add("StaticInitMerger");
        PUBLIC_IDENTIFIERS.add("StatLong");
        PUBLIC_IDENTIFIERS.add("StatRecord");
        PUBLIC_IDENTIFIERS.add("Status");
        PUBLIC_IDENTIFIERS.add("StatusBar");
        PUBLIC_IDENTIFIERS.add("StaxEntityResolverWrapper");
        PUBLIC_IDENTIFIERS.add("StaxErrorReporter");
        PUBLIC_IDENTIFIERS.add("StAXEvent2SAX");
        PUBLIC_IDENTIFIERS.add("StAXResult");
        PUBLIC_IDENTIFIERS.add("StAXSource");
        PUBLIC_IDENTIFIERS.add("StAXStream2SAX");
        PUBLIC_IDENTIFIERS.add("StaxXMLInputSource");
        PUBLIC_IDENTIFIERS.add("Step");
        PUBLIC_IDENTIFIERS.add("StepEvent");
        PUBLIC_IDENTIFIERS.add("StepIterator");
        PUBLIC_IDENTIFIERS.add("StepPattern");
        PUBLIC_IDENTIFIERS.add("StepRequest");
        PUBLIC_IDENTIFIERS.add("stillInsideTimerAction");
        PUBLIC_IDENTIFIERS.add("StopParseException");
        PUBLIC_IDENTIFIERS.add("StoppedException");
        PUBLIC_IDENTIFIERS.add("StorageResolver");
        PUBLIC_IDENTIFIERS.add("StorageResolverException");
        PUBLIC_IDENTIFIERS.add("StorageResolverSpi");
        PUBLIC_IDENTIFIERS.add("StorageType");
        PUBLIC_IDENTIFIERS.add("StoreInstruction");
        PUBLIC_IDENTIFIERS.add("StoreVectorMaskedOperation");
        PUBLIC_IDENTIFIERS.add("StoreVectorOperation");
        PUBLIC_IDENTIFIERS.add("StoreVectorOperationWithMap");
        PUBLIC_IDENTIFIERS.add("Strategy");
        PUBLIC_IDENTIFIERS.add("Stream");
        PUBLIC_IDENTIFIERS.add("StreamableGenerator");
        PUBLIC_IDENTIFIERS.add("StreamCloser");
        PUBLIC_IDENTIFIERS.add("StreamCorruptedException");
        PUBLIC_IDENTIFIERS.add("StreamDecoder");
        PUBLIC_IDENTIFIERS.add("StreamFilter");
        PUBLIC_IDENTIFIERS.add("StreamFinalizer");
        PUBLIC_IDENTIFIERS.add("StreamHandler");
        PUBLIC_IDENTIFIERS.add("StreamingExecutionControl");
        PUBLIC_IDENTIFIERS.add("StreamIterator");
        PUBLIC_IDENTIFIERS.add("StreamMonitor");
        PUBLIC_IDENTIFIERS.add("StreamPrintService");
        PUBLIC_IDENTIFIERS.add("StreamPrintServiceFactory");
        PUBLIC_IDENTIFIERS.add("StreamReaderDelegate");
        PUBLIC_IDENTIFIERS.add("StreamRemoteCall");
        PUBLIC_IDENTIFIERS.add("StreamResult");
        PUBLIC_IDENTIFIERS.add("Streams");
        PUBLIC_IDENTIFIERS.add("StreamSource");
        PUBLIC_IDENTIFIERS.add("StreamTokenizer");
        PUBLIC_IDENTIFIERS.add("StrictMath");
        PUBLIC_IDENTIFIERS.add("STRING");
        PUBLIC_IDENTIFIERS.add("String");
        PUBLIC_IDENTIFIERS.add("StringArgument");
        PUBLIC_IDENTIFIERS.add("StringArrayWrapper");
        PUBLIC_IDENTIFIERS.add("StringBuffer");
        PUBLIC_IDENTIFIERS.add("StringBufferInputStream");
        PUBLIC_IDENTIFIERS.add("StringBufferPool");
        PUBLIC_IDENTIFIERS.add("StringBuilder");
        PUBLIC_IDENTIFIERS.add("StringComparable");
        PUBLIC_IDENTIFIERS.add("StringConcat");
        PUBLIC_IDENTIFIERS.add("StringConcatException");
        PUBLIC_IDENTIFIERS.add("StringCounter");
        PUBLIC_IDENTIFIERS.add("StringDatatypeValidator");
        PUBLIC_IDENTIFIERS.add("StringDV");
        PUBLIC_IDENTIFIERS.add("StringEditor");
        PUBLIC_IDENTIFIERS.add("StringElementHandler");
        PUBLIC_IDENTIFIERS.add("StringHeadTail");
        PUBLIC_IDENTIFIERS.add("StringIndexOutOfBoundsException");
        PUBLIC_IDENTIFIERS.add("StringIterator");
        PUBLIC_IDENTIFIERS.add("StringList");
        PUBLIC_IDENTIFIERS.add("StringMonitor");
        PUBLIC_IDENTIFIERS.add("StringMonitorMBean");
        PUBLIC_IDENTIFIERS.add("StringOverflow");
        PUBLIC_IDENTIFIERS.add("StringPublisher");
        PUBLIC_IDENTIFIERS.add("StringReader");
        PUBLIC_IDENTIFIERS.add("StringRefAddr");
        PUBLIC_IDENTIFIERS.add("StringReference");
        PUBLIC_IDENTIFIERS.add("StringReferenceImpl");
        PUBLIC_IDENTIFIERS.add("StringsCompleter");
        PUBLIC_IDENTIFIERS.add("StringSelection");
        PUBLIC_IDENTIFIERS.add("StringSharingDecompressor");
        PUBLIC_IDENTIFIERS.add("StringSharingDecompressorFactory");
        PUBLIC_IDENTIFIERS.add("StringSharingPlugin");
        PUBLIC_IDENTIFIERS.add("StringsProvider");
        PUBLIC_IDENTIFIERS.add("StringTable");
        PUBLIC_IDENTIFIERS.add("StringToIntTable");
        PUBLIC_IDENTIFIERS.add("StringTokenIterator");
        PUBLIC_IDENTIFIERS.add("StringTokenizer");
        PUBLIC_IDENTIFIERS.add("StringToStringTable");
        PUBLIC_IDENTIFIERS.add("StringToStringTableVector");
        PUBLIC_IDENTIFIERS.add("StringType");
        PUBLIC_IDENTIFIERS.add("StringUIClientPropertyKey");
        PUBLIC_IDENTIFIERS.add("StringUtils");
        PUBLIC_IDENTIFIERS.add("StringValueExp");
        PUBLIC_IDENTIFIERS.add("StringVector");
        PUBLIC_IDENTIFIERS.add("StringWriter");
        PUBLIC_IDENTIFIERS.add("StripFilter");
        PUBLIC_IDENTIFIERS.add("StripWhitespaceFilter");
        PUBLIC_IDENTIFIERS.add("Stroke");
        PUBLIC_IDENTIFIERS.add("StrokeBorder");
        PUBLIC_IDENTIFIERS.add("Struct");
        PUBLIC_IDENTIFIERS.add("StructLayout");
        PUBLIC_IDENTIFIERS.add("StructuralTypeMapping");
        PUBLIC_IDENTIFIERS.add("StructuredTaskScope");
        PUBLIC_IDENTIFIERS.add("StructureViolationExceptions");
        PUBLIC_IDENTIFIERS.add("Stub");
        PUBLIC_IDENTIFIERS.add("StubNotFoundException");
        PUBLIC_IDENTIFIERS.add("StubQueue");
        PUBLIC_IDENTIFIERS.add("StubRoutines");
        PUBLIC_IDENTIFIERS.add("Style");
        PUBLIC_IDENTIFIERS.add("StyleAssociation");
        PUBLIC_IDENTIFIERS.add("StyleConstants");
        PUBLIC_IDENTIFIERS.add("StyleContext");
        PUBLIC_IDENTIFIERS.add("StyledDocument");
        PUBLIC_IDENTIFIERS.add("StyledEditorKit");
        PUBLIC_IDENTIFIERS.add("StyledText");
        PUBLIC_IDENTIFIERS.add("StyledTextAction");
        PUBLIC_IDENTIFIERS.add("StyleResolver");
        PUBLIC_IDENTIFIERS.add("StyleSheet");
        PUBLIC_IDENTIFIERS.add("StyleSheetList");
        PUBLIC_IDENTIFIERS.add("StylesheetPIHandler");
        PUBLIC_IDENTIFIERS.add("SuballocatedByteVector");
        PUBLIC_IDENTIFIERS.add("SuballocatedIntVector");
        PUBLIC_IDENTIFIERS.add("SubContextList");
        PUBLIC_IDENTIFIERS.add("SubjectAlternativeNameExtension");
        PUBLIC_IDENTIFIERS.add("SubjectDelegator");
        PUBLIC_IDENTIFIERS.add("SubjectDomainCombiner");
        PUBLIC_IDENTIFIERS.add("SubjectInfoAccessExtension");
        PUBLIC_IDENTIFIERS.add("SubjectKeyIdentifierExtension");
        PUBLIC_IDENTIFIERS.add("SubmissionPublisher");
        PUBLIC_IDENTIFIERS.add("SubNavLinks");
        PUBLIC_IDENTIFIERS.add("SubOp");
        PUBLIC_IDENTIFIERS.add("SubRegionShowable");
        PUBLIC_IDENTIFIERS.add("Subscriber");
        PUBLIC_IDENTIFIERS.add("SubscriberWrapper");
        PUBLIC_IDENTIFIERS.add("Subscription");
        PUBLIC_IDENTIFIERS.add("SubscriptionBase");
        PUBLIC_IDENTIFIERS.add("Subset");
        PUBLIC_IDENTIFIERS.add("SubstitutionGroupHandler");
        PUBLIC_IDENTIFIERS.add("SubWriterHolderWriter");
        PUBLIC_IDENTIFIERS.add("Success");
        PUBLIC_IDENTIFIERS.add("Suggestion");
        PUBLIC_IDENTIFIERS.add("SummaryAPIListBuilder");
        PUBLIC_IDENTIFIERS.add("SummaryListWriter");
        PUBLIC_IDENTIFIERS.add("SummaryTaglet");
        PUBLIC_IDENTIFIERS.add("SummaryTree");
        PUBLIC_IDENTIFIERS.add("SunAlternateMedia");
        PUBLIC_IDENTIFIERS.add("SunCertPathBuilderException");
        PUBLIC_IDENTIFIERS.add("SunClipboard");
        PUBLIC_IDENTIFIERS.add("SunCommandLineLauncher");
        PUBLIC_IDENTIFIERS.add("SunCompositeContext");
        PUBLIC_IDENTIFIERS.add("SunDisplayChanger");
        PUBLIC_IDENTIFIERS.add("SunDragSourceContextPeer");
        PUBLIC_IDENTIFIERS.add("SunDropTargetContextPeer");
        PUBLIC_IDENTIFIERS.add("SunFontManager");
        PUBLIC_IDENTIFIERS.add("SunGraphicsCallback");
        PUBLIC_IDENTIFIERS.add("SunGraphicsEnvironment");
        PUBLIC_IDENTIFIERS.add("SunHints");
        PUBLIC_IDENTIFIERS.add("SunJmxMBeanServer");
        PUBLIC_IDENTIFIERS.add("SunJSSE");
        PUBLIC_IDENTIFIERS.add("SunPrinterJobService");
        PUBLIC_IDENTIFIERS.add("SunToolkit");
        PUBLIC_IDENTIFIERS.add("SunVolatileImage");
        PUBLIC_IDENTIFIERS.add("SunWritableRaster");
        PUBLIC_IDENTIFIERS.add("SupplementaryLocaleDataProvider");
        PUBLIC_IDENTIFIERS.add("SupplementaryResourceBundleProvider");
        PUBLIC_IDENTIFIERS.add("Supplier");
        PUBLIC_IDENTIFIERS.add("SupportedAnnotationTypes");
        PUBLIC_IDENTIFIERS.add("SupportedOptions");
        PUBLIC_IDENTIFIERS.add("SupportedSourceVersion");
        PUBLIC_IDENTIFIERS.add("SupportedValuesAttribute");
        PUBLIC_IDENTIFIERS.add("SuppressWarnings");
        PUBLIC_IDENTIFIERS.add("Surface");
        PUBLIC_IDENTIFIERS.add("SurfaceData");
        PUBLIC_IDENTIFIERS.add("SurfaceDataProxy");
        PUBLIC_IDENTIFIERS.add("SurfaceManager");
        PUBLIC_IDENTIFIERS.add("SurfaceManagerFactory");
        PUBLIC_IDENTIFIERS.add("Surrogate");
        PUBLIC_IDENTIFIERS.add("SWAP");
        PUBLIC_IDENTIFIERS.add("SwingConstants");
        PUBLIC_IDENTIFIERS.add("SwingContainer");
        PUBLIC_IDENTIFIERS.add("SwingEventMonitor");
        PUBLIC_IDENTIFIERS.add("SwingInterOpUtils");
        PUBLIC_IDENTIFIERS.add("SwingUtilities");
        PUBLIC_IDENTIFIERS.add("SwingUtilities2");
        PUBLIC_IDENTIFIERS.add("SwingUtilities3");
        PUBLIC_IDENTIFIERS.add("SwingWorker");
        PUBLIC_IDENTIFIERS.add("SwitchBootstraps");
        PUBLIC_IDENTIFIERS.add("SwitchExpressionTree");
        PUBLIC_IDENTIFIERS.add("SwitchPoint");
        PUBLIC_IDENTIFIERS.add("SwitchTree");
        PUBLIC_IDENTIFIERS.add("Sym");
        PUBLIC_IDENTIFIERS.add("sym");
        PUBLIC_IDENTIFIERS.add("Symbol");
        PUBLIC_IDENTIFIERS.add("SymbolFinder");
        PUBLIC_IDENTIFIERS.add("SymbolHash");
        PUBLIC_IDENTIFIERS.add("SymbolLookup");
        PUBLIC_IDENTIFIERS.add("SymbolMetadata");
        PUBLIC_IDENTIFIERS.add("SymbolResolutionClosure");
        PUBLIC_IDENTIFIERS.add("SymbolTable");
        PUBLIC_IDENTIFIERS.add("SymbolVisitor");
        PUBLIC_IDENTIFIERS.add("SymLinkResourcePoolEntry");
        PUBLIC_IDENTIFIERS.add("Symtab");
        PUBLIC_IDENTIFIERS.add("SyncFactory");
        PUBLIC_IDENTIFIERS.add("SyncFactoryException");
        PUBLIC_IDENTIFIERS.add("SyncFailedException");
        PUBLIC_IDENTIFIERS.add("SynchronizedTree");
        PUBLIC_IDENTIFIERS.add("SynchronousQueue");
        PUBLIC_IDENTIFIERS.add("SyncMode");
        PUBLIC_IDENTIFIERS.add("SyncProvider");
        PUBLIC_IDENTIFIERS.add("SyncProviderException");
        PUBLIC_IDENTIFIERS.add("SyncResolver");
        PUBLIC_IDENTIFIERS.add("SyncResolverImpl");
        PUBLIC_IDENTIFIERS.add("SyntaxError");
        PUBLIC_IDENTIFIERS.add("SyntaxException");
        PUBLIC_IDENTIFIERS.add("SyntaxTreeNode");
        PUBLIC_IDENTIFIERS.add("SynthButtonUI");
        PUBLIC_IDENTIFIERS.add("SynthCheckBoxMenuItemUI");
        PUBLIC_IDENTIFIERS.add("SynthCheckBoxUI");
        PUBLIC_IDENTIFIERS.add("SynthColorChooserUI");
        PUBLIC_IDENTIFIERS.add("SynthComboBoxUI");
        PUBLIC_IDENTIFIERS.add("SynthConstants");
        PUBLIC_IDENTIFIERS.add("SynthContext");
        PUBLIC_IDENTIFIERS.add("SynthDesktopIconUI");
        PUBLIC_IDENTIFIERS.add("SynthDesktopPaneUI");
        PUBLIC_IDENTIFIERS.add("SynthEditorPaneUI");
        PUBLIC_IDENTIFIERS.add("Synthesizer");
        PUBLIC_IDENTIFIERS.add("Synthetic_attribute");
        PUBLIC_IDENTIFIERS.add("SyntheticRepository");
        PUBLIC_IDENTIFIERS.add("SynthFileChooserUI");
        PUBLIC_IDENTIFIERS.add("SynthFileChooserUIImpl");
        PUBLIC_IDENTIFIERS.add("SynthFormattedTextFieldUI");
        PUBLIC_IDENTIFIERS.add("SynthGraphicsUtils");
        PUBLIC_IDENTIFIERS.add("SynthIcon");
        PUBLIC_IDENTIFIERS.add("SynthInternalFrameUI");
        PUBLIC_IDENTIFIERS.add("SynthLabelUI");
        PUBLIC_IDENTIFIERS.add("SynthListUI");
        PUBLIC_IDENTIFIERS.add("SynthLookAndFeel");
        PUBLIC_IDENTIFIERS.add("SynthMenuBarUI");
        PUBLIC_IDENTIFIERS.add("SynthMenuItemUI");
        PUBLIC_IDENTIFIERS.add("SynthMenuUI");
        PUBLIC_IDENTIFIERS.add("SynthOptionPaneUI");
        PUBLIC_IDENTIFIERS.add("SynthPainter");
        PUBLIC_IDENTIFIERS.add("SynthPanelUI");
        PUBLIC_IDENTIFIERS.add("SynthPasswordFieldUI");
        PUBLIC_IDENTIFIERS.add("SynthPopupMenuUI");
        PUBLIC_IDENTIFIERS.add("SynthProgressBarUI");
        PUBLIC_IDENTIFIERS.add("SynthRadioButtonMenuItemUI");
        PUBLIC_IDENTIFIERS.add("SynthRadioButtonUI");
        PUBLIC_IDENTIFIERS.add("SynthRootPaneUI");
        PUBLIC_IDENTIFIERS.add("SynthScrollBarUI");
        PUBLIC_IDENTIFIERS.add("SynthScrollPaneUI");
        PUBLIC_IDENTIFIERS.add("SynthSeparatorUI");
        PUBLIC_IDENTIFIERS.add("SynthSliderUI");
        PUBLIC_IDENTIFIERS.add("SynthSpinnerUI");
        PUBLIC_IDENTIFIERS.add("SynthSplitPaneUI");
        PUBLIC_IDENTIFIERS.add("SynthStyle");
        PUBLIC_IDENTIFIERS.add("SynthStyleFactory");
        PUBLIC_IDENTIFIERS.add("SynthTabbedPaneUI");
        PUBLIC_IDENTIFIERS.add("SynthTableHeaderUI");
        PUBLIC_IDENTIFIERS.add("SynthTableUI");
        PUBLIC_IDENTIFIERS.add("SynthTextAreaUI");
        PUBLIC_IDENTIFIERS.add("SynthTextFieldUI");
        PUBLIC_IDENTIFIERS.add("SynthTextPaneUI");
        PUBLIC_IDENTIFIERS.add("SynthToggleButtonUI");
        PUBLIC_IDENTIFIERS.add("SynthToolBarUI");
        PUBLIC_IDENTIFIERS.add("SynthToolTipUI");
        PUBLIC_IDENTIFIERS.add("SynthTreeUI");
        PUBLIC_IDENTIFIERS.add("SynthUI");
        PUBLIC_IDENTIFIERS.add("SynthViewportUI");
        PUBLIC_IDENTIFIERS.add("SysexMessage");
        PUBLIC_IDENTIFIERS.add("SysPropsDumper");
        PUBLIC_IDENTIFIERS.add("SysPropsPanel");
        PUBLIC_IDENTIFIERS.add("System");
        PUBLIC_IDENTIFIERS.add("System.Logger");
        PUBLIC_IDENTIFIERS.add("System.Logger.Level");
        PUBLIC_IDENTIFIERS.add("System.LoggerFinder");
        PUBLIC_IDENTIFIERS.add("SystemColorAccessor");
        PUBLIC_IDENTIFIERS.add("SystemCompleter");
        PUBLIC_IDENTIFIERS.add("SystemDictionary");
        PUBLIC_IDENTIFIERS.add("SystemDictionaryHelper");
        PUBLIC_IDENTIFIERS.add("SystemEventListener");
        PUBLIC_IDENTIFIERS.add("SystemIDResolver");
        PUBLIC_IDENTIFIERS.add("SystemMenuBar");
        PUBLIC_IDENTIFIERS.add("SystemMetrics");
        PUBLIC_IDENTIFIERS.add("SystemPropertiesWriter");
        PUBLIC_IDENTIFIERS.add("SystemPropertyTaglet");
        PUBLIC_IDENTIFIERS.add("SystemPropertyTree");
        PUBLIC_IDENTIFIERS.add("SystemSleepListener");
        PUBLIC_IDENTIFIERS.add("SystemTray");
        PUBLIC_IDENTIFIERS.add("SystemTrayAccessor");
        PUBLIC_IDENTIFIERS.add("SystemTrayPeer");
        PUBLIC_IDENTIFIERS.add("SysVVaList");
    }

    private static void addPublicIdentifiersTU() {
        PUBLIC_IDENTIFIERS.add("TextAccessibleContext");
        PUBLIC_IDENTIFIERS.add("Tab");
        PUBLIC_IDENTIFIERS.add("TabableView");
        PUBLIC_IDENTIFIERS.add("TabbedPaneLayout");
        PUBLIC_IDENTIFIERS.add("TabbedPaneUI");
        PUBLIC_IDENTIFIERS.add("TabExpander");
        PUBLIC_IDENTIFIERS.add("Table");
        PUBLIC_IDENTIFIERS.add("TableAccessibleContext");
        PUBLIC_IDENTIFIERS.add("TableCell");
        PUBLIC_IDENTIFIERS.add("TableCellEditor");
        PUBLIC_IDENTIFIERS.add("TableCellRenderer");
        PUBLIC_IDENTIFIERS.add("TableColumn");
        PUBLIC_IDENTIFIERS.add("TableColumnModel");
        PUBLIC_IDENTIFIERS.add("TableColumnModelEvent");
        PUBLIC_IDENTIFIERS.add("TableColumnModelListener");
        PUBLIC_IDENTIFIERS.add("TableElementInfo");
        PUBLIC_IDENTIFIERS.add("TableHeader");
        PUBLIC_IDENTIFIERS.add("TableHeaderBorder");
        PUBLIC_IDENTIFIERS.add("TableHeaderUI");
        PUBLIC_IDENTIFIERS.add("TableModel");
        PUBLIC_IDENTIFIERS.add("TableModelComparator");
        PUBLIC_IDENTIFIERS.add("TableModelEvent");
        PUBLIC_IDENTIFIERS.add("TableModelListener");
        PUBLIC_IDENTIFIERS.add("TableRow");
        PUBLIC_IDENTIFIERS.add("TableRowSorter");
        PUBLIC_IDENTIFIERS.add("TableSorter");
        PUBLIC_IDENTIFIERS.add("TableStringConverter");
        PUBLIC_IDENTIFIERS.add("TABLESWITCH");
        PUBLIC_IDENTIFIERS.add("TableSwitchGenerator");
        PUBLIC_IDENTIFIERS.add("TableSwitchInsnNode");
        PUBLIC_IDENTIFIERS.add("TableUI");
        PUBLIC_IDENTIFIERS.add("TableView");
        PUBLIC_IDENTIFIERS.add("TabsDlg");
        PUBLIC_IDENTIFIERS.add("TabSelectionHandler");
        PUBLIC_IDENTIFIERS.add("TabSet");
        PUBLIC_IDENTIFIERS.add("TabStop");
        PUBLIC_IDENTIFIERS.add("TabularData");
        PUBLIC_IDENTIFIERS.add("TabularDataSupport");
        PUBLIC_IDENTIFIERS.add("TabularType");
        PUBLIC_IDENTIFIERS.add("Tag");
        PUBLIC_IDENTIFIERS.add("TagAction");
        PUBLIC_IDENTIFIERS.add("TagElement");
        PUBLIC_IDENTIFIERS.add("Taglet");
        PUBLIC_IDENTIFIERS.add("TagletManager");
        PUBLIC_IDENTIFIERS.add("TagletWriter");
        PUBLIC_IDENTIFIERS.add("TagletWriterImpl");
        PUBLIC_IDENTIFIERS.add("Target");
        PUBLIC_IDENTIFIERS.add("TargetDataLine");
        PUBLIC_IDENTIFIERS.add("TargetDescription");
        PUBLIC_IDENTIFIERS.add("TargetedNotification");
        PUBLIC_IDENTIFIERS.add("TargetVM");
        PUBLIC_IDENTIFIERS.add("TargetWindowInfo");
        PUBLIC_IDENTIFIERS.add("TarjanNode");
        PUBLIC_IDENTIFIERS.add("Task");
        PUBLIC_IDENTIFIERS.add("Taskbar");
        PUBLIC_IDENTIFIERS.add("TaskbarPeer");
        PUBLIC_IDENTIFIERS.add("TaskListener");
        PUBLIC_IDENTIFIERS.add("TBSCertList");
        PUBLIC_IDENTIFIERS.add("TCPChannel");
        PUBLIC_IDENTIFIERS.add("TCPConnection");
        PUBLIC_IDENTIFIERS.add("TCPDirectSocketFactory");
        PUBLIC_IDENTIFIERS.add("TCPEndpoint");
        PUBLIC_IDENTIFIERS.add("TCPTransport");
        PUBLIC_IDENTIFIERS.add("TeeXMLDocumentFilterImpl");
        PUBLIC_IDENTIFIERS.add("TelnetInputStream");
        PUBLIC_IDENTIFIERS.add("TelnetOutputStream");
        PUBLIC_IDENTIFIERS.add("TelnetProtocolException");
        PUBLIC_IDENTIFIERS.add("Temperature");
        PUBLIC_IDENTIFIERS.add("Templates");
        PUBLIC_IDENTIFIERS.add("TemplatesHandler");
        PUBLIC_IDENTIFIERS.add("TemplatesHandlerImpl");
        PUBLIC_IDENTIFIERS.add("TemplateType");
        PUBLIC_IDENTIFIERS.add("Temporal");
        PUBLIC_IDENTIFIERS.add("TemporalAccessor");
        PUBLIC_IDENTIFIERS.add("TemporalAdjuster");
        PUBLIC_IDENTIFIERS.add("TemporalAmount");
        PUBLIC_IDENTIFIERS.add("TemporalField");
        PUBLIC_IDENTIFIERS.add("TemporalQuery");
        PUBLIC_IDENTIFIERS.add("TemporalUnit");
        PUBLIC_IDENTIFIERS.add("TenuredGeneration");
        PUBLIC_IDENTIFIERS.add("TenuredSpace");
        PUBLIC_IDENTIFIERS.add("Terminal");
        PUBLIC_IDENTIFIERS.add("TerminalFactorySpi");
        PUBLIC_IDENTIFIERS.add("TerminatingThreadLocal");
        PUBLIC_IDENTIFIERS.add("Ternary");
        PUBLIC_IDENTIFIERS.add("TernaryOperation");
        PUBLIC_IDENTIFIERS.add("Test");
        PUBLIC_IDENTIFIERS.add("TestDebugInfo");
        PUBLIC_IDENTIFIERS.add("TestParser");
        PUBLIC_IDENTIFIERS.add("Text");
        PUBLIC_IDENTIFIERS.add("TextAction");
        PUBLIC_IDENTIFIERS.add("TextArea");
        PUBLIC_IDENTIFIERS.add("TextAreaPeer");
        PUBLIC_IDENTIFIERS.add("TextBuilder");
        PUBLIC_IDENTIFIERS.add("TextCallbackHandler");
        PUBLIC_IDENTIFIERS.add("TextComponent");
        PUBLIC_IDENTIFIERS.add("TextComponentPeer");
        PUBLIC_IDENTIFIERS.add("TextComponentPrintable");
        PUBLIC_IDENTIFIERS.add("TextComponentTranslator");
        PUBLIC_IDENTIFIERS.add("TextElementInfo");
        PUBLIC_IDENTIFIERS.add("TextEvent");
        PUBLIC_IDENTIFIERS.add("TextField");
        PUBLIC_IDENTIFIERS.add("TextFieldBorder");
        PUBLIC_IDENTIFIERS.add("TextFieldPeer");
        PUBLIC_IDENTIFIERS.add("Textifier");
        PUBLIC_IDENTIFIERS.add("TextifierSupport");
        PUBLIC_IDENTIFIERS.add("TextImpl");
        PUBLIC_IDENTIFIERS.add("TextInputCallback");
        PUBLIC_IDENTIFIERS.add("TextLabel");
        PUBLIC_IDENTIFIERS.add("TextLineComponent");
        PUBLIC_IDENTIFIERS.add("TextListener");
        PUBLIC_IDENTIFIERS.add("TextOutputCallback");
        PUBLIC_IDENTIFIERS.add("TextPipe");
        PUBLIC_IDENTIFIERS.add("TextRenderer");
        PUBLIC_IDENTIFIERS.add("TextSerializer");
        PUBLIC_IDENTIFIERS.add("TextSource");
        PUBLIC_IDENTIFIERS.add("TextSourceLabel");
        PUBLIC_IDENTIFIERS.add("TextSyntax");
        PUBLIC_IDENTIFIERS.add("TextTree");
        PUBLIC_IDENTIFIERS.add("TextUI");
        PUBLIC_IDENTIFIERS.add("TexturePaint");
        PUBLIC_IDENTIFIERS.add("TGSRep");
        PUBLIC_IDENTIFIERS.add("TGSReq");
        PUBLIC_IDENTIFIERS.add("Thread");
        PUBLIC_IDENTIFIERS.add("Thread.State");
        PUBLIC_IDENTIFIERS.add("Thread.UncaughtExceptionHandler");
        PUBLIC_IDENTIFIERS.add("ThreadAccess");
        PUBLIC_IDENTIFIERS.add("ThreadContainer");
        PUBLIC_IDENTIFIERS.add("ThreadContainers");
        PUBLIC_IDENTIFIERS.add("ThreadContext");
        PUBLIC_IDENTIFIERS.add("ThreadController");
        PUBLIC_IDENTIFIERS.add("ThreadControllerWrapper");
        PUBLIC_IDENTIFIERS.add("ThreadDeath");
        PUBLIC_IDENTIFIERS.add("ThreadDeathEvent");
        PUBLIC_IDENTIFIERS.add("ThreadDeathRequest");
        PUBLIC_IDENTIFIERS.add("ThreadDialog");
        PUBLIC_IDENTIFIERS.add("ThreadDumper");
        PUBLIC_IDENTIFIERS.add("ThreadFactory");
        PUBLIC_IDENTIFIERS.add("ThreadFlock");
        PUBLIC_IDENTIFIERS.add("ThreadGroup");
        PUBLIC_IDENTIFIERS.add("ThreadGroupReference");
        PUBLIC_IDENTIFIERS.add("ThreadGroupReferenceImpl");
        PUBLIC_IDENTIFIERS.add("ThreadImpl");
        PUBLIC_IDENTIFIERS.add("ThreadInfo");
        PUBLIC_IDENTIFIERS.add("ThreadInfoAction");
        PUBLIC_IDENTIFIERS.add("ThreadInfoCompositeData");
        PUBLIC_IDENTIFIERS.add("ThreadInfoPanel");
        PUBLIC_IDENTIFIERS.add("ThreadLocal");
        PUBLIC_IDENTIFIERS.add("ThreadLocalAllocBuffer");
        PUBLIC_IDENTIFIERS.add("ThreadLocalBufferAllocator");
        PUBLIC_IDENTIFIERS.add("ThreadLocalCoders");
        PUBLIC_IDENTIFIERS.add("ThreadMXBean");
        PUBLIC_IDENTIFIERS.add("ThreadPool");
        PUBLIC_IDENTIFIERS.add("ThreadPoolExecutor");
        PUBLIC_IDENTIFIERS.add("ThreadProxy");
        PUBLIC_IDENTIFIERS.add("ThreadReference");
        PUBLIC_IDENTIFIERS.add("ThreadReferenceImpl");
        PUBLIC_IDENTIFIERS.add("Threads");
        PUBLIC_IDENTIFIERS.add("ThreadStackTrace");
        PUBLIC_IDENTIFIERS.add("ThreadStartEvent");
        PUBLIC_IDENTIFIERS.add("ThreadStartRequest");
        PUBLIC_IDENTIFIERS.add("ThreadState");
        PUBLIC_IDENTIFIERS.add("ThreadTracker");
        PUBLIC_IDENTIFIERS.add("Threshold");
        PUBLIC_IDENTIFIERS.add("Throttle");
        PUBLIC_IDENTIFIERS.add("Throwable");
        PUBLIC_IDENTIFIERS.add("ThrowsTaglet");
        PUBLIC_IDENTIFIERS.add("ThrowsTree");
        PUBLIC_IDENTIFIERS.add("ThrowTree");
        PUBLIC_IDENTIFIERS.add("Ticket");
        PUBLIC_IDENTIFIERS.add("TicketFlags");
        PUBLIC_IDENTIFIERS.add("TIFFAttrInfo");
        PUBLIC_IDENTIFIERS.add("TIFFBaseJPEGCompressor");
        PUBLIC_IDENTIFIERS.add("TIFFCIELabColorConverter");
        PUBLIC_IDENTIFIERS.add("TIFFColorConverter");
        PUBLIC_IDENTIFIERS.add("TIFFCompressor");
        PUBLIC_IDENTIFIERS.add("TIFFDecompressor");
        PUBLIC_IDENTIFIERS.add("TIFFDeflateCompressor");
        PUBLIC_IDENTIFIERS.add("TIFFDeflateDecompressor");
        PUBLIC_IDENTIFIERS.add("TIFFDeflater");
        PUBLIC_IDENTIFIERS.add("TIFFDirectory");
        PUBLIC_IDENTIFIERS.add("TIFFExifJPEGCompressor");
        PUBLIC_IDENTIFIERS.add("TIFFFieldNode");
        PUBLIC_IDENTIFIERS.add("TIFFIFD");
        PUBLIC_IDENTIFIERS.add("TIFFImageMetadata");
        PUBLIC_IDENTIFIERS.add("TIFFImageMetadataFormat");
        PUBLIC_IDENTIFIERS.add("TIFFImageMetadataFormatResources");
        PUBLIC_IDENTIFIERS.add("TIFFImageReader");
        PUBLIC_IDENTIFIERS.add("TIFFImageReaderSpi");
        PUBLIC_IDENTIFIERS.add("TIFFImageWriteParam");
        PUBLIC_IDENTIFIERS.add("TIFFImageWriter");
        PUBLIC_IDENTIFIERS.add("TIFFImageWriterSpi");
        PUBLIC_IDENTIFIERS.add("TIFFJPEGCompressor");
        PUBLIC_IDENTIFIERS.add("TIFFJPEGDecompressor");
        PUBLIC_IDENTIFIERS.add("TIFFLSBCompressor");
        PUBLIC_IDENTIFIERS.add("TIFFLSBDecompressor");
        PUBLIC_IDENTIFIERS.add("TIFFLZWCompressor");
        PUBLIC_IDENTIFIERS.add("TIFFMetadataFormat");
        PUBLIC_IDENTIFIERS.add("TIFFNullCompressor");
        PUBLIC_IDENTIFIERS.add("TIFFNullDecompressor");
        PUBLIC_IDENTIFIERS.add("TIFFOldJPEGDecompressor");
        PUBLIC_IDENTIFIERS.add("TIFFPackBitsCompressor");
        PUBLIC_IDENTIFIERS.add("TIFFPackBitsDecompressor");
        PUBLIC_IDENTIFIERS.add("TIFFPackBitsUtil");
        PUBLIC_IDENTIFIERS.add("TIFFRenderedImage");
        PUBLIC_IDENTIFIERS.add("TIFFRLECompressor");
        PUBLIC_IDENTIFIERS.add("TIFFStreamMetadata");
        PUBLIC_IDENTIFIERS.add("TIFFStreamMetadataFormat");
        PUBLIC_IDENTIFIERS.add("TIFFStreamMetadataFormatResources");
        PUBLIC_IDENTIFIERS.add("TIFFT4Compressor");
        PUBLIC_IDENTIFIERS.add("TIFFT6Compressor");
        PUBLIC_IDENTIFIERS.add("TIFFTag");
        PUBLIC_IDENTIFIERS.add("TIFFTagSet");
        PUBLIC_IDENTIFIERS.add("TIFFYCbCrColorConverter");
        PUBLIC_IDENTIFIERS.add("TIFFYCbCrDecompressor");
        PUBLIC_IDENTIFIERS.add("TIFFZLibCompressor");
        PUBLIC_IDENTIFIERS.add("TileObserver");
        PUBLIC_IDENTIFIERS.add("Time");
        PUBLIC_IDENTIFIERS.add("TimeComboBox");
        PUBLIC_IDENTIFIERS.add("TimedEvent");
        PUBLIC_IDENTIFIERS.add("TimeDV");
        PUBLIC_IDENTIFIERS.add("TimedWindowEvent");
        PUBLIC_IDENTIFIERS.add("TimeLimitExceededException");
        PUBLIC_IDENTIFIERS.add("TimeoutException");
        PUBLIC_IDENTIFIERS.add("Timer");
        PUBLIC_IDENTIFIERS.add("TimerMBean");
        PUBLIC_IDENTIFIERS.add("TimerNotification");
        PUBLIC_IDENTIFIERS.add("TimerTask");
        PUBLIC_IDENTIFIERS.add("Timespan");
        PUBLIC_IDENTIFIERS.add("Timestamp");
        PUBLIC_IDENTIFIERS.add("Timestamper");
        PUBLIC_IDENTIFIERS.add("TimestampToken");
        PUBLIC_IDENTIFIERS.add("TimeZone");
        PUBLIC_IDENTIFIERS.add("TimeZoneNameProvider");
        PUBLIC_IDENTIFIERS.add("TimeZoneNameProviderImpl");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_af");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_am");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ar");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_as");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ast");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_az");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_be");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_bg");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_bgc");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_bho");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_bn");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_bo");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_br");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_brx");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_bs");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_bs_Cyrl");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ca");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ccp");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ce");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ceb");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_chr");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_cs");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_cv");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_cy");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_da");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_de");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_de_CH");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_doi");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_dsb");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_dz");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ee");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_el");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_001");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_150");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_AE");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_AU");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_BW");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_CA");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_CM");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_ER");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_GB");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_GH");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_GM");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_GU");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_GY");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_HK");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_IE");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_IN");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_KE");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_LR");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_LS");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_MG");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_MH");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_MO");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_MP");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_MU");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_MW");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_MY");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_NA");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_NG");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_NZ");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_RW");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_SD");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_SG");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_SL");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_SS");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_SZ");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_TZ");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_UG");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_ZA");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_ZM");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_en_ZW");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_es");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_es_419");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_es_AR");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_es_BO");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_es_CL");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_es_CO");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_es_EC");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_es_MX");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_es_PE");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_es_US");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_es_UY");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_es_VE");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_et");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_eu");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_fa");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ff_Adlm");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_fi");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_fil");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_fo");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_fr");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_fr_CA");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_fr_GF");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_fur");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_fy");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ga");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_gd");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_gl");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_gsw");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_gu");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ha");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_haw");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_he");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_hi");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_hi_Latn");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_hr");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_hsb");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_hu");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_hy");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ia");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_id");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ig");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ii");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_is");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_it");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ja");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_jv");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ka");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_kea");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_kgp");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_kk");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_km");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_kn");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ko");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ko_KP");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_kok");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ks");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ks_Deva");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ksh");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ky");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_lb");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_lo");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_lrc");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_lt");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_lv");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_mai");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_mi");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_mk");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ml");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_mn");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_mni");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_mr");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ms");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_mt");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_my");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ne");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ne_IN");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_nl");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_nl_SR");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_nn");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_no");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_or");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_os");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_pa");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_pcm");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_pis");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_pl");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ps");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ps_PK");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_pt");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_pt_AO");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_pt_CV");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_pt_GW");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_pt_MO");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_pt_MZ");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_pt_PT");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_pt_ST");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_pt_TL");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_qu");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_qu_BO");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_qu_EC");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_raj");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_rm");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ro");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ru");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_sa");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_sah");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_sat");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_sc");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_sd");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_sd_Deva");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_se");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_se_FI");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_si");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_sk");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_sl");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_so");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_sq");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_sr");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_sr_Cyrl_BA");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_sr_Latn");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_sr_Latn_BA");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_su");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_sv");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_sw");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_sw_KE");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ta");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ta_MY");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ta_SG");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_te");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_tg");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_th");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ti");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_tk");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_to");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_tr");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_tt");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ug");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_uk");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ur");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_ur_IN");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_uz");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_uz_Arab");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_uz_Cyrl");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_vi");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_wae");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_wo");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_xh");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_yi");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_yo");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_yo_BJ");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_yrl");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_yrl_CO");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_yrl_VE");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_yue");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_yue_Hans");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_zh");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_zh_Hans_SG");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_zh_Hant");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_zh_Hant_HK");
        PUBLIC_IDENTIFIERS.add("TimeZoneNames_zu");
        PUBLIC_IDENTIFIERS.add("TimeZoneNamesBundle");
        PUBLIC_IDENTIFIERS.add("TimeZoneNamesProvider");
        PUBLIC_IDENTIFIERS.add("TIS_620");
        PUBLIC_IDENTIFIERS.add("TitledBorder");
        PUBLIC_IDENTIFIERS.add("TitledBorderUIResource");
        PUBLIC_IDENTIFIERS.add("TitlePaneLayout");
        PUBLIC_IDENTIFIERS.add("TlsChannelBinding");
        PUBLIC_IDENTIFIERS.add("TlsChannelBindingImpl");
        PUBLIC_IDENTIFIERS.add("TlsKeyMaterialParameterSpec");
        PUBLIC_IDENTIFIERS.add("TlsKeyMaterialSpec");
        PUBLIC_IDENTIFIERS.add("TlsMasterSecret");
        PUBLIC_IDENTIFIERS.add("TlsMasterSecretParameterSpec");
        PUBLIC_IDENTIFIERS.add("TlsPrfParameterSpec");
        PUBLIC_IDENTIFIERS.add("TlsRsaPremasterSecretParameterSpec");
        PUBLIC_IDENTIFIERS.add("Todo");
        PUBLIC_IDENTIFIERS.add("ToDoubleBiFunction");
        PUBLIC_IDENTIFIERS.add("ToDoubleFunction");
        PUBLIC_IDENTIFIERS.add("ToggleActionPropertyChangeListener");
        PUBLIC_IDENTIFIERS.add("ToggleButtonBorder");
        PUBLIC_IDENTIFIERS.add("ToggleButtonModel");
        PUBLIC_IDENTIFIERS.add("ToIntBiFunction");
        PUBLIC_IDENTIFIERS.add("ToIntFunction");
        PUBLIC_IDENTIFIERS.add("Token");
        PUBLIC_IDENTIFIERS.add("Tokenizer");
        PUBLIC_IDENTIFIERS.add("TokenMgrError");
        PUBLIC_IDENTIFIERS.add("Tokens");
        PUBLIC_IDENTIFIERS.add("TokenTracker");
        PUBLIC_IDENTIFIERS.add("ToLongBiFunction");
        PUBLIC_IDENTIFIERS.add("ToLongFunction");
        PUBLIC_IDENTIFIERS.add("Tool");
        PUBLIC_IDENTIFIERS.add("ToolBarBorder");
        PUBLIC_IDENTIFIERS.add("ToolBarContListener");
        PUBLIC_IDENTIFIERS.add("ToolBarFocusListener");
        PUBLIC_IDENTIFIERS.add("ToolBarUI");
        PUBLIC_IDENTIFIERS.add("ToolEnvironment");
        PUBLIC_IDENTIFIERS.add("Toolkit");
        PUBLIC_IDENTIFIERS.add("ToolkitAccessor");
        PUBLIC_IDENTIFIERS.add("ToolkitImage");
        PUBLIC_IDENTIFIERS.add("ToolkitThreadBlockedHandler");
        PUBLIC_IDENTIFIERS.add("ToolOptions");
        PUBLIC_IDENTIFIERS.add("ToolProvider");
        PUBLIC_IDENTIFIERS.add("Tooltip");
        PUBLIC_IDENTIFIERS.add("ToolTipUI");
        PUBLIC_IDENTIFIERS.add("TooManyListenersException");
        PUBLIC_IDENTIFIERS.add("ToplevelStateListener");
        PUBLIC_IDENTIFIERS.add("TopLevelWindowListener");
        PUBLIC_IDENTIFIERS.add("ToSAXHandler");
        PUBLIC_IDENTIFIERS.add("ToStream");
        PUBLIC_IDENTIFIERS.add("Tracer");
        PUBLIC_IDENTIFIERS.add("TraceReporter");
        PUBLIC_IDENTIFIERS.add("Trackable");
        PUBLIC_IDENTIFIERS.add("Tracker");
        PUBLIC_IDENTIFIERS.add("TrackListener");
        PUBLIC_IDENTIFIERS.add("TransactionalWriter");
        PUBLIC_IDENTIFIERS.add("TransactionId");
        PUBLIC_IDENTIFIERS.add("Transferable");
        PUBLIC_IDENTIFIERS.add("TransferableProxy");
        PUBLIC_IDENTIFIERS.add("TransferHandler");
        PUBLIC_IDENTIFIERS.add("TransferProtocolClient");
        PUBLIC_IDENTIFIERS.add("TransferQueue");
        PUBLIC_IDENTIFIERS.add("Transform");
        PUBLIC_IDENTIFIERS.add("TransformationException");
        PUBLIC_IDENTIFIERS.add("TransformBase64Decode");
        PUBLIC_IDENTIFIERS.add("TransformBlit");
        PUBLIC_IDENTIFIERS.add("TransformC14N");
        PUBLIC_IDENTIFIERS.add("TransformC14N11");
        PUBLIC_IDENTIFIERS.add("TransformC14N11_WithComments");
        PUBLIC_IDENTIFIERS.add("TransformC14NExclusive");
        PUBLIC_IDENTIFIERS.add("TransformC14NExclusiveWithComments");
        PUBLIC_IDENTIFIERS.add("TransformC14NWithComments");
        PUBLIC_IDENTIFIERS.add("TransformEnvelopedSignature");
        PUBLIC_IDENTIFIERS.add("Transformer");
        PUBLIC_IDENTIFIERS.add("TransformerConfigurationException");
        PUBLIC_IDENTIFIERS.add("TransformerException");
        PUBLIC_IDENTIFIERS.add("TransformerFactory");
        PUBLIC_IDENTIFIERS.add("TransformerFactoryConfigurationError");
        PUBLIC_IDENTIFIERS.add("TransformerFactoryImpl");
        PUBLIC_IDENTIFIERS.add("TransformerHandler");
        PUBLIC_IDENTIFIERS.add("TransformerHandlerImpl");
        PUBLIC_IDENTIFIERS.add("TransformerManager");
        PUBLIC_IDENTIFIERS.add("TransformErrorListener");
        PUBLIC_IDENTIFIERS.add("TransformException");
        PUBLIC_IDENTIFIERS.add("TransformHelper");
        PUBLIC_IDENTIFIERS.add("TransformParam");
        PUBLIC_IDENTIFIERS.add("TransformParameterSpec");
        PUBLIC_IDENTIFIERS.add("Transforms");
        PUBLIC_IDENTIFIERS.add("TransformService");
        PUBLIC_IDENTIFIERS.add("TransformSpi");
        PUBLIC_IDENTIFIERS.add("TransformStateSetter");
        PUBLIC_IDENTIFIERS.add("TransformXPath");
        PUBLIC_IDENTIFIERS.add("TransformXPath2Filter");
        PUBLIC_IDENTIFIERS.add("TransformXSLT");
        PUBLIC_IDENTIFIERS.add("Transient");
        PUBLIC_IDENTIFIERS.add("TransitedEncoding");
        PUBLIC_IDENTIFIERS.add("TransitionFrom");
        PUBLIC_IDENTIFIERS.add("TransitionTo");
        PUBLIC_IDENTIFIERS.add("Translator");
        PUBLIC_IDENTIFIERS.add("Translet");
        PUBLIC_IDENTIFIERS.add("TransletOutputHandlerFactory");
        PUBLIC_IDENTIFIERS.add("Transmitter");
        PUBLIC_IDENTIFIERS.add("Transparency");
        PUBLIC_IDENTIFIERS.add("TransPatterns");
        PUBLIC_IDENTIFIERS.add("Transport");
        PUBLIC_IDENTIFIERS.add("TransportConstants");
        PUBLIC_IDENTIFIERS.add("TransportFactory");
        PUBLIC_IDENTIFIERS.add("TransportFactoryImpl");
        PUBLIC_IDENTIFIERS.add("TransportImpl");
        PUBLIC_IDENTIFIERS.add("TransportService");
        PUBLIC_IDENTIFIERS.add("TransportTimeoutException");
        PUBLIC_IDENTIFIERS.add("TransTypes");
        PUBLIC_IDENTIFIERS.add("TraverseProc");
        PUBLIC_IDENTIFIERS.add("TrAXFilter");
        PUBLIC_IDENTIFIERS.add("TrayIcon");
        PUBLIC_IDENTIFIERS.add("TrayIconAccessor");
        PUBLIC_IDENTIFIERS.add("TrayIconPeer");
        PUBLIC_IDENTIFIERS.add("Tree");
        PUBLIC_IDENTIFIERS.add("TreeCancelEditingAction");
        PUBLIC_IDENTIFIERS.add("TreeCellEditor");
        PUBLIC_IDENTIFIERS.add("TreeCellRenderer");
        PUBLIC_IDENTIFIERS.add("TreeControlIcon");
        PUBLIC_IDENTIFIERS.add("TreeCopier");
        PUBLIC_IDENTIFIERS.add("TreeDiffer");
        PUBLIC_IDENTIFIERS.add("TreeExpansionEvent");
        PUBLIC_IDENTIFIERS.add("TreeExpansionHandler");
        PUBLIC_IDENTIFIERS.add("TreeExpansionListener");
        PUBLIC_IDENTIFIERS.add("TreeFolderIcon");
        PUBLIC_IDENTIFIERS.add("TreeHasher");
        PUBLIC_IDENTIFIERS.add("TreeHomeAction");
        PUBLIC_IDENTIFIERS.add("TreeIncrementAction");
        PUBLIC_IDENTIFIERS.add("TreeInfo");
        PUBLIC_IDENTIFIERS.add("TreeLeafIcon");
        PUBLIC_IDENTIFIERS.add("TreeMaker");
        PUBLIC_IDENTIFIERS.add("TreeMap");
        PUBLIC_IDENTIFIERS.add("TreeModel");
        PUBLIC_IDENTIFIERS.add("TreeModelEvent");
        PUBLIC_IDENTIFIERS.add("TreeModelHandler");
        PUBLIC_IDENTIFIERS.add("TreeModelListener");
        PUBLIC_IDENTIFIERS.add("TreeNode");
        PUBLIC_IDENTIFIERS.add("TreePageAction");
        PUBLIC_IDENTIFIERS.add("TreePath");
        PUBLIC_IDENTIFIERS.add("TreePathScanner");
        PUBLIC_IDENTIFIERS.add("Trees");
        PUBLIC_IDENTIFIERS.add("TreeScanner");
        PUBLIC_IDENTIFIERS.add("TreeSelectionEvent");
        PUBLIC_IDENTIFIERS.add("TreeSelectionHandler");
        PUBLIC_IDENTIFIERS.add("TreeSelectionListener");
        PUBLIC_IDENTIFIERS.add("TreeSelectionModel");
        PUBLIC_IDENTIFIERS.add("TreeSelectionRedirector");
        PUBLIC_IDENTIFIERS.add("TreeSet");
        PUBLIC_IDENTIFIERS.add("TreeTableCellEditor");
        PUBLIC_IDENTIFIERS.add("TreeTableCellRenderer");
        PUBLIC_IDENTIFIERS.add("TreeTableModel");
        PUBLIC_IDENTIFIERS.add("TreeTableModelAdapter");
        PUBLIC_IDENTIFIERS.add("TreeToggleAction");
        PUBLIC_IDENTIFIERS.add("TreeTranslator");
        PUBLIC_IDENTIFIERS.add("TreeTraverseAction");
        PUBLIC_IDENTIFIERS.add("TreeUI");
        PUBLIC_IDENTIFIERS.add("TreeVisitor");
        PUBLIC_IDENTIFIERS.add("TreeWalker");
        PUBLIC_IDENTIFIERS.add("TreeWalkerImpl");
        PUBLIC_IDENTIFIERS.add("TreeWillExpandListener");
        PUBLIC_IDENTIFIERS.add("TreeWriter");
        PUBLIC_IDENTIFIERS.add("Trie");
        PUBLIC_IDENTIFIERS.add("TrueTypeFont");
        PUBLIC_IDENTIFIERS.add("TrueTypeGlyphMapper");
        PUBLIC_IDENTIFIERS.add("TrustAnchor");
        PUBLIC_IDENTIFIERS.add("Trusted");
        PUBLIC_IDENTIFIERS.add("TrustedSubscriber");
        PUBLIC_IDENTIFIERS.add("TrustManager");
        PUBLIC_IDENTIFIERS.add("TrustManagerFactory");
        PUBLIC_IDENTIFIERS.add("TrustManagerFactorySpi");
        PUBLIC_IDENTIFIERS.add("TryBlockWriter");
        PUBLIC_IDENTIFIERS.add("TryCatchBlockNode");
        PUBLIC_IDENTIFIERS.add("TryCatchBlockSorter");
        PUBLIC_IDENTIFIERS.add("TryTree");
        PUBLIC_IDENTIFIERS.add("TSRequest");
        PUBLIC_IDENTIFIERS.add("TTY");
        PUBLIC_IDENTIFIERS.add("TTYResources");
        PUBLIC_IDENTIFIERS.add("TTYResources_de");
        PUBLIC_IDENTIFIERS.add("TTYResources_ja");
        PUBLIC_IDENTIFIERS.add("TTYResources_zh_CN");
        PUBLIC_IDENTIFIERS.add("TubePublisher");
        PUBLIC_IDENTIFIERS.add("TubeSubscriber");
        PUBLIC_IDENTIFIERS.add("Type");
        PUBLIC_IDENTIFIERS.add("Type1Font");
        PUBLIC_IDENTIFIERS.add("TypeAnnotation");
        PUBLIC_IDENTIFIERS.add("TypeAnnotationNode");
        PUBLIC_IDENTIFIERS.add("TypeAnnotationPosition");
        PUBLIC_IDENTIFIERS.add("TypeAnnotations");
        PUBLIC_IDENTIFIERS.add("TypeAnnotationWriter");
        PUBLIC_IDENTIFIERS.add("TypeArgument");
        PUBLIC_IDENTIFIERS.add("TypeArray");
        PUBLIC_IDENTIFIERS.add("TypeArrayKlass");
        PUBLIC_IDENTIFIERS.add("TypeBasedGuardingDynamicLinker");
        PUBLIC_IDENTIFIERS.add("TypeBoundKind");
        PUBLIC_IDENTIFIERS.add("TypeCastTree");
        PUBLIC_IDENTIFIERS.add("TypeCheckError");
        PUBLIC_IDENTIFIERS.add("TypeCode");
        PUBLIC_IDENTIFIERS.add("TypeComponent");
        PUBLIC_IDENTIFIERS.add("TypeComponentImpl");
        PUBLIC_IDENTIFIERS.add("TypeCompound");
        PUBLIC_IDENTIFIERS.add("TypeDataBase");
        PUBLIC_IDENTIFIERS.add("TypeDeclSnippet");
        PUBLIC_IDENTIFIERS.add("TypeDescriptor");
        PUBLIC_IDENTIFIERS.add("TypedInstruction");
        PUBLIC_IDENTIFIERS.add("TypedNamespaceIterator");
        PUBLIC_IDENTIFIERS.add("TypedRootIterator");
        PUBLIC_IDENTIFIERS.add("TypeElement");
        PUBLIC_IDENTIFIERS.add("TypeElementCatalog");
        PUBLIC_IDENTIFIERS.add("TypeEnter");
        PUBLIC_IDENTIFIERS.add("TypeEntries");
        PUBLIC_IDENTIFIERS.add("TypeEntriesAtCall");
        PUBLIC_IDENTIFIERS.add("TypeImpl");
        PUBLIC_IDENTIFIERS.add("TypeIndicators");
        PUBLIC_IDENTIFIERS.add("TypeInfo");
        PUBLIC_IDENTIFIERS.add("TypeInfoProvider");
        PUBLIC_IDENTIFIERS.add("TypeInsnNode");
        PUBLIC_IDENTIFIERS.add("TypeKindVisitor14");
        PUBLIC_IDENTIFIERS.add("TypeKindVisitor6");
        PUBLIC_IDENTIFIERS.add("TypeKindVisitor7");
        PUBLIC_IDENTIFIERS.add("TypeKindVisitor8");
        PUBLIC_IDENTIFIERS.add("TypeKindVisitor9");
        PUBLIC_IDENTIFIERS.add("TypeMapping");
        PUBLIC_IDENTIFIERS.add("TypeMetadata");
        PUBLIC_IDENTIFIERS.add("TypeMirror");
        PUBLIC_IDENTIFIERS.add("TypeNode");
        PUBLIC_IDENTIFIERS.add("TypeNotPresentException");
        PUBLIC_IDENTIFIERS.add("TypeNotPresentExceptionProxy");
        PUBLIC_IDENTIFIERS.add("TypeParameterElement");
        PUBLIC_IDENTIFIERS.add("TypeParameterTree");
        PUBLIC_IDENTIFIERS.add("TypeParamType");
        PUBLIC_IDENTIFIERS.add("TypePathEntry");
        PUBLIC_IDENTIFIERS.add("TypeReference");
        PUBLIC_IDENTIFIERS.add("TypeRelation");
        PUBLIC_IDENTIFIERS.add("Types");
        PUBLIC_IDENTIFIERS.add("TypeSignature");
        PUBLIC_IDENTIFIERS.add("TypeStackSlotEntries");
        PUBLIC_IDENTIFIERS.add("TypeSymbol");
        PUBLIC_IDENTIFIERS.add("TypeTree");
        PUBLIC_IDENTIFIERS.add("TypeTreeVisitor");
        PUBLIC_IDENTIFIERS.add("TypeValidator");
        PUBLIC_IDENTIFIERS.add("TypeVar");
        PUBLIC_IDENTIFIERS.add("TypeVariable");
        PUBLIC_IDENTIFIERS.add("TypeVariableImpl");
        PUBLIC_IDENTIFIERS.add("TypeVariableSignature");
        PUBLIC_IDENTIFIERS.add("TypeVariableSymbol");
        PUBLIC_IDENTIFIERS.add("TypeVisitor");
        PUBLIC_IDENTIFIERS.add("U1Array");
        PUBLIC_IDENTIFIERS.add("U2Array");
        PUBLIC_IDENTIFIERS.add("UCharacterIterator");
        PUBLIC_IDENTIFIERS.add("UCSReader");
        PUBLIC_IDENTIFIERS.add("UIAction");
        PUBLIC_IDENTIFIERS.add("UIClientPropertyKey");
        PUBLIC_IDENTIFIERS.add("UIDefaults");
        PUBLIC_IDENTIFIERS.add("UIDefaultsAccessor");
        PUBLIC_IDENTIFIERS.add("UIEvent");
        PUBLIC_IDENTIFIERS.add("UIManager");
        PUBLIC_IDENTIFIERS.add("UIResource");
        PUBLIC_IDENTIFIERS.add("UnalignedAddressException");
        PUBLIC_IDENTIFIERS.add("Unary");
        PUBLIC_IDENTIFIERS.add("UnaryOperation");
        PUBLIC_IDENTIFIERS.add("UnaryOperator");
        PUBLIC_IDENTIFIERS.add("UnaryTree");
        PUBLIC_IDENTIFIERS.add("UnaryVisitor");
        PUBLIC_IDENTIFIERS.add("UncaughtExceptionHandler");
        PUBLIC_IDENTIFIERS.add("UncheckedDocletException");
        PUBLIC_IDENTIFIERS.add("UncheckedIOException");
        PUBLIC_IDENTIFIERS.add("UncommonTrapBlob");
        PUBLIC_IDENTIFIERS.add("UnconditionalBranch");
        PUBLIC_IDENTIFIERS.add("UndeclaredThrowableException");
        PUBLIC_IDENTIFIERS.add("UnderlineAction");
        PUBLIC_IDENTIFIERS.add("UndetVar");
        PUBLIC_IDENTIFIERS.add("UndetVarListener");
        PUBLIC_IDENTIFIERS.add("UndoableEdit");
        PUBLIC_IDENTIFIERS.add("UndoableEditEvent");
        PUBLIC_IDENTIFIERS.add("UndoableEditListener");
        PUBLIC_IDENTIFIERS.add("UndoableEditLockSupport");
        PUBLIC_IDENTIFIERS.add("UndoableEditSupport");
        PUBLIC_IDENTIFIERS.add("UndoManager");
        PUBLIC_IDENTIFIERS.add("UndoTree");
        PUBLIC_IDENTIFIERS.add("UnexpectedEntry");
        PUBLIC_IDENTIFIERS.add("UnexpectedException");
        PUBLIC_IDENTIFIERS.add("UngrabEvent");
        PUBLIC_IDENTIFIERS.add("UnicastRef");
        PUBLIC_IDENTIFIERS.add("UnicastRef2");
        PUBLIC_IDENTIFIERS.add("UnicastRemoteObject");
        PUBLIC_IDENTIFIERS.add("UnicastServerRef");
        PUBLIC_IDENTIFIERS.add("UnicastServerRef2");
        PUBLIC_IDENTIFIERS.add("UnicodeEncoder");
        PUBLIC_IDENTIFIERS.add("UnicodeLocaleExtension");
        PUBLIC_IDENTIFIERS.add("UnicodeReader");
        PUBLIC_IDENTIFIERS.add("UnicodeSet");
        PUBLIC_IDENTIFIERS.add("UnicodeSetStringSpan");
        PUBLIC_IDENTIFIERS.add("UnImplNode");
        PUBLIC_IDENTIFIERS.add("Uninitialized_variable_info");
        PUBLIC_IDENTIFIERS.add("UnionChildIterator");
        PUBLIC_IDENTIFIERS.add("UnionClassType");
        PUBLIC_IDENTIFIERS.add("UnionDV");
        PUBLIC_IDENTIFIERS.add("UnionLayout");
        PUBLIC_IDENTIFIERS.add("UnionPathIterator");
        PUBLIC_IDENTIFIERS.add("UnionPattern");
        PUBLIC_IDENTIFIERS.add("UnionType");
        PUBLIC_IDENTIFIERS.add("UnionTypeTree");
        PUBLIC_IDENTIFIERS.add("UniqueIdentity");
        PUBLIC_IDENTIFIERS.add("UniqueOrKey");
        PUBLIC_IDENTIFIERS.add("UniqueType");
        PUBLIC_IDENTIFIERS.add("UniqueValueStore");
        PUBLIC_IDENTIFIERS.add("Units");
        PUBLIC_IDENTIFIERS.add("Universe");
        PUBLIC_IDENTIFIERS.add("UniverseExt");
        PUBLIC_IDENTIFIERS.add("UnixFileSystemProvider");
        PUBLIC_IDENTIFIERS.add("UnixLoginModule");
        PUBLIC_IDENTIFIERS.add("UnixNumericGroupPrincipal");
        PUBLIC_IDENTIFIERS.add("UnixNumericUserPrincipal");
        PUBLIC_IDENTIFIERS.add("UnixOperatingSystemMXBean");
        PUBLIC_IDENTIFIERS.add("UnixPrincipal");
        PUBLIC_IDENTIFIERS.add("UnixPrintJob");
        PUBLIC_IDENTIFIERS.add("UnixPrintService");
        PUBLIC_IDENTIFIERS.add("UnixSurfaceManagerFactory");
        PUBLIC_IDENTIFIERS.add("UnixSystem");
        PUBLIC_IDENTIFIERS.add("UNIXToolkit");
        PUBLIC_IDENTIFIERS.add("UnixUserPrincipals");
        PUBLIC_IDENTIFIERS.add("UnknownAnnotationValueException");
        PUBLIC_IDENTIFIERS.add("UnknownAttributeReader");
        PUBLIC_IDENTIFIERS.add("UnknownBlockTagTree");
        PUBLIC_IDENTIFIERS.add("UnknownDirectiveException");
        PUBLIC_IDENTIFIERS.add("UnknownElementException");
        PUBLIC_IDENTIFIERS.add("UnknownEntityException");
        PUBLIC_IDENTIFIERS.add("UnknownError");
        PUBLIC_IDENTIFIERS.add("UnknownFormatConversionException");
        PUBLIC_IDENTIFIERS.add("UnknownFormatFlagsException");
        PUBLIC_IDENTIFIERS.add("UnknownHostException");
        PUBLIC_IDENTIFIERS.add("UnknownInlineTagTree");
        PUBLIC_IDENTIFIERS.add("UnknownOopException");
        PUBLIC_IDENTIFIERS.add("UnknownServiceException");
        PUBLIC_IDENTIFIERS.add("UnknownTag");
        PUBLIC_IDENTIFIERS.add("UnknownType");
        PUBLIC_IDENTIFIERS.add("UnknownTypeException");
        PUBLIC_IDENTIFIERS.add("UnmappableCharacterException");
        PUBLIC_IDENTIFIERS.add("UnmappedAddressException");
        PUBLIC_IDENTIFIERS.add("UnmapperProxy");
        PUBLIC_IDENTIFIERS.add("UnmarshalException");
        PUBLIC_IDENTIFIERS.add("UnmatchedQuote");
        PUBLIC_IDENTIFIERS.add("UnmodifiableClassException");
        PUBLIC_IDENTIFIERS.add("UnmodifiableHeaders");
        PUBLIC_IDENTIFIERS.add("UnmodifiableModuleException");
        PUBLIC_IDENTIFIERS.add("UnmodifiableSetException");
        PUBLIC_IDENTIFIERS.add("UnparsedEntity");
        PUBLIC_IDENTIFIERS.add("UnrecoverableEntryException");
        PUBLIC_IDENTIFIERS.add("UnrecoverableKeyException");
        PUBLIC_IDENTIFIERS.add("Unreferenced");
        PUBLIC_IDENTIFIERS.add("UnresolvedAddressException");
        PUBLIC_IDENTIFIERS.add("UnresolvedClass");
        PUBLIC_IDENTIFIERS.add("UnresolvedReferenceException");
        PUBLIC_IDENTIFIERS.add("UnsatisfiedLinkError");
        PUBLIC_IDENTIFIERS.add("UnsharedNameTable");
        PUBLIC_IDENTIFIERS.add("Unsigned");
        PUBLIC_IDENTIFIERS.add("Unsigned5");
        PUBLIC_IDENTIFIERS.add("UnsolicitedNotification");
        PUBLIC_IDENTIFIERS.add("UnsolicitedNotificationEvent");
        PUBLIC_IDENTIFIERS.add("UnsolicitedNotificationListener");
        PUBLIC_IDENTIFIERS.add("UnsupportedAddressTypeException");
        PUBLIC_IDENTIFIERS.add("UnsupportedAudioFileException");
        PUBLIC_IDENTIFIERS.add("UnsupportedCallbackException");
        PUBLIC_IDENTIFIERS.add("UnsupportedCharsetException");
        PUBLIC_IDENTIFIERS.add("UnsupportedClassVersionError");
        PUBLIC_IDENTIFIERS.add("UnsupportedClassVersionException");
        PUBLIC_IDENTIFIERS.add("UnsupportedEncodingException");
        PUBLIC_IDENTIFIERS.add("UnsupportedFlavorException");
        PUBLIC_IDENTIFIERS.add("UnsupportedLookAndFeelException");
        PUBLIC_IDENTIFIERS.add("UnsupportedOperationException");
        PUBLIC_IDENTIFIERS.add("UnsupportedPlatformException");
        PUBLIC_IDENTIFIERS.add("UnsupportedTagletOperationException");
        PUBLIC_IDENTIFIERS.add("UnsupportedTemporalTypeException");
        PUBLIC_IDENTIFIERS.add("UnsyncBufferedOutputStream");
        PUBLIC_IDENTIFIERS.add("UnsyncByteArrayOutputStream");
        PUBLIC_IDENTIFIERS.add("UpcallLinker");
        PUBLIC_IDENTIFIERS.add("UpdateAction");
        PUBLIC_IDENTIFIERS.add("URI");
        PUBLIC_IDENTIFIERS.add("Uri");
        PUBLIC_IDENTIFIERS.add("URIDereferencer");
        PUBLIC_IDENTIFIERS.add("URIException");
        PUBLIC_IDENTIFIERS.add("URIName");
        PUBLIC_IDENTIFIERS.add("URIParameter");
        PUBLIC_IDENTIFIERS.add("URIReference");
        PUBLIC_IDENTIFIERS.add("URIReferenceException");
        PUBLIC_IDENTIFIERS.add("URIResolver");
        PUBLIC_IDENTIFIERS.add("URISyntax");
        PUBLIC_IDENTIFIERS.add("URISyntaxException");
        PUBLIC_IDENTIFIERS.add("URL");
        PUBLIC_IDENTIFIERS.add("URLClassLoader");
        PUBLIC_IDENTIFIERS.add("URLClassPath");
        PUBLIC_IDENTIFIERS.add("URLConnection");
        PUBLIC_IDENTIFIERS.add("URLDecoder");
        PUBLIC_IDENTIFIERS.add("URLEncoder");
        PUBLIC_IDENTIFIERS.add("URLImageSource");
        PUBLIC_IDENTIFIERS.add("URLJarFile");
        PUBLIC_IDENTIFIERS.add("URLJarFileCallBack");
        PUBLIC_IDENTIFIERS.add("URLJarFileCloseController");
        PUBLIC_IDENTIFIERS.add("URLStreamHandler");
        PUBLIC_IDENTIFIERS.add("URLStreamHandlerFactory");
        PUBLIC_IDENTIFIERS.add("URLStreamHandlerProvider");
        PUBLIC_IDENTIFIERS.add("URLUtil");
        PUBLIC_IDENTIFIERS.add("US_ASCII");
        PUBLIC_IDENTIFIERS.add("UserDataHandler");
        PUBLIC_IDENTIFIERS.add("UserDefinedFileAttributeView");
        PUBLIC_IDENTIFIERS.add("UserException");
        PUBLIC_IDENTIFIERS.add("UserInterruptException");
        PUBLIC_IDENTIFIERS.add("UserPrincipal");
        PUBLIC_IDENTIFIERS.add("UserPrincipalLookupService");
        PUBLIC_IDENTIFIERS.add("UserPrincipalNotFoundException");
        PUBLIC_IDENTIFIERS.add("UserSessionListener");
        PUBLIC_IDENTIFIERS.add("UsesDirective");
        PUBLIC_IDENTIFIERS.add("UsesTree");
        PUBLIC_IDENTIFIERS.add("Ushort4444Argb");
        PUBLIC_IDENTIFIERS.add("Ushort555Rgb");
        PUBLIC_IDENTIFIERS.add("Ushort555Rgbx");
        PUBLIC_IDENTIFIERS.add("Ushort565Rgb");
        PUBLIC_IDENTIFIERS.add("UshortGray");
        PUBLIC_IDENTIFIERS.add("UTF8Reader");
        PUBLIC_IDENTIFIERS.add("UTF_16");
        PUBLIC_IDENTIFIERS.add("UTF_16BE");
        PUBLIC_IDENTIFIERS.add("UTF_16LE");
        PUBLIC_IDENTIFIERS.add("UTF_32");
        PUBLIC_IDENTIFIERS.add("UTF_32BE");
        PUBLIC_IDENTIFIERS.add("UTF_32BE_BOM");
        PUBLIC_IDENTIFIERS.add("UTF_32LE");
        PUBLIC_IDENTIFIERS.add("UTF_32LE_BOM");
        PUBLIC_IDENTIFIERS.add("UTFDataFormatException");
        PUBLIC_IDENTIFIERS.add("Util");
        PUBLIC_IDENTIFIERS.add("Utilities");
        PUBLIC_IDENTIFIERS.add("Utility");
        PUBLIC_IDENTIFIERS.add("Utils");
    }

    private static void addPublicIdentifiersVZ() {
        PUBLIC_IDENTIFIERS.add("V10");
        PUBLIC_IDENTIFIERS.add("V12");
        PUBLIC_IDENTIFIERS.add("ValidatedInfo");
        PUBLIC_IDENTIFIERS.add("ValidatePipe");
        PUBLIC_IDENTIFIERS.add("ValidationContext");
        PUBLIC_IDENTIFIERS.add("ValidationManager");
        PUBLIC_IDENTIFIERS.add("ValidationState");
        PUBLIC_IDENTIFIERS.add("Validator");
        PUBLIC_IDENTIFIERS.add("ValidatorException");
        PUBLIC_IDENTIFIERS.add("ValidatorHandler");
        PUBLIC_IDENTIFIERS.add("VaList");
        PUBLIC_IDENTIFIERS.add("Value");
        PUBLIC_IDENTIFIERS.add("ValueBased");
        PUBLIC_IDENTIFIERS.add("ValueConversionException");
        PUBLIC_IDENTIFIERS.add("ValueConversions");
        PUBLIC_IDENTIFIERS.add("ValueConverter");
        PUBLIC_IDENTIFIERS.add("ValueExp");
        PUBLIC_IDENTIFIERS.add("ValueFilter");
        PUBLIC_IDENTIFIERS.add("ValueKind");
        PUBLIC_IDENTIFIERS.add("ValueKindFactory");
        PUBLIC_IDENTIFIERS.add("ValueLayout");
        PUBLIC_IDENTIFIERS.add("ValueMapper");
        PUBLIC_IDENTIFIERS.add("ValueObject");
        PUBLIC_IDENTIFIERS.add("ValueStore");
        PUBLIC_IDENTIFIERS.add("ValueStoreBase");
        PUBLIC_IDENTIFIERS.add("ValueStoreCache");
        PUBLIC_IDENTIFIERS.add("ValueTaglet");
        PUBLIC_IDENTIFIERS.add("ValueTree");
        PUBLIC_IDENTIFIERS.add("VarHandle");
        PUBLIC_IDENTIFIERS.add("Variability");
        PUBLIC_IDENTIFIERS.add("Variable");
        PUBLIC_IDENTIFIERS.add("VariableElement");
        PUBLIC_IDENTIFIERS.add("VariableGridLayout");
        PUBLIC_IDENTIFIERS.add("VariableHeightLayoutCache");
        PUBLIC_IDENTIFIERS.add("VariableLengthInstruction");
        PUBLIC_IDENTIFIERS.add("VariableSafeAbsRef");
        PUBLIC_IDENTIFIERS.add("VariableStack");
        PUBLIC_IDENTIFIERS.add("VariableTree");
        PUBLIC_IDENTIFIERS.add("VarInsnNode");
        PUBLIC_IDENTIFIERS.add("VarSnippet");
        PUBLIC_IDENTIFIERS.add("VarSymbol");
        PUBLIC_IDENTIFIERS.add("VecExtractOp");
        PUBLIC_IDENTIFIERS.add("VecInsertOp");
        PUBLIC_IDENTIFIERS.add("Vector");
        PUBLIC_IDENTIFIERS.add("VectorBlendOp");
        PUBLIC_IDENTIFIERS.add("VectorBroadcastIntOp");
        PUBLIC_IDENTIFIERS.add("VectorCompareOp");
        PUBLIC_IDENTIFIERS.add("VectorConvertOp");
        PUBLIC_IDENTIFIERS.add("VectorMask");
        PUBLIC_IDENTIFIERS.add("VectorMaskOp");
        PUBLIC_IDENTIFIERS.add("VectorOperators");
        PUBLIC_IDENTIFIERS.add("VectorPayload");
        PUBLIC_IDENTIFIERS.add("VectorRearrangeOp");
        PUBLIC_IDENTIFIERS.add("VectorShuffle");
        PUBLIC_IDENTIFIERS.add("VectorSpecies");
        PUBLIC_IDENTIFIERS.add("VectorSupport");
        PUBLIC_IDENTIFIERS.add("verification_type_info");
        PUBLIC_IDENTIFIERS.add("VerifiedReference");
        PUBLIC_IDENTIFIERS.add("VerifyAccess");
        PUBLIC_IDENTIFIERS.add("VerifyChangePropertyHandler");
        PUBLIC_IDENTIFIERS.add("VerifyError");
        PUBLIC_IDENTIFIERS.add("VerifyType");
        PUBLIC_IDENTIFIERS.add("Version");
        PUBLIC_IDENTIFIERS.add("VersionHelper");
        PUBLIC_IDENTIFIERS.add("VersionTree");
        PUBLIC_IDENTIFIERS.add("VerticalDragController");
        PUBLIC_IDENTIFIERS.add("VetoableChangeListener");
        PUBLIC_IDENTIFIERS.add("VetoableChangeListenerProxy");
        PUBLIC_IDENTIFIERS.add("VetoableChangeSupport");
        PUBLIC_IDENTIFIERS.add("VFrame");
        PUBLIC_IDENTIFIERS.add("View");
        PUBLIC_IDENTIFIERS.add("ViewCSS");
        PUBLIC_IDENTIFIERS.add("ViewFactory");
        PUBLIC_IDENTIFIERS.add("ViewListener");
        PUBLIC_IDENTIFIERS.add("ViewMenu");
        PUBLIC_IDENTIFIERS.add("ViewportChangeHandler");
        PUBLIC_IDENTIFIERS.add("ViewportLayout");
        PUBLIC_IDENTIFIERS.add("ViewportUI");
        PUBLIC_IDENTIFIERS.add("virtual_parse_stack");
        PUBLIC_IDENTIFIERS.add("VirtualBaseConstructor");
        PUBLIC_IDENTIFIERS.add("VirtualCallData");
        PUBLIC_IDENTIFIERS.add("VirtualCallTypeData");
        PUBLIC_IDENTIFIERS.add("VirtualConstructor");
        PUBLIC_IDENTIFIERS.add("VirtualMachine");
        PUBLIC_IDENTIFIERS.add("VirtualMachineDescriptor");
        PUBLIC_IDENTIFIERS.add("VirtualMachineError");
        PUBLIC_IDENTIFIERS.add("VirtualMachineImpl");
        PUBLIC_IDENTIFIERS.add("VirtualMachineManager");
        PUBLIC_IDENTIFIERS.add("VirtualMachineManagerImpl");
        PUBLIC_IDENTIFIERS.add("VirtualMachineManagerService");
        PUBLIC_IDENTIFIERS.add("VirtualSpace");
        PUBLIC_IDENTIFIERS.add("VirtualThreadEndEvent");
        PUBLIC_IDENTIFIERS.add("VirtualThreadPinnedEvent");
        PUBLIC_IDENTIFIERS.add("VirtualThreadStartEvent");
        PUBLIC_IDENTIFIERS.add("VirtualThreadSubmitFailedEvent");
        PUBLIC_IDENTIFIERS.add("Visibility");
        PUBLIC_IDENTIFIERS.add("VisibleMemberCache");
        PUBLIC_IDENTIFIERS.add("VisibleMemberTable");
        PUBLIC_IDENTIFIERS.add("Visitor");
        PUBLIC_IDENTIFIERS.add("Visual");
        PUBLIC_IDENTIFIERS.add("VM");
        PUBLIC_IDENTIFIERS.add("VMCannotBeModifiedException");
        PUBLIC_IDENTIFIERS.add("VMConstant");
        PUBLIC_IDENTIFIERS.add("VMDeathEvent");
        PUBLIC_IDENTIFIERS.add("VMDeathRequest");
        PUBLIC_IDENTIFIERS.add("VMDisconnectedException");
        PUBLIC_IDENTIFIERS.add("VMDisconnectEvent");
        PUBLIC_IDENTIFIERS.add("VmEvent");
        PUBLIC_IDENTIFIERS.add("VMFlagsPanel");
        PUBLIC_IDENTIFIERS.add("VmIdentifier");
        PUBLIC_IDENTIFIERS.add("VMInternalFrame");
        PUBLIC_IDENTIFIERS.add("VmListener");
        PUBLIC_IDENTIFIERS.add("VMManagement");
        PUBLIC_IDENTIFIERS.add("VMMismatchException");
        PUBLIC_IDENTIFIERS.add("VMModifiers");
        PUBLIC_IDENTIFIERS.add("VMNotConnectedException");
        PUBLIC_IDENTIFIERS.add("VMObject");
        PUBLIC_IDENTIFIERS.add("VMObjectFactory");
        PUBLIC_IDENTIFIERS.add("VMOopHandle");
        PUBLIC_IDENTIFIERS.add("VMOption");
        PUBLIC_IDENTIFIERS.add("VMOptionCompositeData");
        PUBLIC_IDENTIFIERS.add("VMOutOfMemoryException");
        PUBLIC_IDENTIFIERS.add("VMPanel");
        PUBLIC_IDENTIFIERS.add("VMReg");
        PUBLIC_IDENTIFIERS.add("VMRegImpl");
        PUBLIC_IDENTIFIERS.add("VMStartEvent");
        PUBLIC_IDENTIFIERS.add("VMStartException");
        PUBLIC_IDENTIFIERS.add("VmStatusChangeEvent");
        PUBLIC_IDENTIFIERS.add("VMSupport");
        PUBLIC_IDENTIFIERS.add("vmSymbols");
        PUBLIC_IDENTIFIERS.add("VMVersionInfoPanel");
        PUBLIC_IDENTIFIERS.add("VMVersionMismatchException");
        PUBLIC_IDENTIFIERS.add("VoiceStatus");
        PUBLIC_IDENTIFIERS.add("Void");
        PUBLIC_IDENTIFIERS.add("VoidDescriptor");
        PUBLIC_IDENTIFIERS.add("VoidType");
        PUBLIC_IDENTIFIERS.add("VoidTypeImpl");
        PUBLIC_IDENTIFIERS.add("VoidValue");
        PUBLIC_IDENTIFIERS.add("VoidValueImpl");
        PUBLIC_IDENTIFIERS.add("VolatileCallSite");
        PUBLIC_IDENTIFIERS.add("VolatileImage");
        PUBLIC_IDENTIFIERS.add("VolatileSurfaceManager");
        PUBLIC_IDENTIFIERS.add("VSBChangeListener");
        PUBLIC_IDENTIFIERS.add("VSyncedBSManager");
        PUBLIC_IDENTIFIERS.add("VtableBlob");
        PUBLIC_IDENTIFIERS.add("VtblAccess");
        PUBLIC_IDENTIFIERS.add("WalkerFactory");
        PUBLIC_IDENTIFIERS.add("WalkingIterator");
        PUBLIC_IDENTIFIERS.add("WalkingIteratorSorted");
        PUBLIC_IDENTIFIERS.add("Warner");
        PUBLIC_IDENTIFIERS.add("Warnings");
        PUBLIC_IDENTIFIERS.add("Watchable");
        PUBLIC_IDENTIFIERS.add("WatcherThread");
        PUBLIC_IDENTIFIERS.add("WatchEvent");
        PUBLIC_IDENTIFIERS.add("WatchKey");
        PUBLIC_IDENTIFIERS.add("WatchpointEvent");
        PUBLIC_IDENTIFIERS.add("WatchpointRequest");
        PUBLIC_IDENTIFIERS.add("WatchService");
        PUBLIC_IDENTIFIERS.add("wav");
        PUBLIC_IDENTIFIERS.add("WBMPImageReader");
        PUBLIC_IDENTIFIERS.add("WBMPImageReaderSpi");
        PUBLIC_IDENTIFIERS.add("WBMPImageWriter");
        PUBLIC_IDENTIFIERS.add("WBMPImageWriterSpi");
        PUBLIC_IDENTIFIERS.add("WBMPMetadata");
        PUBLIC_IDENTIFIERS.add("WBMPMetadataFormat");
        PUBLIC_IDENTIFIERS.add("WeakClassHashMap");
        PUBLIC_IDENTIFIERS.add("WeakHashMap");
        PUBLIC_IDENTIFIERS.add("WeakLink");
        PUBLIC_IDENTIFIERS.add("WeakReference");
        PUBLIC_IDENTIFIERS.add("WebRowSet");
        PUBLIC_IDENTIFIERS.add("WebRowSetImpl");
        PUBLIC_IDENTIFIERS.add("WebRowSetXmlReader");
        PUBLIC_IDENTIFIERS.add("WebRowSetXmlWriter");
        PUBLIC_IDENTIFIERS.add("WebSocket");
        PUBLIC_IDENTIFIERS.add("WebSocketRequest");
        PUBLIC_IDENTIFIERS.add("Where");
        PUBLIC_IDENTIFIERS.add("WhileLoopTree");
        PUBLIC_IDENTIFIERS.add("WhitespaceStrippingElementMatcher");
        PUBLIC_IDENTIFIERS.add("Widget");
        PUBLIC_IDENTIFIERS.add("Wildcard");
        PUBLIC_IDENTIFIERS.add("WildcardTree");
        PUBLIC_IDENTIFIERS.add("WildcardType");
        PUBLIC_IDENTIFIERS.add("WildcardTypeImpl");
        PUBLIC_IDENTIFIERS.add("Win32AARCH64JavaThreadPDAccess");
        PUBLIC_IDENTIFIERS.add("Win32AMD64JavaThreadPDAccess");
        PUBLIC_IDENTIFIERS.add("Win32VtblAccess");
        PUBLIC_IDENTIFIERS.add("Win32X86JavaThreadPDAccess");
        PUBLIC_IDENTIFIERS.add("WindbgAARCH64ThreadFactory");
        PUBLIC_IDENTIFIERS.add("WindbgAMD64ThreadFactory");
        PUBLIC_IDENTIFIERS.add("WindbgDebugger");
        PUBLIC_IDENTIFIERS.add("WindbgDebuggerLocal");
        PUBLIC_IDENTIFIERS.add("WindbgThreadFactory");
        PUBLIC_IDENTIFIERS.add("WindbgX86ThreadFactory");
        PUBLIC_IDENTIFIERS.add("Window");
        PUBLIC_IDENTIFIERS.add("WindowAccessor");
        PUBLIC_IDENTIFIERS.add("WindowAdapter");
        PUBLIC_IDENTIFIERS.add("WindowConstants");
        PUBLIC_IDENTIFIERS.add("WindowEvent");
        PUBLIC_IDENTIFIERS.add("WindowFocusListener");
        PUBLIC_IDENTIFIERS.add("WindowIDProvider");
        PUBLIC_IDENTIFIERS.add("WindowListener");
        PUBLIC_IDENTIFIERS.add("WindowPeer");
        PUBLIC_IDENTIFIERS.add("WindowPropertyGetter");
        PUBLIC_IDENTIFIERS.add("WindowsAMD64CFrame");
        PUBLIC_IDENTIFIERS.add("WindowsKeybindings");
        PUBLIC_IDENTIFIERS.add("WindowsNTSubsystem");
        PUBLIC_IDENTIFIERS.add("WindowsPlacesBar");
        PUBLIC_IDENTIFIERS.add("WindowStateListener");
        PUBLIC_IDENTIFIERS.add("WindowsVersion");
        PUBLIC_IDENTIFIERS.add("WindowsX86CFrame");
        PUBLIC_IDENTIFIERS.add("WindowUpdateFrame");
        PUBLIC_IDENTIFIERS.add("WinListener");
        PUBLIC_IDENTIFIERS.add("WinVaList");
        PUBLIC_IDENTIFIERS.add("WizardDlg");
        PUBLIC_IDENTIFIERS.add("WorkArounds");
        PUBLIC_IDENTIFIERS.add("Worker");
        PUBLIC_IDENTIFIERS.add("WorkerThread");
        PUBLIC_IDENTIFIERS.add("WrappedDiagnosticListener");
        PUBLIC_IDENTIFIERS.add("WrappedFileObject");
        PUBLIC_IDENTIFIERS.add("WrappedJavaFileManager");
        PUBLIC_IDENTIFIERS.add("WrappedJavaFileObject");
        PUBLIC_IDENTIFIERS.add("WrappedPlainView");
        PUBLIC_IDENTIFIERS.add("WrappedRuntimeException");
        PUBLIC_IDENTIFIERS.add("WrappedSAXException");
        PUBLIC_IDENTIFIERS.add("WrappedStandardJavaFileManager");
        PUBLIC_IDENTIFIERS.add("WrappedTaskListener");
        PUBLIC_IDENTIFIERS.add("Wrapper");
        PUBLIC_IDENTIFIERS.add("WrapperInstance");
        PUBLIC_IDENTIFIERS.add("WrappingJavaFileManager");
        PUBLIC_IDENTIFIERS.add("WritableByteChannel");
        PUBLIC_IDENTIFIERS.add("WritableRaster");
        PUBLIC_IDENTIFIERS.add("WritableRasterNative");
        PUBLIC_IDENTIFIERS.add("WritableRenderedImage");
        PUBLIC_IDENTIFIERS.add("WriteableScope");
        PUBLIC_IDENTIFIERS.add("WriteAbortedException");
        PUBLIC_IDENTIFIERS.add("WriteLock");
        PUBLIC_IDENTIFIERS.add("WritePendingException");
        PUBLIC_IDENTIFIERS.add("Writer");
        PUBLIC_IDENTIFIERS.add("WriterFactory");
        PUBLIC_IDENTIFIERS.add("WriterFactoryImpl");
        PUBLIC_IDENTIFIERS.add("WriterOutputStream");
        PUBLIC_IDENTIFIERS.add("WriterUtility");
        PUBLIC_IDENTIFIERS.add("WrongMethodTypeException");
        PUBLIC_IDENTIFIERS.add("WrongNumberArgsException");
        PUBLIC_IDENTIFIERS.add("WrongParserException");
        PUBLIC_IDENTIFIERS.add("WrongTypeException");
        PUBLIC_IDENTIFIERS.add("X11ComponentPeer");
        PUBLIC_IDENTIFIERS.add("X11CustomCursor");
        PUBLIC_IDENTIFIERS.add("X11Dingbats");
        PUBLIC_IDENTIFIERS.add("X11GB18030_0");
        PUBLIC_IDENTIFIERS.add("X11GB18030_1");
        PUBLIC_IDENTIFIERS.add("X11GB2312");
        PUBLIC_IDENTIFIERS.add("X11GBK");
        PUBLIC_IDENTIFIERS.add("X11GraphicsConfig");
        PUBLIC_IDENTIFIERS.add("X11InputMethod");
        PUBLIC_IDENTIFIERS.add("X11InputMethodBase");
        PUBLIC_IDENTIFIERS.add("X11InputMethodDescriptor");
        PUBLIC_IDENTIFIERS.add("X11Johab");
        PUBLIC_IDENTIFIERS.add("X11KSC5601");
        PUBLIC_IDENTIFIERS.add("X11PMBlitBgLoops");
        PUBLIC_IDENTIFIERS.add("X11PMBlitLoops");
        PUBLIC_IDENTIFIERS.add("X11Renderer");
        PUBLIC_IDENTIFIERS.add("X11SunUnicode_0");
        PUBLIC_IDENTIFIERS.add("X11SurfaceData");
        PUBLIC_IDENTIFIERS.add("X11SurfaceDataProxy");
        PUBLIC_IDENTIFIERS.add("X11TextRenderer");
        PUBLIC_IDENTIFIERS.add("X11TracingRenderer");
        PUBLIC_IDENTIFIERS.add("X11VolatileSurfaceManager");
        PUBLIC_IDENTIFIERS.add("X400Address");
        PUBLIC_IDENTIFIERS.add("X500Name");
        PUBLIC_IDENTIFIERS.add("X509Certificate");
        PUBLIC_IDENTIFIERS.add("X509CertificatePair");
        PUBLIC_IDENTIFIERS.add("X509CertificateResolver");
        PUBLIC_IDENTIFIERS.add("X509CertImpl");
        PUBLIC_IDENTIFIERS.add("X509CertInfo");
        PUBLIC_IDENTIFIERS.add("X509CertPath");
        PUBLIC_IDENTIFIERS.add("X509CertSelector");
        PUBLIC_IDENTIFIERS.add("X509CRL");
        PUBLIC_IDENTIFIERS.add("X509CRLEntry");
        PUBLIC_IDENTIFIERS.add("X509CRLEntryImpl");
        PUBLIC_IDENTIFIERS.add("X509CRLImpl");
        PUBLIC_IDENTIFIERS.add("X509CRLSelector");
        PUBLIC_IDENTIFIERS.add("X509Data");
        PUBLIC_IDENTIFIERS.add("X509DigestResolver");
        PUBLIC_IDENTIFIERS.add("X509EncodedKeySpec");
        PUBLIC_IDENTIFIERS.add("X509ExtendedKeyManager");
        PUBLIC_IDENTIFIERS.add("X509ExtendedTrustManager");
        PUBLIC_IDENTIFIERS.add("X509Extension");
        PUBLIC_IDENTIFIERS.add("X509Factory");
        PUBLIC_IDENTIFIERS.add("X509IssuerSerial");
        PUBLIC_IDENTIFIERS.add("X509IssuerSerialResolver");
        PUBLIC_IDENTIFIERS.add("X509Key");
        PUBLIC_IDENTIFIERS.add("X509KeyManager");
        PUBLIC_IDENTIFIERS.add("X509SKIResolver");
        PUBLIC_IDENTIFIERS.add("X509SubjectNameResolver");
        PUBLIC_IDENTIFIERS.add("X509TrustManager");
        PUBLIC_IDENTIFIERS.add("X86_64Architecture");
        PUBLIC_IDENTIFIERS.add("X86CurrentFrameGuess");
        PUBLIC_IDENTIFIERS.add("X86Frame");
        PUBLIC_IDENTIFIERS.add("X86JavaCallWrapper");
        PUBLIC_IDENTIFIERS.add("X86RegisterMap");
        PUBLIC_IDENTIFIERS.add("X86ThreadContext");
        PUBLIC_IDENTIFIERS.add("x_aiff");
        PUBLIC_IDENTIFIERS.add("x_wav");
        PUBLIC_IDENTIFIERS.add("x_xbitmap");
        PUBLIC_IDENTIFIERS.add("x_xpixmap");
        PUBLIC_IDENTIFIERS.add("XAbstractMenuItem");
        PUBLIC_IDENTIFIERS.add("XAConnection");
        PUBLIC_IDENTIFIERS.add("XAConnectionBuilder");
        PUBLIC_IDENTIFIERS.add("XADataSource");
        PUBLIC_IDENTIFIERS.add("XAException");
        PUBLIC_IDENTIFIERS.add("XalanXPathFactory");
        PUBLIC_IDENTIFIERS.add("XAnyEvent");
        PUBLIC_IDENTIFIERS.add("XAResource");
        PUBLIC_IDENTIFIERS.add("XBaseErrorHandler");
        PUBLIC_IDENTIFIERS.add("XBaseMenuWindow");
        PUBLIC_IDENTIFIERS.add("XBaseWindow");
        PUBLIC_IDENTIFIERS.add("Xbgr");
        PUBLIC_IDENTIFIERS.add("XbmImageDecoder");
        PUBLIC_IDENTIFIERS.add("XBoolean");
        PUBLIC_IDENTIFIERS.add("XBooleanStatic");
        PUBLIC_IDENTIFIERS.add("XButtonEvent");
        PUBLIC_IDENTIFIERS.add("XButtonPeer");
        PUBLIC_IDENTIFIERS.add("XcbRequestCounter");
        PUBLIC_IDENTIFIERS.add("XCirculateEvent");
        PUBLIC_IDENTIFIERS.add("XCirculateRequestEvent");
        PUBLIC_IDENTIFIERS.add("XClientMessageEvent");
        PUBLIC_IDENTIFIERS.add("XColor");
        PUBLIC_IDENTIFIERS.add("XColormapEvent");
        PUBLIC_IDENTIFIERS.add("XComponentPeer");
        PUBLIC_IDENTIFIERS.add("XConfigureEvent");
        PUBLIC_IDENTIFIERS.add("XConfigureRequestEvent");
        PUBLIC_IDENTIFIERS.add("XCreateWindowEvent");
        PUBLIC_IDENTIFIERS.add("XCreateWindowParams");
        PUBLIC_IDENTIFIERS.add("XCrossingEvent");
        PUBLIC_IDENTIFIERS.add("XCustomCursor");
        PUBLIC_IDENTIFIERS.add("XDataTransferer");
        PUBLIC_IDENTIFIERS.add("XDataViewer");
        PUBLIC_IDENTIFIERS.add("XDesktopPeer");
        PUBLIC_IDENTIFIERS.add("XDestroyWindowEvent");
        PUBLIC_IDENTIFIERS.add("XDHKeyAgreement");
        PUBLIC_IDENTIFIERS.add("XDHKeyFactory");
        PUBLIC_IDENTIFIERS.add("XDHKeyPairGenerator");
        PUBLIC_IDENTIFIERS.add("XECKey");
        PUBLIC_IDENTIFIERS.add("XECOperations");
        PUBLIC_IDENTIFIERS.add("XECParameters");
        PUBLIC_IDENTIFIERS.add("XECPrivateKey");
        PUBLIC_IDENTIFIERS.add("XECPrivateKeySpec");
        PUBLIC_IDENTIFIERS.add("XECPublicKey");
        PUBLIC_IDENTIFIERS.add("XECPublicKeySpec");
        PUBLIC_IDENTIFIERS.add("XEmbedCanvasPeer");
        PUBLIC_IDENTIFIERS.add("XEmbedChildProxy");
        PUBLIC_IDENTIFIERS.add("XEmbedChildProxyPeer");
        PUBLIC_IDENTIFIERS.add("XEmbedClientHelper");
        PUBLIC_IDENTIFIERS.add("XEmbeddedFrame");
        PUBLIC_IDENTIFIERS.add("XEmbeddedFramePeer");
        PUBLIC_IDENTIFIERS.add("XEmbeddingContainer");
        PUBLIC_IDENTIFIERS.add("XEmbedHelper");
        PUBLIC_IDENTIFIERS.add("XEmbedServerTester");
        PUBLIC_IDENTIFIERS.add("XErrorEvent");
        PUBLIC_IDENTIFIERS.add("XErrorHandler");
        PUBLIC_IDENTIFIERS.add("XEvent");
        PUBLIC_IDENTIFIERS.add("XEventDispatcher");
        PUBLIC_IDENTIFIERS.add("XEventListener");
        PUBLIC_IDENTIFIERS.add("XException");
        PUBLIC_IDENTIFIERS.add("XExposeEvent");
        PUBLIC_IDENTIFIERS.add("XExtData");
        PUBLIC_IDENTIFIERS.add("XFocusChangeEvent");
        PUBLIC_IDENTIFIERS.add("XFocusProxyWindow");
        PUBLIC_IDENTIFIERS.add("XGraphicsExposeEvent");
        PUBLIC_IDENTIFIERS.add("XGravityEvent");
        PUBLIC_IDENTIFIERS.add("XHTMLSerializer");
        PUBLIC_IDENTIFIERS.add("XIconSize");
        PUBLIC_IDENTIFIERS.add("XIconWindow");
        PUBLIC_IDENTIFIERS.add("Xid");
        PUBLIC_IDENTIFIERS.add("XIDGenerator");
        PUBLIC_IDENTIFIERS.add("XInclude11TextReader");
        PUBLIC_IDENTIFIERS.add("XIncludeAwareParserConfiguration");
        PUBLIC_IDENTIFIERS.add("XIncludeHandler");
        PUBLIC_IDENTIFIERS.add("XIncludeMessageFormatter");
        PUBLIC_IDENTIFIERS.add("XIncludeNamespaceSupport");
        PUBLIC_IDENTIFIERS.add("XIncludeParserConfiguration");
        PUBLIC_IDENTIFIERS.add("XIncludeTextReader");
        PUBLIC_IDENTIFIERS.add("XInputMethod");
        PUBLIC_IDENTIFIERS.add("XkbAccessXNotifyEvent");
        PUBLIC_IDENTIFIERS.add("XkbActionMessageEvent");
        PUBLIC_IDENTIFIERS.add("XkbAnyEvent");
        PUBLIC_IDENTIFIERS.add("XkbBellNotifyEvent");
        PUBLIC_IDENTIFIERS.add("XkbCompatMapNotifyEvent");
        PUBLIC_IDENTIFIERS.add("XkbControlsNotifyEvent");
        PUBLIC_IDENTIFIERS.add("XkbEvent");
        PUBLIC_IDENTIFIERS.add("XkbExtensionDeviceNotifyEvent");
        PUBLIC_IDENTIFIERS.add("XkbIndicatorNotifyEvent");
        PUBLIC_IDENTIFIERS.add("XkbMapNotifyEvent");
        PUBLIC_IDENTIFIERS.add("XkbNamesNotifyEvent");
        PUBLIC_IDENTIFIERS.add("XkbNewKeyboardNotifyEvent");
        PUBLIC_IDENTIFIERS.add("XkbStateNotifyEvent");
        PUBLIC_IDENTIFIERS.add("XKeyboardFocusManagerPeer");
        PUBLIC_IDENTIFIERS.add("XKeyEvent");
        PUBLIC_IDENTIFIERS.add("XKeymapEvent");
        PUBLIC_IDENTIFIERS.add("XKeysym");
        PUBLIC_IDENTIFIERS.add("XKeySymConstants");
        PUBLIC_IDENTIFIERS.add("XLayerProtocol");
        PUBLIC_IDENTIFIERS.add("XlibUtil");
        PUBLIC_IDENTIFIERS.add("XLightweightFramePeer");
        PUBLIC_IDENTIFIERS.add("XMapEvent");
        PUBLIC_IDENTIFIERS.add("XMappingEvent");
        PUBLIC_IDENTIFIERS.add("XMapRequestEvent");
        PUBLIC_IDENTIFIERS.add("XMBean");
        PUBLIC_IDENTIFIERS.add("XMBeanAttributes");
        PUBLIC_IDENTIFIERS.add("XMBeanInfo");
        PUBLIC_IDENTIFIERS.add("XMBeanNotifications");
        PUBLIC_IDENTIFIERS.add("XMBeanOperations");
        PUBLIC_IDENTIFIERS.add("XMenuBarPeer");
        PUBLIC_IDENTIFIERS.add("XMenuItemPeer");
        PUBLIC_IDENTIFIERS.add("XMenuPeer");
        PUBLIC_IDENTIFIERS.add("XMenuWindow");
        PUBLIC_IDENTIFIERS.add("XML11Char");
        PUBLIC_IDENTIFIERS.add("XML11Configurable");
        PUBLIC_IDENTIFIERS.add("XML11Configuration");
        PUBLIC_IDENTIFIERS.add("XML11DocumentScannerImpl");
        PUBLIC_IDENTIFIERS.add("XML11DTDConfiguration");
        PUBLIC_IDENTIFIERS.add("XML11DTDDVFactoryImpl");
        PUBLIC_IDENTIFIERS.add("XML11DTDProcessor");
        PUBLIC_IDENTIFIERS.add("XML11DTDScannerImpl");
        PUBLIC_IDENTIFIERS.add("XML11DTDValidator");
        PUBLIC_IDENTIFIERS.add("XML11EntityScanner");
        PUBLIC_IDENTIFIERS.add("XML11IDDatatypeValidator");
        PUBLIC_IDENTIFIERS.add("XML11IDREFDatatypeValidator");
        PUBLIC_IDENTIFIERS.add("XML11NamespaceBinder");
        PUBLIC_IDENTIFIERS.add("XML11NMTOKENDatatypeValidator");
        PUBLIC_IDENTIFIERS.add("XML11NonValidatingConfiguration");
        PUBLIC_IDENTIFIERS.add("XML11NSDocumentScannerImpl");
        PUBLIC_IDENTIFIERS.add("XML11NSDTDValidator");
        PUBLIC_IDENTIFIERS.add("XML11Serializer");
        PUBLIC_IDENTIFIERS.add("XMLAttributeDecl");
        PUBLIC_IDENTIFIERS.add("XMLAttributes");
        PUBLIC_IDENTIFIERS.add("XMLAttributesImpl");
        PUBLIC_IDENTIFIERS.add("XMLAttributesIteratorImpl");
        PUBLIC_IDENTIFIERS.add("XMLBufferListener");
        PUBLIC_IDENTIFIERS.add("XMLChar");
        PUBLIC_IDENTIFIERS.add("XMLCharacterRecognizer");
        PUBLIC_IDENTIFIERS.add("XMLComponent");
        PUBLIC_IDENTIFIERS.add("XMLComponentManager");
        PUBLIC_IDENTIFIERS.add("XMLConfigurationException");
        PUBLIC_IDENTIFIERS.add("XmlConsumer");
        PUBLIC_IDENTIFIERS.add("XMLContentSpec");
        PUBLIC_IDENTIFIERS.add("XMLCryptoContext");
        PUBLIC_IDENTIFIERS.add("XMLDecoder");
        PUBLIC_IDENTIFIERS.add("XMLDocumentFilter");
        PUBLIC_IDENTIFIERS.add("XMLDocumentFilterImpl");
        PUBLIC_IDENTIFIERS.add("XMLDocumentFragmentHandler");
        PUBLIC_IDENTIFIERS.add("XMLDocumentFragmentScannerImpl");
        PUBLIC_IDENTIFIERS.add("XMLDocumentHandler");
        PUBLIC_IDENTIFIERS.add("XMLDocumentParser");
        PUBLIC_IDENTIFIERS.add("XMLDocumentScanner");
        PUBLIC_IDENTIFIERS.add("XMLDocumentScannerImpl");
        PUBLIC_IDENTIFIERS.add("XMLDocumentSource");
        PUBLIC_IDENTIFIERS.add("XMLDOMWriterImpl");
        PUBLIC_IDENTIFIERS.add("XMLDTDContentModelFilter");
        PUBLIC_IDENTIFIERS.add("XMLDTDContentModelHandler");
        PUBLIC_IDENTIFIERS.add("XMLDTDContentModelSource");
        PUBLIC_IDENTIFIERS.add("XMLDTDDescription");
        PUBLIC_IDENTIFIERS.add("XMLDTDFilter");
        PUBLIC_IDENTIFIERS.add("XMLDTDHandler");
        PUBLIC_IDENTIFIERS.add("XMLDTDLoader");
        PUBLIC_IDENTIFIERS.add("XMLDTDProcessor");
        PUBLIC_IDENTIFIERS.add("XMLDTDScanner");
        PUBLIC_IDENTIFIERS.add("XMLDTDScannerImpl");
        PUBLIC_IDENTIFIERS.add("XMLDTDSource");
        PUBLIC_IDENTIFIERS.add("XMLDTDValidator");
        PUBLIC_IDENTIFIERS.add("XMLDTDValidatorFilter");
        PUBLIC_IDENTIFIERS.add("XMLElementDecl");
        PUBLIC_IDENTIFIERS.add("XMLEncoder");
        PUBLIC_IDENTIFIERS.add("XMLEntityDecl");
        PUBLIC_IDENTIFIERS.add("XMLEntityDescription");
        PUBLIC_IDENTIFIERS.add("XMLEntityDescriptionImpl");
        PUBLIC_IDENTIFIERS.add("XMLEntityHandler");
        PUBLIC_IDENTIFIERS.add("XMLEntityManager");
        PUBLIC_IDENTIFIERS.add("XMLEntityReader");
        PUBLIC_IDENTIFIERS.add("XMLEntityResolver");
        PUBLIC_IDENTIFIERS.add("XMLEntityScanner");
        PUBLIC_IDENTIFIERS.add("XMLEntityStorage");
        PUBLIC_IDENTIFIERS.add("XMLErrorHandler");
        PUBLIC_IDENTIFIERS.add("XmlErrorHandler");
        PUBLIC_IDENTIFIERS.add("XMLErrorReporter");
        PUBLIC_IDENTIFIERS.add("XMLErrorResources");
        PUBLIC_IDENTIFIERS.add("XMLErrorResources_de");
        PUBLIC_IDENTIFIERS.add("XMLErrorResources_en");
        PUBLIC_IDENTIFIERS.add("XMLErrorResources_ja");
        PUBLIC_IDENTIFIERS.add("XMLErrorResources_zh_CN");
        PUBLIC_IDENTIFIERS.add("XMLEvent");
        PUBLIC_IDENTIFIERS.add("XMLEventAllocator");
        PUBLIC_IDENTIFIERS.add("XMLEventAllocatorImpl");
        PUBLIC_IDENTIFIERS.add("XMLEventConsumer");
        PUBLIC_IDENTIFIERS.add("XMLEventFactory");
        PUBLIC_IDENTIFIERS.add("XMLEventFactoryImpl");
        PUBLIC_IDENTIFIERS.add("XMLEventReader");
        PUBLIC_IDENTIFIERS.add("XMLEventReaderImpl");
        PUBLIC_IDENTIFIERS.add("XMLEventWriter");
        PUBLIC_IDENTIFIERS.add("XMLEventWriterImpl");
        PUBLIC_IDENTIFIERS.add("XMLFilter");
        PUBLIC_IDENTIFIERS.add("XMLFilterImpl");
        PUBLIC_IDENTIFIERS.add("XMLFormatter");
        PUBLIC_IDENTIFIERS.add("XMLGrammarCachingConfiguration");
        PUBLIC_IDENTIFIERS.add("XMLGrammarDescription");
        PUBLIC_IDENTIFIERS.add("XMLGrammarLoader");
        PUBLIC_IDENTIFIERS.add("XMLGrammarParser");
        PUBLIC_IDENTIFIERS.add("XMLGrammarPool");
        PUBLIC_IDENTIFIERS.add("XMLGrammarPoolImpl");
        PUBLIC_IDENTIFIERS.add("XMLGrammarPreparser");
        PUBLIC_IDENTIFIERS.add("XMLGregorianCalendar");
        PUBLIC_IDENTIFIERS.add("XMLGregorianCalendarImpl");
        PUBLIC_IDENTIFIERS.add("XmlInput");
        PUBLIC_IDENTIFIERS.add("XMLInputFactory");
        PUBLIC_IDENTIFIERS.add("XMLInputFactoryImpl");
        PUBLIC_IDENTIFIERS.add("XMLInputSource");
        PUBLIC_IDENTIFIERS.add("XMLLocator");
        PUBLIC_IDENTIFIERS.add("XMLMessageFormatter");
        PUBLIC_IDENTIFIERS.add("XMLMessageFormatter_de");
        PUBLIC_IDENTIFIERS.add("XMLMessageFormatter_ja");
        PUBLIC_IDENTIFIERS.add("XMLMessageFormatter_zh_CN");
        PUBLIC_IDENTIFIERS.add("XMLMessages");
        PUBLIC_IDENTIFIERS.add("XMLNamespaceBinder");
        PUBLIC_IDENTIFIERS.add("XMLNotationDecl");
        PUBLIC_IDENTIFIERS.add("XMLNSDocumentScannerImpl");
        PUBLIC_IDENTIFIERS.add("XMLNSDTDValidator");
        PUBLIC_IDENTIFIERS.add("XMLObject");
        PUBLIC_IDENTIFIERS.add("XMLOutputFactory");
        PUBLIC_IDENTIFIERS.add("XMLOutputFactoryImpl");
        PUBLIC_IDENTIFIERS.add("XMLOutputSource");
        PUBLIC_IDENTIFIERS.add("XMLParseException");
        PUBLIC_IDENTIFIERS.add("XMLParser");
        PUBLIC_IDENTIFIERS.add("XMLParserConfiguration");
        PUBLIC_IDENTIFIERS.add("XMLParserException");
        PUBLIC_IDENTIFIERS.add("XMLParserImpl");
        PUBLIC_IDENTIFIERS.add("XMLPullParserConfiguration");
        PUBLIC_IDENTIFIERS.add("XMLReader");
        PUBLIC_IDENTIFIERS.add("XmlReader");
        PUBLIC_IDENTIFIERS.add("XMLReaderAdapter");
        PUBLIC_IDENTIFIERS.add("XmlReaderContentHandler");
        PUBLIC_IDENTIFIERS.add("XMLReaderManager");
        PUBLIC_IDENTIFIERS.add("XMLReporter");
        PUBLIC_IDENTIFIERS.add("XMLResolver");
        PUBLIC_IDENTIFIERS.add("XmlResolver");
        PUBLIC_IDENTIFIERS.add("XMLResourceIdentifier");
        PUBLIC_IDENTIFIERS.add("XMLResourceIdentifierImpl");
        PUBLIC_IDENTIFIERS.add("XMLScanner");
        PUBLIC_IDENTIFIERS.add("XMLSchemaDescription");
        PUBLIC_IDENTIFIERS.add("XMLSchemaException");
        PUBLIC_IDENTIFIERS.add("XMLSchemaLoader");
        PUBLIC_IDENTIFIERS.add("XMLSchemaValidator");
        PUBLIC_IDENTIFIERS.add("XMLSecurityException");
        PUBLIC_IDENTIFIERS.add("XMLSerializer");
        PUBLIC_IDENTIFIERS.add("XMLSignature");
        PUBLIC_IDENTIFIERS.add("XMLSignatureException");
        PUBLIC_IDENTIFIERS.add("XMLSignatureFactory");
        PUBLIC_IDENTIFIERS.add("XMLSignatureInput");
        PUBLIC_IDENTIFIERS.add("XMLSignatureInputDebugger");
        PUBLIC_IDENTIFIERS.add("XMLSignContext");
        PUBLIC_IDENTIFIERS.add("XMLSimpleType");
        PUBLIC_IDENTIFIERS.add("XMLStreamConstants");
        PUBLIC_IDENTIFIERS.add("XMLStreamException");
        PUBLIC_IDENTIFIERS.add("XMLStreamFilterImpl");
        PUBLIC_IDENTIFIERS.add("XMLStreamReader");
        PUBLIC_IDENTIFIERS.add("XMLStreamReaderImpl");
        PUBLIC_IDENTIFIERS.add("XMLStreamWriter");
        PUBLIC_IDENTIFIERS.add("XMLStreamWriterBase");
        PUBLIC_IDENTIFIERS.add("XMLStreamWriterImpl");
        PUBLIC_IDENTIFIERS.add("XMLString");
        PUBLIC_IDENTIFIERS.add("XMLStringBuffer");
        PUBLIC_IDENTIFIERS.add("XMLStringDefault");
        PUBLIC_IDENTIFIERS.add("XMLStringFactory");
        PUBLIC_IDENTIFIERS.add("XMLStringFactoryDefault");
        PUBLIC_IDENTIFIERS.add("XMLStringFactoryImpl");
        PUBLIC_IDENTIFIERS.add("XMLStructure");
        PUBLIC_IDENTIFIERS.add("XMLSymbols");
        PUBLIC_IDENTIFIERS.add("XMLValidateContext");
        PUBLIC_IDENTIFIERS.add("XMLVersionDetector");
        PUBLIC_IDENTIFIERS.add("XMLWriter");
        PUBLIC_IDENTIFIERS.add("XmlWriter");
        PUBLIC_IDENTIFIERS.add("XMLX509Certificate");
        PUBLIC_IDENTIFIERS.add("XMLX509CRL");
        PUBLIC_IDENTIFIERS.add("XMLX509DataContent");
        PUBLIC_IDENTIFIERS.add("XMLX509Digest");
        PUBLIC_IDENTIFIERS.add("XMLX509IssuerSerial");
        PUBLIC_IDENTIFIERS.add("XMLX509SKI");
        PUBLIC_IDENTIFIERS.add("XMLX509SubjectName");
        PUBLIC_IDENTIFIERS.add("XModalityProtocol");
        PUBLIC_IDENTIFIERS.add("XModifierKeymap");
        PUBLIC_IDENTIFIERS.add("XMotionEvent");
        PUBLIC_IDENTIFIERS.add("XMSelection");
        PUBLIC_IDENTIFIERS.add("XMSelectionListener");
        PUBLIC_IDENTIFIERS.add("XNIException");
        PUBLIC_IDENTIFIERS.add("XNodeInfo");
        PUBLIC_IDENTIFIERS.add("XNodeSet");
        PUBLIC_IDENTIFIERS.add("XNodeSetForDOM");
        PUBLIC_IDENTIFIERS.add("XNoExposeEvent");
        PUBLIC_IDENTIFIERS.add("XNull");
        PUBLIC_IDENTIFIERS.add("XNumber");
        PUBLIC_IDENTIFIERS.add("XObject");
        PUBLIC_IDENTIFIERS.add("XObjectFactory");
        PUBLIC_IDENTIFIERS.add("XOpenTypeViewer");
        PUBLIC_IDENTIFIERS.add("XOperations");
        PUBLIC_IDENTIFIERS.add("XOption");
        PUBLIC_IDENTIFIERS.add("XorOp");
        PUBLIC_IDENTIFIERS.add("XPanelPeer");
        PUBLIC_IDENTIFIERS.add("XPath");
        PUBLIC_IDENTIFIERS.add("XPathAPI");
        PUBLIC_IDENTIFIERS.add("XPathConstants");
        PUBLIC_IDENTIFIERS.add("XPathContainer");
        PUBLIC_IDENTIFIERS.add("XPathContext");
        PUBLIC_IDENTIFIERS.add("XPathDumper");
        PUBLIC_IDENTIFIERS.add("XPATHErrorResources");
        PUBLIC_IDENTIFIERS.add("XPATHErrorResources_de");
        PUBLIC_IDENTIFIERS.add("XPATHErrorResources_en");
        PUBLIC_IDENTIFIERS.add("XPATHErrorResources_ja");
        PUBLIC_IDENTIFIERS.add("XPATHErrorResources_zh_CN");
        PUBLIC_IDENTIFIERS.add("XPathEvaluationResult");
        PUBLIC_IDENTIFIERS.add("XPathEvaluator");
        PUBLIC_IDENTIFIERS.add("XPathException");
        PUBLIC_IDENTIFIERS.add("XPathExpression");
        PUBLIC_IDENTIFIERS.add("XPathExpressionContext");
        PUBLIC_IDENTIFIERS.add("XPathExpressionException");
        PUBLIC_IDENTIFIERS.add("XPathExpressionImpl");
        PUBLIC_IDENTIFIERS.add("XPathFactory");
        PUBLIC_IDENTIFIERS.add("XPathFactoryConfigurationException");
        PUBLIC_IDENTIFIERS.add("XPathFactoryImpl");
        PUBLIC_IDENTIFIERS.add("XPathFunction");
        PUBLIC_IDENTIFIERS.add("XPathFunctionException");
        PUBLIC_IDENTIFIERS.add("XPathFunctionResolver");
        PUBLIC_IDENTIFIERS.add("XPathImpl");
        PUBLIC_IDENTIFIERS.add("XPathMatcher");
        PUBLIC_IDENTIFIERS.add("XPathMatcherStack");
        PUBLIC_IDENTIFIERS.add("XPATHMessages");
        PUBLIC_IDENTIFIERS.add("XPathNamespace");
        PUBLIC_IDENTIFIERS.add("XPathNodes");
        PUBLIC_IDENTIFIERS.add("XPathNodesImpl");
        PUBLIC_IDENTIFIERS.add("XPathNSResolver");
        PUBLIC_IDENTIFIERS.add("XPathParser");
        PUBLIC_IDENTIFIERS.add("XPathProcessorException");
        PUBLIC_IDENTIFIERS.add("XPathResult");
        PUBLIC_IDENTIFIERS.add("XPathType");
        PUBLIC_IDENTIFIERS.add("XPathVariableResolver");
        PUBLIC_IDENTIFIERS.add("XPathVisitable");
        PUBLIC_IDENTIFIERS.add("XPathVisitor");
        PUBLIC_IDENTIFIERS.add("XPixmapFormatValues");
        PUBLIC_IDENTIFIERS.add("XPlotter");
        PUBLIC_IDENTIFIERS.add("XPlottingViewer");
        PUBLIC_IDENTIFIERS.add("XPointerParserConfiguration");
        PUBLIC_IDENTIFIERS.add("XPointerPart");
        PUBLIC_IDENTIFIERS.add("XPointerProcessor");
        PUBLIC_IDENTIFIERS.add("XPopupMenuPeer");
        PUBLIC_IDENTIFIERS.add("XPropertyCache");
        PUBLIC_IDENTIFIERS.add("XPropertyEvent");
        PUBLIC_IDENTIFIERS.add("XQueryTree");
        PUBLIC_IDENTIFIERS.add("XRBackend");
        PUBLIC_IDENTIFIERS.add("XRBackendNative");
        PUBLIC_IDENTIFIERS.add("XRColor");
        PUBLIC_IDENTIFIERS.add("XRCompositeManager");
        PUBLIC_IDENTIFIERS.add("XRDrawImage");
        PUBLIC_IDENTIFIERS.add("XRDrawLine");
        PUBLIC_IDENTIFIERS.add("XRenderDirectFormat");
        PUBLIC_IDENTIFIERS.add("XRenderPictFormat");
        PUBLIC_IDENTIFIERS.add("XReparentEvent");
        PUBLIC_IDENTIFIERS.add("XResizeRequestEvent");
        PUBLIC_IDENTIFIERS.add("XResourceBundle");
        PUBLIC_IDENTIFIERS.add("XResourceBundleBase");
        PUBLIC_IDENTIFIERS.add("XResources_de");
        PUBLIC_IDENTIFIERS.add("XResources_en");
        PUBLIC_IDENTIFIERS.add("XResources_zh_CN");
        PUBLIC_IDENTIFIERS.add("Xrgb");
        PUBLIC_IDENTIFIERS.add("XRGlyphCache");
        PUBLIC_IDENTIFIERS.add("XRGlyphCacheEntry");
        PUBLIC_IDENTIFIERS.add("XRGraphicsConfig");
        PUBLIC_IDENTIFIERS.add("XRInternalSurfaceData");
        PUBLIC_IDENTIFIERS.add("XRMaskBlit");
        PUBLIC_IDENTIFIERS.add("XRMaskFill");
        PUBLIC_IDENTIFIERS.add("XRMaskImage");
        PUBLIC_IDENTIFIERS.add("XRPixmapSurfaceData");
        PUBLIC_IDENTIFIERS.add("XRRenderer");
        PUBLIC_IDENTIFIERS.add("XRSolidSrcPict");
        PUBLIC_IDENTIFIERS.add("XRSurfaceData");
        PUBLIC_IDENTIFIERS.add("XRSurfaceDataProxy");
        PUBLIC_IDENTIFIERS.add("XRTextRenderer");
        PUBLIC_IDENTIFIERS.add("XRTreeFrag");
        PUBLIC_IDENTIFIERS.add("XRTreeFragSelectWrapper");
        PUBLIC_IDENTIFIERS.add("XRUtils");
        PUBLIC_IDENTIFIERS.add("XRVolatileSurfaceManager");
        PUBLIC_IDENTIFIERS.add("XRWindowSurfaceData");
        PUBLIC_IDENTIFIERS.add("XS10TypeHelper");
        PUBLIC_IDENTIFIERS.add("XSAllCM");
        PUBLIC_IDENTIFIERS.add("XSAnnotation");
        PUBLIC_IDENTIFIERS.add("XSAnnotationImpl");
        PUBLIC_IDENTIFIERS.add("XSAttributeChecker");
        PUBLIC_IDENTIFIERS.add("XSAttributeDecl");
        PUBLIC_IDENTIFIERS.add("XSAttributeDeclaration");
        PUBLIC_IDENTIFIERS.add("XSAttributeGroupDecl");
        PUBLIC_IDENTIFIERS.add("XSAttributeGroupDefinition");
        PUBLIC_IDENTIFIERS.add("XSAttributeUse");
        PUBLIC_IDENTIFIERS.add("XSAttributeUseImpl");
        PUBLIC_IDENTIFIERS.add("XSCMBinOp");
        PUBLIC_IDENTIFIERS.add("XSCMLeaf");
        PUBLIC_IDENTIFIERS.add("XSCMUniOp");
        PUBLIC_IDENTIFIERS.add("XSCMValidator");
        PUBLIC_IDENTIFIERS.add("XSComplexTypeDecl");
        PUBLIC_IDENTIFIERS.add("XSComplexTypeDefinition");
        PUBLIC_IDENTIFIERS.add("XSConstants");
        PUBLIC_IDENTIFIERS.add("XSConstraints");
        PUBLIC_IDENTIFIERS.add("XSDateTime");
        PUBLIC_IDENTIFIERS.add("XSDDescription");
        PUBLIC_IDENTIFIERS.add("XSDecimal");
        PUBLIC_IDENTIFIERS.add("XSDFACM");
        PUBLIC_IDENTIFIERS.add("XSDHandler");
        PUBLIC_IDENTIFIERS.add("XSDouble");
        PUBLIC_IDENTIFIERS.add("XSelectionClearEvent");
        PUBLIC_IDENTIFIERS.add("XSelectionEvent");
        PUBLIC_IDENTIFIERS.add("XSelectionRequestEvent");
        PUBLIC_IDENTIFIERS.add("XSElementDecl");
        PUBLIC_IDENTIFIERS.add("XSElementDeclaration");
        PUBLIC_IDENTIFIERS.add("XSElementDeclHelper");
        PUBLIC_IDENTIFIERS.add("XSEmptyCM");
        PUBLIC_IDENTIFIERS.add("XSettings");
        PUBLIC_IDENTIFIERS.add("XSetWindowAttributes");
        PUBLIC_IDENTIFIERS.add("XSException");
        PUBLIC_IDENTIFIERS.add("XSFacet");
        PUBLIC_IDENTIFIERS.add("XSFacets");
        PUBLIC_IDENTIFIERS.add("XSFloat");
        PUBLIC_IDENTIFIERS.add("XSGrammar");
        PUBLIC_IDENTIFIERS.add("XSGrammarBucket");
        PUBLIC_IDENTIFIERS.add("XSGrammarPool");
        PUBLIC_IDENTIFIERS.add("XSGrammarPoolContainer");
        PUBLIC_IDENTIFIERS.add("XSGroupDecl");
        PUBLIC_IDENTIFIERS.add("XSheet");
        PUBLIC_IDENTIFIERS.add("XSIDCDefinition");
        PUBLIC_IDENTIFIERS.add("XSImplementation");
        PUBLIC_IDENTIFIERS.add("XSImplementationImpl");
        PUBLIC_IDENTIFIERS.add("XSizeHints");
        PUBLIC_IDENTIFIERS.add("XSLMessages");
        PUBLIC_IDENTIFIERS.add("XSLoader");
        PUBLIC_IDENTIFIERS.add("XSLTCDTMManager");
        PUBLIC_IDENTIFIERS.add("XSLTErrorResources");
        PUBLIC_IDENTIFIERS.add("XSLTErrorResources_de");
        PUBLIC_IDENTIFIERS.add("XSLTErrorResources_en");
        PUBLIC_IDENTIFIERS.add("XSLTErrorResources_ja");
        PUBLIC_IDENTIFIERS.add("XSLTErrorResources_zh_CN");
        PUBLIC_IDENTIFIERS.add("XSMessageFormatter");
        PUBLIC_IDENTIFIERS.add("XSModel");
        PUBLIC_IDENTIFIERS.add("XSModelGroup");
        PUBLIC_IDENTIFIERS.add("XSModelGroupDefinition");
        PUBLIC_IDENTIFIERS.add("XSModelGroupImpl");
        PUBLIC_IDENTIFIERS.add("XSMultiValueFacet");
        PUBLIC_IDENTIFIERS.add("XSNamedMap");
        PUBLIC_IDENTIFIERS.add("XSNamedMapImpl");
        PUBLIC_IDENTIFIERS.add("XSNamespaceItem");
        PUBLIC_IDENTIFIERS.add("XSNamespaceItemList");
        PUBLIC_IDENTIFIERS.add("XSNotationDecl");
        PUBLIC_IDENTIFIERS.add("XSNotationDeclaration");
        PUBLIC_IDENTIFIERS.add("XSObject");
        PUBLIC_IDENTIFIERS.add("XSObjectList");
        PUBLIC_IDENTIFIERS.add("XSObjectListImpl");
        PUBLIC_IDENTIFIERS.add("XSParticle");
        PUBLIC_IDENTIFIERS.add("XSParticleDecl");
        PUBLIC_IDENTIFIERS.add("XSQName");
        PUBLIC_IDENTIFIERS.add("XSSimpleType");
        PUBLIC_IDENTIFIERS.add("XSSimpleTypeDecl");
        PUBLIC_IDENTIFIERS.add("XSSimpleTypeDefinition");
        PUBLIC_IDENTIFIERS.add("XSSimpleTypeDelegate");
        PUBLIC_IDENTIFIERS.add("XStateProtocol");
        PUBLIC_IDENTIFIERS.add("XSTerm");
        PUBLIC_IDENTIFIERS.add("XString");
        PUBLIC_IDENTIFIERS.add("XStringForChars");
        PUBLIC_IDENTIFIERS.add("XStringForFSB");
        PUBLIC_IDENTIFIERS.add("XSTypeDefinition");
        PUBLIC_IDENTIFIERS.add("XSurfaceData");
        PUBLIC_IDENTIFIERS.add("XSValue");
        PUBLIC_IDENTIFIERS.add("XSWildcard");
        PUBLIC_IDENTIFIERS.add("XSWildcardDecl");
        PUBLIC_IDENTIFIERS.add("XSystemTrayPeer");
        PUBLIC_IDENTIFIERS.add("XTable");
        PUBLIC_IDENTIFIERS.add("XTextField");
        PUBLIC_IDENTIFIERS.add("XTextFieldEditor");
        PUBLIC_IDENTIFIERS.add("XTranslateCoordinates");
        PUBLIC_IDENTIFIERS.add("XTrayIconPeer");
        PUBLIC_IDENTIFIERS.add("XTree");
        PUBLIC_IDENTIFIERS.add("XTreeRenderer");
        PUBLIC_IDENTIFIERS.add("XUnmapEvent");
        PUBLIC_IDENTIFIERS.add("XVisibilityEvent");
        PUBLIC_IDENTIFIERS.add("XVisualInfo");
        PUBLIC_IDENTIFIERS.add("XWindowAttributes");
        PUBLIC_IDENTIFIERS.add("XWMHints");
        PUBLIC_IDENTIFIERS.add("YearDV");
        PUBLIC_IDENTIFIERS.add("YearMonthDV");
        PUBLIC_IDENTIFIERS.add("YieldTree");
        PUBLIC_IDENTIFIERS.add("ZAttachedArrayForForwarding");
        PUBLIC_IDENTIFIERS.add("ZCollectedHeap");
        PUBLIC_IDENTIFIERS.add("ZExternalBitMap");
        PUBLIC_IDENTIFIERS.add("ZForwarding");
        PUBLIC_IDENTIFIERS.add("ZForwardingEntry");
        PUBLIC_IDENTIFIERS.add("ZForwardingTable");
        PUBLIC_IDENTIFIERS.add("ZGlobals");
        PUBLIC_IDENTIFIERS.add("ZGranuleMapForForwarding");
        PUBLIC_IDENTIFIERS.add("ZGranuleMapForPageTable");
        PUBLIC_IDENTIFIERS.add("ZHeap");
        PUBLIC_IDENTIFIERS.add("ZipEntry");
        PUBLIC_IDENTIFIERS.add("ZipError");
        PUBLIC_IDENTIFIERS.add("ZipException");
        PUBLIC_IDENTIFIERS.add("ZipFile");
        PUBLIC_IDENTIFIERS.add("ZipFileSystemProvider");
        PUBLIC_IDENTIFIERS.add("ZipInfo");
        PUBLIC_IDENTIFIERS.add("ZipInputStream");
        PUBLIC_IDENTIFIERS.add("ZipOutputStream");
        PUBLIC_IDENTIFIERS.add("ZoneId");
        PUBLIC_IDENTIFIERS.add("ZoneInfo");
        PUBLIC_IDENTIFIERS.add("ZoneRulesException");
        PUBLIC_IDENTIFIERS.add("ZoneRulesProvider");
        PUBLIC_IDENTIFIERS.add("ZoneView");
        PUBLIC_IDENTIFIERS.add("ZPage");
        PUBLIC_IDENTIFIERS.add("ZPageAllocator");
        PUBLIC_IDENTIFIERS.add("ZPageTable");
        PUBLIC_IDENTIFIERS.add("ZRelocate");
        PUBLIC_IDENTIFIERS.add("ZVirtualMemory");
    }
}
