#!/usr/bin/env python3

"""
botshot.py 0.2 - Mass Web Screenshot Command Line Script
Copyright (c) 2017-2020 Marco Ivaldi <raptor@0xdeadbeef.info>

"The Other Way to Pen-Test" --HD Moore & Valsmith

Botshot is a Python script that captures screenshots of
websites from the command line. It is useful to automate
mapping of the web attack surface of large networks.

Based on previous work by @federicodotta and @0-duke.

Requirements:
Python 3 (https://pythonclock.org/ is ticking...)
Selenium (https://pypi.python.org/pypi/selenium)
ChromeDriver (https://chromedriver.chromium.org/)

Example usage:
$ ./botshot.py -f urls

TODO:
Implement import from Nmap's XML output files
Add the ability to save output in HTML format
Add the ability to perform nikto/dirb scans
Migrate to Electron (https://electron.atom.io/)?

Get the latest version at:
https://github.com/0xdea/tactical-exploitation/
"""

VERSION = "0.1"
BANNER = """
botshot.py {0} - Mass Web Screenshot Command Line Script
Copyright (c) 2017 Marco Ivaldi <raptor@0xdeadbeef.info>
""".format(VERSION)

import sys
import argparse
import time
import os
import re
from selenium import webdriver

def webshot(args):
    """
    Mass web screenshot function
    """

    targets = [url.rstrip() for url in args.f]
    timeout = args.t

    # chrome webdriver options
    options = webdriver.ChromeOptions()
    options.add_argument("--headless")
    options.add_argument("--ignore-certificate-errors")
    options.add_argument("--no-sandbox")
    #options.add_argument("--disable-dev-shm-usage")

    # set up headless browser
    try:
        browser = webdriver.Chrome(options=options)
        browser.set_page_load_timeout(timeout)
        browser.set_window_size(1920, 1080)
    except Exception as err:
        print("// error: {0}".format(err))
        browser.quit()
        sys.exit(1)

    # create output directory
    outdir = "webshots-" + time.strftime("%Y%m%d-%H%M%S", time.localtime())
    try:
        os.mkdir(outdir, mode=0o755)
    except Exception as err:
        print("// error: {0}".format(err))
        browser.quit()
        sys.exit(1)

    for url in targets:
        print("*** Grabbing screenshot of {0} ***\n".format(url))

        p = re.compile("[:/]+")
        outfile = outdir + "/" + p.sub("_", url) + ".png"

        try:
            browser.get("about:blank")
            browser.get(url)
            time.sleep(1) # workaround for some targets
            browser.save_screenshot(outfile)
        except (KeyboardInterrupt, SystemExit):
            browser.quit()
            sys.exit(1)
        except Exception as err:
            print("// error: {0}".format(err))

    browser.quit()
    return

def get_args():
    """
    Get command line arguments
    """

    parser = argparse.ArgumentParser()
    parser.set_defaults(func=webshot)

    parser.add_argument(
            "-f",
            metavar="FILE",
            type=argparse.FileType("r"),
            required=True,
            help="specify file containing a list of URLs")
    parser.add_argument(
            "-t",
            metavar="TIMEOUT", 
            type=int,
            default=30,
            help="specify timeout in seconds (default: 30)")

    if len(sys.argv) == 1:
        parser.print_help()
        sys.exit(0)

    return parser.parse_args()

def main():
    """
    Main function
    """

    print(BANNER)

    if sys.version_info[0] != 3:
        print("// error: this script requires python 3")
        sys.exit(1)

    args = get_args()
    args.func(args)

if __name__ == "__main__":
    main()
