# frozen_string_literal: true

require_relative '../../spec_helper'
require 'wpxf/modules'

describe Wpxf do
  describe '.build_module_list' do
    it 'builds an array of hashes containing the modules for the specified namespace' do
      result = Wpxf.build_module_list(Wpxf::Exploit, [Wpxf.modules_path])
      expect(result[0]).to include(:class, :name)

      mod = result.find { |m| m[:name] == 'exploit/shell/admin_shell_upload' }
      expect(mod).to_not be_nil
      expect(mod[:class]).to be Wpxf::Exploit::AdminShellUpload
    end
  end

  describe '.modules_path' do
    it 'should return the path to the standard modules directory' do
      path = Wpxf.modules_path
      expect(path).to eql(File.join(Wpxf.app_path, 'lib', 'wpxf', 'modules'))
    end
  end

  describe '.payloads_path' do
    it 'should return the path to the payloads directory' do
      path = Wpxf.payloads_path
      expect(path).to eql(File.join(Wpxf.app_path, 'lib', 'wpxf', 'payloads'))
    end
  end

  describe '.custom_modules_path' do
    it 'should return the path to the custom modules directory' do
      path = Wpxf.custom_modules_path
      expect(path).to eql(File.join(Wpxf.home_directory, 'modules'))
    end
  end

  describe '.load_custom_modules' do
    it 'should load all modules found in the home directory of the current user' do
      allow(Wpxf).to receive(:load)
      allow(Dir).to receive(:glob)
        .with(File.join(Wpxf.home_directory, 'modules', '**', '*.rb'))
        .and_return %w[/path1.rb /path2.rb]

      Wpxf.load_custom_modules
      expect(Wpxf).to have_received(:load)
        .with('/path1.rb')
        .exactly(1).times

      expect(Wpxf).to have_received(:load)
        .with('/path2.rb')
        .exactly(1).times
    end
  end
end

describe 'Wpxf::Auxiliary' do
  describe '.module_list' do
    it 'builds an array of hashes containing the auxiliary modules' do
      list = Wpxf::Auxiliary.module_list
      expect(list).to_not be_nil
      expect(list[0]).to include(:class, :name)
    end
  end

  it 'should contain no duplicate classes' do
    classes = []
    Dir.glob('modules/auxiliary/**/*.rb').each do |f|
      code = File.read(f)
      classes.push(code[/class\s+(.+)?\s/, 1])
    end

    duplicates = classes.detect { |e| classes.count(e) > 1 }
    expect(duplicates).to be_nil
  end
end

describe 'Wpxf::Exploit' do
  it 'should contain no duplicate classes' do
    classes = []
    Dir.glob('modules/exploit/**/*.rb').each do |f|
      code = File.read(f)
      classes.push(code[/class\s+(.+?)\s/, 1])
    end

    duplicates = classes.detect { |e| classes.count(e) > 1 }
    expect(duplicates).to be_nil
  end

  describe '.module_list' do
    it 'builds an array of hashes containing the exploit modules' do
      list = Wpxf::Exploit.module_list
      expect(list).to_not be_nil
      expect(list[0]).to include(:class, :name)
    end
  end
end

describe 'Wpxf::Payloads' do
  describe '.payload_list' do
    it 'builds an array of hashes containing the module payloads' do
      list = Wpxf::Payloads.payload_list
      expect(list).to_not be_nil
      expect(list[0]).to include(:class, :name)
    end
  end
end
