import re
from enum import Enum
from typing import ClassVar, Literal

from pydantic import Field

from core.schemas import entity

vulnerability_matcher = re.compile(
    r"(?P<pre>\W?)(?P<search>CVE-\d{4}-\d{4,7})(?P<post>\W?)"
)


class SeverityType(str, Enum):
    none = "none"
    low = "low"
    medium = "medium"
    high = "high"
    critical = "critical"


class Vulnerability(entity.Entity):
    """
    This class represents a vulnerability in the schema.

    Attributes:
        title: title of the vulnerability.
        base_score : base score of the vulnerability obtained from CVSS metric
                     ranging from 0.0 to 10.0.
        severity: represents the severity of a vulnerability. One of none, low,
                  medium, high, critical.
    """

    _type_filter: ClassVar[str] = "vulnerability"
    type: Literal["vulnerability"] = "vulnerability"

    title: str = ""
    base_score: float = Field(ge=0.0, le=10.0, default=0.0)
    severity: SeverityType = "none"
    reference: str = ""

    @classmethod
    def is_valid(cls, ent: entity.Entity) -> bool:
        if vulnerability_matcher.match(ent.name):
            return True
        return False
